package com.atlassian.cache;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.atlassian.annotations.PublicApi;

/**
 * Interface that defines the common read-through cache operations.
 * <p>
 * Note: {@code null} keys and values are NOT supported. Although using {@code null} values may work for some
 * implementations, it won't work for all implementations and the behaviour may change over time.
 *
 * @param <K> the key type
 * @param <V> the value type
 * @since 6.1
 */
@PublicApi
public interface ReadThroughCache<K, V> {
    /**
     * The name of the cache, uniquely identifies this cache.
     *
     * @return the name of the cache
     */
    @Nonnull
    String getName();

    /**
     * Returns whether an entry exists in the cache under the specified key.
     *
     * <p>Note that:</p>
     *
     * <ul>
     *     <li>If the cache was created with a {@link CacheLoader},
     *     it will not be called. Obviously, any call to {@link #get(Object)}
     *     will call the corresponding {@link CacheLoader} (if required).</li>
     *     <li>If the cache was created with {@link CacheSettings#getReplicateViaCopy()}
     *     set to <code>false</code> and {@link CacheSettings#getLocal()} set to <code>false</code>,
     *     then only the local copy of the cache is checked. A local cache on
     *     another node may contain an entry under the specified key.</li>
     * </ul>
     * @param key the key for the entry to check for containment
     * @return true iff the cache already contains an entry under the specified key
     * @since 2.2.0
     */
    boolean containsKey(@Nonnull K key);

    /**
     * Gets the keys of all objects currently stored in the cache. This will return the keys in a new collection.
     * @return a collection of {@link Object}s keys
     */
    @Nonnull
    Collection<K> getKeys();

    /**
     * Retrieve an object from this cache. Note that a copy of the cached object may be returned. Any changes that are
     * made to the returned object may not be reflected in the cache. Cached values should be considered effectively
     * immutable.
     *
     * @param key the key uniquely identifying the object to be retrieved
     * @return the object from the cache, or {@code null}  if the object is not found
     */
    @Nullable
    V get(@Nonnull K key);

    /**
     * Retrieve an object from this cache. Note that a copy of the cached object may be returned. Any changes that are
     * made to the returned object may not be reflected in the cache. Cached values should be considered effectively
     * immutable.
     * <p>
     * If no value is present in the cache, the {@code valueSupplier} will be used to populate the entry and be counted
     * as a cache miss.
     *
     * @param key the key uniquely identifying the object to be retrieved
     * @param valueSupplier the supplier to call if no value is stored in the cache. the value supplied by the supplier
     * cannot be {@code null}
     * @return the object from the cache, or the newly created value from the supplier
     * @since 2.5
     */
    @Nonnull
    V get(@Nonnull K key, @Nonnull Supplier<? extends V> valueSupplier);

    /**
     * Retrieve multiple entries from the cache. Any entries not present in the cache will be loaded using the given
     * bulk loader function.
     *
     * By default, this method is implemented rather naively based upon calls to {@link #get(Object)} and {@link Map#put(Object, Object)}.
     * This should be fine for the majority of cache implementations, and will still benefit from the bulk value loader.
     * Individual cache implementations, however, may wish to override this method and provide an optimised
     * implementation.
     *
     * @param keys the set of keys for the entries to be fetched from the cache
     * @param valuesSupplier a function that takes the set of keys that were not present in the cache, and returns a map of keys to new values
     * @return a map of keys to values, a combination of entries that were already cached, and entries that had to be loaded
     * @since 5.3
     */
    @Nonnull
    default Map<K, V> getBulk(@Nonnull final Set<K> keys, @Nonnull final Function<Set<K>, Map<K, V>> valuesSupplier) {
        final Map<K, V> result = new HashMap<>();
        final Set<K> keysToLoad = new HashSet<>();

        for (final K key : keys) {
            final V value = get(key);
            if (value != null) {
                result.put(key, value);
            } else {
                keysToLoad.add(key);
            }
        }

        if (!keysToLoad.isEmpty()) {
            final Map<K, V> loadedValues = valuesSupplier.apply(keysToLoad);

            /*
            Use the read-through get to make sure there's an entry in the cache for this key, populating from our newly-loaded
            value if necessary. We discard the result, though, so that we always return the values we just bulk-loaded,
            for consistency.
             */
            loadedValues.forEach((key, value) -> get(key, () -> value));
            loadedValues.forEach(result::put);
        }

        return result;
    }

    /**
     * Remove the object identified by the key from the cache. If no object can be found associated with this key then
     * no action is taken.
     *
     * @param key the key that uniquely identifies the object to be removed
     */
    void remove(@Nonnull K key);

    /**
     * Atomically removes the entry for a key only if currently mapped to a given value.
     * <p>
     * <b>NOTE:</b> Some cache backends (e.g. Ehcache) require this operation to be performed by CAS and will throw an
     * exception when the cache is replicated
     *
     * @param key the key with which the specified value is associated
     * @param value the value expected to be associated with the specified key
     * @return {@code true} if the value was removed, {@code false}  otherwise
     */
    boolean remove(@Nonnull K key, @Nonnull V value);

    /**
     * Remove all of the objects from this cache.
     */
    void removeAll();
}
