package com.atlassian.cache.impl.metrics;

import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.cache.Cache;
import com.atlassian.cache.CachedReference;
import com.atlassian.plugin.util.PluginKeyStack;
import com.atlassian.util.profiling.Metrics;

import static com.atlassian.util.profiling.Metrics.metric;

/**
 * Class which emits JMX metrics
 * @since 5.5.0
 */
public class MetricEmitter {
    public static final String CLASS_NAME_KEY = "className";

    private static final String CACHE_REMOVE_ALL_METRIC_NAME = "cache.removeAll";
    private static final String CACHED_REFERENCE_RESET_METRIC_NAME = "cachedReference.reset";
    private static final String PLUGIN_KEY_AT_CREATION_TAG_KEY = "pluginKeyAtCreation";

    private final String creatorKey;
    private final String invokingClassName;

    private MetricEmitter(String creatorKey, String invokingClassName) {
        this.creatorKey = creatorKey;
        this.invokingClassName = invokingClassName;
    }

    /**
     * Constructs a MetricEmitter
     * @param invokingClassName name of class which we want to track the invocations of
     * @return MetricEmitter
     */
    public static MetricEmitter create(String invokingClassName) {
        return new MetricEmitter(getCreatorKey(), invokingClassName);
    }

    /**
     * Emit a JMX metric that indicates {@link CachedReference#reset()} has been invoked
     */
    public void emitCachedReferenceReset() {
        try {
            tagWithCreatorKeyIfExists(metric(CACHED_REFERENCE_RESET_METRIC_NAME))
                    .tag(CLASS_NAME_KEY, invokingClassName)
                    .withAnalytics()
                    .incrementCounter(1L);
        } catch (NoClassDefFoundError ignored) {
            // CachedReference#reset can be called before the plugin system and the atlassian-profiling plugin
            // is initialised which will cause a NoClassDefFoundError. We do not mind ignoring this because calls before
            // this point are not valuable enough to attempt to emit a metric for
        }
    }

    /**
     * Emit a JMX metric that indicates {@link Cache#removeAll()} has been invoked
     */
    public void emitCacheRemoveAll() {
        try {
            tagWithCreatorKeyIfExists(metric(CACHE_REMOVE_ALL_METRIC_NAME))
                    .tag(CLASS_NAME_KEY, invokingClassName)
                    .withAnalytics()
                    .incrementCounter(1L);
        } catch (NoClassDefFoundError ignored) {
            // Cache#removeAll can be called before the plugin system and the atlassian-profiling plugin
            // is initialised which will cause a NoClassDefFoundError. We do not mind ignoring this because calls before
            // this point are not valuable enough to attempt to emit a metric for
        }
    }

    private static String getCreatorKey() {
        try {
            return PluginKeyStack.getFirstPluginKey();
        } catch (NoClassDefFoundError ignored) {
            return null;
        }
    }

    private Metrics.Builder tagWithCreatorKeyIfExists(Metrics.Builder builder) {
        if (this.creatorKey == null) {
            return builder;
        }
        return builder.tag(PLUGIN_KEY_AT_CREATION_TAG_KEY, this.creatorKey);
    }

    @VisibleForTesting
    String getInvokingClassName() {
        return invokingClassName;
    }
}
