package com.atlassian.cache.ehcache.replication.rmi;

import java.util.Properties;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.sf.ehcache.distribution.CacheReplicator;
import net.sf.ehcache.distribution.RMIAsynchronousCacheReplicator;
import net.sf.ehcache.event.CacheEventListener;
import net.sf.ehcache.event.CacheEventListenerFactory;
import net.sf.ehcache.util.PropertyUtil;

/**
 * A factory for RMI-based {@link CacheReplicator}s, using properties. Config lines look like:
 * <pre>&lt;cacheEventListenerFactory class="com.atlassian.cache.ehcache.RMICacheReplicatorFactory"
 * properties="
 * replicateAsynchronously=true,
 * replicatePuts=true
 * replicateUpdates=true
 * replicateUpdatesViaCopy=true
 * replicateRemovals=true
 * "/&gt;</pre>
 *
 * This factory is a copy of the eponymous class from Ehcache, with the exception that any
 * synchronous RMI replicators it creates do not suffer from https://jira.terracotta.org/jira/browse/EHC-683.
 *
 * @since 2.0
 */
public class RMICacheReplicatorFactory extends CacheEventListenerFactory {

    /**
     * A default for the amount of time the replication thread sleeps after it detects the replicationQueue is empty
     * before checking again.
     */
    protected static final int DEFAULT_ASYNCHRONOUS_REPLICATION_INTERVAL_MILLIS = 1000;

    /**
     * A default for the maximum number of operations in an RMI message.
     */
    protected static final int DEFAULT_ASYNCHRONOUS_REPLICATION_MAXIMUM_BATCH_SIZE = 1000;

    private static final Logger LOG = LoggerFactory.getLogger(RMICacheReplicatorFactory.class.getName());
    private static final String REPLICATE_PUTS = "replicatePuts";
    private static final String REPLICATE_PUTS_VIA_COPY = "replicatePutsViaCopy";
    private static final String REPLICATE_UPDATES = "replicateUpdates";
    private static final String REPLICATE_UPDATES_VIA_COPY = "replicateUpdatesViaCopy";
    private static final String REPLICATE_REMOVALS = "replicateRemovals";
    static final String REPLICATE_ASYNCHRONOUSLY = "replicateAsynchronously";
    private static final String ASYNCHRONOUS_REPLICATION_INTERVAL_MILLIS = "asynchronousReplicationIntervalMillis";
    private static final String ASYNCHRONOUS_REPLICATION_MAXIMUM_BATCH_SIZE = "asynchronousReplicationMaximumBatchSize";
    private static final int MINIMUM_REASONABLE_INTERVAL = 10;

    private static boolean extractBoolean(final String key, final Properties properties) {
        String booleanString = PropertyUtil.extractAndLogProperty(key, properties);
        return booleanString == null || PropertyUtil.parseBoolean(booleanString); // defaults to true
    }

    /**
     * Extracts the value of asynchronousReplicationIntervalMillis. Sets it to 1000ms if
     * either not set or there is a problem parsing the number.
     *
     * @param properties the properties to parse (required)
     */
    private static int extractReplicationIntervalMilis(Properties properties) {
        int asynchronousReplicationIntervalMillis;
        String asynchronousReplicationIntervalMillisString =
                PropertyUtil.extractAndLogProperty(ASYNCHRONOUS_REPLICATION_INTERVAL_MILLIS, properties);
        if (asynchronousReplicationIntervalMillisString != null) {
            try {
                int asynchronousReplicationIntervalMillisCandidate =
                        Integer.parseInt(asynchronousReplicationIntervalMillisString);
                if (asynchronousReplicationIntervalMillisCandidate < MINIMUM_REASONABLE_INTERVAL) {
                    LOG.debug("Trying to set the asynchronousReplicationIntervalMillis to an unreasonable number."
                            + " Using the default instead.");
                    asynchronousReplicationIntervalMillis = DEFAULT_ASYNCHRONOUS_REPLICATION_INTERVAL_MILLIS;
                } else {
                    asynchronousReplicationIntervalMillis = asynchronousReplicationIntervalMillisCandidate;
                }
            } catch (NumberFormatException e) {
                LOG.warn("Number format exception trying to set asynchronousReplicationIntervalMillis. "
                        + "Using the default instead. String value was: '" + asynchronousReplicationIntervalMillisString
                        + "'");
                asynchronousReplicationIntervalMillis = DEFAULT_ASYNCHRONOUS_REPLICATION_INTERVAL_MILLIS;
            }
        } else {
            asynchronousReplicationIntervalMillis = DEFAULT_ASYNCHRONOUS_REPLICATION_INTERVAL_MILLIS;
        }
        return asynchronousReplicationIntervalMillis;
    }

    /**
     * Extracts the value of maximumBatchSize. Sets it to 1024 if
     * either not set or there is a problem parsing the number.
     *
     * @param properties the properties to parse (required)
     */
    private static int extractMaximumBatchSize(final Properties properties) {
        String maximumBatchSizeString =
                PropertyUtil.extractAndLogProperty(ASYNCHRONOUS_REPLICATION_MAXIMUM_BATCH_SIZE, properties);
        if (maximumBatchSizeString == null) {
            return DEFAULT_ASYNCHRONOUS_REPLICATION_MAXIMUM_BATCH_SIZE;
        } else {
            try {
                return Integer.parseInt(maximumBatchSizeString);
            } catch (NumberFormatException e) {
                LOG.warn("Number format exception trying to set maximumBatchSize. "
                        + "Using the default instead. String value was: '" + maximumBatchSizeString + "'");
                return DEFAULT_ASYNCHRONOUS_REPLICATION_MAXIMUM_BATCH_SIZE;
            }
        }
    }

    /**
     * Create a <code>CacheEventListener</code> which is also a CacheReplicator.
     * <p>
     * The defaults if properties are not specified are:
     * <ul>
     * <li>replicatePuts=true
     * <li>replicatePutsViaCopy=true
     * <li>replicateUpdates=true
     * <li>replicateUpdatesViaCopy=true
     * <li>replicateRemovals=true;
     * <li>replicateAsynchronously=true
     * <li>asynchronousReplicationIntervalMillis=1000
     * </ul>
     *
     * @param properties implementation specific properties. These are configured as comma
     *                   separated name value pairs in ehcache.xml e.g.
     *                   <p>
     *                   <code>
     *                   &lt;cacheEventListenerFactory class="net.sf.ehcache.distribution.RMICacheReplicatorFactory"
     *                   properties="
     *                   replicateAsynchronously=true,
     *                   replicatePuts=true
     *                   replicateUpdates=true
     *                   replicateUpdatesViaCopy=true
     *                   replicateRemovals=true
     *                   asynchronousReplicationIntervalMillis=1000
     *                   "/&gt;</code>
     * @return a constructed CacheEventListener
     */
    public CacheEventListener createCacheEventListener(final Properties properties) {
        final boolean replicatePuts = extractBoolean(REPLICATE_PUTS, properties);
        final boolean replicatePutsViaCopy = extractBoolean(REPLICATE_PUTS_VIA_COPY, properties);
        final boolean replicateUpdates = extractBoolean(REPLICATE_UPDATES, properties);
        final boolean replicateUpdatesViaCopy = extractBoolean(REPLICATE_UPDATES_VIA_COPY, properties);
        final boolean replicateRemovals = extractBoolean(REPLICATE_REMOVALS, properties);
        final boolean replicateAsynchronously = extractBoolean(REPLICATE_ASYNCHRONOUSLY, properties);
        final int replicationIntervalMillis = extractReplicationIntervalMilis(properties);
        final int maximumBatchSize = extractMaximumBatchSize(properties);

        if (replicateAsynchronously) {
            return new RMIAsynchronousCacheReplicator(
                    replicatePuts,
                    replicatePutsViaCopy,
                    replicateUpdates,
                    replicateUpdatesViaCopy,
                    replicateRemovals,
                    replicationIntervalMillis,
                    maximumBatchSize);
        }
        // Return our copy of this class in which https://jira.terracotta.org/jira/browse/EHC-683 is fixed
        return new RMISynchronousCacheReplicator(
                replicatePuts, replicatePutsViaCopy, replicateUpdates, replicateUpdatesViaCopy, replicateRemovals);
    }
}
