package com.atlassian.plugin.schema.impl;

import java.util.Set;
import javax.annotation.Nullable;

import com.google.common.collect.ImmutableSet;

import com.atlassian.plugin.ModuleDescriptor;
import com.atlassian.plugin.internal.schema.spi.Schema;
import com.atlassian.plugin.internal.schema.spi.SchemaFactory;
import com.atlassian.plugin.module.ContainerManagedPlugin;
import com.atlassian.plugin.schema.descriptor.DescribedModuleDescriptorFactory;

import static java.util.Collections.singleton;
import static java.util.Objects.requireNonNull;

/**
 * Described module descriptor factory for internal use
 */
final class DescribedModuleTypeDescribedModuleDescriptorFactory<T extends ModuleDescriptor<?>>
        implements DescribedModuleDescriptorFactory {
    private final ContainerManagedPlugin plugin;
    private final String type;
    private final Iterable<String> typeList;
    private final Class<T> moduleDescriptorClass;
    private final SchemaFactory schemaFactory;

    /**
     * Constructs an instance using a specific host container
     *
     * @param type                  The type of module
     * @param moduleDescriptorClass The descriptor class
     * @param schemaFactory         the schema factory
     * @since 3.0.0
     */
    DescribedModuleTypeDescribedModuleDescriptorFactory(
            ContainerManagedPlugin plugin,
            final String type,
            final Class<T> moduleDescriptorClass,
            SchemaFactory schemaFactory) {
        this.plugin = requireNonNull(plugin);
        this.moduleDescriptorClass = moduleDescriptorClass;
        this.type = type;
        this.schemaFactory = schemaFactory;
        this.typeList = singleton(type);
    }

    public ModuleDescriptor getModuleDescriptor(final String type) {
        T result = null;
        if (this.type.equals(type)) {
            result = plugin.getContainerAccessor().createBean(moduleDescriptorClass);
        }
        return result;
    }

    public boolean hasModuleDescriptor(final String type) {
        return (this.type.equals(type));
    }

    @Override
    @Nullable
    public Schema getSchema(String type) {
        return (this.type.equals(type) ? schemaFactory.getSchema() : null);
    }

    @Override
    public Iterable<String> getModuleDescriptorKeys() {
        return typeList;
    }

    public Class<? extends ModuleDescriptor<?>> getModuleDescriptorClass(final String type) {
        return (this.type.equals(type) ? moduleDescriptorClass : null);
    }

    public Set<Class<? extends ModuleDescriptor>> getModuleDescriptorClasses() {
        return ImmutableSet.of(moduleDescriptorClass);
    }
}
