package com.atlassian.plugin.servlet;

import java.util.Enumeration;
import jakarta.servlet.ServletContext;
import jakarta.servlet.http.HttpSession;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.atlassian.plugin.util.ClassLoaderStack;

/**
 * Wraps a HttpSession for consumption by OSGi plugins in order to workaround Weblogic problems caused by setting
 * different Context ClassLoaders.
 * See https://studio.atlassian.com/browse/PLUG-515
 *
 * @since 2.3.9
 */
public class PluginHttpSessionWrapper implements HttpSession {
    private HttpSession delegate;
    private static final Logger log = LoggerFactory.getLogger(PluginHttpSessionWrapper.class);

    public PluginHttpSessionWrapper(final HttpSession session) {
        this.delegate = session;
    }

    public Object getAttribute(final String name) {
        // Trick WLS by putting the WebAppClassLoader back into this thread's ContextClassLoader for the duration of the
        //  getAttribute() call. See PLUG-515.
        ClassLoader classLoader = ClassLoaderStack.pop();
        try {
            if (log.isDebugEnabled()) {
                log.debug(
                        "getAttribute('{}') Popping ClassLoader: {}. New ContextClassLoader: {}",
                        name,
                        classLoader,
                        Thread.currentThread().getContextClassLoader());
            }
            return delegate.getAttribute(name);
        } finally {
            // Reset to the Plugins ClassLoader and let OSGi continue to do its ClassLoader voodoo.
            ClassLoaderStack.push(classLoader);
        }
    }

    public void setAttribute(final String name, final Object value) {
        // Trick WLS by putting the WebAppClassLoader back into this thread's ContextClassLoader for the duration of the
        // method call. See PLUG-515.
        ClassLoader classLoader = ClassLoaderStack.pop();
        try {
            delegate.setAttribute(name, value);
        } finally {
            // Reset to the Plugins ClassLoader and let OSGi continue to do its ClassLoader voodoo.
            ClassLoaderStack.push(classLoader);
        }
    }

    public long getCreationTime() {
        return delegate.getCreationTime();
    }

    public String getId() {
        return delegate.getId();
    }

    public long getLastAccessedTime() {
        return delegate.getLastAccessedTime();
    }

    public ServletContext getServletContext() {
        return delegate.getServletContext();
    }

    public void setMaxInactiveInterval(final int interval) {
        delegate.setMaxInactiveInterval(interval);
    }

    public int getMaxInactiveInterval() {
        return delegate.getMaxInactiveInterval();
    }

    public Enumeration<String> getAttributeNames() {
        return delegate.getAttributeNames();
    }

    public void removeAttribute(final String name) {
        delegate.removeAttribute(name);
    }

    public void invalidate() {
        delegate.invalidate();
    }

    public boolean isNew() {
        return delegate.isNew();
    }
}
