package com.atlassian.pocketknife.internal.querydsl;

import com.atlassian.annotations.nullability.ParametersAreNonnullByDefault;
import com.atlassian.pocketknife.api.querydsl.DatabaseConnection;
import com.atlassian.pocketknife.api.querydsl.DatabaseConnectionConverter;
import com.atlassian.pocketknife.api.querydsl.schema.DialectProvider;

import java.sql.Connection;
import java.sql.SQLException;

/**
 *
 */
@ParametersAreNonnullByDefault
public class DatabaseConnectionConverterImpl implements DatabaseConnectionConverter {
    private final DialectProvider dialectProvider;

    public DatabaseConnectionConverterImpl(final DialectProvider dialectProvider) {
        this.dialectProvider = dialectProvider;
    }

    @Override
    public DatabaseConnection convert(final Connection jdbcConnection) {
        return convertImpl(jdbcConnection, false);
    }

    @Override
    public DatabaseConnection convertExternallyManaged(Connection jdbcConnection) {
        return convertImpl(jdbcConnection, true);
    }

    private DatabaseConnection convertImpl(final Connection jdbcConnection, boolean managedConnection) {
        assertNotClosed(jdbcConnection);

        DialectProvider.Config dialectConfig = getDialectConfig(jdbcConnection);
        SpecificBehaviourConnection specificBehaviourConnection = new SpecificBehaviourConnection(jdbcConnection);
        return new DatabaseConnectionImpl(dialectConfig, specificBehaviourConnection, managedConnection);
    }

    private DialectProvider.Config getDialectConfig(final Connection jdbcConnection) {
        return dialectProvider.getDialectConfig(assertNotClosed(jdbcConnection));
    }

    private Connection assertNotClosed(final Connection jdbcConnection) {
        try {
            if (jdbcConnection.isClosed()) {
                throw new IllegalStateException();
            }
        } catch (SQLException e) {
            throw new IllegalStateException("PKQDSL is unable to assert that the JDBC connection is not closed", e);
        }
        return jdbcConnection;
    }

}
