package com.atlassian.streams.thirdparty.api;

import java.net.URI;
import java.time.ZonedDateTime;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.atlassian.streams.api.Html;
import com.atlassian.streams.api.UserProfile;
import com.atlassian.streams.api.common.Either;
import com.atlassian.streams.api.common.Option;

import static java.util.Objects.requireNonNull;

import static com.atlassian.streams.api.common.Either.left;
import static com.atlassian.streams.api.common.Either.right;
import static com.atlassian.streams.thirdparty.api.ValidationErrors.MAX_CONTENT_LENGTH;
import static com.atlassian.streams.thirdparty.api.ValidationErrors.MAX_STRING_LENGTH;

/**
 * Represents an activity entry provided by a third-party source which can be included in
 * an Activity Streams feed.
 * <p>
 * This class can be used with {@link ActivityService} either to construct a new activity and
 * post it to the feed, or to retrieve properties from existing activities.  To construct a
 * new activity, create a {@link Builder2} instance with {@link #builder2(Application, ZonedDateTime, UserProfile)},
 * configure the builder and call {@link Builder2#build()}, then pass the resulting Activity instance
 * to {@link ActivityService#postActivity(Activity)}.
 * <p>
 * The properties of this class are based on a subset of the {@code activitystrea.ms}
 * specification.
 */
public class Activity {
    private final Long activityId;
    private final Html content;
    private final Application application;
    private final Image icon;
    private final URI id;
    private final ActivityObject object;
    private final ZonedDateTime postedDate;
    private final String poster;
    private final boolean registeredUser;
    private final ActivityObject target;
    private final Html title;
    private final URI url;
    private final UserProfile user;
    private final URI verb;

    /**
     * Returns a {@link Builder2} for constructing an {@link Activity} instance.
     * @param application  an {@link Application} object describing the application that
     *   generated the activity
     * @param postedDate  the activity's timestamp
     * @param user  a {@link UserProfile} describing the user who performed the activity;
     *   if its {@link UserProfile#getUsername() username} property matches a registered
     *   user in the host application, the rest of the profile will be filled in automatically
     * @return  a {@link Builder2} instance
     */
    public static Builder2 builder2(Application application, ZonedDateTime postedDate, UserProfile user) {
        return new Builder2(application, postedDate, user);
    }

    /**
     * Returns a {@link Builder2} for constructing an {@link Activity} instance.  This
     * overload allows you to pass a username rather than a {@link UserProfile}
     * @param application  an {@link Application} object describing the application that
     *   generated the activity
     * @param postedDate  the activity's timestamp
     * @param username  the username of the the user who performed the activity; if this
     *   matches a registered user in the host application, the rest of the user profile
     *   will be filled in automatically
     * @return  a {@link Builder2} instance
     */
    public static Builder2 builder2(Application application, ZonedDateTime postedDate, String username) {
        return new Builder2(application, postedDate, new UserProfile.Builder(username).build());
    }

    private Activity(Builder2 builder) {
        this.activityId = builder.activityId;
        this.content = builder.content;
        this.application = builder.application;
        this.icon = builder.icon;
        this.id = builder.id;
        this.object = builder.object;
        this.postedDate = builder.postedDate;
        this.poster = builder.poster;
        this.registeredUser = builder.registeredUser;
        this.target = builder.target;
        this.title = builder.title;
        this.url = builder.url;
        this.user = builder.user;
        this.verb = builder.verb;
    }

    /**
     * The unique primary key of the activity, if it has been stored in the database.  This is
     * generated by the {@link ActivityService} and cannot be changed.
     * @return  an {@link Option} containing a unique long key, or {@link Option#none()} if
     *   the activity has not been stored
     *   @deprecated - use {@link #getActivityIdOrNull()} instead;
     */
    @Deprecated
    public Option<Long> getActivityId() {
        return Option.option(activityId);
    }

    /**
     * The unique primary key of the activity, if it has been stored in the database.  This is
     * generated by the {@link ActivityService} and cannot be changed.
     * @return  a unique long key, or null if the activity has not been stored
     */
    @Nullable
    public Long getActivityIdOrNull() {
        return activityId;
    }

    /**
     * A description of the application that is the source of the activity.
     * @return  an {@link Application} object; cannot be null
     */
    public Application getApplication() {
        return application;
    }

    /**
     * An optional HTML string that is the detailed description of the activity.
     * @return  an {@link Option} containing an HTML string, or {@link Option#none()} if the
     *   activity has no description
     * @deprecated - use {@link #getContentOrNull()} instead;
     */
    @Deprecated
    public Option<Html> getContent() {
        return Option.option(content);
    }

    /**
     * An nullable HTML string that is the detailed description of the activity.
     * @return  an HTML string, or null if the activity has no description
     */
    @Nullable
    public Html getContentOrNull() {
        return content;
    }

    /**
     * An optional icon representing the activity.
     * @return  an {@link Option} containing an {@link Image} object that describes the icon,
     *   or {@link Option#none()} if no icon was specified
     *   @deprecated - use {@link #getIconOrNull()} instead;
     */
    @Deprecated
    public Option<Image> getIcon() {
        return Option.option(icon);
    }

    /**
     * An nullable icon representing the activity.
     * @return {@link Image} object that describes the icon or null if no icon was specified
     */
    @Nullable
    public Image getIconOrNull() {
        return icon;
    }

    /**
     * A {@link URI} that uniquely identifies the activity, as specified by the application
     * that generated it.  An activity must have either an {@code id} or a {@code url}.
     * @return  an {@link Option} containing the activity's unique identifier, or
     *   {@link Option#none()} if none was specified (in which case {@link #getUrl()} must return
     *   a value)
     * @deprecated - use {@link #getIdOrNull()} instead;
     */
    @Deprecated
    public Option<URI> getId() {
        return Option.option(id);
    }

    /**
     * A nullable {@link URI} that uniquely identifies the activity, as specified by the application
     * that generated it.  An activity must have either an {@code id} or a {@code url}.
     * @return the activity's unique identifier, or null if none was specified (in which case {@link #getUrl()} must return
     *   a value)
     */
    @Nullable
    public URI getIdOrNull() {
        return id;
    }

    /**
     * An optional {@link ActivityObject} describing the activity, with properties that
     * correspond to the activity object in the {@code activitystrea.ms} specification.
     * @return  an {@link Option} containing an {@link ActivityObject}, or {@link Option#none()}
     *   if none was provided
     *   @deprecated - use {@link #getObjectOrNull()} instead;
     */
    @Deprecated
    public Option<ActivityObject> getObject() {
        return Option.option(object);
    }

    /**
     * A nullable {@link ActivityObject} describing the activity, with properties that
     * correspond to the activity object in the {@code activitystrea.ms} specification.
     */
    @Nullable
    public ActivityObject getObjectOrNull() {
        return object;
    }

    /**
     * The date and time that the activity was posted.
     * @return  a {@link ZonedDateTime}; cannot be null
     */
    @Nonnull
    public ZonedDateTime getZonedPostedDate() {
        return postedDate;
    }

    /**
     * The name of the registered user whose credentials were used to post this activity.
     * This is not necessarily the same user identified by {@link #getUser()}.  This property
     * is set by the {@link ActivityService} and cannot be changed.
     * @return  an {@link Option} containing a username, or {@link Option#none()} if unknown
     * @deprecated - use {@link #getPosterOrNull()} instead;
     */
    public Option<String> getPoster() {
        return Option.option(poster);
    }

    /**
     * The name of the registered user whose credentials were used to post this activity.
     * This is not necessarily the same user identified by {@link #getUser()}.  This property
     * is set by the {@link ActivityService} and cannot be changed.
     * @return  username, or null if unknown
     */
    @Nullable
    public String getPosterOrNull() {
        return poster;
    }

    /**
     * True if the user profile returned by {@link #getUser()} is for a registered user.
     * This property is set by the {@link ActivityService} and cannot be changed.
     * @return  {@code true} if the user profile is for a registered user
     */
    public boolean isRegisteredUser() {
        return registeredUser;
    }

    /**
     * An optional {@link ActivityObject} describing the "target" of the activity, with
     * properties that correspond to the activity target object in the {@code activitystrea.ms}
     * specification.  For instance, if the activity is a comment on a JIRA issue, the target
     * is the JIRA issue.
     * @return  an {@link Option} containing an {@link ActivityObject}, or {@link Option#none()}
     *   if none was provided
     *   @deprecated - use {@link #getTargetOrNull()} instead;
     */
    @Deprecated
    public Option<ActivityObject> getTarget() {
        return Option.option(target);
    }

    /**
     * An nullable {@link ActivityObject} describing the "target" of the activity, with
     * properties that correspond to the activity target object in the {@code activitystrea.ms}
     * specification.  For instance, if the activity is a comment on a JIRA issue, the target
     * is the JIRA issue.
     * @return - an {@link ActivityObject}, or null if none was provided
     */
    @Nullable
    public ActivityObject getTargetOrNull() {
        return target;
    }

    /**
     * An optional HTML string that is the title of the entity.
     * @return  an {@link Option} containing an HTML string, or {@link Option#none()} if the
     *   activity has no title
     *   @deprecated - use {@link #getTitleOrNull()} instead;
     */
    @Deprecated
    public Option<Html> getTitle() {
        return Option.option(title);
    }

    /**
     * An nullable HTML string that is the title of the entity.
     * @return HTML string, or null if the activity has no title
     */
    @Nullable
    public Html getTitleOrNull() {
        return title;
    }

    /**
     * A {@link URI} that identifies the activity in a human-readable format on the web, as
     * specified by the application that generated it.  In the Activity Streams feed, the
     * activity's date will be formatted as a permalink to this URI.  If it is omitted, the
     * {@code id} property will be used instead.
     * @return  an {@link Option} containing the activity's URI, or {@link Option#none()} if
     *   none was specified (in which case {@link #getId()} must return a value)
     *   @deprecated - use {@link #getUrlOrNull()} instead;
     */
    @Deprecated
    public Option<URI> getUrl() {
        return Option.option(url);
    }

    /**
     * A {@link URI} that identifies the activity in a human-readable format on the web, as
     * specified by the application that generated it.  In the Activity Streams feed, the
     * activity's date will be formatted as a permalink to this URI.  If it is omitted, the
     * {@code id} property will be used instead.
     * @return  the activity's URI, or null if
     *   none was specified (in which case {@link #getIdOrNull()} must return a value)
     */
    @Nullable
    public URI getUrlOrNull() {
        return url;
    }

    /**
     * A {@link UserProfile} describing the user who performed the activity.  This may be a
     * registered user in the host application, or an unknown user whose name and profile image
     * can be specified but who will not be included in any user-specific search filters.
     * @return  a {@link UserProfile}; cannot be null
     */
    public UserProfile getUser() {
        return user;
    }

    /**
     * An optional {@link URI} that identifies the type of action in the activity.  This is not
     * displayed as human-readable text, but is included in the activity feed and may affect how
     * the activity is formatted by the front end.  It should be either a simple name ("post")
     * or an IRI as described in the {@code activitystrea.ms} specification.
     * @return  an {@link Option} containing a {@link URI}, or {@link Option#none()} if none was
     *   specified
     * @deprecated - use {@link #getVerbOrNull()} instead;
     */
    @Deprecated
    public Option<URI> getVerb() {
        return Option.option(verb);
    }

    /**
     * A nullable {@link URI} that identifies the type of action in the activity.  This is not
     * displayed as human-readable text, but is included in the activity feed and may affect how
     * the activity is formatted by the front end.  It should be either a simple name ("post")
     * or an IRI as described in the {@code activitystrea.ms} specification.
     * @return  the {@link URI}, or null if none was
     *   specified
     */
    @Nullable
    public URI getVerbOrNull() {
        return verb;
    }

    @Override
    public boolean equals(Object other) {
        if (other instanceof Activity) {
            Activity a = (Activity) other;
            return activityId.equals(a.activityId)
                    && application.equals(a.application)
                    && content.equals(a.content)
                    && icon.equals(a.icon)
                    && id.equals(a.id)
                    && object.equals(a.object)
                    && postedDate.equals(a.postedDate)
                    && poster.equals(a.poster)
                    && registeredUser == a.registeredUser
                    && target.equals(a.target)
                    && title.equals(a.title)
                    && url.equals(a.url)
                    && user.equals(a.user)
                    && verb.equals(a.verb);
        }
        return false;
    }

    @Override
    public int hashCode() {
        return activityId.hashCode()
                + 37
                        * (application.hashCode()
                                + 37
                                        * (content.hashCode()
                                                + 37
                                                        * (icon.hashCode()
                                                                + 37
                                                                        * (id.hashCode()
                                                                                + 37
                                                                                        * (object.hashCode()
                                                                                                + 37
                                                                                                        * (postedDate
                                                                                                                        .hashCode()
                                                                                                                + 37
                                                                                                                        * (poster
                                                                                                                                        .hashCode()
                                                                                                                                + 37
                                                                                                                                        * ((registeredUser
                                                                                                                                                        ? 1
                                                                                                                                                        : 0)
                                                                                                                                                + 37
                                                                                                                                                        * (target
                                                                                                                                                                        .hashCode()
                                                                                                                                                                + 37
                                                                                                                                                                        * (title
                                                                                                                                                                                        .hashCode()
                                                                                                                                                                                + 37
                                                                                                                                                                                        * (url
                                                                                                                                                                                                        .hashCode()
                                                                                                                                                                                                + 37
                                                                                                                                                                                                        * (user
                                                                                                                                                                                                                        .hashCode()
                                                                                                                                                                                                                + 37
                                                                                                                                                                                                                        * (verb
                                                                                                                                                                                                                                .hashCode())))))))))))));
    }

    /**
     * Stateful builder class for constructing {@link Activity} instances.
     * <p>
     * The Builder will not allow you to construct an {@link Activity} with invalid properties
     * that cannot be accepted by the {@link ActivityService}.  Value constraints are described
     * in {@link ValidationErrors}.
     */
    public static final class Builder2 {
        private ValidationErrors.Builder errors = new ValidationErrors.Builder();
        private Long activityId = null;
        private Html content = null;
        private final Application application;
        private Image icon = null;
        private URI id = null;
        private ActivityObject object = null;
        private final ZonedDateTime postedDate;
        private String poster = null;
        private boolean registeredUser = false;
        private ActivityObject target = null;
        private Html title = null;
        private URI url = null;
        private final UserProfile user;
        private URI verb = null;

        /**
         * Constructs a new {@link Builder2} instance, specifying all of the required Activity properties.
         *
         * @param application an {@link Application} instance describing the application that created the activity
         * @param postedDate  the activity's timestamp
         * @param user        a {@link UserProfile} instance describing the user who performed the activity
         */
        public Builder2(Application application, ZonedDateTime postedDate, UserProfile user) {
            this(
                    Either.<ValidationErrors, Application>right(application),
                    Either.<ValidationErrors, ZonedDateTime>right(postedDate),
                    Either.<ValidationErrors, UserProfile>right(user));
        }

        /**
         * Constructs a new {@link Builder2} instance, specifying all of the required Activity properties.
         * This overload allows the parameter to be either valid object references or
         * {@link ValidationErrors}, which is useful if the properties are dynamically generated and
         * might be invalid.
         *
         * @param application an {@link Either} containing either an {@link Application} instance or
         *                    {@link ValidationErrors}
         * @param postedDate  an {@link Either} containing either the activity's timestamp or
         *                    {@link ValidationErrors}
         * @param user        an {@link Either} containing either a {@link UserProfile} instance or
         *                    {@link ValidationErrors}
         */
        public Builder2(
                Either<ValidationErrors, Application> application,
                Either<ValidationErrors, ZonedDateTime> postedDate,
                Either<ValidationErrors, UserProfile> user) {
            if (application.isRight()) {
                this.application = application.right().get();
            } else {
                errors.addAll(application.left().get(), "application");
                this.application = null;
            }
            if (postedDate.isRight()) {
                this.postedDate = postedDate.right().get();
            } else {
                errors.addAll(postedDate.left().get(), "postedDate");
                this.postedDate = null;
            }
            if (user.isRight()) {
                this.user = user.right().get();
                errors.checkString(this.user.getUsername(), "user.username");
                errors.checkString(this.user.getFullName(), "user.fullName");
                errors.checkAbsoluteUri(this.user.getProfilePageUri(), "user.profilePageUri");
                errors.checkAbsoluteUri(this.user.getProfilePictureUri(), "user.profilePictureUri");
            } else {
                errors.addAll(user.left().get(), "user");
                this.user = null;
            }
        }

        /**
         * Attempts to create an immutable {@link Activity} instance using the current properties
         * of this {@link Builder2}.
         *
         * @return an {@link Either} containing an {@link Activity} ({@link Either#right()})
         * if successful, or {@link ValidationErrors} ({@link Either#left()}) if any of the
         * builder's properties were invalid
         */
        public Either<ValidationErrors, Activity> build() {
            if (id == null && url == null) {
                errors.addError("activity id and url cannot both be omitted");
            }
            if (errors.isEmpty()) {
                return right(new Activity(this));
            } else {
                return left(errors.build());
            }
        }

        /**
         * Used internally to store the unique primary key of the activity.  Do not try to set
         * this property; it will be overridden by the {@link ActivityService}.
         *
         * @param activityId a unique long key
         * @return the same Builder2 instance
         */
        public Builder2 activityId(long activityId) {
            this.activityId = activityId;
            return this;
        }

        /**
         * Sets an optional HTML string that is the detailed description of the activity.
         *
         * @param content HTML string
         * @return the same Builder2 instance
         */
        public Builder2 content(@Nonnull Html content) {
            this.content = errors.checkHtml(content, "content", MAX_CONTENT_LENGTH);
            return this;
        }

        /**
         * Sets an optional icon representing the activity.
         *
         * @param icon {@link Image} object that describes
         * @return the same Builder2 instance
         */
        public Builder2 icon(@Nonnull Image icon) {
            this.icon = requireNonNull(icon, "icon");
            return this;
        }

        /**
         * Same as {@link #icon}, but accepts an {@link Either} which may contain either an
         * {@link Image} or {@link ValidationErrors}.  In the latter case, the errors are
         * added to the error list for this Builder.  This overload is for convenience, so you can
         * write expressions like this without having to check intermediate results:
         * <pre>
         *     Activity.Builder activityBuilder = Activity.builder().someActivityProperties(...)
         *         .icon(Image.builder(...).someImageProperties(...).build()))
         *         .build();
         * </pre>
         *
         * @param errorsOrImage an {@link Either} containing either an {@link Image} or
         *                      {@link ValidationErrors}
         * @return the same Builder2 instance
         */
        public Builder2 icon(Either<ValidationErrors, Image> errorsOrImage) {
            requireNonNull(errorsOrImage, "errorsOrImage");
            if (errorsOrImage.isRight()) {
                this.icon = errorsOrImage.right().get();
            } else {
                errors.addAll(errorsOrImage.left().get(), "icon");
            }
            return this;
        }

        /**
         * Sets a {@link URI} that uniquely identifies the activity, as specified by the application
         * that generated it.  An activity must have either an {@code id} or a {@code url}.
         *
         * @param id an activity's unique identifier
         * @return the same Builder2 instance
         */
        public Builder2 id(URI id) {
            this.id = errors.checkAbsoluteUri(id, "id");
            return this;
        }

        /**
         * Same as {@link #id}, but specifies the property as a String rather than a URI.  This
         * allows URI syntax errors to be reported by the {@link ValidationErrors} mechanism rather
         * than having to catch URISyntaxExceptions.
         *
         * @param id an {@link Option} containing the activity's ID as a string, or {@link Option#none()}
         *            if none is specified
         * @return the same Builder2 instance
         */
        public Builder2 idString(String id) {
            this.id = errors.checkAbsoluteUriString(id, "id");
            return this;
        }

        /**
         * Sets an {@link ActivityObject} describing the activity, with properties that
         * correspond to the activity object in the {@code activitystrea.ms} specification.
         *
         * @param object an {@link ActivityObject}
         * @return the same Builder2 instance
         */
        public Builder2 object(@Nonnull ActivityObject object) {
            this.object = requireNonNull(object, "object");
            return this;
        }

        /**
         * Same as {@link #object}, but accepts an {@link Either} which may contain either an
         * {@link ActivityObject} or {@link ValidationErrors}.  In the latter case, the errors are
         * added to the error list for this Builder.  This overload is for convenience, so you can
         * write expressions like this without having to check intermediate results:
         * <pre>
         *     Activity.Builder activityBuilder = Activity.builder().someActivityProperties(...)
         *         .object(ActivityObject.builder().someObjectProperties(...).build()))
         *         .build();
         * </pre>
         *
         * @param errorsOrObject an {@link Either} containing either an {@link ActivityObject} or
         *                       {@link ValidationErrors}
         * @return the same Builder2 instance
         */
        public Builder2 object(Either<ValidationErrors, ActivityObject> errorsOrObject) {
            requireNonNull(errorsOrObject, "targetOrErrors");
            if (errorsOrObject.isRight()) {
                this.object = errorsOrObject.right().get();
            } else {
                errors.addAll(errorsOrObject.left().get(), "object");
            }
            return this;
        }

        /**
         * Used internally to store the name of the registered user whose credentials were used
         * to post this activity.  Do not try to set this property; it will be overridden by the
         * {@link ActivityService}.
         *
         * @param poster a username
         * @return the same Builder2 instance
         */
        public Builder2 poster(@Nonnull String poster) {
            this.poster = errors.checkString(poster, "poster");
            return this;
        }

        /**
         * Used internally to indicate whether the user profile is for a registered user.  Do not
         * try to set this property; it will be overridden by the {@link ActivityService}.
         *
         * @param registeredUser true if the user profile is for a registered user
         * @return the same Builder2 instance
         */
        public Builder2 registeredUser(boolean registeredUser) {
            this.registeredUser = registeredUser;
            return this;
        }

        /**
         * Sets an {@link ActivityObject} describing the "target" of the activity, with
         * properties that correspond to the activity target object in the {@code activitystrea.ms}
         * specification.  For instance, if the activity is a comment on a JIRA issue, the target
         * is the JIRA issue.
         * <p>
         * To refer to specific entities defined by the host application -- such as an issue or
         * project in JIRA, or a space in Confluence -- you need only set the {@link ActivityObject#getUrl() url}
         * property of the target, either to the URL of the entity or just to its unique identifier
         * (e.g. a JIRA issue key), and the {@link ActivityService} will fill in the complete URL
         * and object type for you.  This will also cause the activity to be correctly found by
         * search filters that are based on issue or project keys.
         *
         * @param target an {@link ActivityObject}
         * @return the same Builder2 instance
         */
        public Builder2 target(@Nonnull ActivityObject target) {
            this.target = requireNonNull(target, "target");
            return this;
        }

        /**
         * Same as {@link #target}, but accepts an {@link Either} which may contain either an
         * {@link ActivityObject} or {@link ValidationErrors}.  In the latter case, the errors are
         * added to the error list for this Builder.  This overload is for convenience, so you can
         * write expressions like this without having to check intermediate results:
         * <pre>
         *     Activity.Builder activityBuilder = Activity.builder().someActivityProperties(...)
         *         .target(ActivityObject.builder().someObjectProperties(...).build()))
         *         .build();
         * </pre>
         *
         * @param targetOrErrors an {@link Either} containing either an {@link ActivityObject} or
         *                       {@link ValidationErrors}
         * @return the same Builder instance
         */
        public Builder2 target(Either<ValidationErrors, ActivityObject> targetOrErrors) {
            requireNonNull(targetOrErrors, "targetOrErrors");
            if (targetOrErrors.isRight()) {
                this.target = targetOrErrors.right().get();
            } else {
                errors.addAll(targetOrErrors.left().get(), "target");
            }
            return this;
        }

        /**
         * Sets an optional HTML string that is the title of the entity.
         *
         * @param title an HTML string
         * @return the same Builder instance
         */
        public Builder2 title(@Nonnull Html title) {
            this.title = errors.checkHtml(title, "title", MAX_STRING_LENGTH);
            return this;
        }

        /**
         * Sets a {@link URI} that identifies the activity in a human-readable format on the web, as
         * specified by the application that generated it.  In the Activity Streams feed, the
         * activity's date will be formatted as a permalink to this URI.  If it is omitted, the
         * {@link #id} property will be used instead and must have a value.
         *
         * @param url the activity's URI
         * @return the same Builder instance
         */
        public Builder2 url(@Nonnull URI url) {
            this.url = errors.checkAbsoluteUri(url, "url");
            return this;
        }

        /**
         * Same as {@link #url}, but specifies the property as a String rather than a URI.  This
         * allows URI syntax errors to be reported by the {@link ValidationErrors} mechanism rather
         * than having to catch URISyntaxExceptions.
         *
         * @param url activity's URI as a string
         * @return the same Builder instance
         */
        public Builder2 urlString(@Nonnull String url) {
            this.url = errors.checkAbsoluteUriString(url, "url");
            return this;
        }

        /**
         * Sets nullable {@link URI} that identifies the type of action in the activity.  This is not
         * displayed as human-readable text, but is included in the activity feed and may affect how
         * the activity is formatted by the front end.  It should be either a simple name ("post")
         * or an IRI as described in the {@code activitystrea.ms} specification.
         *
         * @param verb a string, or {@link Option#none()} if none is
         *             specified (in which case it will default to "post" in the output feed)
         * @return the same Builder instance
         */
        public Builder2 verb(URI verb) {
            this.verb = errors.checkSimpleNameOrAbsoluteUri(verb, "verb");
            return this;
        }

        /**
         * Same as {@link #verb}, but specifies the property as a String rather than a URI.  This
         * allows URI syntax errors to be reported by the {@link ValidationErrors} mechanism rather
         * than having to catch URISyntaxExceptions.
         *
         * @param verb a nonnull String
         * @return the same Builder instance
         */
        public Builder2 verbString(@Nonnull String verb) {
            this.verb = errors.checkSimpleNameOrAbsoluteUriString(verb, "verb");
            return this;
        }
    }
}
