package com.dji.wpmzsdk.common.utils.kml.model

import android.content.Context

import dji.sdk.wpmz.value.mission.WaylineActionGroup
import dji.sdk.wpmz.value.mission.WaylineTemplate
import java.math.BigDecimal

/**
 * Description :
 *
 * @filename : MissionWaypointFault
 * @author : devin.xu
 * @time : 2022/12/16
 *
 * Copyright (c) 2016, DJI All Rights Reserved.
 **/
sealed class MissionFaultSegment {

    open fun getContent(context: Context): String {
        return ""
    }

    override fun equals(other: Any?): Boolean {
        return this === other
    }

    override fun hashCode(): Int {
        return System.identityHashCode(this)
    }
}

class TemplateSeg : MissionFaultSegment()

class WaypointTemplateSeg : MissionFaultSegment()

class WaylineWaypointSeg(val tag: String) : MissionFaultSegment()

class ActionGroupSeg(val tag: String, actionGroups: List<WaylineActionGroup>) : MissionFaultSegment() {

    var pointIndex = 0

    init {
        val splitStrs = tag.split("_")
        if (splitStrs.size > 1) {
            val index = splitStrs[1].toInt()
            pointIndex = actionGroups[index].startIndex
        }
    }

    override fun getContent(context: Context): String {
        return "${"Waypoints"}${pointIndex + 1}"
    }
}

class ActionSeg(val tag: String) : MissionFaultSegment() {
    override fun getContent(context: Context): String {
        return ""
    }
}

class ActionGimbalRotateYawSeg(var value: String) : MissionFaultSegment() {
    init {
        try {
            value = BigDecimal(value).setScale(1, BigDecimal.ROUND_HALF_UP).toString()
        } catch (e: Throwable) {
            // ignore
        }
    }
    override fun getContent(context: Context): String {
        return "${"Gimbal Yaw Rotation"} $value"
    }
}

class ActionGimbalRotatePitchSeg(var value: String) : MissionFaultSegment() {
    init {
        try {
            value = BigDecimal(value).setScale(1, BigDecimal.ROUND_HALF_UP).toString()
        } catch (e: Throwable) {
            // ignore
        }
    }
    override fun getContent(context: Context): String {
        return "${"Gimbal Pitch Rotation"} $value"
    }
}

class ActionAircraftYawSeg(var value: String) : MissionFaultSegment() {
    init {
        try {
            value = BigDecimal(value).setScale(1, BigDecimal.ROUND_HALF_UP).toString()
        } catch (e: Throwable) {
            // ignore
        }
    }
    override fun getContent(context: Context): String {
        return "${"Aircraft Rotate"} $value"
    }
}

class ActionZoomSeg(var value: String) : MissionFaultSegment() {
    init {
        try {
            value = BigDecimal(value).setScale(1, BigDecimal.ROUND_HALF_UP).toString()
        } catch (e: Throwable) {
            // ignore
        }
    }
    override fun getContent(context: Context): String {
        return "${"Camera Zoom"} $value"
    }
}

class ActionGimbalEvenlyRotateSeg(var value: String) : MissionFaultSegment() {
    init {
        try {
            value = BigDecimal(value).setScale(1, BigDecimal.ROUND_HALF_UP).toString()
        } catch (e: Throwable) {
            // ignore
        }
    }
    override fun getContent(context: Context): String {
        return "${"Gimbal Control"} " +
                "${"Gimbal Pitch Rotation"} $value"
    }
}

class IgnoreSeg(val tag: String) : MissionFaultSegment()

fun parseSeg(tag: String, value: String, template: WaylineTemplate): List<MissionFaultSegment> {
    val segList = tag.split(".").map {
        when {
            it.startsWith("template") -> TemplateSeg()
            it.startsWith("waypointInfo") -> WaypointTemplateSeg()
            it.startsWith("actionGroups") -> ActionGroupSeg(it, template.waypointInfo.actionGroups)
            it.startsWith("actions") -> ActionSeg(it)
            it.startsWith("yaw") && tag.contains("gimbalRotateParam") -> ActionGimbalRotateYawSeg(value)
            it.startsWith("pitch") && tag.contains("gimbalRotateParam") -> ActionGimbalRotatePitchSeg(value)
            it.startsWith("aircraftRotateYawParam") -> ActionAircraftYawSeg(value)
            it.startsWith("zoomParam") -> ActionZoomSeg(value)
            it.startsWith("gimbalEvenlyRotateParam") -> ActionGimbalEvenlyRotateSeg(value)
            else -> IgnoreSeg(it)
        }
    }
    return segList
}




