package com.dji.wpmzsdk.common.utils.kml.transfrom


import com.dji.industry.pilot.data.cache.model.EdgePointModelGreenDao
import com.dji.industry.pilot.data.cache.model.MappingMissionModelGreenDao
import com.dji.industry.pilot.data.cache.model.MappingWaylineModelGreenDao

import com.dji.industry.pilot.mission.transfrom.*
import com.dji.wpmzsdk.common.utils.kml.GpsUtils
import com.dji.wpmzsdk.common.utils.kml.data.MissionType
import com.dji.wpmzsdk.common.utils.kml.model.MappingCameraType
import com.dji.wpmzsdk.common.utils.kml.model.MissionInfoModelGreenDao

import dji.sdk.wpmz.value.mission.*

/**
 * Description :
 *
 * @filename : MappingMissionEntityTransform
 * @author : devin.xu
 * @time : 2022/11/30
 *
 * Copyright (c) 2016, DJI All Rights Reserved.
 **/
class MappingMissionEntityTransform {

    private val mFinishActionTransform = WaylineFinishActionTransform()
    private val mAltitudeModeTransform = MappingAltitudeModeTransform()
    private val mShootTypeTransform = MappingShootTypeTransform()
    private val mFocusModeTransform = WaylineMappingFocusModeTransform()
    private val mEchoModeTransform = WaylineEchoModeTransform()
    private val mWaylineScanModeTransform = WaylineScanModeTransform()
    private val mMappingCameraTransform = MappingPayloadInfoTransform()
    private val mWaylinePayloadParamTransform = WaylinePaylodParamTransform()

    fun transFrom(model: MappingMissionModelGreenDao): MissionTransformData {
        val mappingWaylineModel = model.mappingWayline!!
        val edgePointModels = model.edgePoints!!
        val template = WaylineTemplate()
        template.autoFlightSpeed = mappingWaylineModel.speed.toDouble()
        template.transitionalSpeed = mappingWaylineModel.takeoffSpeed.toDouble()
        template.coordinateParam = transCoordinateParamFrom(mappingWaylineModel)
        template.payloadParam = transPayloadParamsFrom(mappingWaylineModel)
        if (model.missionInfo!!.type == MissionType.Mapping2D) {
            template.templateType = WaylineTemplateType.MAPPING2D
            template.mapping2DInfo = trans2DTemplateFrom(mappingWaylineModel, edgePointModels)
        } else {
            template.templateType = WaylineTemplateType.MAPPING3D
            template.mapping3DInfo = trans3DTemplateFrom(mappingWaylineModel, edgePointModels)
        }
        val waylinePayloadParam: List<WaylinePayloadParam> =
            transPayloadParamsFrom(mappingWaylineModel)
        template.payloadParam = waylinePayloadParam
        val missionConfig: WaylineMissionConfig = transConfigFrom(mappingWaylineModel)
        val transformData = MissionTransformData()
        transformData.mission = createWaylineMission(model.missionInfo!!)
        transformData.setTemplate(template)
        transformData.missionConfig = missionConfig
        transformData.executeMissionConfig = createExecuteConfig(missionConfig)
        return transformData
    }

    private fun createWaylineMission(infoModel: MissionInfoModelGreenDao): WaylineMission {
        val mission = WaylineMission()
        mission.updateTime = infoModel.updateTime.toDouble()
        mission.createTime = infoModel.createTime.toDouble()
        return mission
    }

    private fun transConfigFrom(waylineModel: MappingWaylineModelGreenDao): WaylineMissionConfig {
        val config = WaylineMissionConfig()
        config.flyToWaylineMode = WaylineFlyToWaylineMode.SAFELY

        config.exitOnRCLostBehavior = WaylineExitOnRCLostBehavior.EXCUTE_RC_LOST_ACTION
        config.exitOnRCLostType = WaylineExitOnRCLostAction.GO_BACK
        config.globalTransitionalSpeed = waylineModel.takeoffSpeed.toDouble()
        config.isTakeOffPositionRefSet = false
        config.isGlobalRTHHeightSet = false
        config.finishAction = mFinishActionTransform.transFrom(waylineModel.actionOnFinish)

        if (waylineModel.camera != null) {
            val cameraType = MappingCameraType.find(waylineModel.camera!!.name)
            config.droneInfo = getDroneInfoFromCamera(cameraType)
        }
        val payloadInfos: MutableList<WaylinePayloadInfo> = ArrayList()
        if (waylineModel.camera != null) {
            payloadInfos.add(mMappingCameraTransform.transPayloadInfoFrom(waylineModel.camera!!))
        }
        config.payloadInfo = payloadInfos
        return config
    }

    private fun transCoordinateParamFrom(waylineModel: MappingWaylineModelGreenDao): WaylineCoordinateParam {
        val coordinateParam = WaylineCoordinateParam()
        coordinateParam.coordinateMode = WaylineCoordinateMode.WGS84
        coordinateParam.isWaylinePositioningTypeSet = false
        coordinateParam.altitudeMode = mAltitudeModeTransform.transFrom(waylineModel.altitudeMode)
        if (!waylineModel.enableDsm) {
            val isAbsolute = waylineModel.altitudeMode == MappingAltitudeModeTransform.ABSOLUTE
            val globalShootHeight: Double = if (isAbsolute) waylineModel.relativeDistance.toDouble() else (waylineModel.altitude - waylineModel.relativeDistance).toDouble()
            coordinateParam.globalShootHeight = globalShootHeight
            coordinateParam.isGlobalShootHeightSet = true
        } else {
            coordinateParam.isSurfaceFollowParamSet = true
            val surfaceFollowParam = SurfaceFollowParam()
            if (waylineModel.dsmPath != null) {
                surfaceFollowParam.dsmFiles = waylineModel.dsmPath
            } else {
                surfaceFollowParam.dsmFiles = mutableListOf()
            }
            surfaceFollowParam.surfaceFollowModeEnable = waylineModel.enableDsm
            surfaceFollowParam.surfaceRelativeHeight = waylineModel.dsmAltitude.toDouble()
            coordinateParam.surfaceFollowParam = surfaceFollowParam
            coordinateParam.globalShootHeight = waylineModel.dsmAltitude.toDouble()
            coordinateParam.isGlobalShootHeightSet = true
        }
        return coordinateParam
    }

    private fun transPayloadParamsFrom(waylineModel: MappingWaylineModelGreenDao): List<WaylinePayloadParam> {
        val payloadParams: MutableList<WaylinePayloadParam> = mutableListOf()
        val payloadParam = WaylinePayloadParam()
        payloadParam.focusMode = mFocusModeTransform.transFrom(waylineModel.focusMode)
        payloadParam.isDewarpingEnableSet = true
        payloadParam.dewarpingEnable = waylineModel.enableDewarping
        payloadParam.isModelColoringEnableSet = true
        payloadParam.modelColoringEnable = waylineModel.needVariegation
        payloadParam.returnMode = mEchoModeTransform.transFrom(waylineModel.echoMode)
        payloadParam.samplingRate = mWaylinePayloadParamTransform.transSampleRateFrom(
            waylineModel.sampleRate,
            payloadParam.returnMode
        )
        payloadParam.scanningMode = mWaylineScanModeTransform.transFrom(waylineModel.scanMode)
        if (waylineModel.camera != null) {
            val cameraType = MappingCameraType.find(waylineModel.camera!!.name)
            payloadParam.imageFormat =
                mWaylinePayloadParamTransform.transImageFormatFrom(cameraType)
            if (!MappingCameraType.isEP600(cameraType)) {
                payloadParam.focusMode = WaylinePayloadFocusMode.UNKNOWN
            }
        }
        payloadParams.add(payloadParam)
        return payloadParams
    }

    private fun trans2DTemplateFrom(
        waylineModel: MappingWaylineModelGreenDao,
        edgePointModels: List<EdgePointModelGreenDao>
    ): WaylineTemplateMapping2DInfo {
        val template2DInfo = WaylineTemplateMapping2DInfo()
        template2DInfo.coordinates = transEdgePoint(edgePointModels)
        template2DInfo.direction = waylineModel.direction
        template2DInfo.margin = waylineModel.margin.toDouble()
        template2DInfo.caliFlightEnable = waylineModel.enableCalibrate
        template2DInfo.height = waylineModel.altitude.toDouble()
        template2DInfo.ellipsoidHeight = waylineModel.altitude.toDouble()
        if (waylineModel.altitudeMode == 1 /*AltitudeMode.ABSOLUTE*/ && edgePointModels.isNotEmpty()) {
            val firstPoint = edgePointModels[0]
            template2DInfo.height = GpsUtils.egm96Altitude(waylineModel.altitude.toDouble(), firstPoint.latitude, firstPoint.longitude)
        }

        val overlap = WaylineOverlap()
        overlap.isOrthoCameraOverlapHSet = true
        overlap.orthoCameraOverlapH = waylineModel.overlapH
        overlap.isOrthoCameraOverlapWSet = true
        // PPAL实际计算目前都只根据CameraOverlap计算。LidarOverlap目前仅用来存储激光的重叠率数据，用于设置和显示。
        var overlapW = waylineModel.overlapW
        if (isLidarCamera(waylineModel.camera)) {
            overlapW = getVisibleOverlapWValue(waylineModel.overlapW)
        }
        overlap.orthoCameraOverlapW = overlapW
        overlap.isOrthoLidarOverlapHSet = true
        overlap.orthoLidarOverlapH = waylineModel.overlapH
        overlap.isOrthoLidarOverlapWSet = true
        overlap.orthoLidarOverlapW = waylineModel.overlapW
        template2DInfo.overlap = overlap
        template2DInfo.elevationOptimizeEnable = waylineModel.elevationOptimize
        template2DInfo.isSmartObliqueGimbalPitchSet = true
        template2DInfo.smartObliqueGimbalPitch = waylineModel.fiveWayPoseWithGimbalPitch
        template2DInfo.shootType = mShootTypeTransform.transFrom(waylineModel.photoMode)
        template2DInfo.smartObliqueEnable = waylineModel.fiveWayPose
        return template2DInfo
    }

    private fun trans3DTemplateFrom(
        waylineModel: MappingWaylineModelGreenDao,
        edgePointModels: List<EdgePointModelGreenDao>
    ): WaylineTemplateMapping3DInfo? {
        val template3DInfo = WaylineTemplateMapping3DInfo()
        template3DInfo.coordinates = transEdgePoint(edgePointModels)
        template3DInfo.direction = waylineModel.direction
        template3DInfo.margin = waylineModel.margin.toDouble()
        template3DInfo.height = waylineModel.altitude.toDouble()
        if (waylineModel.altitudeMode == 1 /*AltitudeMode.ABSOLUTE*/) {
            template3DInfo.ellipsoidHeight = waylineModel.altitude.toDouble()
            if (edgePointModels.isNotEmpty()) {
                val firstPoint = edgePointModels[0]
                template3DInfo.height = GpsUtils.egm96Altitude(waylineModel.altitude.toDouble(), firstPoint.latitude, firstPoint.longitude)
                template3DInfo.ellipsoidHeight = waylineModel.altitude.toDouble()
            }
        } else {
            template3DInfo.ellipsoidHeight = waylineModel.altitude.toDouble()
        }
        val overlap = WaylineOverlap()
        overlap.isOrthoCameraOverlapHSet = true
        overlap.orthoCameraOverlapH = waylineModel.overlapH
        overlap.isOrthoCameraOverlapWSet = true
        overlap.orthoCameraOverlapW = waylineModel.overlapW
        overlap.isOrthoLidarOverlapHSet = true
        overlap.orthoLidarOverlapH = waylineModel.overlapH
        overlap.isOrthoLidarOverlapWSet = true
        overlap.orthoLidarOverlapW = waylineModel.overlapW
        overlap.isInclinedCameraOverlapHSet = true
        overlap.inclinedCameraOverlapH = waylineModel.inclineOverlapH
        overlap.isInclinedCameraOverlapWSet = true
        overlap.inclinedCameraOverlapW = waylineModel.inclineOverlapW
        overlap.isInclinedLidarOverlapHSet = true
        overlap.inclinedLidarOverlapH = waylineModel.inclineOverlapH
        overlap.isInclinedLidarOverlapWSet = true
        overlap.inclinedLidarOverlapW = waylineModel.inclineOverlapW
        template3DInfo.overlap = overlap
        template3DInfo.caliFlightEnable = waylineModel.enableCalibrate
        template3DInfo.inclinedFlightSpeed = waylineModel.inclineSpeed.toDouble()
        template3DInfo.inclinedGimbalPitch = waylineModel.gimbalPitch
        template3DInfo.shootType = mShootTypeTransform.transFrom(waylineModel.photoMode)
        template3DInfo.isShootTypeSet = true
        return template3DInfo
    }

    private fun transEdgePoint(points: List<EdgePointModelGreenDao>): MutableList<WaylineLocationCoordinate3D> {
        val edgePoint: MutableList<WaylineLocationCoordinate3D> = mutableListOf()
        for (item in points) {
            val location = WaylineLocationCoordinate3D()
            location.latitude = item.latitude
            location.longitude = item.longitude
            edgePoint.add(location)
        }
        return edgePoint
    }

}