package com.dji.wpmzsdk.common.utils.kml.transfrom

import com.dji.industry.pilot.data.cache.model.MappingCameraModelGreenDao
import com.dji.wpmzsdk.common.utils.kml.model.MappingCameraModel
import com.dji.wpmzsdk.common.utils.kml.model.MappingCameraType

import dji.sdk.wpmz.value.mission.WaylinePayloadInfo
import dji.sdk.wpmz.value.mission.WaylinePayloadParam
import dji.sdk.wpmz.value.mission.WaylinePayloadType

/**
 * Description :
 *
 * @filename : MappingPayloadInfoTransform
 * @author : devin.xu
 * @time : 2021/9/15
 *
 * Copyright (c) 2016, DJI All Rights Reserved.
 **/
class MappingPayloadInfoTransform {

    companion object SubType {
        const val P1_24MM: Int = 0
        const val P1_35MM: Int = 1
        const val P1_50MM: Int = 2

        const val L1_VISUAL: Int = 0
        const val L1_LIDAR: Int = 1

        const val H20_WIDE: Int = 0
        const val H20_ZOOM: Int = 1
        const val H20_IR: Int = 2

        const val X7_24MM: Int = 0;
        const val X7_35MM: Int = 1;

        const val M30_WIDE: Int = 0
        const val M30_ZOOM: Int = 1
        const val M30_IR: Int = 2

        const val M3_WIDE: Int = 0
        const val M3_IR: Int = 2
        const val M3_NARROWBAND: Int = 3
    }

    fun transPayloadInfoFrom(cameraModel: MappingCameraModelGreenDao) : WaylinePayloadInfo {
        val payloadInfo = WaylinePayloadInfo()
        val cameraType = MappingCameraType.find(cameraModel.name)
        val payloadPair = transPayloadTypeFrom(cameraType)
        payloadInfo.payloadType = payloadPair.first
        payloadInfo.payloadSubType = payloadPair.second
        if (cameraType == MappingCameraType.OTHER || cameraType == MappingCameraType.SHARE_102S) {
            payloadInfo.isSetThirdPartyCameraParam = true
            payloadInfo.focalLength = cameraModel.focalLength.toDouble()
            payloadInfo.sensorW = cameraModel.sensorWidth.toDouble()
            payloadInfo.sensorH = cameraModel.sensorHeight.toDouble()
            payloadInfo.imageW = cameraModel.imageWidth
            payloadInfo.imageH = cameraModel.imageHeight
            payloadInfo.minShootInterval = cameraModel.shotInterval.toDouble()
            payloadInfo.payloadName = cameraModel.name
        }
        return payloadInfo
    }

    fun transPayloadInfoFrom(cameraModel: MappingCameraModel) : WaylinePayloadInfo {
        val payloadInfo = WaylinePayloadInfo()
        val cameraType = MappingCameraType.find(cameraModel.name)
        val payloadPair = transPayloadTypeFrom(cameraType)
        payloadInfo.payloadType = payloadPair.first
        payloadInfo.payloadSubType = payloadPair.second
        if (cameraType == MappingCameraType.OTHER || cameraType == MappingCameraType.SHARE_102S) {
            payloadInfo.isSetThirdPartyCameraParam = true
            payloadInfo.focalLength = cameraModel.focalLength.toDouble()
            payloadInfo.sensorW = cameraModel.sensorWidth.toDouble()
            payloadInfo.sensorH = cameraModel.sensorHeight.toDouble()
            payloadInfo.imageW = cameraModel.imageWidth
            payloadInfo.imageH = cameraModel.imageHeight
            payloadInfo.minShootInterval = cameraModel.shotInterval.toDouble()
            payloadInfo.payloadName = cameraModel.name
        }
        return payloadInfo
    }

    private fun transPayloadTypeFrom(cameraType: MappingCameraType) : Pair<WaylinePayloadType, Int> {
        return when (cameraType) {
            MappingCameraType.PM320 -> Pair(WaylinePayloadType.PM320_DUAL, M30_WIDE)
            MappingCameraType.PM320T_WIDE -> Pair(WaylinePayloadType.PM320_TRIPLE, M30_WIDE)
            MappingCameraType.PM320_IR -> Pair(WaylinePayloadType.PM320_TRIPLE, M30_IR)
            MappingCameraType.PM320_WIDE_IR -> Pair(WaylinePayloadType.PM320_TRIPLE, M30_IR)
            MappingCameraType.WM265_WIDE -> Pair(WaylinePayloadType.WM265E, M3_WIDE)
            MappingCameraType.WM265_T_WIDE -> Pair(WaylinePayloadType.WM265T, M3_WIDE)
            MappingCameraType.WM265_IR,
            MappingCameraType.WM265_WIDE_IR -> Pair(WaylinePayloadType.WM265T, M3_IR)
            MappingCameraType.WM265_M_WIDE -> Pair(WaylinePayloadType.WM265M, M3_WIDE)
            MappingCameraType.WM265_M_NARROWBAND,
            MappingCameraType.WM265_M_WIDE_NARROWBAND -> Pair(WaylinePayloadType.WM265M, M3_NARROWBAND)
            MappingCameraType.EP600_24MM -> Pair(WaylinePayloadType.ZENMUSE_P1, P1_24MM)
            MappingCameraType.EP600_35MM -> Pair(WaylinePayloadType.ZENMUSE_P1, P1_35MM)
            MappingCameraType.EP600_50MM -> Pair(WaylinePayloadType.ZENMUSE_P1, P1_50MM)
            MappingCameraType.EP800_LIDAR -> Pair(WaylinePayloadType.ZENMUSE_L1, L1_LIDAR)
            MappingCameraType.EP800_VISUAL -> Pair(WaylinePayloadType.ZENMUSE_L1, L1_VISUAL)
            MappingCameraType.GD610 -> Pair(WaylinePayloadType.ZENMUSE_H20, H20_WIDE)
            MappingCameraType.GD610T_WIDE -> Pair(WaylinePayloadType.ZENMUSE_H20T, H20_WIDE)
            MappingCameraType.GD610_IR -> Pair(WaylinePayloadType.ZENMUSE_H20T, H20_IR)
            MappingCameraType.GD610_WIDE_IR -> Pair(WaylinePayloadType.ZENMUSE_H20T, H20_IR)
            MappingCameraType.GD612_WIDE -> Pair(WaylinePayloadType.ZENMUSE_H20N, H20_WIDE)
            MappingCameraType.GD612_IR -> Pair(WaylinePayloadType.ZENMUSE_H20N, H20_IR)
            MappingCameraType.GD612_WIDE_IR -> Pair(WaylinePayloadType.ZENMUSE_H20N, H20_IR)
            MappingCameraType.P4 -> Pair(WaylinePayloadType.P4, 0)
            MappingCameraType.P4A -> Pair(WaylinePayloadType.P4A, 0)
            MappingCameraType.P4P -> Pair(WaylinePayloadType.P4P_PRO, 0)
            MappingCameraType.P4PV2 -> Pair(WaylinePayloadType.P4P_PRO_V2, 0)
            MappingCameraType.P4R -> Pair(WaylinePayloadType.P4R, 0)
            MappingCameraType.WM245_ZOOM -> Pair(WaylinePayloadType.WM245_ZOOM, 0)
            MappingCameraType.WM246 -> Pair(WaylinePayloadType.WM246, 0)
            MappingCameraType.X7_24MM -> Pair(WaylinePayloadType.ZENMUSE_X7, X7_24MM)
            MappingCameraType.X7_35MM -> Pair(WaylinePayloadType.ZENMUSE_X7, X7_35MM)
            else -> Pair(WaylinePayloadType.UNKNOWN, 0)
        }
    }

    fun transPayloadInfoTo(payloadInfo: WaylinePayloadInfo, payloadParam: WaylinePayloadParam) : MappingCameraModel {
        val cameraModel = MappingCameraModel()
        val doubleImage = payloadParam.imageFormat.size > 1
        var cameraType = transPayloadTypeTo(payloadInfo.payloadType, payloadInfo.payloadSubType, doubleImage)
        if (payloadInfo.payloadName == MappingCameraType.SHARE_102S.nameStr) {
            cameraType = MappingCameraType.SHARE_102S
        }

        if (payloadInfo.isSetThirdPartyCameraParam) {
            cameraModel.focalLength = payloadInfo.focalLength.toFloat()
            cameraModel.sensorWidth = payloadInfo.sensorW.toFloat()
            cameraModel.sensorHeight = payloadInfo.sensorH.toFloat()
            cameraModel.imageWidth = payloadInfo.imageW
            cameraModel.imageHeight = payloadInfo.imageH
            cameraModel.shotInterval = payloadInfo.minShootInterval.toFloat()
            cameraModel.name = payloadInfo.payloadName
        } else {
            cameraModel.focalLength = cameraType.focalLength
            cameraModel.sensorWidth = cameraType.sensorW
            cameraModel.sensorHeight = cameraType.sensorH
            cameraModel.imageWidth = cameraType.imageWidth
            cameraModel.imageHeight = cameraType.imageHeight
            cameraModel.name = cameraType.nameStr
        }

        return cameraModel
    }

    fun transPayloadTypeTo(payloadType: WaylinePayloadType, subType: Int, doubleImage: Boolean): MappingCameraType {
        var cameraType = trans265PayloadTypeTo(payloadType, subType, doubleImage)

        if (cameraType == MappingCameraType.OTHER) {
            cameraType = transPM430PayloadTypeTo(payloadType, subType, doubleImage)
        }

        if (cameraType == MappingCameraType.OTHER) {
            cameraType = when {
                payloadType == WaylinePayloadType.PM320_DUAL && subType == M30_WIDE -> MappingCameraType.PM320
                payloadType == WaylinePayloadType.PM320_TRIPLE && subType == M30_WIDE -> MappingCameraType.PM320T_WIDE
                payloadType == WaylinePayloadType.PM320_TRIPLE && subType == M30_IR && !doubleImage -> MappingCameraType.PM320_IR
                payloadType == WaylinePayloadType.PM320_TRIPLE && subType == M30_IR && doubleImage -> MappingCameraType.PM320_WIDE_IR
                else -> transOtherPayloadTypeTo(payloadType, subType)
            }
        }
        return cameraType
    }

    private fun transPM430PayloadTypeTo(payloadType: WaylinePayloadType, subType: Int, doubleImage: Boolean): MappingCameraType {
        return when {
            payloadType == WaylinePayloadType.ZENMUSE_P1 && subType == P1_24MM -> MappingCameraType.EP600_24MM
            payloadType == WaylinePayloadType.ZENMUSE_P1 && subType == P1_35MM -> MappingCameraType.EP600_35MM
            payloadType == WaylinePayloadType.ZENMUSE_P1 && subType == P1_50MM -> MappingCameraType.EP600_50MM
            payloadType == WaylinePayloadType.ZENMUSE_L1 && subType == L1_LIDAR -> MappingCameraType.EP800_LIDAR
            payloadType == WaylinePayloadType.ZENMUSE_L1 && subType == L1_VISUAL -> MappingCameraType.EP800_VISUAL
            payloadType == WaylinePayloadType.ZENMUSE_H20 && subType == H20_WIDE -> MappingCameraType.GD610
            payloadType == WaylinePayloadType.ZENMUSE_H20T && subType == H20_WIDE -> MappingCameraType.GD610T_WIDE
            payloadType == WaylinePayloadType.ZENMUSE_H20T && subType == H20_IR && !doubleImage -> MappingCameraType.GD610_IR
            payloadType == WaylinePayloadType.ZENMUSE_H20T && subType == H20_IR && doubleImage -> MappingCameraType.GD610_WIDE_IR
            payloadType == WaylinePayloadType.ZENMUSE_H20N && subType == H20_WIDE -> MappingCameraType.GD612_WIDE
            payloadType == WaylinePayloadType.ZENMUSE_H20N && subType == H20_IR && !doubleImage -> MappingCameraType.GD612_IR
            payloadType == WaylinePayloadType.ZENMUSE_H20N && subType == H20_IR && doubleImage -> MappingCameraType.GD612_WIDE_IR
            else -> MappingCameraType.OTHER
        }
    }

    private fun transOtherPayloadTypeTo(payloadType: WaylinePayloadType, subType: Int): MappingCameraType {
        return when {
            payloadType == WaylinePayloadType.P4 -> MappingCameraType.P4
            payloadType == WaylinePayloadType.P4P_PRO -> MappingCameraType.P4P
            payloadType == WaylinePayloadType.P4P_PRO_V2 -> MappingCameraType.P4PV2
            payloadType == WaylinePayloadType.P4A -> MappingCameraType.P4A
            payloadType == WaylinePayloadType.P4R -> MappingCameraType.P4R
            payloadType == WaylinePayloadType.WM245_ZOOM -> MappingCameraType.WM245_ZOOM
            payloadType == WaylinePayloadType.WM246 -> MappingCameraType.WM246
            payloadType == WaylinePayloadType.ZENMUSE_X7 && subType == X7_24MM -> MappingCameraType.X7_24MM
            payloadType == WaylinePayloadType.ZENMUSE_X7 && subType == X7_35MM -> MappingCameraType.X7_35MM
            else -> MappingCameraType.OTHER
        }
    }

    private fun trans265PayloadTypeTo(payloadType: WaylinePayloadType, subType: Int, doubleImage: Boolean): MappingCameraType {
        return when {
            payloadType == WaylinePayloadType.WM265E && subType == M3_WIDE -> MappingCameraType.WM265_WIDE
            payloadType == WaylinePayloadType.WM265T -> if (subType == M3_IR) {
                if (doubleImage) MappingCameraType.WM265_WIDE_IR else MappingCameraType.WM265_IR
            } else {
                MappingCameraType.WM265_T_WIDE
            }
            payloadType == WaylinePayloadType.WM265M -> if (subType == M3_NARROWBAND) {
                if (doubleImage) MappingCameraType.WM265_M_WIDE_NARROWBAND else MappingCameraType.WM265_M_NARROWBAND
            } else {
                MappingCameraType.WM265_M_WIDE
            }
            else -> MappingCameraType.OTHER
        }
    }

}