package com.dji.wpmzsdk.common.utils.kml.transfrom


import com.dji.wpmzsdk.common.utils.kml.model.MappingCameraType
import com.dji.wpmzsdk.common.utils.kml.model.PayloadConfigInfoModel
import com.dji.wpmzsdk.common.utils.kml.model.ScanMode

import dji.sdk.wpmz.value.mission.*

/**
 * Description :
 *
 * @filename : WaylinePaylodParamTransform
 * @author : devin.xu
 * @time : 2021/9/14
 *
 * Copyright (c) 2016, DJI All Rights Reserved.
 **/
class WaylinePaylodParamTransform : ModelTransform<PayloadConfigInfoModel, WaylinePayloadParam> {

    override fun transFrom(configInfoModel: PayloadConfigInfoModel): WaylinePayloadParam {
        val payloadParam = WaylinePayloadParam()
        payloadParam.meteringMode = transMeteringModeFrom(configInfoModel.meteringMode)
        payloadParam.isDewarpingEnableSet = true
        payloadParam.dewarpingEnable = configInfoModel.isEnableDewarp
        payloadParam.returnMode = transEchoModeFrom(configInfoModel.echoMode)
        payloadParam.samplingRate = transSampleRateFrom(configInfoModel.sampleRate, payloadParam.returnMode)
        payloadParam.scanningMode = transScanModeFrom(configInfoModel.scanMode)
        payloadParam.isModelColoringEnableSet = true
        payloadParam.modelColoringEnable = configInfoModel.isNeedVariegation
        return payloadParam
    }

    override fun transTo(r: WaylinePayloadParam): PayloadConfigInfoModel {
        val configInfoModel = PayloadConfigInfoModel()
        configInfoModel.meteringMode = transMeteringModeTo(r.meteringMode)
        configInfoModel.isEnableDewarp = r.dewarpingEnable ?: false
        configInfoModel.echoMode = transEchoModeTo(r.returnMode)
        configInfoModel.sampleRate = transSampleRateTo(r.samplingRate, r.returnMode)
        configInfoModel.scanMode = transScanModeTo(r.scanningMode)
        configInfoModel.isNeedVariegation = r.modelColoringEnable ?: false
        return configInfoModel
    }

    private fun transMeteringModeFrom(meteringMode: Int) : WaylinePayloadMeteringMode {
        return when (meteringMode) {
            PayloadConfigInfoModel.METERING_MODE_POINT -> WaylinePayloadMeteringMode.SPOT
            PayloadConfigInfoModel.METERING_MODE_GLOBAL -> WaylinePayloadMeteringMode.AVERAGE
            else -> WaylinePayloadMeteringMode.AVERAGE
        }
    }

    private fun transMeteringModeTo(meteringMode: WaylinePayloadMeteringMode) : Int {
        return when (meteringMode) {
            WaylinePayloadMeteringMode.AVERAGE -> PayloadConfigInfoModel.METERING_MODE_GLOBAL
            WaylinePayloadMeteringMode.SPOT -> PayloadConfigInfoModel.METERING_MODE_POINT
            else -> PayloadConfigInfoModel.METERING_MODE_GLOBAL
        }
    }

    private fun transEchoModeFrom(echoMode: Int) : WaylinePayloadReturnMode {
        return when (echoMode) {
            PayloadConfigInfoModel.ECHO_MODE_TREBLE -> WaylinePayloadReturnMode.TRIPLE_RETURN
            PayloadConfigInfoModel.ECHO_MODE_DOUBLE -> WaylinePayloadReturnMode.DUAL_RETURN
            PayloadConfigInfoModel.ECHO_MODE_SINGLE_STRONG -> WaylinePayloadReturnMode.SINGLE_RETURN_STRONGEST
            PayloadConfigInfoModel.ECHO_MODE_SINGLE_FIRST -> WaylinePayloadReturnMode.SINGLE_RETURN_FIRST
            else -> WaylinePayloadReturnMode.SINGLE_RETURN_STRONGEST
        }
    }

    private fun transEchoModeTo(echoMode: WaylinePayloadReturnMode) : Int {
        return when (echoMode) {
            WaylinePayloadReturnMode.TRIPLE_RETURN -> PayloadConfigInfoModel.ECHO_MODE_TREBLE
            WaylinePayloadReturnMode.DUAL_RETURN -> PayloadConfigInfoModel.ECHO_MODE_DOUBLE
            WaylinePayloadReturnMode.SINGLE_RETURN_STRONGEST -> PayloadConfigInfoModel.ECHO_MODE_SINGLE_STRONG
            WaylinePayloadReturnMode.SINGLE_RETURN_FIRST -> PayloadConfigInfoModel.ECHO_MODE_SINGLE_FIRST
            else -> PayloadConfigInfoModel.ECHO_MODE_SINGLE_STRONG
        }
    }

//    <string-array name="lidar_sample_rate_general">
//    <item>240KHz</item>
//    <item>180KHz</item>
//    <item>120KHz</item>
//    <item>60KHz</item>
//    </string-array>
//
//    <string-array name="lidar_sample_rate_triple">
//    <item>160KHz</item>
//    <item>120KHz</item>
//    <item>80KHz</item>
//    </string-array>

    fun transSampleRateFrom(sampleRate: Int, echoMode: WaylinePayloadReturnMode) : WaylinePayloadRate {
        return when {
            echoMode == WaylinePayloadReturnMode.TRIPLE_RETURN && sampleRate == 0 -> WaylinePayloadRate.RATE_160KHz
            echoMode == WaylinePayloadReturnMode.TRIPLE_RETURN && sampleRate == 1 -> WaylinePayloadRate.RATE_120KHz
            echoMode == WaylinePayloadReturnMode.TRIPLE_RETURN && sampleRate == 2 -> WaylinePayloadRate.RATE_80KHz
            sampleRate == 0 -> WaylinePayloadRate.RATE_240KHz
            sampleRate == 1 -> WaylinePayloadRate.RATE_180KHz
            sampleRate == 2 -> WaylinePayloadRate.RATE_120KHz
            sampleRate == 3 -> WaylinePayloadRate.RATE_60KHz
            else -> WaylinePayloadRate.UNKNOWN
        }
    }

    fun transSampleRateTo(sampleRate: WaylinePayloadRate, echoMode: WaylinePayloadReturnMode) : Int {
        return when {
            echoMode == WaylinePayloadReturnMode.TRIPLE_RETURN && sampleRate == WaylinePayloadRate.RATE_160KHz -> 0
            echoMode == WaylinePayloadReturnMode.TRIPLE_RETURN && sampleRate == WaylinePayloadRate.RATE_120KHz -> 1
            echoMode == WaylinePayloadReturnMode.TRIPLE_RETURN && sampleRate == WaylinePayloadRate.RATE_80KHz -> 2
            sampleRate == WaylinePayloadRate.RATE_240KHz -> 0
            sampleRate == WaylinePayloadRate.RATE_180KHz -> 1
            sampleRate == WaylinePayloadRate.RATE_120KHz -> 2
            sampleRate == WaylinePayloadRate.RATE_60KHz -> 3
            else -> 0
        }
    }

    private fun transScanModeFrom(scanMode: Int) : WaylinePayloadScanningMode {
        return when (scanMode) {
            ScanMode.REPEAT.value() -> WaylinePayloadScanningMode.REPETITIVE
            ScanMode.NON_REPEAT.value() -> WaylinePayloadScanningMode.NON_REPETITIVE
            else -> WaylinePayloadScanningMode.REPETITIVE
        }
    }

    private fun transScanModeTo(scanMode: WaylinePayloadScanningMode) : Int {
        return when (scanMode) {
            WaylinePayloadScanningMode.REPETITIVE -> ScanMode.REPEAT.value()
            WaylinePayloadScanningMode.NON_REPETITIVE -> ScanMode.NON_REPEAT.value()
            else -> ScanMode.NON_REPEAT.value()
        }
    }

    fun transImageFormatFrom(cameraType: MappingCameraType): List<CameraLensType> {
        val imageFormat = mutableListOf<CameraLensType>()
        when (cameraType) {
            MappingCameraType.GD610,
            MappingCameraType.GD610T_WIDE,
            MappingCameraType.GD612_WIDE,
            MappingCameraType.PM320,
            MappingCameraType.PM320T_WIDE,
            MappingCameraType.WM265_WIDE,
            MappingCameraType.WM265_T_WIDE,
            MappingCameraType.WM265_M_WIDE -> imageFormat.add(CameraLensType.WIDE)
            MappingCameraType.GD610_IR,
            MappingCameraType.GD612_IR,
            MappingCameraType.PM320_IR,
            MappingCameraType.WM265_IR -> imageFormat.add(CameraLensType.IR)
            MappingCameraType.GD610_WIDE_IR,
            MappingCameraType.GD612_WIDE_IR,
            MappingCameraType.PM320_WIDE_IR,
            MappingCameraType.WM265_WIDE_IR -> {
                imageFormat.add(CameraLensType.WIDE)
                imageFormat.add(CameraLensType.IR)
            }
            MappingCameraType.WM265_M_NARROWBAND -> imageFormat.add(CameraLensType.NARROW_BAND)
            MappingCameraType.WM265_M_WIDE_NARROWBAND -> {
                imageFormat.add(CameraLensType.WIDE)
                imageFormat.add(CameraLensType.NARROW_BAND)
            }
            else -> Unit
        }
        return imageFormat.toList()
    }
}