package com.dji.wpmzsdk.common.utils.kml.transfrom


import com.dji.industry.pilot.data.cache.model.WaypointActionModelGreenDao
import com.dji.wpmzsdk.common.utils.kml.model.WaypointActionType
import com.dji.wpmzsdk.common.utils.kml.model.PreciseShotInfo
import dji.sdk.wpmz.value.mission.*

/**
 * Description :
 *
 * @filename : WaypointActionEntityTransform
 * @author : devin.xu
 * @time : 2022/11/30
 *
 * Copyright (c) 2016, DJI All Rights Reserved.
 **/
class WaypointActionEntityTransform {

    fun trans(actionModel: WaypointActionModelGreenDao, preciseInfos: List<PreciseShotInfo>): WaylineActionInfo? {
        return when (actionModel.actionType) {
            WaypointActionType.STAY -> transStay(actionModel)
            WaypointActionType.START_TIME_INTERVAL_SHOT, WaypointActionType.START_DISTANCE_INTERVAL_SHOT, WaypointActionType.STOP_INTERVAL_SHOT -> null
            WaypointActionType.START_TAKE_PHOTO -> transTakePhoto(actionModel)
            WaypointActionType.START_RECORD -> transStartRecord(actionModel)
            WaypointActionType.STOP_RECORD -> transStopRecord(actionModel)
            WaypointActionType.ROTATE_AIRCRAFT -> transRotateAircraft(actionModel)
            WaypointActionType.GIMBAL_PITCH -> transGimbalPitch(actionModel)
            WaypointActionType.GIMBAL_YAW -> transGimbalYaw(actionModel)
            WaypointActionType.CAMERA_ZOOM -> transCameraZoom(actionModel)
            WaypointActionType.PRECISE_SHOT -> transPreciseShot(actionModel, preciseInfos, true)
            WaypointActionType.NORMAL_PRECISE_SHOT -> transPreciseShot(actionModel, preciseInfos, false)
            WaypointActionType.CAMERA_MKDIR -> transCameraMkdir(actionModel)
            else -> transPSDKActions(actionModel.actionType, actionModel)
        }
    }

    private fun transPSDKActions(actionType: WaypointActionType?, actionModel: WaypointActionModelGreenDao): WaylineActionInfo? {
        return when (actionType) {
            WaypointActionType.PAYLOAD_BUTTON -> transPayloadButton(actionModel)
            WaypointActionType.PAYLOAD_SWITCH_ON -> transPayloadSwitchOn(actionModel)
            WaypointActionType.PAYLOAD_SWITCH_OFF -> transPayloadSwitchOff(actionModel)
            WaypointActionType.PAYLOAD_SEEK_VALUE -> transPayloadSeekValue(actionModel)
            WaypointActionType.START_POINT_CLOUD,
            WaypointActionType.PAUSE_POINT_CLOUD,
            WaypointActionType.CONTINUE_POINT_CLOUD,
            WaypointActionType.FINISH_POINT_CLOUD -> transPointCloudAction(actionType, actionModel)
            else -> null
        }
    }

    private fun transTakePhoto(actionModel: WaypointActionModelGreenDao): WaylineActionInfo {
        val info = WaylineActionInfo()
        info.actionType = WaylineActionType.TAKE_PHOTO
        val photoTypes: MutableList<CameraLensType> = ArrayList()

        val param = ActionTakePhotoParam(
            actionModel.cameraIndex,
            false,
            photoTypes,
            actionModel.mediaPathName
        )
        param.payloadPositionIndex = actionModel.cameraIndex
        info.takePhotoParam = param
        return info
    }

    private fun transStartRecord(actionModel: WaypointActionModelGreenDao): WaylineActionInfo {
        val info = WaylineActionInfo()
        info.actionType = WaylineActionType.START_RECORD
        val photoTypes: MutableList<CameraLensType> = ArrayList()

        val param = ActionStartRecordParam(
            actionModel.cameraIndex,
            false, photoTypes, actionModel.mediaPathName
        )
        info.startRecordParam = param
        return info
    }

    private fun transStopRecord(actionModel: WaypointActionModelGreenDao): WaylineActionInfo {
        val info = WaylineActionInfo()
        info.actionType = WaylineActionType.STOP_RECORD
        val param = ActionStopRecordParam()
        param.payloadPositionIndex = actionModel.cameraIndex
        info.stopRecordParam = param
        return info
    }

    private fun transStay(actionModel: WaypointActionModelGreenDao): WaylineActionInfo {
        val info = WaylineActionInfo()
        info.actionType = WaylineActionType.HOVER
        val param = ActionAircraftHoverParam()
        param.hoverTime = actionModel.getRealParam() / 1000.0
        info.aircraftHoverParam = param
        return info
    }

    private fun transRotateAircraft(actionModel: WaypointActionModelGreenDao): WaylineActionInfo {
        val info = WaylineActionInfo()
        info.actionType = WaylineActionType.ROTATE_YAW
        val param = ActionAircraftRotateYawParam()
        param.heading = actionModel.param.toDouble()
        param.pathMode = WaylineWaypointYawPathMode.FOLLOW_BAD_ARC
        info.aircraftRotateYawParam = param
        return info
    }

    private fun transGimbalPitch(actionModel: WaypointActionModelGreenDao): WaylineActionInfo {
        val info = WaylineActionInfo()
        info.actionType = WaylineActionType.GIMBAL_ROTATE
        val param = ActionGimbalRotateParam()
        param.enablePitch = true
        param.pitch = actionModel.getRealParam().toDouble()
        param.rotateMode = WaylineGimbalActuatorRotateMode.ABSOLUTE_ANGLE
        param.payloadPositionIndex = actionModel.cameraIndex
        info.gimbalRotateParam = param
        return info
    }

    private fun transGimbalYaw(actionModel: WaypointActionModelGreenDao): WaylineActionInfo {
        val info = WaylineActionInfo()
        info.actionType = WaylineActionType.GIMBAL_ROTATE
        val param = ActionGimbalRotateParam()
        param.enableYaw = true
        param.yaw = actionModel.getRealParam().toDouble()
        param.rotateMode = WaylineGimbalActuatorRotateMode.ABSOLUTE_ANGLE
        param.payloadPositionIndex = actionModel.cameraIndex
        info.gimbalRotateParam = param
        return info
    }

    private fun transCameraZoom(actionModel: WaypointActionModelGreenDao): WaylineActionInfo {
        val info = WaylineActionInfo()
        info.actionType = WaylineActionType.ZOOM
        val param = ActionZoomParam()
        param.focalLength = actionModel.getRealParam().toDouble()
        param.payloadPositionIndex = actionModel.cameraIndex
        info.zoomParam = param
        return info
    }

    private fun transPreciseShot(
        actionModel: WaypointActionModelGreenDao,
        preciseInfos: List<PreciseShotInfo>,
        frameValid: Boolean
    ): WaylineActionInfo {
        val info = WaylineActionInfo()
        info.actionType = WaylineActionType.ACCURATE_PHOTO
        val param = ActionAccuratePhotoParam()
        val shotInfo = preciseInfos.find { it.cacheFileName == actionModel.preciseInfoName }
        shotInfo?.also { preciseShotInfo ->
            val cameraInfo = WaylineActionAccurateShootPhotoCameraStateInformation()
            cameraInfo.focal_distance = preciseShotInfo.focalDistance
            cameraInfo.aF_position = preciseShotInfo.afPosition
            param.camera_infos = cameraInfo
            val deviceInfo = WaylineActionAccurateShootPhotoDeviceInformation()
            deviceInfo.camera_type = preciseShotInfo.cameraType
            deviceInfo.gimbal_port = preciseShotInfo.cameraIndex
            param.device_infos = deviceInfo
            val fileInfo = WaylineActionAccurateShootPhotoFileInformation()
            fileInfo.file_size = preciseShotInfo.fileSize.toInt()
            fileInfo.file_path = preciseShotInfo.cacheFileName
            fileInfo.file_md5 = preciseShotInfo.fileMD5
            fileInfo.fileSuffix = actionModel.mediaPathName
            param.file_infos = fileInfo
            val gimbalInfo = WaylineActionAccurateShootPhotoGimbalStateInformation()
            gimbalInfo.pitch = preciseShotInfo.gimbalPitch.toDouble()
            gimbalInfo.roll = preciseShotInfo.gimbalRoll.toDouble()
            gimbalInfo.yaw = preciseShotInfo.gimbalYaw.toDouble()
            param.gimbal_infos = gimbalInfo
            val imageSize = WaylineActionAccurateShootSize()
            imageSize.height = preciseShotInfo.imageHeight
            imageSize.width = preciseShotInfo.imageWidth
            param.image_size = imageSize
            val targetRect = WaylineActionAccurateShootPhotoTargetRect()
            targetRect.angle = preciseShotInfo.cropAngle.toDouble()
            val targetPos = WaylineActionAccurateShootPoint()
            targetPos.x = preciseShotInfo.cropPosX.toInt()
            targetPos.y = preciseShotInfo.cropPosY.toInt()
            targetRect.position = targetPos
            val cropSize = WaylineActionAccurateShootSize()
            cropSize.width = preciseShotInfo.cropWidth.toInt()
            cropSize.height = preciseShotInfo.cropHeight.toInt()
            targetRect.size = cropSize
            param.target_rect = targetRect
            param.heading = preciseShotInfo.droneYaw.toDouble()
            param.isFrameValid = frameValid
            param.actionUUID = preciseShotInfo.uuid
        }

        param.useGlobalPayloadLensIndex = false
        val photoTypes: MutableList<CameraLensType> = java.util.ArrayList()
        param.payloadLensIndex = photoTypes
        info.accuratePhotoParam = param
        return info
    }

    private fun transCameraMkdir(actionModel: WaypointActionModelGreenDao): WaylineActionInfo {
        val info = WaylineActionInfo()
        info.actionType = WaylineActionType.CUSTOM_DIR_NAME
        val param = ActionCustomDirNameParam()
        param.directoryName = actionModel.mediaPathName
        param.payloadPositionIndex = actionModel.cameraIndex
        info.customDirNameParam = param
        return info
    }

    private fun transPayloadButton(actionModel: WaypointActionModelGreenDao): WaylineActionInfo {
        val info = WaylineActionInfo()
        info.actionType = WaylineActionType.PSDK_ACTION
        val param = ActionPSDKParam()
        param.actionType = actionModel.payloadType
        param.actionIndex = actionModel.payloadIndex
        param.actionParam = 1.0
        param.payloadPositionIndex = actionModel.cameraIndex
        info.psdkParam = param
        return info
    }

    private fun transPayloadSwitchOn(actionModel: WaypointActionModelGreenDao): WaylineActionInfo {
        val info = WaylineActionInfo()
        info.actionType = WaylineActionType.PSDK_ACTION
        val param = ActionPSDKParam()
        param.actionType = actionModel.payloadType
        param.actionIndex = actionModel.payloadIndex
        param.actionParam = 1.0
        param.payloadPositionIndex = actionModel.cameraIndex
        info.psdkParam = param
        return info
    }

    private fun transPayloadSwitchOff(actionModel: WaypointActionModelGreenDao): WaylineActionInfo {
        val info = WaylineActionInfo()
        info.actionType = WaylineActionType.PSDK_ACTION
        val param = ActionPSDKParam()
        param.actionType = actionModel.payloadType
        param.actionIndex = actionModel.payloadIndex
        param.actionParam = 0.0
        param.payloadPositionIndex = actionModel.cameraIndex
        info.psdkParam = param
        return info
    }

    private fun transPayloadSeekValue(actionModel: WaypointActionModelGreenDao): WaylineActionInfo {
        val info = WaylineActionInfo()
        info.actionType = WaylineActionType.PSDK_ACTION
        val param = ActionPSDKParam()
        param.actionType = actionModel.payloadType
        param.actionIndex = actionModel.payloadIndex
        param.actionParam = actionModel.param.toDouble()
        param.payloadPositionIndex = actionModel.cameraIndex
        info.psdkParam = param
        return info
    }

    private fun transPointCloudAction(
        actionType: WaypointActionType,
        actionModel: WaypointActionModelGreenDao
    ): WaylineActionInfo {
        val info = WaylineActionInfo()
        info.actionType = WaylineActionType.RECORD_POINT_CLOUD
        val param = ActionRecordPointCloudParam()
        if (actionType == WaypointActionType.START_POINT_CLOUD) {
            param.pointCloudOperateType = WaylineActionRecordPointCloudOperateType.START_RECORD
        } else if (actionType == WaypointActionType.PAUSE_POINT_CLOUD) {
            param.pointCloudOperateType = WaylineActionRecordPointCloudOperateType.PAUSE_RECORD
        } else if (actionType == WaypointActionType.CONTINUE_POINT_CLOUD) {
            param.pointCloudOperateType = WaylineActionRecordPointCloudOperateType.RESUME_RECORD
        } else if (actionType == WaypointActionType.FINISH_POINT_CLOUD) {
            param.pointCloudOperateType = WaylineActionRecordPointCloudOperateType.STOP_RECORD
        }
        param.payloadPositionIndex = actionModel.cameraIndex
        info.recordPointCloudParam = param
        return info
    }


}