package com.dji.wpmzsdk.common.utils.kml.transfrom

import android.util.Pair

import com.dji.industry.pilot.data.cache.model.*

import com.dji.wpmzsdk.common.utils.kml.GpsUtils
import com.dji.wpmzsdk.common.utils.kml.data.DroneInfoModel
import com.dji.wpmzsdk.common.utils.kml.mission.RTKReferenceStationSource
import com.dji.wpmzsdk.common.utils.kml.mission.WaypointMissionHeadingMode
import com.dji.wpmzsdk.common.utils.kml.mission.WaypointV2MissionTypes
import com.dji.wpmzsdk.common.utils.kml.model.*

import dji.sdk.wpmz.value.mission.*

/**
 * Description :
 *
 * @filename : WaypointMissionEntityTransform
 * @author : devin.xu
 * @time : 2022/11/30
 *
 * Copyright (c) 2016, DJI All Rights Reserved.
 **/
class WaypointMissionEntityTransform {

    private val mFinishActionTransform = WaylineFinishActionTransform()
    private val mToFirstPointTransform = WaylineToFirstPointTransform()
    private val mWaylineRtkTypeTransform = WaylineRtkTypeTransform()
    private val mWaylineTurnModeTransform = WaylineTurnModeTransform()
    private val mWaylineHeadingModeTransform = WaylineHeadingModeTransform()
    private val mWaylineDroneTypeTransform = WaylineDroneTypeTransform()
    private val mWaylinePayloadInfoTransform = WaylinePayloadInfoTransform()
    private val mWaylinePaylodParamTransform = WaylinePaylodParamTransform()

    fun trans(waypointMission: WaypointMissionModelGreenDao): MissionTransformData {
        val waypointEntities = waypointMission.waypoints
        val waylineEntity = waypointMission.wayline!!

        val template = WaylineTemplate()
        template.templateId = 0
        template.templateType = WaylineTemplateType.WAYPOINT

        val waypointInfo: WaylineTemplateWaypointInfo = transWaylineTemplateFrom(waylineEntity, waypointEntities)
        template.waypointInfo = waypointInfo

        val coordinateParam: WaylineCoordinateParam = transCoordinateParamFrom(
            waypointMission.missionInfo!!.rtkStation,
            waylineEntity
        )
        template.coordinateParam = coordinateParam

        template.transitionalSpeed = waylineEntity.autoFlightSpeed.toDouble()
        template.autoFlightSpeed = waylineEntity.autoFlightSpeed.toDouble()

        val payloadParams: List<WaylinePayloadParam> = transPayloadParamsFrom(waylineEntity.droneInfo!!)
        template.payloadParam = payloadParams

        heightModeCheck(waypointMission, waypointInfo)

        val config: WaylineMissionConfig = transConfigFrom(waylineEntity)

        val data = MissionTransformData()
        data.setTemplate(template)
        data.missionConfig = config
        data.mission = createWaylineMission(waypointMission.missionInfo!!)
        data.executeMissionConfig = createExecuteConfig(config)
        return data
    }

    private fun heightModeCheck(waypointMission: WaypointMissionModelGreenDao, waypointInfo: WaylineTemplateWaypointInfo) {
        val droneHeightModel = waypointMission.wayline!!.droneInfo!!.droneHeight
        if (!droneHeightModel.isUseAbsolute) {
            waypointInfo.waypoints.forEach {
                if (it.useGlobalFlightHeight) {
                    it.ellipsoidHeight = waypointMission.wayline!!.altitude.toDouble()
                    it.height = it.ellipsoidHeight
                } else {
                    it.ellipsoidHeight = it.height
                    it.height = it.ellipsoidHeight
                }
            }
        } else {
            val takeoffHeight: Double = if (droneHeightModel.isHasTakeoffHeight) droneHeightModel.takeoffHeight.toDouble() else 0.0
            val waylineWgs84Height = waypointMission.wayline!!.altitude + takeoffHeight
            if (waypointMission.waypoints.isNotEmpty()) {
                waypointInfo.globalFlightHeight = GpsUtils.egm96Altitude(waylineWgs84Height, waypointMission.waypoints[0].latitude, waypointMission.waypoints[0].longitude)
            } else {
                waypointInfo.globalFlightHeight= waylineWgs84Height
            }
            waypointInfo.waypoints.forEach {
                if (it.useGlobalFlightHeight) {
                    it.ellipsoidHeight = waylineWgs84Height
                    it.height = GpsUtils.egm96Altitude(it.ellipsoidHeight, it.location.latitude, it.location.longitude)
                } else {
                    it.ellipsoidHeight = it.height + takeoffHeight
                    it.height = GpsUtils.egm96Altitude(it.ellipsoidHeight, it.location.latitude, it.location.longitude)
                }
            }
        }
    }

    private fun transWaylineTemplateFrom(waylineModel: WaylineModelGreenDao, waypointModels: List<WaypointModelGreenDao>): WaylineTemplateWaypointInfo {
        val waypoints: MutableList<WaylineWaypoint> = ArrayList()
        for (i in waypointModels.indices) {
            val waypointModel = waypointModels[i]
            val waypoint: WaylineWaypoint = transWaylineWaypointFrom(waylineModel, waypointModel)
            waypoint.waypointIndex = i
            waypoints.add(waypoint)
        }
        val waypointInfo = WaylineTemplateWaypointInfo()
        waypointInfo.waypoints = waypoints
        waypointInfo.actionGroups = transformActionsFrom(waypointModels)
        waypointInfo.globalFlightHeight = waylineModel.altitude.toDouble()
        waypointInfo.isGlobalFlightHeightSet = true
        waypointInfo.isGlobalActionSet = false

        val turnModePair: Pair<WaylineWaypointTurnMode, Boolean> = transWaypointType(waylineModel.waypointType)
        waypointInfo.globalTurnMode = turnModePair.first
        waypointInfo.useStraightLine = turnModePair.second
        waypointInfo.isTemplateGlobalTurnModeSet = true
        val yawParam = WaylineWaypointYawParam()
        yawParam.yawMode = mWaylineHeadingModeTransform.transFrom(waylineModel.headingMode)
        if (yawParam.yawMode == WaylineWaypointYawMode.TOWARD_POI) {
            yawParam.poiLocation = WaylineLocationCoordinate3D(waylineModel.poiLatitude, waylineModel.poiLongitude, 0.0)
        }
        waypointInfo.globalYawParam = yawParam
        waypointInfo.isTemplateGlobalYawParamSet = true
        waypointInfo.pitchMode = if (waylineModel.gimbalPitchRotationEnable) WaylineWaypointPitchMode.USE_POINT_SETTING else WaylineWaypointPitchMode.MANUALLY
        waypointInfo.caliFlightEnable = waylineModel.flightCali
        return waypointInfo
    }

    private fun transWaypointType(waypointTurnMode: WaypointV2MissionTypes.WaypointV2FlightPathMode?): Pair<WaylineWaypointTurnMode, Boolean> {
        var turnMode = WaylineWaypointTurnMode.TO_POINT_AND_STOP_WITH_DISCONTINUITY_CURVATURE
        var useStraightLine = true
        if (waypointTurnMode != null) {
            when (waypointTurnMode) {
                WaypointV2MissionTypes.WaypointV2FlightPathMode.CURVATURE_CONTINUOUS_PASSED -> {
                    turnMode = WaylineWaypointTurnMode.TO_POINT_AND_PASS_WITH_CONTINUITY_CURVATURE
                    useStraightLine = false
                }
                WaypointV2MissionTypes.WaypointV2FlightPathMode.GOTO_POINT_CURVE_AND_STOP -> {
                    turnMode = WaylineWaypointTurnMode.TO_POINT_AND_STOP_WITH_CONTINUITY_CURVATURE
                    useStraightLine = false
                }
                WaypointV2MissionTypes.WaypointV2FlightPathMode.COORDINATE_TURN -> {
                    turnMode = WaylineWaypointTurnMode.COORDINATE_TURN
                    useStraightLine = true
                }
            }
        }
        return Pair.create(turnMode, useStraightLine)
    }

    private fun transWaylineWaypointFrom(waylineModel: WaylineModelGreenDao, waypointModel: WaypointModelGreenDao): WaylineWaypoint {
        val waypoint = WaylineWaypoint()
        val location = WaylineLocationCoordinate2D()
        location.longitude = waypointModel.longitude
        location.latitude = waypointModel.latitude
        waypoint.location = location
        waypoint.height = waypointModel.altitude.toDouble()
        waypoint.useGlobalFlightHeight = waypointModel.useWaylineAltitude

        if (!waypointModel.useWaylineSpeed) {
            waypoint.speed = waypointModel.speed.toDouble()
        }
        waypoint.useGlobalAutoFlightSpeed = waypointModel.useWaylineSpeed
        waypoint.gimbalPitchAngle = waypointModel.gimbalPitch.toDouble()
        val setWaypointYawParam =
            !waypointModel.useWaylineHeadMode || waylineModel.headingMode == WaypointMissionHeadingMode.USING_WAYPOINT_HEADING
        if (setWaypointYawParam) {
            var headingMode = waypointModel.headingMode
            if (waypointModel.useWaylineHeadMode && waylineModel.headingMode == WaypointMissionHeadingMode.USING_WAYPOINT_HEADING) {
                headingMode = WaypointMissionHeadingMode.USING_WAYPOINT_HEADING
            }
            val yawParam = WaylineWaypointYawParam()
            yawParam.enableYawAngle =
                headingMode == WaypointMissionHeadingMode.USING_WAYPOINT_HEADING
            yawParam.yawAngle = waypointModel.heading.toDouble()
            yawParam.yawMode = mWaylineHeadingModeTransform.transFrom(headingMode)
            yawParam.yawPathMode = mWaylineTurnModeTransform.transFrom(waypointModel.turnMode)
            yawParam.poiLocation = WaylineLocationCoordinate3D(waylineModel.poiLatitude, waylineModel.poiLongitude, 0.0)
            waypoint.yawParam = yawParam
        }
        waypoint.isWaylineWaypointYawParamSet = setWaypointYawParam
        waypoint.useGlobalYawParam = !setWaypointYawParam
        if (!waypointModel.useWaylinePointType) {
            val turnParam = WaylineWaypointTurnParam()
            turnParam.turnDampingDistance = waypointModel.cornerRadius.toDouble()
            val turnModePair: Pair<WaylineWaypointTurnMode, Boolean> = transWaypointType(waypointModel.waypointType)
            turnParam.turnMode = turnModePair.first
            waypoint.useStraightLine = turnModePair.second
            waypoint.turnParam = turnParam
        }
        waypoint.isWaylineWaypointTurnParamSet = !waypointModel.useWaylinePointType
        waypoint.useGlobalTurnParam = waypointModel.useWaylinePointType
        return waypoint
    }

    private fun transCoordinateParamFrom(rtkSource: RTKReferenceStationSource, waylineModel: WaylineModelGreenDao): WaylineCoordinateParam {
        val coordinateParam = WaylineCoordinateParam()
        coordinateParam.coordinateMode = WaylineCoordinateMode.WGS84
        coordinateParam.positioningType = mWaylineRtkTypeTransform.transFrom(rtkSource)
        coordinateParam.isGlobalShootHeightSet = false
        coordinateParam.isWaylinePositioningTypeSet = true
        coordinateParam.isSurfaceFollowParamSet = false
        val useAbsolute = waylineModel.droneInfo!!.droneHeight.isUseAbsolute
        coordinateParam.altitudeMode =
            if (useAbsolute) WaylineAltitudeMode.EGM96 else WaylineAltitudeMode.RELATIVE_TO_START_POINT
        return coordinateParam
    }

    private fun transPayloadParamsFrom(droneInfoModel: DroneInfoModel): List<WaylinePayloadParam> {
        val payloadParams: MutableList<WaylinePayloadParam> = mutableListOf()
        val droneCameras = droneInfoModel.cameras
        for (cameraModel in droneCameras) {
            var payloadParam = WaylinePayloadParam()
            if (cameraModel.payloadConfigInfo != null) {
                payloadParam = mWaylinePaylodParamTransform.transFrom(cameraModel.payloadConfigInfo)
            }
            if (cameraModel.photoTypes != null) {
                payloadParam.imageFormat = java.util.ArrayList(cameraModel.photoTypes)
            }
            payloadParam.payloadPositionIndex = cameraModel.cameraIndex
            payloadParam.isPayloadPositionIndexSet = true
            payloadParams.add(payloadParam)
        }
        return payloadParams
    }

    private fun transConfigFrom(waylineModel: WaylineModelGreenDao): WaylineMissionConfig {
        val config = WaylineMissionConfig()
        config.flyToWaylineMode = mToFirstPointTransform.transFrom(waylineModel.gotoFirstPointMode)
        config.finishAction = mFinishActionTransform.transFrom(waylineModel.actionOnFinish)
        config.droneInfo = mWaylineDroneTypeTransform.transFrom(waylineModel.droneInfo!!.droneType)
        config.isSecurityTakeOffHeightSet = false
        val goOnExecute = waylineModel.exitOnRCLost.not()
        config.exitOnRCLostBehavior = if (goOnExecute) WaylineExitOnRCLostBehavior.GO_ON else WaylineExitOnRCLostBehavior.EXCUTE_RC_LOST_ACTION
        config.exitOnRCLostType = WaylineExitOnRCLostAction.GO_BACK

        config.globalTransitionalSpeed = waylineModel.autoFlightSpeed.toDouble()
        val droneHeightModel = waylineModel.droneInfo!!.droneHeight
        if (droneHeightModel.isHasTakeoffHeight) {
            config.isTakeOffPositionRefSet = true
            val takeoffLocation = WaylineLocationCoordinate3D(
                droneHeightModel.takeoffLat,
                droneHeightModel.takeoffLng, droneHeightModel.takeoffHeight.toDouble()
            )
            config.takeOffPositionRef = takeoffLocation
        } else {
            config.isTakeOffPositionRefSet = false
        }
        config.isGlobalRTHHeightSet = false
        val payloadInfos: MutableList<WaylinePayloadInfo> = mutableListOf()
        val droneCameras = waylineModel.droneInfo!!.cameras
        for (cameraModel in droneCameras) {
            val payloadInfo: WaylinePayloadInfo = mWaylinePayloadInfoTransform.transFrom(cameraModel)
            payloadInfos.add(payloadInfo)
        }
        config.payloadInfo = payloadInfos
        return config
    }

    private fun createWaylineMission(infoModel: MissionInfoModelGreenDao): WaylineMission {
        val mission = WaylineMission()
        mission.updateTime = infoModel.updateTime.toDouble()
        mission.createTime = infoModel.createTime.toDouble()
        return mission
    }

    private fun transformActionsFrom(waypoints: List<WaypointModelGreenDao>): List<WaylineActionGroup> {
        val actionGroups: MutableList<WaylineActionGroup> = mutableListOf()
        for (i in waypoints.indices) {
            val waypoint = waypoints[i]
            val actionModels = waypoint.actions
            val actionInfos: MutableList<WaylineActionInfo> = mutableListOf()
            for (item in actionModels) {
                val actionInfo = WaypointActionEntityTransform().trans(item, waypoint.preciseShotInfo)
                if (actionInfo != null) {
                    actionInfos.add(actionInfo)
                }
            }
            if (actionInfos.size > 0) {
                val actionGroup = WaylineActionGroup()
                val trigger = WaylineActionTrigger()
                trigger.triggerType = WaylineActionTriggerType.REACH_POINT
                actionGroup.trigger = trigger
                actionGroup.groupId = actionGroups.size
                actionGroup.startIndex = i
                actionGroup.endIndex = i
                actionGroups.add(actionGroup)
                actionGroup.actions = actionInfos
                val nodeLists: MutableList<WaylineActionNodeList> = mutableListOf()
                val root = WaylineActionNodeList()
                val treeNodes: MutableList<WaylineActionTreeNode> = mutableListOf()
                val rootNode = WaylineActionTreeNode()
                rootNode.nodeType = WaylineActionsRelationType.SEQUENCE
                rootNode.childrenNum = actionInfos.size
                treeNodes.add(rootNode)
                root.nodes = treeNodes
                nodeLists.add(root)
                val children = WaylineActionNodeList()
                val childrenNodeList: MutableList<WaylineActionTreeNode> = mutableListOf()
                for (j in actionInfos.indices) {
                    val child = WaylineActionTreeNode()
                    child.nodeType = WaylineActionsRelationType.LEAF
                    child.actionIndex = j
                    childrenNodeList.add(child)
                }
                children.nodes = childrenNodeList
                nodeLists.add(children)
                actionGroup.nodeLists = nodeLists
            }
        }

        // 处理定时定距动作
        transFromIntervalActions(waypoints, actionGroups)
        return actionGroups
    }

    private fun transFromIntervalActions(waypoints: List<WaypointModelGreenDao>, actionGroups: MutableList<WaylineActionGroup>) {
        var intervalShotAction: WaylineActionGroup? = null
        for (i in waypoints.indices) {
            val actionModels = waypoints[i].actions
            for (item in actionModels) {
                if (intervalShotAction == null &&
                    (item.actionType == WaypointActionType.START_TIME_INTERVAL_SHOT
                            || item.actionType == WaypointActionType.START_DISTANCE_INTERVAL_SHOT)
                ) {
                    intervalShotAction = getIntevalShootActionGroup(actionGroups, i, item)
                } else if (item.actionType == WaypointActionType.STOP_INTERVAL_SHOT && intervalShotAction != null) {
                    intervalShotAction.endIndex = i
                    actionGroups.add(intervalShotAction)
                    intervalShotAction = null
                }
            }
        }
        if (intervalShotAction != null) {
            intervalShotAction.endIndex = waypoints.size - 1
            actionGroups.add(intervalShotAction)
        }
    }

    private fun getIntevalShootActionGroup(actionGroups: List<WaylineActionGroup>, waypointIndex: Int, item: WaypointActionModelGreenDao): WaylineActionGroup {
        val actionGroup = WaylineActionGroup()
        actionGroup.groupId = actionGroups.size
        actionGroup.startIndex = waypointIndex
        actionGroup.trigger = getActionTrigger(item)
        val actionInfos: MutableList<WaylineActionInfo> = mutableListOf()
        val actionInfo = WaylineActionInfo()
        actionInfo.actionId = actionInfos.size
        actionInfo.actionType = WaylineActionType.TAKE_PHOTO
        val photoTypes: MutableList<CameraLensType> = mutableListOf()
        val param = ActionTakePhotoParam(item.cameraIndex, false, photoTypes, null)

        actionInfo.takePhotoParam = param
        actionInfos.add(actionInfo)
        actionGroup.actions = actionInfos
        val nodeLists: MutableList<WaylineActionNodeList> = mutableListOf()
        val root = WaylineActionNodeList()
        val treeNodes: MutableList<WaylineActionTreeNode> = mutableListOf()
        val rootNode = WaylineActionTreeNode()
        rootNode.nodeType = WaylineActionsRelationType.SEQUENCE
        rootNode.childrenNum = actionInfos.size
        treeNodes.add(rootNode)
        root.nodes = treeNodes
        nodeLists.add(root)
        val children = WaylineActionNodeList()
        val childrenNodeList: MutableList<WaylineActionTreeNode> = mutableListOf()
        for (j in actionInfos.indices) {
            val child = WaylineActionTreeNode()
            child.nodeType = WaylineActionsRelationType.LEAF
            child.actionIndex = j
            childrenNodeList.add(child)
        }
        children.nodes = childrenNodeList
        nodeLists.add(children)
        actionGroup.nodeLists = nodeLists
        return actionGroup
    }

    private fun getActionTrigger(item: WaypointActionModelGreenDao): WaylineActionTrigger {
        val trigger = WaylineActionTrigger()
        if (item.actionType == WaypointActionType.START_TIME_INTERVAL_SHOT) {
            trigger.triggerType = WaylineActionTriggerType.MULTIPLE_TIMING
            trigger.timeInterval = item.param.toDouble()
        } else {
            trigger.triggerType = WaylineActionTriggerType.MULTIPLE_DISTANCE
            trigger.distanceInterval = item.param.toDouble()
        }
        return trigger
    }

}