package com.dji.wpmzsdk.common.utils.kml.transfrom;

import android.util.Pair;


import com.dji.wpmzsdk.common.utils.kml.data.DroneInfoModel;
import com.dji.wpmzsdk.common.utils.kml.mission.RTKReferenceStationSource;
import com.dji.wpmzsdk.common.utils.kml.mission.WaypointMissionHeadingMode;
import com.dji.wpmzsdk.common.utils.kml.model.DroneCameraModel;
import com.dji.wpmzsdk.common.utils.kml.model.DroneHeightModel;
import com.dji.wpmzsdk.common.utils.kml.model.PayloadConfigInfoModel;
import com.dji.wpmzsdk.common.utils.kml.model.WaypointActionType;
import com.dji.wpmzsdk.common.utils.kml.model.PointTurnMode;
import com.dji.wpmzsdk.common.utils.kml.model.WaylineModel;
import com.dji.wpmzsdk.common.utils.kml.model.WaypointActionModel;
import com.dji.wpmzsdk.common.utils.kml.model.WaypointMissionModel;
import com.dji.wpmzsdk.common.utils.kml.model.WaypointModel;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;


import dji.sdk.wpmz.value.mission.ActionTakePhotoParam;
import dji.sdk.wpmz.value.mission.CameraLensType;
import dji.sdk.wpmz.value.mission.WaylineActionGroup;
import dji.sdk.wpmz.value.mission.WaylineActionInfo;
import dji.sdk.wpmz.value.mission.WaylineActionNodeList;
import dji.sdk.wpmz.value.mission.WaylineActionTreeNode;
import dji.sdk.wpmz.value.mission.WaylineActionTrigger;
import dji.sdk.wpmz.value.mission.WaylineActionTriggerType;
import dji.sdk.wpmz.value.mission.WaylineActionType;
import dji.sdk.wpmz.value.mission.WaylineActionsRelationType;
import dji.sdk.wpmz.value.mission.WaylineAltitudeMode;
import dji.sdk.wpmz.value.mission.WaylineCoordinateMode;
import dji.sdk.wpmz.value.mission.WaylineCoordinateParam;
import dji.sdk.wpmz.value.mission.WaylineExitOnRCLostAction;
import dji.sdk.wpmz.value.mission.WaylineExitOnRCLostBehavior;
import dji.sdk.wpmz.value.mission.WaylineLocationCoordinate2D;
import dji.sdk.wpmz.value.mission.WaylineLocationCoordinate3D;
import dji.sdk.wpmz.value.mission.WaylineMissionConfig;
import dji.sdk.wpmz.value.mission.WaylinePayloadInfo;
import dji.sdk.wpmz.value.mission.WaylinePayloadParam;
import dji.sdk.wpmz.value.mission.WaylineTemplate;
import dji.sdk.wpmz.value.mission.WaylineTemplateType;
import dji.sdk.wpmz.value.mission.WaylineTemplateWaypointInfo;
import dji.sdk.wpmz.value.mission.WaylineWaypoint;
import dji.sdk.wpmz.value.mission.WaylineWaypointPitchMode;
import dji.sdk.wpmz.value.mission.WaylineWaypointTurnMode;
import dji.sdk.wpmz.value.mission.WaylineWaypointTurnParam;
import dji.sdk.wpmz.value.mission.WaylineWaypointYawMode;
import dji.sdk.wpmz.value.mission.WaylineWaypointYawParam;

/**
 * Description :
 * 航点飞行模板转换
 * @author : devin.xu
 * @filename : WaypointTemplateTransform
 * @time : 2021/2/22
 * <p>
 * Copyright (c) 2016, DJI All Rights Reserved.
 **/
public class WaypointTemplateTransform implements ModelTransform<WaypointMissionModel, MissionTransformData>{

    private final WaylineFinishActionTransform mFinishActionTransform = new WaylineFinishActionTransform();
    private final WaylineToFirstPointTransform mToFirstPointTransform = new WaylineToFirstPointTransform();
    private final WaylineRtkTypeTransform mWaylineRtkTypeTransform = new WaylineRtkTypeTransform();
    private final WaylineTurnModeTransform mWaylineTurnModeTransform = new WaylineTurnModeTransform();
    private final WaylineWaypointTypeTransform mWaylineWaypointTypeTransform = new WaylineWaypointTypeTransform();
    private final WaylineHeadingModeTransform mWaylineHeadingModeTransform = new WaylineHeadingModeTransform();
    private final WaypointActionTemplateTransform mWaypointActionTemplateTransform = new WaypointActionTemplateTransform();
    private final WaylineActionTemplateTransform mWaylineActionTemplateTransform = new WaylineActionTemplateTransform();
    private final WaylineDroneTypeTransform mWaylineDroneTypeTransform = new WaylineDroneTypeTransform();
    private final WaylinePayloadInfoTransform mWaylinePayloadInfoTransform = new WaylinePayloadInfoTransform();
    private final WaylinePaylodParamTransform mWaylinePaylodParamTransform = new WaylinePaylodParamTransform();

    @Override
    public MissionTransformData transFrom(WaypointMissionModel waypointMissionModel) {
        List<WaypointModel> waypointModels = waypointMissionModel.getWaypoints();
        WaylineModel waylineModel =waypointMissionModel.getWayline();

        WaylineTemplate template = new WaylineTemplate();
        template.setTemplateId(0);
        template.setTemplateType(WaylineTemplateType.WAYPOINT);

        WaylineTemplateWaypointInfo waypointInfo = transWaylineTemplateFrom(waylineModel, waypointModels);
        template.setWaypointInfo(waypointInfo);

        WaylineCoordinateParam coordinateParam = transCoordinateParamFrom(waypointMissionModel.getMissionInfo().getRtkStation(), waylineModel);
        template.setCoordinateParam(coordinateParam);
        template.setUseGlobalTransitionalSpeed(true);
        template.setAutoFlightSpeed((double) waylineModel.getAutoFlightSpeed());

        List<WaylinePayloadParam> payloadParams = transPayloadParamsFrom(waylineModel.getDroneInfo());
        template.setPayloadParam(payloadParams);

        WaylineMissionConfig config = transConfigFrom(waylineModel);

        MissionTransformData data = new MissionTransformData();
        data.setTemplate(template);
        data.setMissionConfig(config);
        data.setMission(WaylineInfoTransformKt.createWaylineMission(waypointMissionModel.getMissionInfo()));
        data.setExecuteMissionConfig(WaylineInfoTransformKt.createExecuteConfig(config));
        return data;
    }

    private WaylineWaypoint transWaylineWaypointFrom(WaylineModel waylineModel, WaypointModel waypointModel) {
        WaylineWaypoint waypoint = new WaylineWaypoint();

        WaylineLocationCoordinate2D location = new WaylineLocationCoordinate2D();
        location.setLongitude(waypointModel.getLongitude());
        location.setLatitude(waypointModel.getLatitude());
        waypoint.setLocation(location);

        waypoint.setHeight((double) waypointModel.getAltitude());
        waypoint.setEllipsoidHeight((double) waypointModel.getWgs84Altitude());
        waypoint.setUseGlobalFlightHeight(waypointModel.isUseWaylineAltitude());

        if (!waypointModel.isUseWaylineSpeed()) {
            waypoint.setSpeed((double) waypointModel.getSpeed());
        }
        waypoint.setUseGlobalAutoFlightSpeed(waypointModel.isUseWaylineSpeed());

        waypoint.setGimbalPitchAngle((double) waypointModel.getGimbalPitch());

        waypoint.setUseGlobalActionGroup(waypointModel.isUseWaylineAction());

        boolean setWaypointYawParam = !waypointModel.isUseWaylineHeadMode() || waylineModel.getHeadingMode() == WaypointMissionHeadingMode.USING_WAYPOINT_HEADING;
        if (setWaypointYawParam) {
            WaypointMissionHeadingMode headingMode = waypointModel.getHeadingMode();
            if (waypointModel.isUseWaylineHeadMode() && waylineModel.getHeadingMode() == WaypointMissionHeadingMode.USING_WAYPOINT_HEADING) {
                headingMode = WaypointMissionHeadingMode.USING_WAYPOINT_HEADING;
            }

            WaylineWaypointYawParam yawParam = new WaylineWaypointYawParam();
            yawParam.setEnableYawAngle(headingMode == WaypointMissionHeadingMode.USING_WAYPOINT_HEADING);
            yawParam.setYawAngle((double) waypointModel.getHeading());
            yawParam.setYawMode(mWaylineHeadingModeTransform.transFrom(headingMode));
            yawParam.setYawPathMode(mWaylineTurnModeTransform.transFrom(waypointModel.getTurnMode()));
            yawParam.setPoiLocation(new WaylineLocationCoordinate3D(waylineModel.getPoiLatitude(), waylineModel.getPoiLongitude(), 0d));
            waypoint.setYawParam(yawParam);
        }
        waypoint.setIsWaylineWaypointYawParamSet(setWaypointYawParam);
        waypoint.setUseGlobalYawParam(!setWaypointYawParam);

        if (!waypointModel.isUseWaylinePointType()) {
            WaylineWaypointTurnParam turnParam = new WaylineWaypointTurnParam();
            turnParam.setTurnDampingDistance((double) waypointModel.getCornerRadius());
            Pair<WaylineWaypointTurnMode, Boolean> turnModePair = mWaylineWaypointTypeTransform.transFrom(waypointModel.getWaypointTurnMode());
            turnParam.setTurnMode(turnModePair.first);
            waypoint.setUseStraightLine(turnModePair.second);
            waypoint.setTurnParam(turnParam);
        }
        waypoint.setIsWaylineWaypointTurnParamSet(!waypointModel.isUseWaylinePointType());
        waypoint.setUseGlobalTurnParam(waypointModel.isUseWaylinePointType());

        return waypoint;
    }

    private WaylineTemplateWaypointInfo transWaylineTemplateFrom(WaylineModel waylineModel, List<WaypointModel> waypointModels) {
        List<WaylineWaypoint> waypoints = new ArrayList<>();
        for (int i = 0; i < waypointModels.size(); ++i) {
            WaypointModel waypointModel = waypointModels.get(i);
            WaylineWaypoint waypoint = transWaylineWaypointFrom(waylineModel, waypointModel);
            waypoint.setWaypointIndex(i);
            waypoints.add(waypoint);
        }
        WaylineTemplateWaypointInfo waypointInfo = new WaylineTemplateWaypointInfo();
        waypointInfo.setWaypoints(waypoints);
        waypointInfo.setActionGroups(transformActionsFrom(waypointModels));
        waypointInfo.setGlobalFlightHeight((double) waylineModel.getAltitude());
        waypointInfo.setIsGlobalFlightHeightSet(true);

        if (waylineModel.getActions() == null || waylineModel.getActions().isEmpty()) {
            waypointInfo.setIsGlobalActionSet(false);
        } else {
            WaylineActionGroup globalAction = transWaylineActionGroupFrom(waylineModel.getActions());
            waypointInfo.setGlobalAction(globalAction);
            waypointInfo.setIsGlobalActionSet(true);
        }

        Pair<WaylineWaypointTurnMode, Boolean> turnModePair = mWaylineWaypointTypeTransform.transFrom(waylineModel.getWaypointTurnMode());
        waypointInfo.setGlobalTurnMode(turnModePair.first);
        waypointInfo.setUseStraightLine(turnModePair.second);
        waypointInfo.setIsTemplateGlobalTurnModeSet(true);

        WaylineWaypointYawParam yawParam = new WaylineWaypointYawParam();
        yawParam.setYawMode(mWaylineHeadingModeTransform.transFrom(waylineModel.getHeadingMode()));
        yawParam.setPoiLocation(new WaylineLocationCoordinate3D(waylineModel.getPoiLatitude(), waylineModel.getPoiLongitude(), 0d));
        waypointInfo.setGlobalYawParam(yawParam);
        waypointInfo.setIsTemplateGlobalYawParamSet(true);

        waypointInfo.setPitchMode(waylineModel.isGimbalPitchRotationEnable() ? WaylineWaypointPitchMode.USE_POINT_SETTING : WaylineWaypointPitchMode.MANUALLY);
        waypointInfo.setCaliFlightEnable(waylineModel.isFlightCali());
        return waypointInfo;
    }

    private WaylineActionGroup transWaylineActionGroupFrom(List<WaypointActionModel> actions) {
        final WaylineActionGroup actionGroup = new WaylineActionGroup();

        // action Info
        final List<WaylineActionInfo> actionInfo = new ArrayList<>();
        for (WaypointActionModel item : actions) {
            WaylineActionInfo info = mWaypointActionTemplateTransform.trans(item);
            if (info != null && info.getActionType() != WaylineActionType.UNKNOWN) {
                actionInfo.add(info);
            }
        }
        final WaylineActionTrigger trigger = new WaylineActionTrigger();
        trigger.setTriggerType(WaylineActionTriggerType.REACH_POINT);
        actionGroup.setTrigger(trigger);
        actionGroup.setActions(actionInfo);

        final List<WaylineActionNodeList> nodeLists = new ArrayList<>();

        final WaylineActionNodeList root = new WaylineActionNodeList();
        final List<WaylineActionTreeNode> treeNodes = new ArrayList<>();
        final WaylineActionTreeNode rootNode = new WaylineActionTreeNode();
        rootNode.setNodeType(WaylineActionsRelationType.SEQUENCE);
        rootNode.setChildrenNum(actionInfo.size());

        treeNodes.add(rootNode);
        root.setNodes(treeNodes);
        nodeLists.add(root);

        final WaylineActionNodeList children = new WaylineActionNodeList();
        final List<WaylineActionTreeNode> childrenNodeList = new ArrayList<>();
        for (int i = 0; i < actionInfo.size(); i++) {
            final WaylineActionTreeNode child = new WaylineActionTreeNode();
            child.setNodeType(WaylineActionsRelationType.LEAF);
            child.setActionIndex(i);
            childrenNodeList.add(child);
        }
        children.setNodes(childrenNodeList);
        nodeLists.add(children);

        actionGroup.setNodeLists(nodeLists);
        return actionGroup;
    }

    private WaylineMissionConfig transConfigFrom(WaylineModel waylineModel) {
        WaylineMissionConfig config = new WaylineMissionConfig();
        config.setFlyToWaylineMode(mToFirstPointTransform.transFrom(waylineModel.getGotoFirstPointMode()));
        config.setFinishAction(mFinishActionTransform.transFrom(waylineModel.getActionOnFinish()));
        config.setDroneInfo(mWaylineDroneTypeTransform.transFrom(waylineModel.getDroneInfo().getDroneType()));
        config.setSecurityTakeOffHeight((double) waylineModel.getSecureTakeoffHeight());
        config.setIsSecurityTakeOffHeightSet(true);

        boolean goOnExecute = waylineModel.getLostAction() == null || waylineModel.getLostAction() == WaylineExitOnRCLostAction.UNKNOWN;
        config.setExitOnRCLostBehavior(goOnExecute
                ? WaylineExitOnRCLostBehavior.GO_ON : WaylineExitOnRCLostBehavior.EXCUTE_RC_LOST_ACTION);
        if (waylineModel.getLostAction() != null) {
            config.setExitOnRCLostType(waylineModel.getLostAction());
        }
        config.setGlobalTransitionalSpeed(15d);

        DroneHeightModel droneHeightModel = waylineModel.getDroneInfo().getDroneHeight();
        if (droneHeightModel.isHasTakeoffHeight()) {
            config.setIsTakeOffPositionRefSet(true);
            WaylineLocationCoordinate3D takeoffLocation =
                    new WaylineLocationCoordinate3D(droneHeightModel.getTakeoffLat(),
                            droneHeightModel.getTakeoffLng(), (double) droneHeightModel.getTakeoffHeight());
            config.setTakeOffPositionRef(takeoffLocation);
        } else {
            config.setIsTakeOffPositionRefSet(false);
        }
        config.setIsGlobalRTHHeightSet(false);

        List<WaylinePayloadInfo> payloadInfos = new ArrayList<>();
        List<DroneCameraModel> droneCameras = waylineModel.getDroneInfo().getCameras();
        for (DroneCameraModel cameraModel : droneCameras) {
            WaylinePayloadInfo payloadInfo = mWaylinePayloadInfoTransform.transFrom(cameraModel);
            payloadInfos.add(payloadInfo);
        }
        config.setPayloadInfo(payloadInfos);
        return config;
    }



    private WaylineCoordinateParam transCoordinateParamFrom(RTKReferenceStationSource rtkSource, WaylineModel waylineModel) {
        WaylineCoordinateParam coordinateParam = new WaylineCoordinateParam();
        coordinateParam.setCoordinateMode(WaylineCoordinateMode.WGS84);
        coordinateParam.setPositioningType(mWaylineRtkTypeTransform.transFrom(rtkSource));
        coordinateParam.setIsGlobalShootHeightSet(false);
        coordinateParam.setIsWaylinePositioningTypeSet(true);
        coordinateParam.setIsSurfaceFollowParamSet(false);
        boolean useAbsolute = waylineModel.getDroneInfo().getDroneHeight().isUseAbsolute();
        coordinateParam.setAltitudeMode(useAbsolute ? WaylineAltitudeMode.EGM96 : WaylineAltitudeMode.RELATIVE_TO_START_POINT);

        return coordinateParam;
    }

    private List<WaylinePayloadParam> transPayloadParamsFrom(DroneInfoModel droneInfoModel) {
        List<WaylinePayloadParam> payloadParams = new ArrayList<>();
        List<DroneCameraModel> droneCameras = droneInfoModel.getCameras();
        for (DroneCameraModel cameraModel : droneCameras) {
            WaylinePayloadParam payloadParam = new WaylinePayloadParam();
            if (cameraModel.getPayloadConfigInfo() != null) {
                payloadParam = mWaylinePaylodParamTransform.transFrom(cameraModel.getPayloadConfigInfo());
            }
            if (cameraModel.getPhotoTypes() != null) {
                payloadParam.setImageFormat(new ArrayList<>(cameraModel.getPhotoTypes()));
            }
            payloadParam.setPayloadPositionIndex(cameraModel.getCameraIndex());
            payloadParam.setIsPayloadPositionIndexSet(true);
            payloadParams.add(payloadParam);
        }
        return payloadParams;
    }

    @Override
    public WaypointMissionModel transTo(MissionTransformData data) {
        WaylineTemplate template = data.getTemplates().get(0);
        WaylineMissionConfig config = data.getMissionConfig();

        WaypointMissionModel model = new WaypointMissionModel();

        // 转换航线配置
        WaylineModel waylineModel = transWaylineTemplateTo(template, config);
        model.setWayline(waylineModel);

        // 转换航点配置
        List<WaypointModel> waypointModels = transWaypointTemplateTo(template.getWaypointInfo());
        model.setWaypoints(waypointModels);
        return model;
    }

    private WaylineModel transWaylineTemplateTo(WaylineTemplate template, WaylineMissionConfig config) {
        WaylineModel waylineModel = new WaylineModel();
        waylineModel.setAltitude(template.getWaypointInfo().getGlobalFlightHeight().floatValue());
        waylineModel.setAutoFlightSpeed(template.getAutoFlightSpeed().floatValue());
        waylineModel.setGotoFirstPointMode(mToFirstPointTransform.transTo(config.getFlyToWaylineMode()));
        waylineModel.setActionOnFinish(mFinishActionTransform.transTo(config.getFinishAction()));
        if (config.getExitOnRCLostBehavior() == WaylineExitOnRCLostBehavior.EXCUTE_RC_LOST_ACTION) {
            waylineModel.setLostAction(config.getExitOnRCLostType());
        } else {
            waylineModel.setLostAction(WaylineExitOnRCLostAction.UNKNOWN);
        }
        Pair<WaylineWaypointTurnMode, Boolean> turnModePair = Pair.create(template.getWaypointInfo().getGlobalTurnMode(), template.getWaypointInfo().getUseStraightLine());
        waylineModel.setWaypointTurnMode(mWaylineWaypointTypeTransform.transTo(turnModePair));
        WaylineWaypointPitchMode pitchMode = template.getWaypointInfo().getPitchMode();
        waylineModel.setGimbalPitchRotationEnable(pitchMode == WaylineWaypointPitchMode.USE_POINT_SETTING);
        waylineModel.setSecureTakeoffHeight(config.getSecurityTakeOffHeight().floatValue());
        waylineModel.setFlightCali(template.getWaypointInfo().getCaliFlightEnable());
        waylineModel.setActions(transWaylineGlobalActionsTo(template.getWaypointInfo().getGlobalAction()));

        final WaylineWaypointYawParam yawParam = template.getWaypointInfo().getGlobalYawParam();
        waylineModel.setHeadingMode(mWaylineHeadingModeTransform.transTo(yawParam.getYawMode()));
        if (waylineModel.getHeadingMode() == WaypointMissionHeadingMode.TOWARD_POINT_OF_INTEREST
                && yawParam.getPoiLocation() != null) {
            waylineModel.setPoiLatitude(yawParam.getPoiLocation().getLatitude());
            waylineModel.setPoiLongitude(yawParam.getPoiLocation().getLongitude());
        }

        List<WaylinePayloadInfo> payloadInfos = config.getPayloadInfo();
        LinkedHashMap<Integer, DroneCameraModel> droneCameras = new LinkedHashMap<>();
        for (int i = 0; i < payloadInfos.size(); ++i) {
            DroneCameraModel droneCameraModel = mWaylinePayloadInfoTransform.transTo(payloadInfos.get(i));
            droneCameras.put(droneCameraModel.getCameraIndex(), droneCameraModel);
        }

        List<WaylinePayloadParam> payloadParams = template.getPayloadParam();
        for (WaylinePayloadParam payloadParam : payloadParams) {
            int index = payloadParam.getPayloadPositionIndex();
            PayloadConfigInfoModel configInfoModel = mWaylinePaylodParamTransform.transTo(payloadParam);
            DroneCameraModel cameraModel = droneCameras.get(index);
            if (cameraModel != null) {
                cameraModel.setPayloadConfigInfo(configInfoModel);
                if (payloadParam.getImageFormat() != null) {
                    cameraModel.setPhotoTypes(new HashSet<>(payloadParam.getImageFormat()));
                }
            }
        }

        DroneHeightModel droneHeightModel = new DroneHeightModel();
        droneHeightModel.setUseAbsolute(template.getCoordinateParam().getAltitudeMode() == WaylineAltitudeMode.EGM96);
        droneHeightModel.setAboveGroundMode(template.getCoordinateParam().getAltitudeMode() == WaylineAltitudeMode.ABOVE_GROUND_LEVEL);

        if (config.getIsTakeOffPositionRefSet()) {
            droneHeightModel.setTakeoffHeight(config.getTakeOffPositionRef().getAltitude().floatValue());
            droneHeightModel.setTakeoffLat(config.getTakeOffPositionRef().getLatitude());
            droneHeightModel.setTakeoffLng(config.getTakeOffPositionRef().getLongitude());
            droneHeightModel.setHasTakeoffHeight(true);
        }

        DroneInfoModel droneInfoModel = new DroneInfoModel();
        droneInfoModel.setDroneCamera(new ArrayList<>(droneCameras.values()));
        droneInfoModel.setDroneHeight(droneHeightModel);
        droneInfoModel.setDroneType(mWaylineDroneTypeTransform.transTo(config.getDroneInfo()));
        waylineModel.setDroneInfo(droneInfoModel);

        return waylineModel;
    }

    private List<WaypointActionModel> transWaylineGlobalActionsTo(WaylineActionGroup globalAction) {
        final List<WaypointActionModel> actionModels = new ArrayList<>();
        if (globalAction == null) {
            return actionModels;
        }

        final WaylineActionTrigger trigger = globalAction.getTrigger();
        final WaylineActionTriggerType triggerType = trigger.getTriggerType();

        if (triggerType == WaylineActionTriggerType.REACH_POINT) {
            final List<WaylineActionInfo> actionInfos = globalAction.getActions();
            for (WaylineActionInfo actionInfo : actionInfos) {
                final WaypointActionModel actionModel = mWaylineActionTemplateTransform.trans(actionInfo);
                if (actionModel != null) {
                    actionModels.add(actionModel);
                }
            }
        }
        return actionModels;
    }

    private List<WaypointModel> transWaypointTemplateTo(WaylineTemplateWaypointInfo template) {
        List<WaypointModel> waypointModels = new ArrayList<>();
        for (int i = 0; i < template.getWaypoints().size(); ++i) {
            WaylineWaypoint waypoint = template.getWaypoints().get(i);
            WaypointModel waypointModel = new WaypointModel();
            waypointModel.setLatitude(waypoint.getLocation().getLatitude());
            waypointModel.setLongitude(waypoint.getLocation().getLongitude());
            waypointModel.setUseWaylineAltitude(waypoint.getUseGlobalFlightHeight());
            waypointModel.setUseWaylineHeadMode(waypoint.getUseGlobalYawParam());
            waypointModel.setUseWaylinePointType(waypoint.getUseGlobalTurnParam());
            waypointModel.setUseWaylineSpeed(waypoint.getUseGlobalAutoFlightSpeed());
            waypointModel.setAltitude(waypoint.getHeight().floatValue());
            waypointModel.setWgs84Altitude(waypoint.getEllipsoidHeight().floatValue());

            if (waypoint.getIsWaylineWaypointTurnParamSet()) {
                waypointModel.setCornerRadius(waypoint.getTurnParam().getTurnDampingDistance().floatValue());
                Pair<WaylineWaypointTurnMode, Boolean> turnModePair = Pair.create(waypoint.getTurnParam().getTurnMode(), waypoint.getUseStraightLine());
                waypointModel.setWaypointTurnMode(mWaylineWaypointTypeTransform.transTo(turnModePair));
            }
            if (waypoint.getIsWaylineWaypointYawParamSet()) {
                waypointModel.setTurnMode(mWaylineTurnModeTransform.transTo(waypoint.getYawParam().getYawPathMode()));
                waypointModel.setHeadingMode(mWaylineHeadingModeTransform.transTo(waypoint.getYawParam().getYawMode()));
                waypointModel.setHeading(waypoint.getYawParam().getYawAngle().intValue());
            } else {
                waypointModel.setTurnMode(PointTurnMode.CLOCKWISE);
            }

            if (!waypoint.getUseGlobalAutoFlightSpeed()) {
                waypointModel.setSpeed(waypoint.getSpeed().floatValue());
            }
            waypointModel.setGimbalPitch(waypoint.getGimbalPitchAngle().floatValue());
            waypointModel.setIndex(waypoint.getWaypointIndex());
            waypointModel.setActions(new ArrayList<>());
            waypointModels.add(waypointModel);
        }
        transformActionTo(template.getActionGroups(), waypointModels);
        return waypointModels;
    }

    private List<WaylineActionGroup> transformActionsFrom(List<WaypointModel> waypoints) {
        List<WaylineActionGroup> actionGroups = new ArrayList<>();

        for (int i = 0; i < waypoints.size(); ++i) {
            WaypointModel waypoint = waypoints.get(i);
            List<WaypointActionModel> actionModels = waypoint.getActions();
            List<WaylineActionInfo> actionInfos = new ArrayList<>();
            for (WaypointActionModel item : actionModels) {
                WaylineActionInfo actionInfo = mWaypointActionTemplateTransform.trans(item);
                if (actionInfo != null) {
                    actionInfos.add(actionInfo);
                }
            }

            if (actionInfos.size() > 0) {
                WaylineActionGroup actionGroup = new WaylineActionGroup();
                WaylineActionTrigger trigger = new WaylineActionTrigger();
                trigger.setTriggerType(WaylineActionTriggerType.REACH_POINT);
                actionGroup.setTrigger(trigger);
                actionGroup.setGroupId(actionGroups.size());
                actionGroup.setStartIndex(i);
                actionGroup.setEndIndex(i);
                actionGroups.add(actionGroup);
                actionGroup.setActions(actionInfos);

                List<WaylineActionNodeList> nodeLists = new ArrayList<>();

                WaylineActionNodeList root = new WaylineActionNodeList();
                List<WaylineActionTreeNode> treeNodes = new ArrayList<>();
                WaylineActionTreeNode rootNode = new WaylineActionTreeNode();
                rootNode.setNodeType(WaylineActionsRelationType.SEQUENCE);
                rootNode.setChildrenNum(actionInfos.size());
                treeNodes.add(rootNode);
                root.setNodes(treeNodes);
                nodeLists.add(root);

                WaylineActionNodeList children = new WaylineActionNodeList();
                List<WaylineActionTreeNode> childrenNodeList = new ArrayList<>();
                for (int j = 0; j <  actionInfos.size(); ++j) {
                    WaylineActionTreeNode child = new WaylineActionTreeNode();
                    child.setNodeType(WaylineActionsRelationType.LEAF);
                    child.setActionIndex(j);
                    childrenNodeList.add(child);
                }
                children.setNodes(childrenNodeList);
                nodeLists.add(children);

                actionGroup.setNodeLists(nodeLists);
            }
        }

        // 处理定时定距动作
        transFromIntervalActions(waypoints, actionGroups);

        return actionGroups;
    }

    private void transFromIntervalActions(List<WaypointModel> waypoints, List<WaylineActionGroup> actionGroups) {
        WaylineActionGroup intervalShotAction = null;
        for (int i = 0; i < waypoints.size(); ++i) {
            List<WaypointActionModel> actionModels = waypoints.get(i).getActions();

            for (WaypointActionModel item : actionModels) {
                if (intervalShotAction == null &&
                        (item.getActionType() == WaypointActionType.START_TIME_INTERVAL_SHOT
                                || item.getActionType() == WaypointActionType.START_DISTANCE_INTERVAL_SHOT)) {
                    intervalShotAction = getIntevalShootActionGroup(actionGroups, i, item);

                } else if (item.getActionType() == WaypointActionType.STOP_INTERVAL_SHOT && intervalShotAction != null) {
                    intervalShotAction.setEndIndex(i);
                    actionGroups.add(intervalShotAction);
                    intervalShotAction = null;
                }
            }
        }

        if (intervalShotAction != null) {
            intervalShotAction.setEndIndex(waypoints.size() - 1);
            actionGroups.add(intervalShotAction);
        }
    }

    private WaylineActionGroup getIntevalShootActionGroup(List<WaylineActionGroup> actionGroups, int waypointIndex, WaypointActionModel item) {
        WaylineActionGroup actionGroup = new WaylineActionGroup();
        actionGroup.setGroupId(actionGroups.size());
        actionGroup.setStartIndex(waypointIndex);

        actionGroup.setTrigger(getActionTrigger(item));

        List<WaylineActionInfo> actionInfos = new ArrayList<>();
        WaylineActionInfo actionInfo = new WaylineActionInfo();
        actionInfo.setActionId(actionInfos.size());
        actionInfo.setActionType(WaylineActionType.TAKE_PHOTO);
        List<CameraLensType> photoTypes = new ArrayList<>();
        if (item.getPhotoTypes() != null) {
            photoTypes.addAll(item.getPhotoTypes());
        }
        ActionTakePhotoParam param = new ActionTakePhotoParam(item.getCameraIndex(), item.isFollowPhotoType(), photoTypes, null);
        actionInfo.setTakePhotoParam(param);
        actionInfos.add(actionInfo);
        actionGroup.setActions(actionInfos);

        List<WaylineActionNodeList> nodeLists = new ArrayList<>();
        WaylineActionNodeList root = new WaylineActionNodeList();
        List<WaylineActionTreeNode> treeNodes = new ArrayList<>();
        WaylineActionTreeNode rootNode = new WaylineActionTreeNode();
        rootNode.setNodeType(WaylineActionsRelationType.SEQUENCE);
        rootNode.setChildrenNum(actionInfos.size());
        treeNodes.add(rootNode);
        root.setNodes(treeNodes);
        nodeLists.add(root);

        WaylineActionNodeList children = new WaylineActionNodeList();
        List<WaylineActionTreeNode> childrenNodeList = new ArrayList<>();
        for (int j = 0; j <  actionInfos.size(); ++j) {
            WaylineActionTreeNode child = new WaylineActionTreeNode();
            child.setNodeType(WaylineActionsRelationType.LEAF);
            child.setActionIndex(j);
            childrenNodeList.add(child);
        }
        children.setNodes(childrenNodeList);
        nodeLists.add(children);
        actionGroup.setNodeLists(nodeLists);
        return actionGroup;
    }

    private WaylineActionTrigger getActionTrigger(WaypointActionModel item) {
        WaylineActionTrigger trigger = new WaylineActionTrigger();
        if (item.getActionType() == WaypointActionType.START_TIME_INTERVAL_SHOT) {
            trigger.setTriggerType(WaylineActionTriggerType.MULTIPLE_TIMING);
            trigger.setTimeInterval((double) item.getParam());
        } else {
            trigger.setTriggerType(WaylineActionTriggerType.MULTIPLE_DISTANCE);
            trigger.setDistanceInterval((double) item.getParam());
        }
        return trigger;
    }

    private void transformActionTo(List<WaylineActionGroup> waylineActionGroups, List<WaypointModel> waypointModels) {
        for (WaylineActionGroup actionGroup : waylineActionGroups) {
            WaylineActionTrigger trigger = actionGroup.getTrigger();
            WaylineActionTriggerType triggerType = trigger.getTriggerType();
            int pointIndex = actionGroup.getStartIndex();
            WaypointModel waypointModel = waypointModels.get(pointIndex);
            if (waypointModel.getActions() == null) {
                waypointModel.setActions(new ArrayList<>());
            }

            List<WaypointActionModel> actionModels = waypointModel.getActions();
            if (triggerType == WaylineActionTriggerType.REACH_POINT) {
                transReachPointActionsTo(actionGroup, waypointModel, actionModels);

            } else if (triggerType == WaylineActionTriggerType.MULTIPLE_TIMING) {
                transMultiTimeActionsTo(actionGroup, waypointModels,
                        trigger.getTimeInterval().intValue(), actionModels);

            } else if (triggerType == WaylineActionTriggerType.MULTIPLE_DISTANCE) {
                transMultiDistanceActionsTo(actionGroup, waypointModels,
                        trigger.getDistanceInterval().intValue(), actionModels);
            }
        }
    }

    private void transReachPointActionsTo(WaylineActionGroup actionGroup,
                                                         WaypointModel waypointModel,
                                                         List<WaypointActionModel> actionModels) {
        List<WaylineActionInfo> actionInfos = actionGroup.getActions();
        for (WaylineActionInfo actionInfo : actionInfos) {
            WaypointActionModel actionModel = mWaylineActionTemplateTransform.trans(actionInfo);
            if (actionModel != null) {
                actionModels.add(actionModel);
            }
        }
    }

    private void transMultiTimeActionsTo(WaylineActionGroup actionGroup,
                                         List<WaypointModel> waypointModels,
                                         int timeInterval,
                                         List<WaypointActionModel> actionModels) {
        List<WaylineActionInfo> infoList = actionGroup.getActions();
        ActionTakePhotoParam takePhotoParam = infoList.get(0).getTakePhotoParam();
        int actuatorIndex = takePhotoParam.getPayloadPositionIndex();

        WaypointActionModel actionModel = mWaylineActionTemplateTransform.getTimeIntervalShotAction(actuatorIndex, timeInterval);
        actionModel.setFollowPhotoType(takePhotoParam.getUseGlobalPayloadLensIndex());
        actionModel.setPhotoTypes(new HashSet<>(takePhotoParam.getPayloadLensIndex()));
        actionModels.add(actionModel);

        int endIndex = actionGroup.getEndIndex();
        WaypointModel endWaypointModel = waypointModels.get(endIndex);
        if (endWaypointModel.getActions() == null) {
            endWaypointModel.setActions(new ArrayList<>());
        }
        List<WaypointActionModel> endWaypointActions = endWaypointModel.getActions();
        endWaypointActions.add(mWaylineActionTemplateTransform.getIntervalShotStopAction(actuatorIndex));
    }

    private void transMultiDistanceActionsTo(WaylineActionGroup actionGroup,
                                             List<WaypointModel> waypointModels,
                                             int distanceInterval,
                                             List<WaypointActionModel> actionModels) {
        List<WaylineActionInfo> infoList = actionGroup.getActions();
        ActionTakePhotoParam takePhotoParam = infoList.get(0).getTakePhotoParam();
        int actuatorIndex = takePhotoParam.getPayloadPositionIndex();

        WaypointActionModel actionModel = mWaylineActionTemplateTransform.getDistanceIntervalShotAction(actuatorIndex, distanceInterval);
        actionModel.setFollowPhotoType(takePhotoParam.getUseGlobalPayloadLensIndex());
        actionModel.setPhotoTypes(new HashSet<>(takePhotoParam.getPayloadLensIndex()));
        actionModels.add(actionModel);

        int endIndex = actionGroup.getEndIndex();
        WaypointModel endWaypointModel = waypointModels.get(endIndex);
        if (endWaypointModel.getActions() == null) {
            endWaypointModel.setActions(new ArrayList<>());
        }
        List<WaypointActionModel> endWaypointActions = endWaypointModel.getActions();
        endWaypointActions.add(mWaylineActionTemplateTransform.getIntervalShotStopAction(actuatorIndex));
    }
}
