/*
 * Copyright 2015 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.google.cloud.bigquery;

import static com.google.common.base.Preconditions.checkNotNull;

import com.google.api.services.bigquery.model.JobReference;
import com.google.auto.value.AutoValue;
import java.io.Serializable;
import java.util.UUID;
import javax.annotation.Nullable;

/** Google BigQuery Job identity. */
@AutoValue
public abstract class JobId implements Serializable {

  private static final long serialVersionUID = 1225914835379688977L;

  JobId() {
    // Users cannot extend this, but AutoValue can.
  }

  /**
   * Returns job's project id.
   *
   * <p>When sending requests with null project, the client will attempt to infer the project name
   * from the environment.
   */
  @Nullable
  public abstract String getProject();

  /**
   * Returns the job's id.
   *
   * <p>The server returns null job id for dry-run queries.
   */
  @Nullable
  public abstract String getJob();

  /**
   * Returns the job's location.
   *
   * <p>When sending requests, the location must be specified for jobs whose location not "US" or
   * "EU".
   */
  @Nullable
  public abstract String getLocation();

  public abstract Builder toBuilder();

  public static Builder newBuilder() {
    return new AutoValue_JobId.Builder();
  }

  @AutoValue.Builder
  public abstract static class Builder {
    public abstract Builder setProject(String project);

    public abstract Builder setJob(String job);

    /** {@code setJob} to a pseudo-random string. */
    public Builder setRandomJob() {
      return setJob(UUID.randomUUID().toString());
    }

    public abstract Builder setLocation(String location);

    public abstract JobId build();
  }

  /**
   * Creates a job identity given project's and job's user-defined id.
   */
  public static JobId of(String project, String job) {
    return newBuilder().setProject(checkNotNull(project)).setJob(checkNotNull(job)).build();
  }

  /**
   * Creates a job identity given only its user-defined id.
   */
  public static JobId of(String job) {
    return newBuilder().setJob(checkNotNull(job)).build();
  }

  /**
   * Creates a job identity with autogenerated id and no project specified.
   */
  public static JobId of() {
    return newBuilder().setRandomJob().build();
  }

  JobId setProjectId(String projectId) {
    return getProject() != null ? this : toBuilder().setProject(projectId).build();
  }

  JobReference toPb() {
    return new JobReference()
        .setProjectId(getProject())
        .setJobId(getJob())
        .setLocation(getLocation());
  }

  static JobId fromPb(JobReference jobRef) {
    return newBuilder()
        .setProject(jobRef.getProjectId())
        .setJob(jobRef.getJobId())
        .setLocation(jobRef.getLocation())
        .build();
  }
}
