/**
 * Copyright (c) 2016, 2024, Oracle and/or its affiliates.  All rights reserved.
 * This software is dual-licensed to you under the Universal Permissive License (UPL) 1.0 as shown at https://oss.oracle.com/licenses/upl or Apache License 2.0 as shown at http://www.apache.org/licenses/LICENSE-2.0. You may choose either license.
 */
package com.oracle.bmc.vulnerabilityscanning;

import com.oracle.bmc.util.internal.Validate;
import com.oracle.bmc.vulnerabilityscanning.requests.*;
import com.oracle.bmc.vulnerabilityscanning.responses.*;

import java.util.Objects;

/**
 * Async client implementation for VulnerabilityScanning service. <br>
 * There are two ways to use async client: 1. Use AsyncHandler: using AsyncHandler, if the response
 * to the call is an {@link java.io.InputStream}, like getObject Api in object storage service,
 * developers need to process the stream in AsyncHandler, and not anywhere else, because the stream
 * will be closed right after the AsyncHandler is invoked. <br>
 * 2. Use Java Future: using Java Future, developers need to close the stream after they are done
 * with the Java Future.<br>
 * Accessing the result should be done in a mutually exclusive manner, either through the Future or
 * the AsyncHandler, but not both. If the Future is used, the caller should pass in null as the
 * AsyncHandler. If the AsyncHandler is used, it is still safe to use the Future to determine
 * whether or not the request was completed via Future.isDone/isCancelled.<br>
 * Please refer to
 * https://github.com/oracle/oci-java-sdk/blob/master/bmc-examples/src/main/java/ResteasyClientWithObjectStorageExample.java
 */
@jakarta.annotation.Generated(value = "OracleSDKGenerator", comments = "API Version: 20210215")
public class VulnerabilityScanningAsyncClient extends com.oracle.bmc.http.internal.BaseAsyncClient
        implements VulnerabilityScanningAsync {
    /** Service instance for VulnerabilityScanning. */
    public static final com.oracle.bmc.Service SERVICE =
            com.oracle.bmc.Services.serviceBuilder()
                    .serviceName("VULNERABILITYSCANNING")
                    .serviceEndpointPrefix("")
                    .serviceEndpointTemplate("https://vss-cp-api.{region}.oci.{secondLevelDomain}")
                    .build();

    private static final org.slf4j.Logger LOG =
            org.slf4j.LoggerFactory.getLogger(VulnerabilityScanningAsyncClient.class);

    VulnerabilityScanningAsyncClient(
            com.oracle.bmc.common.ClientBuilderBase<?, ?> builder,
            com.oracle.bmc.auth.AbstractAuthenticationDetailsProvider
                    authenticationDetailsProvider) {
        this(builder, authenticationDetailsProvider, true);
    }

    VulnerabilityScanningAsyncClient(
            com.oracle.bmc.common.ClientBuilderBase<?, ?> builder,
            com.oracle.bmc.auth.AbstractAuthenticationDetailsProvider authenticationDetailsProvider,
            boolean isStreamWarningEnabled) {
        super(builder, authenticationDetailsProvider);

        if (isStreamWarningEnabled && com.oracle.bmc.util.StreamUtils.isExtraStreamLogsEnabled()) {
            LOG.warn(
                    com.oracle.bmc.util.StreamUtils.getStreamWarningMessage(
                            "VulnerabilityScanningAsyncClient",
                            "exportHostAgentScanResultCsv,exportHostVulnerabilityCsv"));
        }
    }

    /**
     * Create a builder for this client.
     *
     * @return builder
     */
    public static Builder builder() {
        return new Builder(SERVICE);
    }

    /**
     * Builder class for this client. The "authenticationDetailsProvider" is required and must be
     * passed to the {@link #build(AbstractAuthenticationDetailsProvider)} method.
     */
    public static class Builder
            extends com.oracle.bmc.common.RegionalClientBuilder<
                    Builder, VulnerabilityScanningAsyncClient> {
        private boolean isStreamWarningEnabled = true;

        private Builder(com.oracle.bmc.Service service) {
            super(service);
            final String packageName = "vulnerabilityscanning";
            com.oracle.bmc.internal.Alloy.throwDisabledServiceExceptionIfAppropriate(packageName);
            requestSignerFactory =
                    new com.oracle.bmc.http.signing.internal.DefaultRequestSignerFactory(
                            com.oracle.bmc.http.signing.SigningStrategy.STANDARD);
        }

        /**
         * Enable/disable the stream warnings for the client
         *
         * @param isStreamWarningEnabled executorService
         * @return this builder
         */
        public Builder isStreamWarningEnabled(boolean isStreamWarningEnabled) {
            this.isStreamWarningEnabled = isStreamWarningEnabled;
            return this;
        }

        /**
         * Build the client.
         *
         * @param authenticationDetailsProvider authentication details provider
         * @return the client
         */
        public VulnerabilityScanningAsyncClient build(
                @jakarta.annotation.Nonnull
                        com.oracle.bmc.auth.AbstractAuthenticationDetailsProvider
                                authenticationDetailsProvider) {
            return new VulnerabilityScanningAsyncClient(
                    this, authenticationDetailsProvider, isStreamWarningEnabled);
        }
    }

    @Override
    public void setRegion(com.oracle.bmc.Region region) {
        super.setRegion(region);
    }

    @Override
    public void setRegion(String regionId) {
        super.setRegion(regionId);
    }

    @Override
    public java.util.concurrent.Future<ChangeContainerScanRecipeCompartmentResponse>
            changeContainerScanRecipeCompartment(
                    ChangeContainerScanRecipeCompartmentRequest request,
                    final com.oracle.bmc.responses.AsyncHandler<
                                    ChangeContainerScanRecipeCompartmentRequest,
                                    ChangeContainerScanRecipeCompartmentResponse>
                            handler) {

        Validate.notBlank(
                request.getContainerScanRecipeId(), "containerScanRecipeId must not be blank");
        Objects.requireNonNull(
                request.getChangeContainerScanRecipeCompartmentDetails(),
                "changeContainerScanRecipeCompartmentDetails is required");

        return clientCall(request, ChangeContainerScanRecipeCompartmentResponse::builder)
                .logger(LOG, "changeContainerScanRecipeCompartment")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ChangeContainerScanRecipeCompartment",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/ContainerScanRecipe/ChangeContainerScanRecipeCompartment")
                .method(com.oracle.bmc.http.client.Method.POST)
                .requestBuilder(ChangeContainerScanRecipeCompartmentRequest::builder)
                .basePath("/20210215")
                .appendPathParam("containerScanRecipes")
                .appendPathParam(request.getContainerScanRecipeId())
                .appendPathParam("actions")
                .appendPathParam("changeCompartment")
                .accept("application/json")
                .appendHeader("if-match", request.getIfMatch())
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .appendHeader("opc-retry-token", request.getOpcRetryToken())
                .hasBody()
                .handleResponseHeaderString(
                        "opc-request-id",
                        ChangeContainerScanRecipeCompartmentResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ChangeContainerScanResultCompartmentResponse>
            changeContainerScanResultCompartment(
                    ChangeContainerScanResultCompartmentRequest request,
                    final com.oracle.bmc.responses.AsyncHandler<
                                    ChangeContainerScanResultCompartmentRequest,
                                    ChangeContainerScanResultCompartmentResponse>
                            handler) {

        Validate.notBlank(
                request.getContainerScanResultId(), "containerScanResultId must not be blank");
        Objects.requireNonNull(
                request.getChangeContainerScanResultCompartmentDetails(),
                "changeContainerScanResultCompartmentDetails is required");

        return clientCall(request, ChangeContainerScanResultCompartmentResponse::builder)
                .logger(LOG, "changeContainerScanResultCompartment")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ChangeContainerScanResultCompartment",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/ContainerScanResult/ChangeContainerScanResultCompartment")
                .method(com.oracle.bmc.http.client.Method.POST)
                .requestBuilder(ChangeContainerScanResultCompartmentRequest::builder)
                .basePath("/20210215")
                .appendPathParam("containerScanResults")
                .appendPathParam(request.getContainerScanResultId())
                .appendPathParam("actions")
                .appendPathParam("changeCompartment")
                .accept("application/json")
                .appendHeader("if-match", request.getIfMatch())
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .appendHeader("opc-retry-token", request.getOpcRetryToken())
                .hasBody()
                .handleResponseHeaderString(
                        "opc-request-id",
                        ChangeContainerScanResultCompartmentResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ChangeContainerScanTargetCompartmentResponse>
            changeContainerScanTargetCompartment(
                    ChangeContainerScanTargetCompartmentRequest request,
                    final com.oracle.bmc.responses.AsyncHandler<
                                    ChangeContainerScanTargetCompartmentRequest,
                                    ChangeContainerScanTargetCompartmentResponse>
                            handler) {

        Validate.notBlank(
                request.getContainerScanTargetId(), "containerScanTargetId must not be blank");
        Objects.requireNonNull(
                request.getChangeContainerScanTargetCompartmentDetails(),
                "changeContainerScanTargetCompartmentDetails is required");

        return clientCall(request, ChangeContainerScanTargetCompartmentResponse::builder)
                .logger(LOG, "changeContainerScanTargetCompartment")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ChangeContainerScanTargetCompartment",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/ContainerScanTarget/ChangeContainerScanTargetCompartment")
                .method(com.oracle.bmc.http.client.Method.POST)
                .requestBuilder(ChangeContainerScanTargetCompartmentRequest::builder)
                .basePath("/20210215")
                .appendPathParam("containerScanTargets")
                .appendPathParam(request.getContainerScanTargetId())
                .appendPathParam("actions")
                .appendPathParam("changeCompartment")
                .accept("application/json")
                .appendHeader("if-match", request.getIfMatch())
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .appendHeader("opc-retry-token", request.getOpcRetryToken())
                .hasBody()
                .handleResponseHeaderString(
                        "opc-request-id",
                        ChangeContainerScanTargetCompartmentResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ChangeHostAgentScanResultCompartmentResponse>
            changeHostAgentScanResultCompartment(
                    ChangeHostAgentScanResultCompartmentRequest request,
                    final com.oracle.bmc.responses.AsyncHandler<
                                    ChangeHostAgentScanResultCompartmentRequest,
                                    ChangeHostAgentScanResultCompartmentResponse>
                            handler) {

        Validate.notBlank(
                request.getHostAgentScanResultId(), "hostAgentScanResultId must not be blank");
        Objects.requireNonNull(
                request.getChangeHostAgentScanResultCompartmentDetails(),
                "changeHostAgentScanResultCompartmentDetails is required");

        return clientCall(request, ChangeHostAgentScanResultCompartmentResponse::builder)
                .logger(LOG, "changeHostAgentScanResultCompartment")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ChangeHostAgentScanResultCompartment",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostAgentScanResult/ChangeHostAgentScanResultCompartment")
                .method(com.oracle.bmc.http.client.Method.POST)
                .requestBuilder(ChangeHostAgentScanResultCompartmentRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostAgentScanResults")
                .appendPathParam(request.getHostAgentScanResultId())
                .appendPathParam("actions")
                .appendPathParam("changeCompartment")
                .accept("application/json")
                .appendHeader("if-match", request.getIfMatch())
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .appendHeader("opc-retry-token", request.getOpcRetryToken())
                .hasBody()
                .handleResponseHeaderString(
                        "opc-request-id",
                        ChangeHostAgentScanResultCompartmentResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ChangeHostCisBenchmarkScanResultCompartmentResponse>
            changeHostCisBenchmarkScanResultCompartment(
                    ChangeHostCisBenchmarkScanResultCompartmentRequest request,
                    final com.oracle.bmc.responses.AsyncHandler<
                                    ChangeHostCisBenchmarkScanResultCompartmentRequest,
                                    ChangeHostCisBenchmarkScanResultCompartmentResponse>
                            handler) {

        Validate.notBlank(
                request.getHostCisBenchmarkScanResultId(),
                "hostCisBenchmarkScanResultId must not be blank");
        Objects.requireNonNull(
                request.getChangeHostCisBenchmarkScanResultCompartmentDetails(),
                "changeHostCisBenchmarkScanResultCompartmentDetails is required");

        return clientCall(request, ChangeHostCisBenchmarkScanResultCompartmentResponse::builder)
                .logger(LOG, "changeHostCisBenchmarkScanResultCompartment")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ChangeHostCisBenchmarkScanResultCompartment",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostCisBenchmarkScanResult/ChangeHostCisBenchmarkScanResultCompartment")
                .method(com.oracle.bmc.http.client.Method.POST)
                .requestBuilder(ChangeHostCisBenchmarkScanResultCompartmentRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostCisBenchmarkScanResults")
                .appendPathParam(request.getHostCisBenchmarkScanResultId())
                .appendPathParam("actions")
                .appendPathParam("changeCompartment")
                .accept("application/json")
                .appendHeader("if-match", request.getIfMatch())
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .appendHeader("opc-retry-token", request.getOpcRetryToken())
                .hasBody()
                .handleResponseHeaderString(
                        "opc-request-id",
                        ChangeHostCisBenchmarkScanResultCompartmentResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ChangeHostEndpointProtectionScanResultCompartmentResponse>
            changeHostEndpointProtectionScanResultCompartment(
                    ChangeHostEndpointProtectionScanResultCompartmentRequest request,
                    final com.oracle.bmc.responses.AsyncHandler<
                                    ChangeHostEndpointProtectionScanResultCompartmentRequest,
                                    ChangeHostEndpointProtectionScanResultCompartmentResponse>
                            handler) {

        Validate.notBlank(
                request.getHostEndpointProtectionScanResultId(),
                "hostEndpointProtectionScanResultId must not be blank");
        Objects.requireNonNull(
                request.getChangeHostEndpointProtectionScanResultCompartmentDetails(),
                "changeHostEndpointProtectionScanResultCompartmentDetails is required");

        return clientCall(
                        request, ChangeHostEndpointProtectionScanResultCompartmentResponse::builder)
                .logger(LOG, "changeHostEndpointProtectionScanResultCompartment")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ChangeHostEndpointProtectionScanResultCompartment",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostEndpointProtectionScanResult/ChangeHostEndpointProtectionScanResultCompartment")
                .method(com.oracle.bmc.http.client.Method.POST)
                .requestBuilder(ChangeHostEndpointProtectionScanResultCompartmentRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostEndpointProtectionScanResults")
                .appendPathParam(request.getHostEndpointProtectionScanResultId())
                .appendPathParam("actions")
                .appendPathParam("changeCompartment")
                .accept("application/json")
                .appendHeader("if-match", request.getIfMatch())
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .appendHeader("opc-retry-token", request.getOpcRetryToken())
                .hasBody()
                .handleResponseHeaderString(
                        "opc-request-id",
                        ChangeHostEndpointProtectionScanResultCompartmentResponse.Builder
                                ::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ChangeHostPortScanResultCompartmentResponse>
            changeHostPortScanResultCompartment(
                    ChangeHostPortScanResultCompartmentRequest request,
                    final com.oracle.bmc.responses.AsyncHandler<
                                    ChangeHostPortScanResultCompartmentRequest,
                                    ChangeHostPortScanResultCompartmentResponse>
                            handler) {

        Validate.notBlank(
                request.getHostPortScanResultId(), "hostPortScanResultId must not be blank");
        Objects.requireNonNull(
                request.getChangeHostPortScanResultCompartmentDetails(),
                "changeHostPortScanResultCompartmentDetails is required");

        return clientCall(request, ChangeHostPortScanResultCompartmentResponse::builder)
                .logger(LOG, "changeHostPortScanResultCompartment")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ChangeHostPortScanResultCompartment",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostPortScanResult/ChangeHostPortScanResultCompartment")
                .method(com.oracle.bmc.http.client.Method.POST)
                .requestBuilder(ChangeHostPortScanResultCompartmentRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostPortScanResults")
                .appendPathParam(request.getHostPortScanResultId())
                .appendPathParam("actions")
                .appendPathParam("changeCompartment")
                .accept("application/json")
                .appendHeader("if-match", request.getIfMatch())
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .appendHeader("opc-retry-token", request.getOpcRetryToken())
                .hasBody()
                .handleResponseHeaderString(
                        "opc-request-id",
                        ChangeHostPortScanResultCompartmentResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ChangeHostScanRecipeCompartmentResponse>
            changeHostScanRecipeCompartment(
                    ChangeHostScanRecipeCompartmentRequest request,
                    final com.oracle.bmc.responses.AsyncHandler<
                                    ChangeHostScanRecipeCompartmentRequest,
                                    ChangeHostScanRecipeCompartmentResponse>
                            handler) {

        Validate.notBlank(request.getHostScanRecipeId(), "hostScanRecipeId must not be blank");
        Objects.requireNonNull(
                request.getChangeHostScanRecipeCompartmentDetails(),
                "changeHostScanRecipeCompartmentDetails is required");

        return clientCall(request, ChangeHostScanRecipeCompartmentResponse::builder)
                .logger(LOG, "changeHostScanRecipeCompartment")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ChangeHostScanRecipeCompartment",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostScanRecipe/ChangeHostScanRecipeCompartment")
                .method(com.oracle.bmc.http.client.Method.POST)
                .requestBuilder(ChangeHostScanRecipeCompartmentRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostScanRecipes")
                .appendPathParam(request.getHostScanRecipeId())
                .appendPathParam("actions")
                .appendPathParam("changeCompartment")
                .accept("application/json")
                .appendHeader("if-match", request.getIfMatch())
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .appendHeader("opc-retry-token", request.getOpcRetryToken())
                .hasBody()
                .handleResponseHeaderString(
                        "opc-request-id",
                        ChangeHostScanRecipeCompartmentResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ChangeHostScanTargetCompartmentResponse>
            changeHostScanTargetCompartment(
                    ChangeHostScanTargetCompartmentRequest request,
                    final com.oracle.bmc.responses.AsyncHandler<
                                    ChangeHostScanTargetCompartmentRequest,
                                    ChangeHostScanTargetCompartmentResponse>
                            handler) {

        Validate.notBlank(request.getHostScanTargetId(), "hostScanTargetId must not be blank");
        Objects.requireNonNull(
                request.getChangeHostScanTargetCompartmentDetails(),
                "changeHostScanTargetCompartmentDetails is required");

        return clientCall(request, ChangeHostScanTargetCompartmentResponse::builder)
                .logger(LOG, "changeHostScanTargetCompartment")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ChangeHostScanTargetCompartment",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostScanTarget/ChangeHostScanTargetCompartment")
                .method(com.oracle.bmc.http.client.Method.POST)
                .requestBuilder(ChangeHostScanTargetCompartmentRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostScanTargets")
                .appendPathParam(request.getHostScanTargetId())
                .appendPathParam("actions")
                .appendPathParam("changeCompartment")
                .accept("application/json")
                .appendHeader("if-match", request.getIfMatch())
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .appendHeader("opc-retry-token", request.getOpcRetryToken())
                .hasBody()
                .handleResponseHeaderString(
                        "opc-request-id",
                        ChangeHostScanTargetCompartmentResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<CreateContainerScanRecipeResponse> createContainerScanRecipe(
            CreateContainerScanRecipeRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            CreateContainerScanRecipeRequest, CreateContainerScanRecipeResponse>
                    handler) {
        Objects.requireNonNull(
                request.getCreateContainerScanRecipeDetails(),
                "createContainerScanRecipeDetails is required");

        return clientCall(request, CreateContainerScanRecipeResponse::builder)
                .logger(LOG, "createContainerScanRecipe")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "CreateContainerScanRecipe",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/ContainerScanRecipe/CreateContainerScanRecipe")
                .method(com.oracle.bmc.http.client.Method.POST)
                .requestBuilder(CreateContainerScanRecipeRequest::builder)
                .basePath("/20210215")
                .appendPathParam("containerScanRecipes")
                .accept("application/json")
                .appendHeader("opc-retry-token", request.getOpcRetryToken())
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .hasBody()
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model.ContainerScanRecipe.class,
                        CreateContainerScanRecipeResponse.Builder::containerScanRecipe)
                .handleResponseHeaderString("etag", CreateContainerScanRecipeResponse.Builder::etag)
                .handleResponseHeaderString(
                        "opc-work-request-id",
                        CreateContainerScanRecipeResponse.Builder::opcWorkRequestId)
                .handleResponseHeaderString(
                        "opc-request-id", CreateContainerScanRecipeResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<CreateContainerScanTargetResponse> createContainerScanTarget(
            CreateContainerScanTargetRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            CreateContainerScanTargetRequest, CreateContainerScanTargetResponse>
                    handler) {
        Objects.requireNonNull(
                request.getCreateContainerScanTargetDetails(),
                "createContainerScanTargetDetails is required");

        return clientCall(request, CreateContainerScanTargetResponse::builder)
                .logger(LOG, "createContainerScanTarget")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "CreateContainerScanTarget",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/ContainerScanTarget/CreateContainerScanTarget")
                .method(com.oracle.bmc.http.client.Method.POST)
                .requestBuilder(CreateContainerScanTargetRequest::builder)
                .basePath("/20210215")
                .appendPathParam("containerScanTargets")
                .accept("application/json")
                .appendHeader("opc-retry-token", request.getOpcRetryToken())
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .hasBody()
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model.ContainerScanTarget.class,
                        CreateContainerScanTargetResponse.Builder::containerScanTarget)
                .handleResponseHeaderString("etag", CreateContainerScanTargetResponse.Builder::etag)
                .handleResponseHeaderString(
                        "opc-work-request-id",
                        CreateContainerScanTargetResponse.Builder::opcWorkRequestId)
                .handleResponseHeaderString(
                        "opc-request-id", CreateContainerScanTargetResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<CreateHostScanRecipeResponse> createHostScanRecipe(
            CreateHostScanRecipeRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            CreateHostScanRecipeRequest, CreateHostScanRecipeResponse>
                    handler) {
        Objects.requireNonNull(
                request.getCreateHostScanRecipeDetails(),
                "createHostScanRecipeDetails is required");

        return clientCall(request, CreateHostScanRecipeResponse::builder)
                .logger(LOG, "createHostScanRecipe")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "CreateHostScanRecipe",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostScanRecipe/CreateHostScanRecipe")
                .method(com.oracle.bmc.http.client.Method.POST)
                .requestBuilder(CreateHostScanRecipeRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostScanRecipes")
                .accept("application/json")
                .appendHeader("opc-retry-token", request.getOpcRetryToken())
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .hasBody()
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model.HostScanRecipe.class,
                        CreateHostScanRecipeResponse.Builder::hostScanRecipe)
                .handleResponseHeaderString("etag", CreateHostScanRecipeResponse.Builder::etag)
                .handleResponseHeaderString(
                        "opc-work-request-id",
                        CreateHostScanRecipeResponse.Builder::opcWorkRequestId)
                .handleResponseHeaderString(
                        "opc-request-id", CreateHostScanRecipeResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<CreateHostScanTargetResponse> createHostScanTarget(
            CreateHostScanTargetRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            CreateHostScanTargetRequest, CreateHostScanTargetResponse>
                    handler) {
        Objects.requireNonNull(
                request.getCreateHostScanTargetDetails(),
                "createHostScanTargetDetails is required");

        return clientCall(request, CreateHostScanTargetResponse::builder)
                .logger(LOG, "createHostScanTarget")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "CreateHostScanTarget",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostScanTarget/CreateHostScanTarget")
                .method(com.oracle.bmc.http.client.Method.POST)
                .requestBuilder(CreateHostScanTargetRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostScanTargets")
                .accept("application/json")
                .appendHeader("opc-retry-token", request.getOpcRetryToken())
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .hasBody()
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model.HostScanTarget.class,
                        CreateHostScanTargetResponse.Builder::hostScanTarget)
                .handleResponseHeaderString("etag", CreateHostScanTargetResponse.Builder::etag)
                .handleResponseHeaderString(
                        "opc-work-request-id",
                        CreateHostScanTargetResponse.Builder::opcWorkRequestId)
                .handleResponseHeaderString(
                        "opc-request-id", CreateHostScanTargetResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<DeleteContainerScanRecipeResponse> deleteContainerScanRecipe(
            DeleteContainerScanRecipeRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            DeleteContainerScanRecipeRequest, DeleteContainerScanRecipeResponse>
                    handler) {

        Validate.notBlank(
                request.getContainerScanRecipeId(), "containerScanRecipeId must not be blank");

        return clientCall(request, DeleteContainerScanRecipeResponse::builder)
                .logger(LOG, "deleteContainerScanRecipe")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "DeleteContainerScanRecipe",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/ContainerScanRecipe/DeleteContainerScanRecipe")
                .method(com.oracle.bmc.http.client.Method.DELETE)
                .requestBuilder(DeleteContainerScanRecipeRequest::builder)
                .basePath("/20210215")
                .appendPathParam("containerScanRecipes")
                .appendPathParam(request.getContainerScanRecipeId())
                .accept("application/json")
                .appendHeader("if-match", request.getIfMatch())
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleResponseHeaderString(
                        "opc-work-request-id",
                        DeleteContainerScanRecipeResponse.Builder::opcWorkRequestId)
                .handleResponseHeaderString(
                        "opc-request-id", DeleteContainerScanRecipeResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<DeleteContainerScanResultResponse> deleteContainerScanResult(
            DeleteContainerScanResultRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            DeleteContainerScanResultRequest, DeleteContainerScanResultResponse>
                    handler) {

        Validate.notBlank(
                request.getContainerScanResultId(), "containerScanResultId must not be blank");

        return clientCall(request, DeleteContainerScanResultResponse::builder)
                .logger(LOG, "deleteContainerScanResult")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "DeleteContainerScanResult",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/ContainerScanResult/DeleteContainerScanResult")
                .method(com.oracle.bmc.http.client.Method.DELETE)
                .requestBuilder(DeleteContainerScanResultRequest::builder)
                .basePath("/20210215")
                .appendPathParam("containerScanResults")
                .appendPathParam(request.getContainerScanResultId())
                .accept("application/json")
                .appendHeader("if-match", request.getIfMatch())
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleResponseHeaderString(
                        "opc-work-request-id",
                        DeleteContainerScanResultResponse.Builder::opcWorkRequestId)
                .handleResponseHeaderString(
                        "opc-request-id", DeleteContainerScanResultResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<DeleteContainerScanTargetResponse> deleteContainerScanTarget(
            DeleteContainerScanTargetRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            DeleteContainerScanTargetRequest, DeleteContainerScanTargetResponse>
                    handler) {

        Validate.notBlank(
                request.getContainerScanTargetId(), "containerScanTargetId must not be blank");

        return clientCall(request, DeleteContainerScanTargetResponse::builder)
                .logger(LOG, "deleteContainerScanTarget")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "DeleteContainerScanTarget",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/ContainerScanTarget/DeleteContainerScanTarget")
                .method(com.oracle.bmc.http.client.Method.DELETE)
                .requestBuilder(DeleteContainerScanTargetRequest::builder)
                .basePath("/20210215")
                .appendPathParam("containerScanTargets")
                .appendPathParam(request.getContainerScanTargetId())
                .accept("application/json")
                .appendHeader("if-match", request.getIfMatch())
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleResponseHeaderString(
                        "opc-work-request-id",
                        DeleteContainerScanTargetResponse.Builder::opcWorkRequestId)
                .handleResponseHeaderString(
                        "opc-request-id", DeleteContainerScanTargetResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<DeleteHostAgentScanResultResponse> deleteHostAgentScanResult(
            DeleteHostAgentScanResultRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            DeleteHostAgentScanResultRequest, DeleteHostAgentScanResultResponse>
                    handler) {

        Validate.notBlank(
                request.getHostAgentScanResultId(), "hostAgentScanResultId must not be blank");

        return clientCall(request, DeleteHostAgentScanResultResponse::builder)
                .logger(LOG, "deleteHostAgentScanResult")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "DeleteHostAgentScanResult",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostAgentScanResult/DeleteHostAgentScanResult")
                .method(com.oracle.bmc.http.client.Method.DELETE)
                .requestBuilder(DeleteHostAgentScanResultRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostAgentScanResults")
                .appendPathParam(request.getHostAgentScanResultId())
                .accept("application/json")
                .appendHeader("if-match", request.getIfMatch())
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleResponseHeaderString(
                        "opc-work-request-id",
                        DeleteHostAgentScanResultResponse.Builder::opcWorkRequestId)
                .handleResponseHeaderString(
                        "opc-request-id", DeleteHostAgentScanResultResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<DeleteHostCisBenchmarkScanResultResponse>
            deleteHostCisBenchmarkScanResult(
                    DeleteHostCisBenchmarkScanResultRequest request,
                    final com.oracle.bmc.responses.AsyncHandler<
                                    DeleteHostCisBenchmarkScanResultRequest,
                                    DeleteHostCisBenchmarkScanResultResponse>
                            handler) {

        Validate.notBlank(
                request.getHostCisBenchmarkScanResultId(),
                "hostCisBenchmarkScanResultId must not be blank");

        return clientCall(request, DeleteHostCisBenchmarkScanResultResponse::builder)
                .logger(LOG, "deleteHostCisBenchmarkScanResult")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "DeleteHostCisBenchmarkScanResult",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostCisBenchmarkScanResult/DeleteHostCisBenchmarkScanResult")
                .method(com.oracle.bmc.http.client.Method.DELETE)
                .requestBuilder(DeleteHostCisBenchmarkScanResultRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostCisBenchmarkScanResults")
                .appendPathParam(request.getHostCisBenchmarkScanResultId())
                .accept("application/json")
                .appendHeader("if-match", request.getIfMatch())
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleResponseHeaderString(
                        "opc-work-request-id",
                        DeleteHostCisBenchmarkScanResultResponse.Builder::opcWorkRequestId)
                .handleResponseHeaderString(
                        "opc-request-id",
                        DeleteHostCisBenchmarkScanResultResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<DeleteHostEndpointProtectionScanResultResponse>
            deleteHostEndpointProtectionScanResult(
                    DeleteHostEndpointProtectionScanResultRequest request,
                    final com.oracle.bmc.responses.AsyncHandler<
                                    DeleteHostEndpointProtectionScanResultRequest,
                                    DeleteHostEndpointProtectionScanResultResponse>
                            handler) {

        Validate.notBlank(
                request.getHostEndpointProtectionScanResultId(),
                "hostEndpointProtectionScanResultId must not be blank");

        return clientCall(request, DeleteHostEndpointProtectionScanResultResponse::builder)
                .logger(LOG, "deleteHostEndpointProtectionScanResult")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "DeleteHostEndpointProtectionScanResult",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostEndpointProtectionScanResult/DeleteHostEndpointProtectionScanResult")
                .method(com.oracle.bmc.http.client.Method.DELETE)
                .requestBuilder(DeleteHostEndpointProtectionScanResultRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostEndpointProtectionScanResults")
                .appendPathParam(request.getHostEndpointProtectionScanResultId())
                .accept("application/json")
                .appendHeader("if-match", request.getIfMatch())
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleResponseHeaderString(
                        "opc-work-request-id",
                        DeleteHostEndpointProtectionScanResultResponse.Builder::opcWorkRequestId)
                .handleResponseHeaderString(
                        "opc-request-id",
                        DeleteHostEndpointProtectionScanResultResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<DeleteHostPortScanResultResponse> deleteHostPortScanResult(
            DeleteHostPortScanResultRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            DeleteHostPortScanResultRequest, DeleteHostPortScanResultResponse>
                    handler) {

        Validate.notBlank(
                request.getHostPortScanResultId(), "hostPortScanResultId must not be blank");

        return clientCall(request, DeleteHostPortScanResultResponse::builder)
                .logger(LOG, "deleteHostPortScanResult")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "DeleteHostPortScanResult",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostPortScanResult/DeleteHostPortScanResult")
                .method(com.oracle.bmc.http.client.Method.DELETE)
                .requestBuilder(DeleteHostPortScanResultRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostPortScanResults")
                .appendPathParam(request.getHostPortScanResultId())
                .accept("application/json")
                .appendHeader("if-match", request.getIfMatch())
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleResponseHeaderString(
                        "opc-work-request-id",
                        DeleteHostPortScanResultResponse.Builder::opcWorkRequestId)
                .handleResponseHeaderString(
                        "opc-request-id", DeleteHostPortScanResultResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<DeleteHostScanRecipeResponse> deleteHostScanRecipe(
            DeleteHostScanRecipeRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            DeleteHostScanRecipeRequest, DeleteHostScanRecipeResponse>
                    handler) {

        Validate.notBlank(request.getHostScanRecipeId(), "hostScanRecipeId must not be blank");

        return clientCall(request, DeleteHostScanRecipeResponse::builder)
                .logger(LOG, "deleteHostScanRecipe")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "DeleteHostScanRecipe",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostScanRecipe/DeleteHostScanRecipe")
                .method(com.oracle.bmc.http.client.Method.DELETE)
                .requestBuilder(DeleteHostScanRecipeRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostScanRecipes")
                .appendPathParam(request.getHostScanRecipeId())
                .accept("application/json")
                .appendHeader("if-match", request.getIfMatch())
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleResponseHeaderString(
                        "opc-work-request-id",
                        DeleteHostScanRecipeResponse.Builder::opcWorkRequestId)
                .handleResponseHeaderString(
                        "opc-request-id", DeleteHostScanRecipeResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<DeleteHostScanTargetResponse> deleteHostScanTarget(
            DeleteHostScanTargetRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            DeleteHostScanTargetRequest, DeleteHostScanTargetResponse>
                    handler) {

        Validate.notBlank(request.getHostScanTargetId(), "hostScanTargetId must not be blank");

        return clientCall(request, DeleteHostScanTargetResponse::builder)
                .logger(LOG, "deleteHostScanTarget")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "DeleteHostScanTarget",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostScanTarget/DeleteHostScanTarget")
                .method(com.oracle.bmc.http.client.Method.DELETE)
                .requestBuilder(DeleteHostScanTargetRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostScanTargets")
                .appendPathParam(request.getHostScanTargetId())
                .accept("application/json")
                .appendHeader("if-match", request.getIfMatch())
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleResponseHeaderString(
                        "opc-work-request-id",
                        DeleteHostScanTargetResponse.Builder::opcWorkRequestId)
                .handleResponseHeaderString(
                        "opc-request-id", DeleteHostScanTargetResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ExportHostAgentScanResultCsvResponse>
            exportHostAgentScanResultCsv(
                    ExportHostAgentScanResultCsvRequest request,
                    final com.oracle.bmc.responses.AsyncHandler<
                                    ExportHostAgentScanResultCsvRequest,
                                    ExportHostAgentScanResultCsvResponse>
                            handler) {
        Objects.requireNonNull(
                request.getExportHostAgentScanResultCsvDetails(),
                "exportHostAgentScanResultCsvDetails is required");

        return clientCall(request, ExportHostAgentScanResultCsvResponse::builder)
                .logger(LOG, "exportHostAgentScanResultCsv")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ExportHostAgentScanResultCsv",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostAgentScanResult/ExportHostAgentScanResultCsv")
                .method(com.oracle.bmc.http.client.Method.POST)
                .requestBuilder(ExportHostAgentScanResultCsvRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostAgentScanResults")
                .appendPathParam("actions")
                .appendPathParam("exportCsv")
                .accept("text/csv")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .hasBody()
                .handleBody(
                        java.io.InputStream.class,
                        ExportHostAgentScanResultCsvResponse.Builder::inputStream)
                .handleResponseHeaderString(
                        "opc-request-id",
                        ExportHostAgentScanResultCsvResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ExportHostVulnerabilityCsvResponse>
            exportHostVulnerabilityCsv(
                    ExportHostVulnerabilityCsvRequest request,
                    final com.oracle.bmc.responses.AsyncHandler<
                                    ExportHostVulnerabilityCsvRequest,
                                    ExportHostVulnerabilityCsvResponse>
                            handler) {
        Objects.requireNonNull(
                request.getExportHostVulnerabilityCsvDetails(),
                "exportHostVulnerabilityCsvDetails is required");

        return clientCall(request, ExportHostVulnerabilityCsvResponse::builder)
                .logger(LOG, "exportHostVulnerabilityCsv")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ExportHostVulnerabilityCsv",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostVulnerability/ExportHostVulnerabilityCsv")
                .method(com.oracle.bmc.http.client.Method.POST)
                .requestBuilder(ExportHostVulnerabilityCsvRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostVulnerabilities")
                .appendPathParam("actions")
                .appendPathParam("exportCsv")
                .accept("text/csv")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .hasBody()
                .handleBody(
                        java.io.InputStream.class,
                        ExportHostVulnerabilityCsvResponse.Builder::inputStream)
                .handleResponseHeaderString(
                        "opc-request-id", ExportHostVulnerabilityCsvResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<GetContainerScanRecipeResponse> getContainerScanRecipe(
            GetContainerScanRecipeRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            GetContainerScanRecipeRequest, GetContainerScanRecipeResponse>
                    handler) {

        Validate.notBlank(
                request.getContainerScanRecipeId(), "containerScanRecipeId must not be blank");

        return clientCall(request, GetContainerScanRecipeResponse::builder)
                .logger(LOG, "getContainerScanRecipe")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "GetContainerScanRecipe",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/ContainerScanRecipe/GetContainerScanRecipe")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(GetContainerScanRecipeRequest::builder)
                .basePath("/20210215")
                .appendPathParam("containerScanRecipes")
                .appendPathParam(request.getContainerScanRecipeId())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model.ContainerScanRecipe.class,
                        GetContainerScanRecipeResponse.Builder::containerScanRecipe)
                .handleResponseHeaderString("etag", GetContainerScanRecipeResponse.Builder::etag)
                .handleResponseHeaderString(
                        "opc-request-id", GetContainerScanRecipeResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<GetContainerScanResultResponse> getContainerScanResult(
            GetContainerScanResultRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            GetContainerScanResultRequest, GetContainerScanResultResponse>
                    handler) {

        Validate.notBlank(
                request.getContainerScanResultId(), "containerScanResultId must not be blank");

        return clientCall(request, GetContainerScanResultResponse::builder)
                .logger(LOG, "getContainerScanResult")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "GetContainerScanResult",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/ContainerScanResult/GetContainerScanResult")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(GetContainerScanResultRequest::builder)
                .basePath("/20210215")
                .appendPathParam("containerScanResults")
                .appendPathParam(request.getContainerScanResultId())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model.ContainerScanResult.class,
                        GetContainerScanResultResponse.Builder::containerScanResult)
                .handleResponseHeaderString("etag", GetContainerScanResultResponse.Builder::etag)
                .handleResponseHeaderString(
                        "opc-request-id", GetContainerScanResultResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<GetContainerScanTargetResponse> getContainerScanTarget(
            GetContainerScanTargetRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            GetContainerScanTargetRequest, GetContainerScanTargetResponse>
                    handler) {

        Validate.notBlank(
                request.getContainerScanTargetId(), "containerScanTargetId must not be blank");

        return clientCall(request, GetContainerScanTargetResponse::builder)
                .logger(LOG, "getContainerScanTarget")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "GetContainerScanTarget",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/ContainerScanTarget/GetContainerScanTarget")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(GetContainerScanTargetRequest::builder)
                .basePath("/20210215")
                .appendPathParam("containerScanTargets")
                .appendPathParam(request.getContainerScanTargetId())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model.ContainerScanTarget.class,
                        GetContainerScanTargetResponse.Builder::containerScanTarget)
                .handleResponseHeaderString("etag", GetContainerScanTargetResponse.Builder::etag)
                .handleResponseHeaderString(
                        "opc-request-id", GetContainerScanTargetResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<GetHostAgentScanResultResponse> getHostAgentScanResult(
            GetHostAgentScanResultRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            GetHostAgentScanResultRequest, GetHostAgentScanResultResponse>
                    handler) {

        Validate.notBlank(
                request.getHostAgentScanResultId(), "hostAgentScanResultId must not be blank");

        return clientCall(request, GetHostAgentScanResultResponse::builder)
                .logger(LOG, "getHostAgentScanResult")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "GetHostAgentScanResult",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostAgentScanResult/GetHostAgentScanResult")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(GetHostAgentScanResultRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostAgentScanResults")
                .appendPathParam(request.getHostAgentScanResultId())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model.HostAgentScanResult.class,
                        GetHostAgentScanResultResponse.Builder::hostAgentScanResult)
                .handleResponseHeaderString("etag", GetHostAgentScanResultResponse.Builder::etag)
                .handleResponseHeaderString(
                        "opc-request-id", GetHostAgentScanResultResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<GetHostCisBenchmarkScanResultResponse>
            getHostCisBenchmarkScanResult(
                    GetHostCisBenchmarkScanResultRequest request,
                    final com.oracle.bmc.responses.AsyncHandler<
                                    GetHostCisBenchmarkScanResultRequest,
                                    GetHostCisBenchmarkScanResultResponse>
                            handler) {

        Validate.notBlank(
                request.getHostCisBenchmarkScanResultId(),
                "hostCisBenchmarkScanResultId must not be blank");

        return clientCall(request, GetHostCisBenchmarkScanResultResponse::builder)
                .logger(LOG, "getHostCisBenchmarkScanResult")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "GetHostCisBenchmarkScanResult",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostCisBenchmarkScanResult/GetHostCisBenchmarkScanResult")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(GetHostCisBenchmarkScanResultRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostCisBenchmarkScanResults")
                .appendPathParam(request.getHostCisBenchmarkScanResultId())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model.HostCisBenchmarkScanResult.class,
                        GetHostCisBenchmarkScanResultResponse.Builder::hostCisBenchmarkScanResult)
                .handleResponseHeaderString(
                        "etag", GetHostCisBenchmarkScanResultResponse.Builder::etag)
                .handleResponseHeaderString(
                        "opc-request-id",
                        GetHostCisBenchmarkScanResultResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<GetHostEndpointProtectionScanResultResponse>
            getHostEndpointProtectionScanResult(
                    GetHostEndpointProtectionScanResultRequest request,
                    final com.oracle.bmc.responses.AsyncHandler<
                                    GetHostEndpointProtectionScanResultRequest,
                                    GetHostEndpointProtectionScanResultResponse>
                            handler) {

        Validate.notBlank(
                request.getHostEndpointProtectionScanResultId(),
                "hostEndpointProtectionScanResultId must not be blank");

        return clientCall(request, GetHostEndpointProtectionScanResultResponse::builder)
                .logger(LOG, "getHostEndpointProtectionScanResult")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "GetHostEndpointProtectionScanResult",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostEndpointProtectionScanResult/GetHostEndpointProtectionScanResult")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(GetHostEndpointProtectionScanResultRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostEndpointProtectionScanResults")
                .appendPathParam(request.getHostEndpointProtectionScanResultId())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model.HostEndpointProtectionScanResult
                                .class,
                        GetHostEndpointProtectionScanResultResponse.Builder
                                ::hostEndpointProtectionScanResult)
                .handleResponseHeaderString(
                        "etag", GetHostEndpointProtectionScanResultResponse.Builder::etag)
                .handleResponseHeaderString(
                        "opc-request-id",
                        GetHostEndpointProtectionScanResultResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<GetHostPortScanResultResponse> getHostPortScanResult(
            GetHostPortScanResultRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            GetHostPortScanResultRequest, GetHostPortScanResultResponse>
                    handler) {

        Validate.notBlank(
                request.getHostPortScanResultId(), "hostPortScanResultId must not be blank");

        return clientCall(request, GetHostPortScanResultResponse::builder)
                .logger(LOG, "getHostPortScanResult")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "GetHostPortScanResult",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostPortScanResult/GetHostPortScanResult")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(GetHostPortScanResultRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostPortScanResults")
                .appendPathParam(request.getHostPortScanResultId())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model.HostPortScanResult.class,
                        GetHostPortScanResultResponse.Builder::hostPortScanResult)
                .handleResponseHeaderString("etag", GetHostPortScanResultResponse.Builder::etag)
                .handleResponseHeaderString(
                        "opc-request-id", GetHostPortScanResultResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<GetHostScanRecipeResponse> getHostScanRecipe(
            GetHostScanRecipeRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            GetHostScanRecipeRequest, GetHostScanRecipeResponse>
                    handler) {

        Validate.notBlank(request.getHostScanRecipeId(), "hostScanRecipeId must not be blank");

        return clientCall(request, GetHostScanRecipeResponse::builder)
                .logger(LOG, "getHostScanRecipe")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "GetHostScanRecipe",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostScanRecipe/GetHostScanRecipe")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(GetHostScanRecipeRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostScanRecipes")
                .appendPathParam(request.getHostScanRecipeId())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model.HostScanRecipe.class,
                        GetHostScanRecipeResponse.Builder::hostScanRecipe)
                .handleResponseHeaderString("etag", GetHostScanRecipeResponse.Builder::etag)
                .handleResponseHeaderString(
                        "opc-request-id", GetHostScanRecipeResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<GetHostScanTargetResponse> getHostScanTarget(
            GetHostScanTargetRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            GetHostScanTargetRequest, GetHostScanTargetResponse>
                    handler) {

        Validate.notBlank(request.getHostScanTargetId(), "hostScanTargetId must not be blank");

        return clientCall(request, GetHostScanTargetResponse::builder)
                .logger(LOG, "getHostScanTarget")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "GetHostScanTarget",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostScanTarget/GetHostScanTarget")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(GetHostScanTargetRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostScanTargets")
                .appendPathParam(request.getHostScanTargetId())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model.HostScanTarget.class,
                        GetHostScanTargetResponse.Builder::hostScanTarget)
                .handleResponseHeaderString("etag", GetHostScanTargetResponse.Builder::etag)
                .handleResponseHeaderString(
                        "opc-request-id", GetHostScanTargetResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<GetHostVulnerabilityResponse> getHostVulnerability(
            GetHostVulnerabilityRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            GetHostVulnerabilityRequest, GetHostVulnerabilityResponse>
                    handler) {

        Validate.notBlank(
                request.getHostVulnerabilityId(), "hostVulnerabilityId must not be blank");

        return clientCall(request, GetHostVulnerabilityResponse::builder)
                .logger(LOG, "getHostVulnerability")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "GetHostVulnerability",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostVulnerability/GetHostVulnerability")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(GetHostVulnerabilityRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostVulnerabilities")
                .appendPathParam(request.getHostVulnerabilityId())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model.HostVulnerability.class,
                        GetHostVulnerabilityResponse.Builder::hostVulnerability)
                .handleResponseHeaderString("etag", GetHostVulnerabilityResponse.Builder::etag)
                .handleResponseHeaderString(
                        "opc-request-id", GetHostVulnerabilityResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<GetVulnerabilityResponse> getVulnerability(
            GetVulnerabilityRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            GetVulnerabilityRequest, GetVulnerabilityResponse>
                    handler) {

        Validate.notBlank(request.getVulnerabilityId(), "vulnerabilityId must not be blank");

        return clientCall(request, GetVulnerabilityResponse::builder)
                .logger(LOG, "getVulnerability")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "GetVulnerability",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/Vulnerability/GetVulnerability")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(GetVulnerabilityRequest::builder)
                .basePath("/20210215")
                .appendPathParam("vulnerabilities")
                .appendPathParam(request.getVulnerabilityId())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model.Vulnerability.class,
                        GetVulnerabilityResponse.Builder::vulnerability)
                .handleResponseHeaderString("etag", GetVulnerabilityResponse.Builder::etag)
                .handleResponseHeaderString(
                        "opc-request-id", GetVulnerabilityResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<GetWorkRequestResponse> getWorkRequest(
            GetWorkRequestRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            GetWorkRequestRequest, GetWorkRequestResponse>
                    handler) {

        Validate.notBlank(request.getWorkRequestId(), "workRequestId must not be blank");

        return clientCall(request, GetWorkRequestResponse::builder)
                .logger(LOG, "getWorkRequest")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "GetWorkRequest",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/WorkRequest/GetWorkRequest")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(GetWorkRequestRequest::builder)
                .basePath("/20210215")
                .appendPathParam("workRequests")
                .appendPathParam(request.getWorkRequestId())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model.WorkRequest.class,
                        GetWorkRequestResponse.Builder::workRequest)
                .handleResponseHeaderString(
                        "opc-request-id", GetWorkRequestResponse.Builder::opcRequestId)
                .handleResponseHeaderFloat(
                        "retry-after", GetWorkRequestResponse.Builder::retryAfter)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ListContainerScanRecipesResponse> listContainerScanRecipes(
            ListContainerScanRecipesRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            ListContainerScanRecipesRequest, ListContainerScanRecipesResponse>
                    handler) {
        Objects.requireNonNull(request.getCompartmentId(), "compartmentId is required");

        return clientCall(request, ListContainerScanRecipesResponse::builder)
                .logger(LOG, "listContainerScanRecipes")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ListContainerScanRecipes",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/ContainerScanRecipe/ListContainerScanRecipes")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(ListContainerScanRecipesRequest::builder)
                .basePath("/20210215")
                .appendPathParam("containerScanRecipes")
                .appendQueryParam("compartmentId", request.getCompartmentId())
                .appendEnumQueryParam("lifecycleState", request.getLifecycleState())
                .appendQueryParam("displayName", request.getDisplayName())
                .appendQueryParam("limit", request.getLimit())
                .appendQueryParam("page", request.getPage())
                .appendEnumQueryParam("sortOrder", request.getSortOrder())
                .appendEnumQueryParam("sortBy", request.getSortBy())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model
                                .ContainerScanRecipeSummaryCollection.class,
                        ListContainerScanRecipesResponse.Builder
                                ::containerScanRecipeSummaryCollection)
                .handleResponseHeaderString(
                        "opc-request-id", ListContainerScanRecipesResponse.Builder::opcRequestId)
                .handleResponseHeaderString(
                        "opc-next-page", ListContainerScanRecipesResponse.Builder::opcNextPage)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ListContainerScanResultsResponse> listContainerScanResults(
            ListContainerScanResultsRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            ListContainerScanResultsRequest, ListContainerScanResultsResponse>
                    handler) {
        Objects.requireNonNull(request.getCompartmentId(), "compartmentId is required");

        return clientCall(request, ListContainerScanResultsResponse::builder)
                .logger(LOG, "listContainerScanResults")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ListContainerScanResults",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/ContainerScanResult/ListContainerScanResults")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(ListContainerScanResultsRequest::builder)
                .basePath("/20210215")
                .appendPathParam("containerScanResults")
                .appendQueryParam("compartmentId", request.getCompartmentId())
                .appendQueryParam("repository", request.getRepository())
                .appendQueryParam("image", request.getImage())
                .appendQueryParam(
                        "areSubcompartmentsIncluded", request.getAreSubcompartmentsIncluded())
                .appendEnumQueryParam("highestProblemSeverity", request.getHighestProblemSeverity())
                .appendQueryParam(
                        "timeStartedGreaterThanOrEqualTo",
                        request.getTimeStartedGreaterThanOrEqualTo())
                .appendQueryParam(
                        "timeStartedLessThanOrEqualTo", request.getTimeStartedLessThanOrEqualTo())
                .appendEnumQueryParam("sortOrder", request.getSortOrder())
                .appendEnumQueryParam("sortBy", request.getSortBy())
                .appendQueryParam("limit", request.getLimit())
                .appendQueryParam("page", request.getPage())
                .appendQueryParam("isLatestOnly", request.getIsLatestOnly())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model
                                .ContainerScanResultSummaryCollection.class,
                        ListContainerScanResultsResponse.Builder
                                ::containerScanResultSummaryCollection)
                .handleResponseHeaderString(
                        "opc-request-id", ListContainerScanResultsResponse.Builder::opcRequestId)
                .handleResponseHeaderString(
                        "opc-next-page", ListContainerScanResultsResponse.Builder::opcNextPage)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ListContainerScanTargetsResponse> listContainerScanTargets(
            ListContainerScanTargetsRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            ListContainerScanTargetsRequest, ListContainerScanTargetsResponse>
                    handler) {
        Objects.requireNonNull(request.getCompartmentId(), "compartmentId is required");

        return clientCall(request, ListContainerScanTargetsResponse::builder)
                .logger(LOG, "listContainerScanTargets")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ListContainerScanTargets",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/ContainerScanTarget/ListContainerScanTargets")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(ListContainerScanTargetsRequest::builder)
                .basePath("/20210215")
                .appendPathParam("containerScanTargets")
                .appendQueryParam("compartmentId", request.getCompartmentId())
                .appendEnumQueryParam("lifecycleState", request.getLifecycleState())
                .appendQueryParam("displayName", request.getDisplayName())
                .appendQueryParam("limit", request.getLimit())
                .appendQueryParam("page", request.getPage())
                .appendEnumQueryParam("sortOrder", request.getSortOrder())
                .appendEnumQueryParam("sortBy", request.getSortBy())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model
                                .ContainerScanTargetSummaryCollection.class,
                        ListContainerScanTargetsResponse.Builder
                                ::containerScanTargetSummaryCollection)
                .handleResponseHeaderString(
                        "opc-request-id", ListContainerScanTargetsResponse.Builder::opcRequestId)
                .handleResponseHeaderString(
                        "opc-next-page", ListContainerScanTargetsResponse.Builder::opcNextPage)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ListHostAgentScanResultsResponse> listHostAgentScanResults(
            ListHostAgentScanResultsRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            ListHostAgentScanResultsRequest, ListHostAgentScanResultsResponse>
                    handler) {
        Objects.requireNonNull(request.getCompartmentId(), "compartmentId is required");

        return clientCall(request, ListHostAgentScanResultsResponse::builder)
                .logger(LOG, "listHostAgentScanResults")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ListHostAgentScanResults",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostAgentScanResult/ListHostAgentScanResults")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(ListHostAgentScanResultsRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostAgentScanResults")
                .appendQueryParam("compartmentId", request.getCompartmentId())
                .appendQueryParam("instanceId", request.getInstanceId())
                .appendEnumQueryParam("highestProblemSeverity", request.getHighestProblemSeverity())
                .appendQueryParam("operatingSystem", request.getOperatingSystem())
                .appendQueryParam(
                        "timeStartedGreaterThanOrEqualTo",
                        request.getTimeStartedGreaterThanOrEqualTo())
                .appendQueryParam(
                        "timeStartedLessThanOrEqualTo", request.getTimeStartedLessThanOrEqualTo())
                .appendEnumQueryParam("sortOrder", request.getSortOrder())
                .appendEnumQueryParam("sortBy", request.getSortBy())
                .appendQueryParam("limit", request.getLimit())
                .appendQueryParam("page", request.getPage())
                .appendQueryParam("displayName", request.getDisplayName())
                .appendQueryParam("isLatestOnly", request.getIsLatestOnly())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model
                                .HostAgentScanResultSummaryCollection.class,
                        ListHostAgentScanResultsResponse.Builder
                                ::hostAgentScanResultSummaryCollection)
                .handleResponseHeaderString(
                        "opc-request-id", ListHostAgentScanResultsResponse.Builder::opcRequestId)
                .handleResponseHeaderString(
                        "opc-next-page", ListHostAgentScanResultsResponse.Builder::opcNextPage)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ListHostCisBenchmarkScanResultsResponse>
            listHostCisBenchmarkScanResults(
                    ListHostCisBenchmarkScanResultsRequest request,
                    final com.oracle.bmc.responses.AsyncHandler<
                                    ListHostCisBenchmarkScanResultsRequest,
                                    ListHostCisBenchmarkScanResultsResponse>
                            handler) {
        Objects.requireNonNull(request.getCompartmentId(), "compartmentId is required");

        return clientCall(request, ListHostCisBenchmarkScanResultsResponse::builder)
                .logger(LOG, "listHostCisBenchmarkScanResults")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ListHostCisBenchmarkScanResults",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostCisBenchmarkScanResult/ListHostCisBenchmarkScanResults")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(ListHostCisBenchmarkScanResultsRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostCisBenchmarkScanResults")
                .appendQueryParam("compartmentId", request.getCompartmentId())
                .appendQueryParam("instanceId", request.getInstanceId())
                .appendEnumQueryParam("highestProblemSeverity", request.getHighestProblemSeverity())
                .appendQueryParam(
                        "timeStartedGreaterThanOrEqualTo",
                        request.getTimeStartedGreaterThanOrEqualTo())
                .appendQueryParam(
                        "timeStartedLessThanOrEqualTo", request.getTimeStartedLessThanOrEqualTo())
                .appendEnumQueryParam("sortOrder", request.getSortOrder())
                .appendEnumQueryParam("sortBy", request.getSortBy())
                .appendQueryParam("limit", request.getLimit())
                .appendQueryParam("page", request.getPage())
                .appendQueryParam("displayName", request.getDisplayName())
                .appendQueryParam("isLatestOnly", request.getIsLatestOnly())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model
                                .HostCisBenchmarkScanResultSummaryCollection.class,
                        ListHostCisBenchmarkScanResultsResponse.Builder
                                ::hostCisBenchmarkScanResultSummaryCollection)
                .handleResponseHeaderString(
                        "opc-request-id",
                        ListHostCisBenchmarkScanResultsResponse.Builder::opcRequestId)
                .handleResponseHeaderString(
                        "opc-next-page",
                        ListHostCisBenchmarkScanResultsResponse.Builder::opcNextPage)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ListHostEndpointProtectionScanResultsResponse>
            listHostEndpointProtectionScanResults(
                    ListHostEndpointProtectionScanResultsRequest request,
                    final com.oracle.bmc.responses.AsyncHandler<
                                    ListHostEndpointProtectionScanResultsRequest,
                                    ListHostEndpointProtectionScanResultsResponse>
                            handler) {
        Objects.requireNonNull(request.getCompartmentId(), "compartmentId is required");

        return clientCall(request, ListHostEndpointProtectionScanResultsResponse::builder)
                .logger(LOG, "listHostEndpointProtectionScanResults")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ListHostEndpointProtectionScanResults",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostEndpointProtectionScanResult/ListHostEndpointProtectionScanResults")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(ListHostEndpointProtectionScanResultsRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostEndpointProtectionScanResults")
                .appendQueryParam("compartmentId", request.getCompartmentId())
                .appendQueryParam("instanceId", request.getInstanceId())
                .appendEnumQueryParam("highestProblemSeverity", request.getHighestProblemSeverity())
                .appendQueryParam(
                        "timeStartedGreaterThanOrEqualTo",
                        request.getTimeStartedGreaterThanOrEqualTo())
                .appendQueryParam(
                        "timeStartedLessThanOrEqualTo", request.getTimeStartedLessThanOrEqualTo())
                .appendEnumQueryParam("sortOrder", request.getSortOrder())
                .appendEnumQueryParam("sortBy", request.getSortBy())
                .appendQueryParam("limit", request.getLimit())
                .appendQueryParam("page", request.getPage())
                .appendQueryParam("displayName", request.getDisplayName())
                .appendQueryParam("isLatestOnly", request.getIsLatestOnly())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model
                                .HostEndpointProtectionScanResultSummaryCollection.class,
                        ListHostEndpointProtectionScanResultsResponse.Builder
                                ::hostEndpointProtectionScanResultSummaryCollection)
                .handleResponseHeaderString(
                        "opc-request-id",
                        ListHostEndpointProtectionScanResultsResponse.Builder::opcRequestId)
                .handleResponseHeaderString(
                        "opc-next-page",
                        ListHostEndpointProtectionScanResultsResponse.Builder::opcNextPage)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ListHostPortScanResultsResponse> listHostPortScanResults(
            ListHostPortScanResultsRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            ListHostPortScanResultsRequest, ListHostPortScanResultsResponse>
                    handler) {
        Objects.requireNonNull(request.getCompartmentId(), "compartmentId is required");

        return clientCall(request, ListHostPortScanResultsResponse::builder)
                .logger(LOG, "listHostPortScanResults")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ListHostPortScanResults",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostPortScanResult/ListHostPortScanResults")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(ListHostPortScanResultsRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostPortScanResults")
                .appendQueryParam("compartmentId", request.getCompartmentId())
                .appendQueryParam("instanceId", request.getInstanceId())
                .appendEnumQueryParam("highestProblemSeverity", request.getHighestProblemSeverity())
                .appendQueryParam(
                        "timeStartedGreaterThanOrEqualTo",
                        request.getTimeStartedGreaterThanOrEqualTo())
                .appendQueryParam(
                        "timeStartedLessThanOrEqualTo", request.getTimeStartedLessThanOrEqualTo())
                .appendEnumQueryParam("sortOrder", request.getSortOrder())
                .appendEnumQueryParam("sortBy", request.getSortBy())
                .appendQueryParam("limit", request.getLimit())
                .appendQueryParam("page", request.getPage())
                .appendQueryParam("displayName", request.getDisplayName())
                .appendQueryParam("isLatestOnly", request.getIsLatestOnly())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model
                                .HostPortScanResultSummaryCollection.class,
                        ListHostPortScanResultsResponse.Builder
                                ::hostPortScanResultSummaryCollection)
                .handleResponseHeaderString(
                        "opc-request-id", ListHostPortScanResultsResponse.Builder::opcRequestId)
                .handleResponseHeaderString(
                        "opc-next-page", ListHostPortScanResultsResponse.Builder::opcNextPage)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ListHostScanRecipesResponse> listHostScanRecipes(
            ListHostScanRecipesRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            ListHostScanRecipesRequest, ListHostScanRecipesResponse>
                    handler) {
        Objects.requireNonNull(request.getCompartmentId(), "compartmentId is required");

        return clientCall(request, ListHostScanRecipesResponse::builder)
                .logger(LOG, "listHostScanRecipes")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ListHostScanRecipes",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostScanRecipe/ListHostScanRecipes")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(ListHostScanRecipesRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostScanRecipes")
                .appendQueryParam("compartmentId", request.getCompartmentId())
                .appendEnumQueryParam("lifecycleState", request.getLifecycleState())
                .appendQueryParam("displayName", request.getDisplayName())
                .appendQueryParam("limit", request.getLimit())
                .appendQueryParam("page", request.getPage())
                .appendEnumQueryParam("sortOrder", request.getSortOrder())
                .appendEnumQueryParam("sortBy", request.getSortBy())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model.HostScanRecipeSummaryCollection
                                .class,
                        ListHostScanRecipesResponse.Builder::hostScanRecipeSummaryCollection)
                .handleResponseHeaderString(
                        "opc-request-id", ListHostScanRecipesResponse.Builder::opcRequestId)
                .handleResponseHeaderString(
                        "opc-next-page", ListHostScanRecipesResponse.Builder::opcNextPage)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ListHostScanTargetErrorsResponse> listHostScanTargetErrors(
            ListHostScanTargetErrorsRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            ListHostScanTargetErrorsRequest, ListHostScanTargetErrorsResponse>
                    handler) {
        Objects.requireNonNull(request.getCompartmentId(), "compartmentId is required");

        Validate.notBlank(request.getHostScanTargetId(), "hostScanTargetId must not be blank");

        return clientCall(request, ListHostScanTargetErrorsResponse::builder)
                .logger(LOG, "listHostScanTargetErrors")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ListHostScanTargetErrors",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostScanTargetErrorSummaryCollection/ListHostScanTargetErrors")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(ListHostScanTargetErrorsRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostScanTargets")
                .appendPathParam(request.getHostScanTargetId())
                .appendPathParam("errors")
                .appendQueryParam("compartmentId", request.getCompartmentId())
                .appendEnumQueryParam("lifecycleState", request.getLifecycleState())
                .appendEnumQueryParam("sortOrder", request.getSortOrder())
                .appendEnumQueryParam("sortBy", request.getSortBy())
                .appendQueryParam("limit", request.getLimit())
                .appendQueryParam("page", request.getPage())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model
                                .HostScanTargetErrorSummaryCollection.class,
                        ListHostScanTargetErrorsResponse.Builder
                                ::hostScanTargetErrorSummaryCollection)
                .handleResponseHeaderString(
                        "opc-request-id", ListHostScanTargetErrorsResponse.Builder::opcRequestId)
                .handleResponseHeaderString(
                        "opc-next-page", ListHostScanTargetErrorsResponse.Builder::opcNextPage)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ListHostScanTargetsResponse> listHostScanTargets(
            ListHostScanTargetsRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            ListHostScanTargetsRequest, ListHostScanTargetsResponse>
                    handler) {
        Objects.requireNonNull(request.getCompartmentId(), "compartmentId is required");

        return clientCall(request, ListHostScanTargetsResponse::builder)
                .logger(LOG, "listHostScanTargets")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ListHostScanTargets",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostScanTarget/ListHostScanTargets")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(ListHostScanTargetsRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostScanTargets")
                .appendQueryParam("compartmentId", request.getCompartmentId())
                .appendEnumQueryParam("lifecycleState", request.getLifecycleState())
                .appendQueryParam("displayName", request.getDisplayName())
                .appendQueryParam("limit", request.getLimit())
                .appendQueryParam("page", request.getPage())
                .appendEnumQueryParam("sortOrder", request.getSortOrder())
                .appendEnumQueryParam("sortBy", request.getSortBy())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model.HostScanTargetSummaryCollection
                                .class,
                        ListHostScanTargetsResponse.Builder::hostScanTargetSummaryCollection)
                .handleResponseHeaderString(
                        "opc-request-id", ListHostScanTargetsResponse.Builder::opcRequestId)
                .handleResponseHeaderString(
                        "opc-next-page", ListHostScanTargetsResponse.Builder::opcNextPage)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ListHostVulnerabilitiesResponse> listHostVulnerabilities(
            ListHostVulnerabilitiesRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            ListHostVulnerabilitiesRequest, ListHostVulnerabilitiesResponse>
                    handler) {
        Objects.requireNonNull(request.getCompartmentId(), "compartmentId is required");

        return clientCall(request, ListHostVulnerabilitiesResponse::builder)
                .logger(LOG, "listHostVulnerabilities")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ListHostVulnerabilities",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostVulnerability/ListHostVulnerabilities")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(ListHostVulnerabilitiesRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostVulnerabilities")
                .appendQueryParam("compartmentId", request.getCompartmentId())
                .appendQueryParam("limit", request.getLimit())
                .appendQueryParam("page", request.getPage())
                .appendEnumQueryParam("severity", request.getSeverity())
                .appendQueryParam("name", request.getName())
                .appendQueryParam("cveReference", request.getCveReference())
                .appendEnumQueryParam("vulnerabilityType", request.getVulnerabilityType())
                .appendEnumQueryParam("sortOrder", request.getSortOrder())
                .appendEnumQueryParam("sortBy", request.getSortBy())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model
                                .HostVulnerabilitySummaryCollection.class,
                        ListHostVulnerabilitiesResponse.Builder::hostVulnerabilitySummaryCollection)
                .handleResponseHeaderString(
                        "opc-request-id", ListHostVulnerabilitiesResponse.Builder::opcRequestId)
                .handleResponseHeaderString(
                        "opc-next-page", ListHostVulnerabilitiesResponse.Builder::opcNextPage)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ListHostVulnerabilityImpactedHostsResponse>
            listHostVulnerabilityImpactedHosts(
                    ListHostVulnerabilityImpactedHostsRequest request,
                    final com.oracle.bmc.responses.AsyncHandler<
                                    ListHostVulnerabilityImpactedHostsRequest,
                                    ListHostVulnerabilityImpactedHostsResponse>
                            handler) {

        Validate.notBlank(
                request.getHostVulnerabilityId(), "hostVulnerabilityId must not be blank");

        return clientCall(request, ListHostVulnerabilityImpactedHostsResponse::builder)
                .logger(LOG, "listHostVulnerabilityImpactedHosts")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ListHostVulnerabilityImpactedHosts",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostVulnerability/ListHostVulnerabilityImpactedHosts")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(ListHostVulnerabilityImpactedHostsRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostVulnerabilities")
                .appendPathParam(request.getHostVulnerabilityId())
                .appendPathParam("impactedHosts")
                .appendEnumQueryParam("sortOrder", request.getSortOrder())
                .appendEnumQueryParam("sortBy", request.getSortBy())
                .appendQueryParam("limit", request.getLimit())
                .appendQueryParam("page", request.getPage())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model
                                .HostVulnerabilityImpactedHostSummaryCollection.class,
                        ListHostVulnerabilityImpactedHostsResponse.Builder
                                ::hostVulnerabilityImpactedHostSummaryCollection)
                .handleResponseHeaderString(
                        "opc-request-id",
                        ListHostVulnerabilityImpactedHostsResponse.Builder::opcRequestId)
                .handleResponseHeaderString(
                        "opc-next-page",
                        ListHostVulnerabilityImpactedHostsResponse.Builder::opcNextPage)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ListVulnerabilitiesResponse> listVulnerabilities(
            ListVulnerabilitiesRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            ListVulnerabilitiesRequest, ListVulnerabilitiesResponse>
                    handler) {
        Objects.requireNonNull(request.getCompartmentId(), "compartmentId is required");

        return clientCall(request, ListVulnerabilitiesResponse::builder)
                .logger(LOG, "listVulnerabilities")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ListVulnerabilities",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/Vulnerability/ListVulnerabilities")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(ListVulnerabilitiesRequest::builder)
                .basePath("/20210215")
                .appendPathParam("vulnerabilities")
                .appendQueryParam("compartmentId", request.getCompartmentId())
                .appendQueryParam("limit", request.getLimit())
                .appendQueryParam("page", request.getPage())
                .appendEnumQueryParam("severity", request.getSeverity())
                .appendQueryParam("name", request.getName())
                .appendQueryParam("vulnerabilityReference", request.getVulnerabilityReference())
                .appendEnumQueryParam("vulnerabilityType", request.getVulnerabilityType())
                .appendEnumQueryParam("sortOrder", request.getSortOrder())
                .appendEnumQueryParam("sortBy", request.getSortBy())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model.VulnerabilitySummaryCollection
                                .class,
                        ListVulnerabilitiesResponse.Builder::vulnerabilitySummaryCollection)
                .handleResponseHeaderString(
                        "opc-request-id", ListVulnerabilitiesResponse.Builder::opcRequestId)
                .handleResponseHeaderString(
                        "opc-next-page", ListVulnerabilitiesResponse.Builder::opcNextPage)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ListVulnerabilityImpactedContainersResponse>
            listVulnerabilityImpactedContainers(
                    ListVulnerabilityImpactedContainersRequest request,
                    final com.oracle.bmc.responses.AsyncHandler<
                                    ListVulnerabilityImpactedContainersRequest,
                                    ListVulnerabilityImpactedContainersResponse>
                            handler) {

        Validate.notBlank(request.getVulnerabilityId(), "vulnerabilityId must not be blank");

        return clientCall(request, ListVulnerabilityImpactedContainersResponse::builder)
                .logger(LOG, "listVulnerabilityImpactedContainers")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ListVulnerabilityImpactedContainers",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/Vulnerability/ListVulnerabilityImpactedContainers")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(ListVulnerabilityImpactedContainersRequest::builder)
                .basePath("/20210215")
                .appendPathParam("vulnerabilities")
                .appendPathParam(request.getVulnerabilityId())
                .appendPathParam("impactedContainers")
                .appendEnumQueryParam("sortOrder", request.getSortOrder())
                .appendQueryParam("limit", request.getLimit())
                .appendQueryParam("page", request.getPage())
                .appendEnumQueryParam("sortBy", request.getSortBy())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model
                                .VulnerabilityImpactedContainerSummaryCollection.class,
                        ListVulnerabilityImpactedContainersResponse.Builder
                                ::vulnerabilityImpactedContainerSummaryCollection)
                .handleResponseHeaderString(
                        "opc-request-id",
                        ListVulnerabilityImpactedContainersResponse.Builder::opcRequestId)
                .handleResponseHeaderString(
                        "opc-next-page",
                        ListVulnerabilityImpactedContainersResponse.Builder::opcNextPage)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ListVulnerabilityImpactedHostsResponse>
            listVulnerabilityImpactedHosts(
                    ListVulnerabilityImpactedHostsRequest request,
                    final com.oracle.bmc.responses.AsyncHandler<
                                    ListVulnerabilityImpactedHostsRequest,
                                    ListVulnerabilityImpactedHostsResponse>
                            handler) {

        Validate.notBlank(request.getVulnerabilityId(), "vulnerabilityId must not be blank");

        return clientCall(request, ListVulnerabilityImpactedHostsResponse::builder)
                .logger(LOG, "listVulnerabilityImpactedHosts")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ListVulnerabilityImpactedHosts",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/Vulnerability/ListVulnerabilityImpactedHosts")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(ListVulnerabilityImpactedHostsRequest::builder)
                .basePath("/20210215")
                .appendPathParam("vulnerabilities")
                .appendPathParam(request.getVulnerabilityId())
                .appendPathParam("impactedHosts")
                .appendEnumQueryParam("sortOrder", request.getSortOrder())
                .appendEnumQueryParam("sortBy", request.getSortBy())
                .appendQueryParam("limit", request.getLimit())
                .appendQueryParam("page", request.getPage())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model
                                .VulnerabilityImpactedHostSummaryCollection.class,
                        ListVulnerabilityImpactedHostsResponse.Builder
                                ::vulnerabilityImpactedHostSummaryCollection)
                .handleResponseHeaderString(
                        "opc-request-id",
                        ListVulnerabilityImpactedHostsResponse.Builder::opcRequestId)
                .handleResponseHeaderString(
                        "opc-next-page",
                        ListVulnerabilityImpactedHostsResponse.Builder::opcNextPage)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ListWorkRequestErrorsResponse> listWorkRequestErrors(
            ListWorkRequestErrorsRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            ListWorkRequestErrorsRequest, ListWorkRequestErrorsResponse>
                    handler) {

        Validate.notBlank(request.getWorkRequestId(), "workRequestId must not be blank");

        return clientCall(request, ListWorkRequestErrorsResponse::builder)
                .logger(LOG, "listWorkRequestErrors")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ListWorkRequestErrors",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/WorkRequestError/ListWorkRequestErrors")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(ListWorkRequestErrorsRequest::builder)
                .basePath("/20210215")
                .appendPathParam("workRequests")
                .appendPathParam(request.getWorkRequestId())
                .appendPathParam("errors")
                .appendQueryParam("page", request.getPage())
                .appendQueryParam("limit", request.getLimit())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model.WorkRequestErrorCollection.class,
                        ListWorkRequestErrorsResponse.Builder::workRequestErrorCollection)
                .handleResponseHeaderString(
                        "opc-next-page", ListWorkRequestErrorsResponse.Builder::opcNextPage)
                .handleResponseHeaderString(
                        "opc-request-id", ListWorkRequestErrorsResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ListWorkRequestLogsResponse> listWorkRequestLogs(
            ListWorkRequestLogsRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            ListWorkRequestLogsRequest, ListWorkRequestLogsResponse>
                    handler) {

        Validate.notBlank(request.getWorkRequestId(), "workRequestId must not be blank");

        return clientCall(request, ListWorkRequestLogsResponse::builder)
                .logger(LOG, "listWorkRequestLogs")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ListWorkRequestLogs",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/WorkRequestLogEntry/ListWorkRequestLogs")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(ListWorkRequestLogsRequest::builder)
                .basePath("/20210215")
                .appendPathParam("workRequests")
                .appendPathParam(request.getWorkRequestId())
                .appendPathParam("logs")
                .appendQueryParam("page", request.getPage())
                .appendQueryParam("limit", request.getLimit())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model.WorkRequestLogEntryCollection
                                .class,
                        ListWorkRequestLogsResponse.Builder::workRequestLogEntryCollection)
                .handleResponseHeaderString(
                        "opc-next-page", ListWorkRequestLogsResponse.Builder::opcNextPage)
                .handleResponseHeaderString(
                        "opc-request-id", ListWorkRequestLogsResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<ListWorkRequestsResponse> listWorkRequests(
            ListWorkRequestsRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            ListWorkRequestsRequest, ListWorkRequestsResponse>
                    handler) {
        Objects.requireNonNull(request.getCompartmentId(), "compartmentId is required");

        return clientCall(request, ListWorkRequestsResponse::builder)
                .logger(LOG, "listWorkRequests")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "ListWorkRequests",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/WorkRequest/ListWorkRequests")
                .method(com.oracle.bmc.http.client.Method.GET)
                .requestBuilder(ListWorkRequestsRequest::builder)
                .basePath("/20210215")
                .appendPathParam("workRequests")
                .appendQueryParam("compartmentId", request.getCompartmentId())
                .appendQueryParam("page", request.getPage())
                .appendQueryParam("limit", request.getLimit())
                .accept("application/json")
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .handleBody(
                        com.oracle.bmc.vulnerabilityscanning.model.WorkRequestCollection.class,
                        ListWorkRequestsResponse.Builder::workRequestCollection)
                .handleResponseHeaderString(
                        "opc-request-id", ListWorkRequestsResponse.Builder::opcRequestId)
                .handleResponseHeaderString(
                        "opc-next-page", ListWorkRequestsResponse.Builder::opcNextPage)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<UpdateContainerScanRecipeResponse> updateContainerScanRecipe(
            UpdateContainerScanRecipeRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            UpdateContainerScanRecipeRequest, UpdateContainerScanRecipeResponse>
                    handler) {

        Validate.notBlank(
                request.getContainerScanRecipeId(), "containerScanRecipeId must not be blank");
        Objects.requireNonNull(
                request.getUpdateContainerScanRecipeDetails(),
                "updateContainerScanRecipeDetails is required");

        return clientCall(request, UpdateContainerScanRecipeResponse::builder)
                .logger(LOG, "updateContainerScanRecipe")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "UpdateContainerScanRecipe",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/ContainerScanRecipe/UpdateContainerScanRecipe")
                .method(com.oracle.bmc.http.client.Method.PUT)
                .requestBuilder(UpdateContainerScanRecipeRequest::builder)
                .basePath("/20210215")
                .appendPathParam("containerScanRecipes")
                .appendPathParam(request.getContainerScanRecipeId())
                .accept("application/json")
                .appendHeader("if-match", request.getIfMatch())
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .hasBody()
                .handleResponseHeaderString(
                        "opc-work-request-id",
                        UpdateContainerScanRecipeResponse.Builder::opcWorkRequestId)
                .handleResponseHeaderString(
                        "opc-request-id", UpdateContainerScanRecipeResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<UpdateContainerScanTargetResponse> updateContainerScanTarget(
            UpdateContainerScanTargetRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            UpdateContainerScanTargetRequest, UpdateContainerScanTargetResponse>
                    handler) {

        Validate.notBlank(
                request.getContainerScanTargetId(), "containerScanTargetId must not be blank");
        Objects.requireNonNull(
                request.getUpdateContainerScanTargetDetails(),
                "updateContainerScanTargetDetails is required");

        return clientCall(request, UpdateContainerScanTargetResponse::builder)
                .logger(LOG, "updateContainerScanTarget")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "UpdateContainerScanTarget",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/ContainerScanTarget/UpdateContainerScanTarget")
                .method(com.oracle.bmc.http.client.Method.PUT)
                .requestBuilder(UpdateContainerScanTargetRequest::builder)
                .basePath("/20210215")
                .appendPathParam("containerScanTargets")
                .appendPathParam(request.getContainerScanTargetId())
                .accept("application/json")
                .appendHeader("if-match", request.getIfMatch())
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .hasBody()
                .handleResponseHeaderString(
                        "opc-work-request-id",
                        UpdateContainerScanTargetResponse.Builder::opcWorkRequestId)
                .handleResponseHeaderString(
                        "opc-request-id", UpdateContainerScanTargetResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<UpdateHostScanRecipeResponse> updateHostScanRecipe(
            UpdateHostScanRecipeRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            UpdateHostScanRecipeRequest, UpdateHostScanRecipeResponse>
                    handler) {

        Validate.notBlank(request.getHostScanRecipeId(), "hostScanRecipeId must not be blank");
        Objects.requireNonNull(
                request.getUpdateHostScanRecipeDetails(),
                "updateHostScanRecipeDetails is required");

        return clientCall(request, UpdateHostScanRecipeResponse::builder)
                .logger(LOG, "updateHostScanRecipe")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "UpdateHostScanRecipe",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostScanRecipe/UpdateHostScanRecipe")
                .method(com.oracle.bmc.http.client.Method.PUT)
                .requestBuilder(UpdateHostScanRecipeRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostScanRecipes")
                .appendPathParam(request.getHostScanRecipeId())
                .accept("application/json")
                .appendHeader("if-match", request.getIfMatch())
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .hasBody()
                .handleResponseHeaderString(
                        "opc-work-request-id",
                        UpdateHostScanRecipeResponse.Builder::opcWorkRequestId)
                .handleResponseHeaderString(
                        "opc-request-id", UpdateHostScanRecipeResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    @Override
    public java.util.concurrent.Future<UpdateHostScanTargetResponse> updateHostScanTarget(
            UpdateHostScanTargetRequest request,
            final com.oracle.bmc.responses.AsyncHandler<
                            UpdateHostScanTargetRequest, UpdateHostScanTargetResponse>
                    handler) {

        Validate.notBlank(request.getHostScanTargetId(), "hostScanTargetId must not be blank");
        Objects.requireNonNull(
                request.getUpdateHostScanTargetDetails(),
                "updateHostScanTargetDetails is required");

        return clientCall(request, UpdateHostScanTargetResponse::builder)
                .logger(LOG, "updateHostScanTarget")
                .serviceDetails(
                        "VulnerabilityScanning",
                        "UpdateHostScanTarget",
                        "https://docs.oracle.com/iaas/api/#/en/scanning/20210215/HostScanTarget/UpdateHostScanTarget")
                .method(com.oracle.bmc.http.client.Method.PUT)
                .requestBuilder(UpdateHostScanTargetRequest::builder)
                .basePath("/20210215")
                .appendPathParam("hostScanTargets")
                .appendPathParam(request.getHostScanTargetId())
                .accept("application/json")
                .appendHeader("if-match", request.getIfMatch())
                .appendHeader("opc-request-id", request.getOpcRequestId())
                .hasBody()
                .handleResponseHeaderString(
                        "opc-work-request-id",
                        UpdateHostScanTargetResponse.Builder::opcWorkRequestId)
                .handleResponseHeaderString(
                        "opc-request-id", UpdateHostScanTargetResponse.Builder::opcRequestId)
                .callAsync(handler);
    }

    /**
     * Create a new client instance.
     *
     * @param authenticationDetailsProvider The authentication details (see {@link Builder#build})
     * @deprecated Use the {@link #builder() builder} instead.
     */
    @Deprecated
    public VulnerabilityScanningAsyncClient(
            com.oracle.bmc.auth.BasicAuthenticationDetailsProvider authenticationDetailsProvider) {
        this(builder(), authenticationDetailsProvider);
    }

    /**
     * Create a new client instance.
     *
     * @param authenticationDetailsProvider The authentication details (see {@link Builder#build})
     * @param configuration {@link Builder#configuration}
     * @deprecated Use the {@link #builder() builder} instead.
     */
    @Deprecated
    public VulnerabilityScanningAsyncClient(
            com.oracle.bmc.auth.BasicAuthenticationDetailsProvider authenticationDetailsProvider,
            com.oracle.bmc.ClientConfiguration configuration) {
        this(builder().configuration(configuration), authenticationDetailsProvider);
    }

    /**
     * Create a new client instance.
     *
     * @param authenticationDetailsProvider The authentication details (see {@link Builder#build})
     * @param configuration {@link Builder#configuration}
     * @param clientConfigurator {@link Builder#clientConfigurator}
     * @deprecated Use the {@link #builder() builder} instead.
     */
    @Deprecated
    public VulnerabilityScanningAsyncClient(
            com.oracle.bmc.auth.BasicAuthenticationDetailsProvider authenticationDetailsProvider,
            com.oracle.bmc.ClientConfiguration configuration,
            com.oracle.bmc.http.ClientConfigurator clientConfigurator) {
        this(
                builder().configuration(configuration).clientConfigurator(clientConfigurator),
                authenticationDetailsProvider);
    }

    /**
     * Create a new client instance.
     *
     * @param authenticationDetailsProvider The authentication details (see {@link Builder#build})
     * @param configuration {@link Builder#configuration}
     * @param clientConfigurator {@link Builder#clientConfigurator}
     * @param defaultRequestSignerFactory {@link Builder#requestSignerFactory}
     * @deprecated Use the {@link #builder() builder} instead.
     */
    @Deprecated
    public VulnerabilityScanningAsyncClient(
            com.oracle.bmc.auth.AbstractAuthenticationDetailsProvider authenticationDetailsProvider,
            com.oracle.bmc.ClientConfiguration configuration,
            com.oracle.bmc.http.ClientConfigurator clientConfigurator,
            com.oracle.bmc.http.signing.RequestSignerFactory defaultRequestSignerFactory) {
        this(
                builder()
                        .configuration(configuration)
                        .clientConfigurator(clientConfigurator)
                        .requestSignerFactory(defaultRequestSignerFactory),
                authenticationDetailsProvider);
    }

    /**
     * Create a new client instance.
     *
     * @param authenticationDetailsProvider The authentication details (see {@link Builder#build})
     * @param configuration {@link Builder#configuration}
     * @param clientConfigurator {@link Builder#clientConfigurator}
     * @param defaultRequestSignerFactory {@link Builder#requestSignerFactory}
     * @param additionalClientConfigurators {@link Builder#additionalClientConfigurators}
     * @deprecated Use the {@link #builder() builder} instead.
     */
    @Deprecated
    public VulnerabilityScanningAsyncClient(
            com.oracle.bmc.auth.AbstractAuthenticationDetailsProvider authenticationDetailsProvider,
            com.oracle.bmc.ClientConfiguration configuration,
            com.oracle.bmc.http.ClientConfigurator clientConfigurator,
            com.oracle.bmc.http.signing.RequestSignerFactory defaultRequestSignerFactory,
            java.util.List<com.oracle.bmc.http.ClientConfigurator> additionalClientConfigurators) {
        this(
                builder()
                        .configuration(configuration)
                        .clientConfigurator(clientConfigurator)
                        .requestSignerFactory(defaultRequestSignerFactory)
                        .additionalClientConfigurators(additionalClientConfigurators),
                authenticationDetailsProvider);
    }

    /**
     * Create a new client instance.
     *
     * @param authenticationDetailsProvider The authentication details (see {@link Builder#build})
     * @param configuration {@link Builder#configuration}
     * @param clientConfigurator {@link Builder#clientConfigurator}
     * @param defaultRequestSignerFactory {@link Builder#requestSignerFactory}
     * @param additionalClientConfigurators {@link Builder#additionalClientConfigurators}
     * @param endpoint {@link Builder#endpoint}
     * @deprecated Use the {@link #builder() builder} instead.
     */
    @Deprecated
    public VulnerabilityScanningAsyncClient(
            com.oracle.bmc.auth.AbstractAuthenticationDetailsProvider authenticationDetailsProvider,
            com.oracle.bmc.ClientConfiguration configuration,
            com.oracle.bmc.http.ClientConfigurator clientConfigurator,
            com.oracle.bmc.http.signing.RequestSignerFactory defaultRequestSignerFactory,
            java.util.List<com.oracle.bmc.http.ClientConfigurator> additionalClientConfigurators,
            String endpoint) {
        this(
                builder()
                        .configuration(configuration)
                        .clientConfigurator(clientConfigurator)
                        .requestSignerFactory(defaultRequestSignerFactory)
                        .additionalClientConfigurators(additionalClientConfigurators)
                        .endpoint(endpoint),
                authenticationDetailsProvider);
    }

    /**
     * Create a new client instance.
     *
     * @param authenticationDetailsProvider The authentication details (see {@link Builder#build})
     * @param configuration {@link Builder#configuration}
     * @param clientConfigurator {@link Builder#clientConfigurator}
     * @param defaultRequestSignerFactory {@link Builder#requestSignerFactory}
     * @param additionalClientConfigurators {@link Builder#additionalClientConfigurators}
     * @param endpoint {@link Builder#endpoint}
     * @param signingStrategyRequestSignerFactories {@link
     *     Builder#signingStrategyRequestSignerFactories}
     * @deprecated Use the {@link #builder() builder} instead.
     */
    @Deprecated
    public VulnerabilityScanningAsyncClient(
            com.oracle.bmc.auth.AbstractAuthenticationDetailsProvider authenticationDetailsProvider,
            com.oracle.bmc.ClientConfiguration configuration,
            com.oracle.bmc.http.ClientConfigurator clientConfigurator,
            com.oracle.bmc.http.signing.RequestSignerFactory defaultRequestSignerFactory,
            java.util.Map<
                            com.oracle.bmc.http.signing.SigningStrategy,
                            com.oracle.bmc.http.signing.RequestSignerFactory>
                    signingStrategyRequestSignerFactories,
            java.util.List<com.oracle.bmc.http.ClientConfigurator> additionalClientConfigurators,
            String endpoint) {
        this(
                builder()
                        .configuration(configuration)
                        .clientConfigurator(clientConfigurator)
                        .requestSignerFactory(defaultRequestSignerFactory)
                        .additionalClientConfigurators(additionalClientConfigurators)
                        .endpoint(endpoint)
                        .signingStrategyRequestSignerFactories(
                                signingStrategyRequestSignerFactories),
                authenticationDetailsProvider);
    }
}
