/*-
 * See the file LICENSE for redistribution information.
 *
 * Copyright (c) 2002, 2012 Oracle and/or its affiliates.  All rights reserved.
 *
 */

package com.sleepycat.je.util;

import java.io.File;

import com.sleepycat.je.Environment;
import com.sleepycat.je.EnvironmentConfig;
import com.sleepycat.je.DbInternal;
import com.sleepycat.je.utilint.CmdUtil;

/**
 * In JE 5.0 the internal log format changed.  Before upgrading environments to
 * JE 5.0, this utility must first be run using JE 4.1.
 *
 * <p> This utility is used for standalone environments. For replicated 
 * environments, please use 
 * {@link com.sleepycat.je.rep.util.DbRepPreUpgrade_4_1} instead.</p>
 *
 * The command line for this utility:
 * <pre>
 * java com.sleepycat.je.util.DbPreUpgrade_4_1
 *   -h &lt;dir&gt;        # environment home directory
 * </pre>
 * See {@link DbPreUpgrade_4_1#main} for a full description of the command line
 * arguments.
 */
public class DbPreUpgrade_4_1 {
    private File envHome;
    private Environment env;

    private static final String usageString =
        "usage: " + CmdUtil.getJavaCommand(DbPreUpgrade_4_1.class) + "\n" +
        "  -h <dir> # environment home directory\n";

    private DbPreUpgrade_4_1() {
    }

    /**
     * Create a DbPreUpgrade_4_1 object for a specific environment.
     *
     * @param envHome The home directory of the Environment that needs to be
     * upgraded to JE 5.0.
     */
    public DbPreUpgrade_4_1(File envHome) {
        this.envHome = envHome;
    }

    public static void main(String[] args)
        throws Exception {

        DbPreUpgrade_4_1 upgrader = new DbPreUpgrade_4_1();
        upgrader.parseArgs(args);

        try {
            upgrader.preUpgrade();
        } catch (Throwable t) {
            t.printStackTrace();
        }
    }

    private void printUsage(String msg) {
        System.err.println(msg);
        System.err.println(usageString);
        System.exit(-1);
    }

    private void parseArgs(String[] args) {
        int argc = 0;
        int nArgs = args.length;

        if (nArgs != 2) {
            printUsage("Invalid argument");
        }

        while (argc < nArgs) {
            String thisArg = args[argc++].trim();
            if (thisArg.equals("-h")) {
                if (argc < nArgs) {
                    envHome = new File(args[argc++]);
                } else {
                    printUsage("-h requires an argument");
                }
            } else {
                printUsage("Invalid argument");
            }
        }

        if (envHome == null) {
            printUsage("-h is a required argument");
        }
    }

    /**
     * Ensure that an Environment can be upgraded to JE 5.0.
     */
    public void preUpgrade() {
        EnvironmentConfig envConfig = new EnvironmentConfig();

        /* Shared standalone and replication configuration. */
        DbPreUpgrade_4_1.configureForPreUpgrade(envConfig);

        Environment env = new Environment(envHome, envConfig);

        /* Shared standalone and replication pre-upgrade. */
        DbPreUpgrade_4_1.performPreUpgrade(env);

        env.close();
    }

    /**
     * @hidden
     * For internal use only.
     * Shared with DbRepPreUpgrade_4_1.
     */
    public static void configureForPreUpgrade(EnvironmentConfig envConfig) {

        /*
         * Should be unnecessary, but also doesn't hurt.  Leave this alone to
         * avoid having to retest without it.
         */
        envConfig.setAllowCreate(true);

        /* Disable all daemons to avoid writing dup DB nodes. */
        envConfig.setConfigParam(EnvironmentConfig.ENV_RUN_CLEANER, "false");
        envConfig.setConfigParam(EnvironmentConfig.ENV_RUN_CHECKPOINTER,
                                 "false");
        envConfig.setConfigParam(EnvironmentConfig.ENV_RUN_EVICTOR, "false");
        envConfig.setConfigParam(EnvironmentConfig.ENV_RUN_IN_COMPRESSOR,
                                 "false");
    }
    
    /**
     * @hidden
     * For internal use only.
     * Shared with DbRepPreUpgrade_4_1.
     */
    public static void performPreUpgrade(Environment env) {

        /*
         * Daemons are disabled but there is still the possibility of
         * eviction.  Eviction should not be needed, since recovery has
         * finished and we will only be performing a sync/checkpoint next.
         * However, to be paranoid we shutdown the evictor before the sync.
         */
        DbInternal.getEnvironmentImpl(env).getEvictor().shutdown();

        /*
         * Sync twice for good measure. Another checkpoint will occur when the
         * env is closed as well.
         */
        env.sync();
        env.sync();
    }
}
