/*
 * Decompiled with CFR 0.152.
 */
package com.sleepycat.je.log;

import com.sleepycat.je.DatabaseException;
import com.sleepycat.je.EnvironmentFailureException;
import com.sleepycat.je.StatsConfig;
import com.sleepycat.je.cleaner.DbFileSummary;
import com.sleepycat.je.cleaner.LocalUtilizationTracker;
import com.sleepycat.je.cleaner.TrackedFileSummary;
import com.sleepycat.je.cleaner.UtilizationTracker;
import com.sleepycat.je.config.EnvironmentParams;
import com.sleepycat.je.dbi.DatabaseImpl;
import com.sleepycat.je.dbi.DbConfigManager;
import com.sleepycat.je.dbi.EnvironmentFailureReason;
import com.sleepycat.je.dbi.EnvironmentImpl;
import com.sleepycat.je.log.ChecksumException;
import com.sleepycat.je.log.ChecksumValidator;
import com.sleepycat.je.log.FSyncManager;
import com.sleepycat.je.log.FileHandleSource;
import com.sleepycat.je.log.FileManager;
import com.sleepycat.je.log.FileSource;
import com.sleepycat.je.log.LogBuffer;
import com.sleepycat.je.log.LogBufferPool;
import com.sleepycat.je.log.LogBufferSegment;
import com.sleepycat.je.log.LogEntryHeader;
import com.sleepycat.je.log.LogEntryType;
import com.sleepycat.je.log.LogItem;
import com.sleepycat.je.log.LogParams;
import com.sleepycat.je.log.LogSource;
import com.sleepycat.je.log.LogStatDefinition;
import com.sleepycat.je.log.Provisional;
import com.sleepycat.je.log.ReplicationContext;
import com.sleepycat.je.log.WholeEntry;
import com.sleepycat.je.log.entry.LogEntry;
import com.sleepycat.je.recovery.Checkpointer;
import com.sleepycat.je.txn.WriteLockInfo;
import com.sleepycat.je.utilint.DbLsn;
import com.sleepycat.je.utilint.LSNStat;
import com.sleepycat.je.utilint.LongStat;
import com.sleepycat.je.utilint.StatGroup;
import com.sleepycat.je.utilint.TestHook;
import com.sleepycat.je.utilint.TestHookExecute;
import com.sleepycat.je.utilint.VLSN;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.util.Collection;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.CountDownLatch;

public class LogManager {
    private static final String DEBUG_NAME = LogManager.class.getName();
    private final LogBufferPool logBufferPool;
    private final Object logWriteMutex;
    private final boolean doChecksumOnRead;
    private final FileManager fileManager;
    private final FSyncManager grpManager;
    private final EnvironmentImpl envImpl;
    private final boolean readOnly;
    private final int readBufferSize;
    private long lastLsnAtRecovery = -1L;
    private final StatGroup stats;
    private final LongStat nRepeatFaultReads;
    private final LongStat nTempBufferWrites;
    private final LSNStat endOfLog;
    private LogBuffer prevLogBuffer = null;
    private TestHook readHook;
    private TestHook<Object> delayVLSNRegisterHook;
    private TestHook<CountDownLatch> flushHook;
    private final Queue<LazyQueueEntry> lazyLogQueue = new ConcurrentLinkedQueue<LazyQueueEntry>();

    public LogManager(EnvironmentImpl envImpl, boolean readOnly) throws DatabaseException {
        this.envImpl = envImpl;
        this.fileManager = envImpl.getFileManager();
        this.grpManager = new FSyncManager(this.envImpl);
        DbConfigManager configManager = envImpl.getConfigManager();
        this.readOnly = readOnly;
        this.logBufferPool = new LogBufferPool(this.fileManager, envImpl);
        this.doChecksumOnRead = configManager.getBoolean(EnvironmentParams.LOG_CHECKSUM_READ);
        this.logWriteMutex = new Object();
        this.readBufferSize = configManager.getInt(EnvironmentParams.LOG_FAULT_READ_SIZE);
        this.stats = new StatGroup("I/O", "Log file opens, fsyncs, reads, writes, cache misses.");
        this.nRepeatFaultReads = new LongStat(this.stats, LogStatDefinition.LOGMGR_REPEAT_FAULT_READS);
        this.nTempBufferWrites = new LongStat(this.stats, LogStatDefinition.LOGMGR_TEMP_BUFFER_WRITES);
        this.endOfLog = new LSNStat(this.stats, LogStatDefinition.LOGMGR_END_OF_LOG);
    }

    public boolean getChecksumOnRead() {
        return this.doChecksumOnRead;
    }

    public long getLastLsnAtRecovery() {
        return this.lastLsnAtRecovery;
    }

    public void setLastLsnAtRecovery(long lastLsnAtRecovery) {
        this.lastLsnAtRecovery = lastLsnAtRecovery;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void resetPool(DbConfigManager configManager) throws DatabaseException {
        Object object = this.logWriteMutex;
        synchronized (object) {
            this.logBufferPool.reset(configManager);
        }
    }

    public long logForceFlush(LogEntry entry, boolean fsyncRequired, ReplicationContext repContext) throws DatabaseException {
        return this.log(entry, Provisional.NO, true, fsyncRequired, false, repContext);
    }

    public long logForceFlip(LogEntry entry) throws DatabaseException {
        return this.log(entry, Provisional.NO, true, false, true, ReplicationContext.NO_REPLICATE);
    }

    public long log(LogEntry entry, ReplicationContext repContext) throws DatabaseException {
        return this.log(entry, Provisional.NO, false, false, false, repContext);
    }

    public void logLazily(LogEntry entry, ReplicationContext repContext) {
        this.lazyLogQueue.add(new LazyQueueEntry(entry, repContext));
    }

    private long log(LogEntry entry, Provisional provisional, boolean flushRequired, boolean fsyncRequired, boolean forceNewLogFile, ReplicationContext repContext) throws DatabaseException {
        LogParams params = new LogParams();
        params.entry = entry;
        params.provisional = provisional;
        params.repContext = repContext;
        params.flushRequired = flushRequired;
        params.fsyncRequired = fsyncRequired;
        params.forceNewLogFile = forceNewLogFile;
        LogItem item = this.log(params);
        return item.lsn;
    }

    public LogItem log(LogParams params) throws DatabaseException {
        LogItem item = new LogItem();
        if (this.readOnly) {
            return item;
        }
        try {
            LazyQueueEntry lqe = this.lazyLogQueue.poll();
            while (lqe != null) {
                LogParams lqeParams = new LogParams();
                lqeParams.entry = lqe.entry;
                lqeParams.provisional = Provisional.NO;
                lqeParams.repContext = lqe.repContext;
                this.logItem(new LogItem(), lqeParams);
                lqe = this.lazyLogQueue.poll();
            }
            LogEntry logEntry = params.entry;
            if (logEntry.getLogType().marshallOutsideLatch()) {
                item.header = new LogEntryHeader(logEntry, params.provisional, params.repContext);
                item.buffer = this.marshallIntoBuffer(item.header, logEntry);
            }
            this.logItem(item, params);
            if (params.fsyncRequired || params.flushRequired) {
                this.grpManager.sync(params.fsyncRequired);
            } else if (params.switchedLogBuffer) {
                this.logBufferPool.writeDirty(false);
            }
            TestHookExecute.doHookIfSet(this.flushHook);
            if (params.repContext.inReplicationStream()) {
                assert (item.header.getVLSN() != null) : "Unexpected null vlsn: " + item.header + " " + params.repContext;
                TestHookExecute.doHookIfSet(this.delayVLSNRegisterHook);
                this.envImpl.registerVLSN(item);
            }
        }
        catch (EnvironmentFailureException e) {
            if (!this.envImpl.isValid()) {
                throw e;
            }
            throw EnvironmentFailureException.unexpectedException(this.envImpl, (Exception)e);
        }
        catch (Exception e) {
            throw EnvironmentFailureException.unexpectedException(this.envImpl, e);
        }
        catch (Error e) {
            this.envImpl.invalidate(e);
            throw e;
        }
        Checkpointer ckpter = this.envImpl.getCheckpointer();
        if (ckpter != null) {
            ckpter.wakeupAfterWrite();
        }
        if (params.wakeupCleaner) {
            this.envImpl.getUtilizationTracker().activateCleaner();
        }
        if (params.backgroundIO) {
            this.envImpl.updateBackgroundWrites(params.totalNewSize, this.logBufferPool.getLogBufferSize());
        }
        return item;
    }

    private void logItem(LogItem item, LogParams params) throws IOException, DatabaseException {
        UtilizationTracker tracker;
        boolean flushRequired = params.flushRequired && !params.fsyncRequired;
        LogWriteInfo lwi = this.serialLog(item, params, params.forceNewLogFile, flushRequired, tracker = this.envImpl.getUtilizationTracker());
        if (lwi != null) {
            item.buffer = item.header.addPostMarshallingInfo(item.buffer, lwi.fileOffset, lwi.vlsn);
            lwi.lbs.put(item.buffer);
        }
        this.updateObsolete(params, tracker);
    }

    private LogWriteInfo serialLog(LogItem item, LogParams params, boolean forceNewLogFile, boolean flushRequired, UtilizationTracker tracker) throws IOException {
        Object object = this.logWriteMutex;
        synchronized (object) {
            this.envImpl.checkIfInvalid();
            try {
                return this.serialLogWork(item, params, forceNewLogFile, flushRequired, tracker);
            }
            catch (EnvironmentFailureException e) {
                if (!this.envImpl.isValid()) {
                    throw e;
                }
                throw EnvironmentFailureException.unexpectedException(this.envImpl, (Exception)e);
            }
            catch (Exception e) {
                throw EnvironmentFailureException.unexpectedException(this.envImpl, e);
            }
            catch (Error e) {
                this.envImpl.invalidate(e);
                throw e;
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private LogWriteInfo serialLogWork(LogItem item, LogParams params, boolean forceNewLogFile, boolean flushRequired, UtilizationTracker tracker) throws IOException {
        LogBufferSegment useBuffer;
        int entrySize;
        boolean usedTemporaryBuffer = false;
        LogBuffer lastLogBuffer = null;
        VLSN vlsn = null;
        boolean marshallOutsideLatch = item.buffer != null;
        LogEntryType entryType = params.entry.getLogType();
        if (!DbLsn.isTransientOrNull(params.oldLsn)) {
            if (params.obsoleteDupsAllowed) {
                tracker.countObsoleteNodeDupsAllowed(params.oldLsn, entryType, params.oldSize, params.nodeDb);
            } else {
                tracker.countObsoleteNode(params.oldLsn, entryType, params.oldSize, params.nodeDb);
            }
        }
        if (!DbLsn.isTransientOrNull(params.auxOldLsn)) {
            if (params.obsoleteDupsAllowed) {
                tracker.countObsoleteNodeDupsAllowed(params.auxOldLsn, entryType, 0, params.nodeDb);
            } else {
                tracker.countObsoleteNode(params.auxOldLsn, entryType, 0, params.nodeDb);
            }
        }
        if (marshallOutsideLatch) {
            entrySize = item.buffer.limit();
            assert (item.header != null);
        } else {
            assert (item.header == null);
            item.header = new LogEntryHeader(params.entry, params.provisional, params.repContext);
            entrySize = item.header.getEntrySize();
        }
        if (forceNewLogFile) {
            this.fileManager.forceNewLogFile();
        }
        boolean flippedFile = this.fileManager.bumpLsn(entrySize);
        usedTemporaryBuffer = false;
        long currentLsn = this.fileManager.getLastUsedLsn();
        if (tracker.countNewLogEntry(currentLsn, entryType, entrySize, params.nodeDb)) {
            params.wakeupCleaner = true;
        }
        if (params.entry.isImmediatelyObsolete(params.nodeDb)) {
            tracker.countObsoleteNodeInexact(currentLsn, entryType, entrySize, params.nodeDb);
        }
        if (!marshallOutsideLatch) {
            assert (item.buffer == null);
            item.buffer = this.marshallIntoBuffer(item.header, params.entry);
        }
        if (entrySize != item.buffer.limit()) {
            throw EnvironmentFailureException.unexpectedState("Logged entry entrySize= " + entrySize + " but marshalledSize=" + item.buffer.limit() + " type=" + entryType + " currentLsn=" + DbLsn.getNoFormatString(currentLsn));
        }
        lastLogBuffer = this.logBufferPool.getWriteBuffer(entrySize, flippedFile);
        if (lastLogBuffer != this.prevLogBuffer) {
            params.switchedLogBuffer = true;
        }
        this.prevLogBuffer = lastLogBuffer;
        long fileOffset = this.fileManager.getPrevEntryOffset();
        if (params.repContext.getClientVLSN() != null || params.repContext.mustGenerateVLSN()) {
            vlsn = params.repContext.mustGenerateVLSN() ? this.envImpl.bumpVLSN() : params.repContext.getClientVLSN();
        }
        lastLogBuffer.latchForWrite();
        try {
            useBuffer = lastLogBuffer.allocate(entrySize);
            if (useBuffer == null) {
                item.buffer = item.header.addPostMarshallingInfo(item.buffer, fileOffset, vlsn);
                this.fileManager.writeLogBuffer(new LogBuffer(item.buffer, currentLsn), flushRequired);
                usedTemporaryBuffer = true;
                assert (lastLogBuffer.getDataBuffer().position() == 0);
                this.nTempBufferWrites.increment();
            }
        }
        finally {
            lastLogBuffer.release();
        }
        if (!usedTemporaryBuffer) {
            lastLogBuffer.registerLsn(currentLsn);
        }
        params.entry.postLogWork(item.header, currentLsn, vlsn);
        item.lsn = currentLsn;
        item.size = entrySize;
        params.totalNewSize += entrySize;
        return useBuffer == null ? null : new LogWriteInfo(useBuffer, vlsn, fileOffset);
    }

    private ByteBuffer marshallIntoBuffer(LogEntryHeader header, LogEntry entry) {
        int entrySize = header.getSize() + header.getItemSize();
        ByteBuffer destBuffer = ByteBuffer.allocate(entrySize);
        header.writeToLog(destBuffer);
        entry.writeEntry(destBuffer);
        destBuffer.flip();
        return destBuffer;
    }

    ByteBuffer putIntoBuffer(LogEntry entry, long prevLogEntryOffset) {
        LogEntryHeader header = new LogEntryHeader(entry, Provisional.NO, ReplicationContext.NO_REPLICATE);
        assert (!entry.getLogType().isTransactional());
        ByteBuffer destBuffer = this.marshallIntoBuffer(header, entry);
        return header.addPostMarshallingInfo(destBuffer, prevLogEntryOffset, null);
    }

    public LogEntry getLogEntry(long lsn) throws FileNotFoundException {
        return this.getLogEntry(lsn, false).getEntry();
    }

    public WholeEntry getWholeLogEntry(long lsn) throws FileNotFoundException {
        return this.getLogEntry(lsn, false);
    }

    public WholeEntry getLogEntryAllowInvisibleAtRecovery(long lsn) throws FileNotFoundException {
        return this.getLogEntry(lsn, this.envImpl.isInInit());
    }

    public WholeEntry getLogEntryAllowInvisible(long lsn) throws FileNotFoundException {
        return this.getLogEntry(lsn, true);
    }

    private WholeEntry getLogEntry(long lsn, boolean invisibleReadAllowed) throws FileNotFoundException {
        this.envImpl.checkIfInvalid();
        try {
            LogSource logSource = this.getLogSource(lsn);
            return this.getLogEntryFromLogSource(lsn, logSource, invisibleReadAllowed);
        }
        catch (ChecksumException e) {
            throw new EnvironmentFailureException(this.envImpl, EnvironmentFailureReason.LOG_CHECKSUM, (Throwable)e);
        }
    }

    public LogEntry getLogEntryHandleFileNotFound(long lsn) throws DatabaseException {
        try {
            return this.getLogEntry(lsn);
        }
        catch (FileNotFoundException e) {
            throw new EnvironmentFailureException(this.envImpl, EnvironmentFailureReason.LOG_FILE_NOT_FOUND, (Throwable)e);
        }
    }

    public WholeEntry getWholeLogEntryHandleFileNotFound(long lsn) throws DatabaseException {
        try {
            return this.getWholeLogEntry(lsn);
        }
        catch (FileNotFoundException e) {
            throw new EnvironmentFailureException(this.envImpl, EnvironmentFailureReason.LOG_FILE_NOT_FOUND, (Throwable)e);
        }
    }

    LogEntry getLogEntryAllowChecksumException(long lsn) throws ChecksumException, FileNotFoundException, DatabaseException {
        return this.getLogEntryFromLogSource(lsn, this.getLogSource(lsn), false).getEntry();
    }

    LogEntry getLogEntryAllowChecksumException(long lsn, RandomAccessFile file, int logVersion) throws ChecksumException, DatabaseException {
        return this.getLogEntryFromLogSource(lsn, new FileSource(file, this.readBufferSize, this.fileManager, DbLsn.getFileNumber(lsn), logVersion), false).getEntry();
    }

    WholeEntry getLogEntryFromLogSource(long lsn, LogSource logSource, boolean invisibleReadAllowed) throws ChecksumException, DatabaseException {
        try {
            long fileOffset = DbLsn.getFileOffset(lsn);
            ByteBuffer entryBuffer = logSource.getBytes(fileOffset);
            if (entryBuffer.remaining() < 14) {
                throw new ChecksumException("Incomplete log entry header, size=" + entryBuffer.remaining() + " lsn=" + DbLsn.getNoFormatString(lsn));
            }
            LogEntryHeader header = new LogEntryHeader(entryBuffer, logSource.getLogVersion());
            if (header.isVariableLength()) {
                if (entryBuffer.remaining() < header.getVariablePortionSize()) {
                    throw new ChecksumException("Incomplete log entry header, size=" + entryBuffer.remaining() + " varSize=" + header.getVariablePortionSize() + " lsn=" + DbLsn.getNoFormatString(lsn));
                }
                header.readVariablePortion(entryBuffer);
            }
            ChecksumValidator validator = null;
            if (this.doChecksumOnRead) {
                int itemStart = entryBuffer.position();
                if (header.isInvisible()) {
                    LogEntryHeader.turnOffInvisible(entryBuffer, itemStart - header.getSize());
                }
                validator = new ChecksumValidator();
                int headerSizeMinusChecksum = header.getSizeMinusChecksum();
                entryBuffer.position(itemStart - headerSizeMinusChecksum);
                validator.update(entryBuffer, headerSizeMinusChecksum);
                entryBuffer.position(itemStart);
            }
            int itemSize = header.getItemSize();
            if (entryBuffer.remaining() < itemSize) {
                entryBuffer = logSource.getBytes(fileOffset + (long)header.getSize(), itemSize);
                this.nRepeatFaultReads.increment();
            }
            if (this.doChecksumOnRead) {
                validator.update(entryBuffer, itemSize);
                validator.validate(header.getChecksum(), lsn);
            }
            if (header.isInvisible() && !invisibleReadAllowed) {
                throw new EnvironmentFailureException(this.envImpl, EnvironmentFailureReason.LOG_INTEGRITY, "Read invisible log entry at " + DbLsn.getNoFormatString(lsn) + " " + header);
            }
            assert (LogEntryType.isValidType(header.getType())) : "Read non-valid log entry type: " + header.getType();
            LogEntry logEntry = LogEntryType.findType(header.getType()).getNewLogEntry();
            logEntry.readEntry(this.envImpl, header, entryBuffer);
            if (this.readHook != null) {
                try {
                    this.readHook.doIOHook();
                }
                catch (IOException e) {
                    throw new EnvironmentFailureException(this.envImpl, EnvironmentFailureReason.LOG_READ, (Throwable)e);
                }
            }
            WholeEntry wholeEntry = new WholeEntry(header, logEntry);
            return wholeEntry;
        }
        catch (Error e) {
            this.envImpl.invalidate(e);
            throw e;
        }
        finally {
            if (logSource != null) {
                logSource.release();
            }
        }
    }

    public ByteBuffer getByteBufferFromLog(long lsn) throws DatabaseException {
        this.envImpl.checkIfInvalid();
        LogSource logSource = null;
        try {
            logSource = this.getLogSource(lsn);
            long fileOffset = DbLsn.getFileOffset(lsn);
            ByteBuffer entryBuffer = logSource.getBytes(fileOffset);
            int startingPosition = entryBuffer.position();
            int amountRemaining = entryBuffer.remaining();
            assert (amountRemaining >= 22);
            LogEntryHeader header = new LogEntryHeader(entryBuffer, logSource.getLogVersion());
            int totalSize = header.getSize() + header.getItemSize();
            if (amountRemaining < totalSize) {
                entryBuffer = logSource.getBytes(fileOffset, totalSize);
                this.nRepeatFaultReads.increment();
            }
            entryBuffer.position(startingPosition);
            ByteBuffer singleEntryBuffer = ByteBuffer.allocate(totalSize);
            entryBuffer.limit(startingPosition + totalSize);
            singleEntryBuffer.put(entryBuffer);
            singleEntryBuffer.position(0);
            ByteBuffer byteBuffer = singleEntryBuffer;
            return byteBuffer;
        }
        catch (FileNotFoundException e) {
            throw new EnvironmentFailureException(this.envImpl, EnvironmentFailureReason.LOG_FILE_NOT_FOUND, (Throwable)e);
        }
        catch (ChecksumException e) {
            throw new EnvironmentFailureException(this.envImpl, EnvironmentFailureReason.LOG_CHECKSUM, (Throwable)e);
        }
        finally {
            logSource.release();
        }
    }

    public Object getEntry(long lsn) throws FileNotFoundException, DatabaseException {
        LogEntry entry = this.getLogEntry(lsn);
        return entry.getMainItem();
    }

    public Object getEntryHandleFileNotFound(long lsn) {
        LogEntry entry = this.getLogEntryHandleFileNotFound(lsn);
        return entry.getMainItem();
    }

    public LogSource getLogSource(long lsn) throws FileNotFoundException, ChecksumException, DatabaseException {
        LogBuffer logBuffer = this.logBufferPool.getReadBufferByLsn(lsn);
        if (logBuffer == null) {
            try {
                long fileNum = DbLsn.getFileNumber(lsn);
                return new FileHandleSource(this.fileManager.getFileHandle(fileNum), this.readBufferSize, this.fileManager);
            }
            catch (DatabaseException e) {
                e.addErrorMessage("lsn= " + DbLsn.getNoFormatString(lsn));
                throw e;
            }
        }
        return logBuffer;
    }

    public LogBuffer getReadBufferByLsn(long lsn) {
        assert (DbLsn.getFileOffset(lsn) != 0L) : "Read of lsn " + DbLsn.getNoFormatString(lsn) + " is illegal because file header entry is not in the log buffer";
        return this.logBufferPool.getReadBufferByLsn(lsn);
    }

    public void flush() throws DatabaseException {
        if (!this.readOnly) {
            this.flushInternal(false);
            this.fileManager.syncLogEnd();
        }
    }

    public void flushNoSync() throws DatabaseException {
        if (!this.readOnly) {
            this.flushInternal(false);
        }
    }

    public void flushWriteNoSync() throws DatabaseException {
        if (!this.readOnly) {
            this.flushInternal(true);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void flushInternal(boolean flushRequired) throws DatabaseException {
        Object object = this.logWriteMutex;
        synchronized (object) {
            if (!this.logBufferPool.bumpCurrent(0)) {
                this.logBufferPool.bumpAndWriteSynced(0, flushRequired);
                return;
            }
        }
        this.logBufferPool.writeDirty(flushRequired);
    }

    public StatGroup loadStats(StatsConfig config) throws DatabaseException {
        if (!config.getFast()) {
            this.loadEndOfLogStat();
        }
        StatGroup copyStats = this.stats.cloneGroup(config.getClear());
        copyStats.addAll(this.logBufferPool.loadStats(config));
        copyStats.addAll(this.fileManager.loadStats(config));
        copyStats.addAll(this.grpManager.loadStats(config));
        return copyStats;
    }

    public long getNCacheMiss() {
        return this.logBufferPool.getNCacheMiss();
    }

    public StatGroup getBufferPoolLatchStats() {
        return this.logBufferPool.getBufferPoolLatchStats();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public TrackedFileSummary getUnflushableTrackedSummary(long file) {
        Object object = this.logWriteMutex;
        synchronized (object) {
            return this.envImpl.getUtilizationTracker().getUnflushableTrackedSummary(file);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void removeTrackedFile(TrackedFileSummary tfs) {
        Object object = this.logWriteMutex;
        synchronized (object) {
            tfs.reset();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void updateObsolete(LogParams params, UtilizationTracker tracker) {
        Object object = this.logWriteMutex;
        synchronized (object) {
            if (params.packedObsoleteInfo != null) {
                params.packedObsoleteInfo.countObsoleteInfo(tracker, params.nodeDb);
            }
            if (params.obsoleteWriteLockInfo != null) {
                for (WriteLockInfo info : params.obsoleteWriteLockInfo) {
                    tracker.countObsoleteNode(info.getAbortLsn(), null, info.getAbortLogSize(), info.getDb());
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void countObsoleteNode(long lsn, LogEntryType type, int size, DatabaseImpl nodeDb, boolean countExact) {
        Object object = this.logWriteMutex;
        synchronized (object) {
            UtilizationTracker tracker = this.envImpl.getUtilizationTracker();
            if (countExact) {
                tracker.countObsoleteNode(lsn, type, size, nodeDb);
            } else {
                tracker.countObsoleteNodeInexact(lsn, type, size, nodeDb);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void countObsoleteNodeDupsAllowed(long lsn, LogEntryType type, int size, DatabaseImpl nodeDb) {
        Object object = this.logWriteMutex;
        synchronized (object) {
            UtilizationTracker tracker = this.envImpl.getUtilizationTracker();
            tracker.countObsoleteNodeDupsAllowed(lsn, type, size, nodeDb);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void transferToUtilizationTracker(LocalUtilizationTracker localTracker) throws DatabaseException {
        Object object = this.logWriteMutex;
        synchronized (object) {
            UtilizationTracker tracker = this.envImpl.getUtilizationTracker();
            localTracker.transferToUtilizationTracker(tracker);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void countObsoleteDb(DatabaseImpl db) {
        Object object = this.logWriteMutex;
        synchronized (object) {
            db.countObsoleteDb(this.envImpl.getUtilizationTracker(), -1L);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean removeDbFileSummaries(DatabaseImpl db, Collection<Long> fileNums) {
        Object object = this.logWriteMutex;
        synchronized (object) {
            return db.removeDbFileSummaries(fileNums);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Map<Long, DbFileSummary> cloneDbFileSummaries(DatabaseImpl db) {
        Object object = this.logWriteMutex;
        synchronized (object) {
            return db.cloneDbFileSummariesInternal();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void loadEndOfLogStat() {
        Object object = this.logWriteMutex;
        synchronized (object) {
            this.endOfLog.set(this.fileManager.getLastUsedLsn());
        }
    }

    public void setReadHook(TestHook hook) {
        this.readHook = hook;
    }

    public void setDelayVLSNRegisterHook(TestHook<Object> hook) {
        this.delayVLSNRegisterHook = hook;
    }

    public void setFlushLogHook(TestHook<CountDownLatch> hook) {
        this.flushHook = hook;
        this.grpManager.setFlushLogHook(hook);
    }

    private class LogWriteInfo {
        final LogBufferSegment lbs;
        final VLSN vlsn;
        final long fileOffset;

        LogWriteInfo(LogBufferSegment bs, VLSN vlsn, long fileOffset) {
            this.lbs = bs;
            this.vlsn = vlsn;
            this.fileOffset = fileOffset;
        }
    }

    private static class LazyQueueEntry {
        private final LogEntry entry;
        private final ReplicationContext repContext;

        private LazyQueueEntry(LogEntry entry, ReplicationContext repContext) {
            this.entry = entry;
            this.repContext = repContext;
        }
    }
}

