package de.flapdoodle.embed.mongo.packageresolver;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link VersionRange}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableVersionRange.builder()}.
 * Use the static factory method to create immutable instances:
 * {@code ImmutableVersionRange.of()}.
 */
@SuppressWarnings({"all"})
public final class ImmutableVersionRange extends VersionRange {
  private final NumericVersion min;
  private final NumericVersion max;

  private ImmutableVersionRange(
      NumericVersion min,
      NumericVersion max) {
    this.min = Objects.requireNonNull(min, "min");
    this.max = Objects.requireNonNull(max, "max");
  }

  private ImmutableVersionRange(
      ImmutableVersionRange original,
      NumericVersion min,
      NumericVersion max) {
    this.min = min;
    this.max = max;
  }

  /**
   * @return The value of the {@code min} attribute
   */
  @Override
  public NumericVersion min() {
    return min;
  }

  /**
   * @return The value of the {@code max} attribute
   */
  @Override
  public NumericVersion max() {
    return max;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link VersionRange#min() min} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for min
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableVersionRange withMin(NumericVersion value) {
    if (this.min == value) return this;
    NumericVersion newValue = Objects.requireNonNull(value, "min");
    return validate(new ImmutableVersionRange(this, newValue, this.max));
  }

  /**
   * Copy the current immutable object by setting a value for the {@link VersionRange#max() max} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for max
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableVersionRange withMax(NumericVersion value) {
    if (this.max == value) return this;
    NumericVersion newValue = Objects.requireNonNull(value, "max");
    return validate(new ImmutableVersionRange(this, this.min, newValue));
  }

  /**
   * This instance is equal to all instances of {@code ImmutableVersionRange} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableVersionRange
        && equalsByValue((ImmutableVersionRange) another);
  }

  private boolean equalsByValue(ImmutableVersionRange another) {
    return min.equals(another.min)
        && max.equals(another.max);
  }

  /**
   * Computes a hash code from attributes: {@code min}, {@code max}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + min.hashCode();
    h += (h << 5) + max.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code VersionRange} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "VersionRange{"
        + "min=" + min
        + ", max=" + max
        + "}";
  }

  /**
   * Construct a new immutable {@code VersionRange} instance.
   * @param min The value for the {@code min} attribute
   * @param max The value for the {@code max} attribute
   * @return An immutable VersionRange instance
   */
  public static ImmutableVersionRange of(NumericVersion min, NumericVersion max) {
    return validate(new ImmutableVersionRange(min, max));
  }

  private static ImmutableVersionRange validate(ImmutableVersionRange instance) {
    instance.check();
    return instance;
  }

  /**
   * Creates an immutable copy of a {@link VersionRange} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable VersionRange instance
   */
  public static ImmutableVersionRange copyOf(VersionRange instance) {
    if (instance instanceof ImmutableVersionRange) {
      return (ImmutableVersionRange) instance;
    }
    return ImmutableVersionRange.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableVersionRange ImmutableVersionRange}.
   * <pre>
   * ImmutableVersionRange.builder()
   *    .min(de.flapdoodle.embed.mongo.packageresolver.NumericVersion) // required {@link VersionRange#min() min}
   *    .max(de.flapdoodle.embed.mongo.packageresolver.NumericVersion) // required {@link VersionRange#max() max}
   *    .build();
   * </pre>
   * @return A new ImmutableVersionRange builder
   */
  public static ImmutableVersionRange.Builder builder() {
    return new ImmutableVersionRange.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableVersionRange ImmutableVersionRange}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private static final long INIT_BIT_MIN = 0x1L;
    private static final long INIT_BIT_MAX = 0x2L;
    private long initBits = 0x3L;

    private NumericVersion min;
    private NumericVersion max;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code VersionRange} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(VersionRange instance) {
      Objects.requireNonNull(instance, "instance");
      this.min(instance.min());
      this.max(instance.max());
      return this;
    }

    /**
     * Initializes the value for the {@link VersionRange#min() min} attribute.
     * @param min The value for min 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder min(NumericVersion min) {
      this.min = Objects.requireNonNull(min, "min");
      initBits &= ~INIT_BIT_MIN;
      return this;
    }

    /**
     * Initializes the value for the {@link VersionRange#max() max} attribute.
     * @param max The value for max 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder max(NumericVersion max) {
      this.max = Objects.requireNonNull(max, "max");
      initBits &= ~INIT_BIT_MAX;
      return this;
    }

    /**
     * Builds a new {@link ImmutableVersionRange ImmutableVersionRange}.
     * @return An immutable instance of VersionRange
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableVersionRange build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return ImmutableVersionRange.validate(new ImmutableVersionRange(null, min, max));
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_MIN) != 0) attributes.add("min");
      if ((initBits & INIT_BIT_MAX) != 0) attributes.add("max");
      return "Cannot build VersionRange, some of required attributes are not set " + attributes;
    }
  }
}
