package de.flapdoodle.embed.process.transitions;

import de.flapdoodle.embed.process.archives.ExtractedFileSet;
import de.flapdoodle.embed.process.config.SupportConfig;
import de.flapdoodle.embed.process.config.store.Package;
import de.flapdoodle.embed.process.distribution.Distribution;
import de.flapdoodle.embed.process.distribution.Version;
import de.flapdoodle.embed.process.io.ProcessOutput;
import de.flapdoodle.embed.process.io.directories.PersistentDir;
import de.flapdoodle.embed.process.io.directories.TempDir;
import de.flapdoodle.embed.process.io.progress.ProgressListener;
import de.flapdoodle.embed.process.store.DownloadCache;
import de.flapdoodle.embed.process.store.ExtractedFileSetStore;
import de.flapdoodle.embed.process.types.*;
import de.flapdoodle.embed.process.types.ExecutedProcess;
import de.flapdoodle.embed.process.types.ProcessConfig;
import de.flapdoodle.os.OS;
import de.flapdoodle.reverse.Transition;
import de.flapdoodle.reverse.transitions.Start;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;
import java.util.function.Supplier;

/**
 * Immutable implementation of {@link ProcessFactory}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableProcessFactory.builder()}.
 */
@SuppressWarnings({"all"})
public final class ImmutableProcessFactory extends ProcessFactory {
  private final Version version;
  private final Transition<Name> name;
  private final Transition<TempDir> initTempDirectory;
  private final Transition<ProcessWorkingDir> processWorkingDir;
  private final Start<ProcessConfig> processConfig;
  private final Transition<ProcessEnv> processEnv;
  private final Transition<ProcessArguments> processArguments;
  private final Transition<ProcessOutput> processOutput;
  private final Transition<ProgressListener> progressListener;
  private final Transition<SupportConfig> supportConfig;
  private final Transition<PersistentDir> persistentBaseDir;
  private final Transition<DownloadCache> downloadCache;
  private final Transition<ExtractedFileSetStore> extractedFileSetStore;
  private final Transition<ExtractedFileSet> extractPackage;
  private final Transition<Archive> downloadPackage;
  private final Transition<Distribution> distribution;
  private final Function<Distribution, Package> packageInformation;
  private final Supplier<Collection<? extends OS>> osList;
  private final Transition<ExecutedProcess> executer;

  private ImmutableProcessFactory(ImmutableProcessFactory.Builder builder) {
    this.version = builder.version;
    this.name = builder.name;
    this.processArguments = builder.processArguments;
    this.persistentBaseDir = builder.persistentBaseDir;
    this.packageInformation = builder.packageInformation;
    this.osList = builder.osList;
    if (builder.initTempDirectory != null) {
      initShim.initTempDirectory(builder.initTempDirectory);
    }
    if (builder.processWorkingDir != null) {
      initShim.processWorkingDir(builder.processWorkingDir);
    }
    if (builder.processConfig != null) {
      initShim.processConfig(builder.processConfig);
    }
    if (builder.processEnv != null) {
      initShim.processEnv(builder.processEnv);
    }
    if (builder.processOutput != null) {
      initShim.processOutput(builder.processOutput);
    }
    if (builder.progressListener != null) {
      initShim.progressListener(builder.progressListener);
    }
    if (builder.supportConfig != null) {
      initShim.supportConfig(builder.supportConfig);
    }
    if (builder.downloadCache != null) {
      initShim.downloadCache(builder.downloadCache);
    }
    if (builder.extractedFileSetStore != null) {
      initShim.extractedFileSetStore(builder.extractedFileSetStore);
    }
    if (builder.extractPackage != null) {
      initShim.extractPackage(builder.extractPackage);
    }
    if (builder.downloadPackage != null) {
      initShim.downloadPackage(builder.downloadPackage);
    }
    if (builder.distribution != null) {
      initShim.distribution(builder.distribution);
    }
    if (builder.executer != null) {
      initShim.executer(builder.executer);
    }
    this.initTempDirectory = initShim.initTempDirectory();
    this.processWorkingDir = initShim.processWorkingDir();
    this.processConfig = initShim.processConfig();
    this.processEnv = initShim.processEnv();
    this.processOutput = initShim.processOutput();
    this.progressListener = initShim.progressListener();
    this.supportConfig = initShim.supportConfig();
    this.downloadCache = initShim.downloadCache();
    this.extractedFileSetStore = initShim.extractedFileSetStore();
    this.extractPackage = initShim.extractPackage();
    this.downloadPackage = initShim.downloadPackage();
    this.distribution = initShim.distribution();
    this.executer = initShim.executer();
    this.initShim = null;
  }

  private ImmutableProcessFactory(
      Version version,
      Transition<Name> name,
      Transition<TempDir> initTempDirectory,
      Transition<ProcessWorkingDir> processWorkingDir,
      Start<ProcessConfig> processConfig,
      Transition<ProcessEnv> processEnv,
      Transition<ProcessArguments> processArguments,
      Transition<ProcessOutput> processOutput,
      Transition<ProgressListener> progressListener,
      Transition<SupportConfig> supportConfig,
      Transition<PersistentDir> persistentBaseDir,
      Transition<DownloadCache> downloadCache,
      Transition<ExtractedFileSetStore> extractedFileSetStore,
      Transition<ExtractedFileSet> extractPackage,
      Transition<Archive> downloadPackage,
      Transition<Distribution> distribution,
      Function<Distribution, Package> packageInformation,
      Supplier<Collection<? extends OS>> osList,
      Transition<ExecutedProcess> executer) {
    this.version = version;
    this.name = name;
    this.initTempDirectory = initTempDirectory;
    this.processWorkingDir = processWorkingDir;
    this.processConfig = processConfig;
    this.processEnv = processEnv;
    this.processArguments = processArguments;
    this.processOutput = processOutput;
    this.progressListener = progressListener;
    this.supportConfig = supportConfig;
    this.persistentBaseDir = persistentBaseDir;
    this.downloadCache = downloadCache;
    this.extractedFileSetStore = extractedFileSetStore;
    this.extractPackage = extractPackage;
    this.downloadPackage = downloadPackage;
    this.distribution = distribution;
    this.packageInformation = packageInformation;
    this.osList = osList;
    this.executer = executer;
    this.initShim = null;
  }

  private static final byte STAGE_INITIALIZING = -1;
  private static final byte STAGE_UNINITIALIZED = 0;
  private static final byte STAGE_INITIALIZED = 1;
  private transient volatile InitShim initShim = new InitShim();

  private final class InitShim {
    private byte initTempDirectoryBuildStage = STAGE_UNINITIALIZED;
    private Transition<TempDir> initTempDirectory;

    Transition<TempDir> initTempDirectory() {
      if (initTempDirectoryBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (initTempDirectoryBuildStage == STAGE_UNINITIALIZED) {
        initTempDirectoryBuildStage = STAGE_INITIALIZING;
        this.initTempDirectory = Objects.requireNonNull(ImmutableProcessFactory.super.initTempDirectory(), "initTempDirectory");
        initTempDirectoryBuildStage = STAGE_INITIALIZED;
      }
      return this.initTempDirectory;
    }

    void initTempDirectory(Transition<TempDir> initTempDirectory) {
      this.initTempDirectory = initTempDirectory;
      initTempDirectoryBuildStage = STAGE_INITIALIZED;
    }

    private byte processWorkingDirBuildStage = STAGE_UNINITIALIZED;
    private Transition<ProcessWorkingDir> processWorkingDir;

    Transition<ProcessWorkingDir> processWorkingDir() {
      if (processWorkingDirBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (processWorkingDirBuildStage == STAGE_UNINITIALIZED) {
        processWorkingDirBuildStage = STAGE_INITIALIZING;
        this.processWorkingDir = Objects.requireNonNull(ImmutableProcessFactory.super.processWorkingDir(), "processWorkingDir");
        processWorkingDirBuildStage = STAGE_INITIALIZED;
      }
      return this.processWorkingDir;
    }

    void processWorkingDir(Transition<ProcessWorkingDir> processWorkingDir) {
      this.processWorkingDir = processWorkingDir;
      processWorkingDirBuildStage = STAGE_INITIALIZED;
    }

    private byte processConfigBuildStage = STAGE_UNINITIALIZED;
    private Start<ProcessConfig> processConfig;

    Start<ProcessConfig> processConfig() {
      if (processConfigBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (processConfigBuildStage == STAGE_UNINITIALIZED) {
        processConfigBuildStage = STAGE_INITIALIZING;
        this.processConfig = Objects.requireNonNull(ImmutableProcessFactory.super.processConfig(), "processConfig");
        processConfigBuildStage = STAGE_INITIALIZED;
      }
      return this.processConfig;
    }

    void processConfig(Start<ProcessConfig> processConfig) {
      this.processConfig = processConfig;
      processConfigBuildStage = STAGE_INITIALIZED;
    }

    private byte processEnvBuildStage = STAGE_UNINITIALIZED;
    private Transition<ProcessEnv> processEnv;

    Transition<ProcessEnv> processEnv() {
      if (processEnvBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (processEnvBuildStage == STAGE_UNINITIALIZED) {
        processEnvBuildStage = STAGE_INITIALIZING;
        this.processEnv = Objects.requireNonNull(ImmutableProcessFactory.super.processEnv(), "processEnv");
        processEnvBuildStage = STAGE_INITIALIZED;
      }
      return this.processEnv;
    }

    void processEnv(Transition<ProcessEnv> processEnv) {
      this.processEnv = processEnv;
      processEnvBuildStage = STAGE_INITIALIZED;
    }

    private byte processOutputBuildStage = STAGE_UNINITIALIZED;
    private Transition<ProcessOutput> processOutput;

    Transition<ProcessOutput> processOutput() {
      if (processOutputBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (processOutputBuildStage == STAGE_UNINITIALIZED) {
        processOutputBuildStage = STAGE_INITIALIZING;
        this.processOutput = Objects.requireNonNull(ImmutableProcessFactory.super.processOutput(), "processOutput");
        processOutputBuildStage = STAGE_INITIALIZED;
      }
      return this.processOutput;
    }

    void processOutput(Transition<ProcessOutput> processOutput) {
      this.processOutput = processOutput;
      processOutputBuildStage = STAGE_INITIALIZED;
    }

    private byte progressListenerBuildStage = STAGE_UNINITIALIZED;
    private Transition<ProgressListener> progressListener;

    Transition<ProgressListener> progressListener() {
      if (progressListenerBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (progressListenerBuildStage == STAGE_UNINITIALIZED) {
        progressListenerBuildStage = STAGE_INITIALIZING;
        this.progressListener = Objects.requireNonNull(ImmutableProcessFactory.super.progressListener(), "progressListener");
        progressListenerBuildStage = STAGE_INITIALIZED;
      }
      return this.progressListener;
    }

    void progressListener(Transition<ProgressListener> progressListener) {
      this.progressListener = progressListener;
      progressListenerBuildStage = STAGE_INITIALIZED;
    }

    private byte supportConfigBuildStage = STAGE_UNINITIALIZED;
    private Transition<SupportConfig> supportConfig;

    Transition<SupportConfig> supportConfig() {
      if (supportConfigBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (supportConfigBuildStage == STAGE_UNINITIALIZED) {
        supportConfigBuildStage = STAGE_INITIALIZING;
        this.supportConfig = Objects.requireNonNull(ImmutableProcessFactory.super.supportConfig(), "supportConfig");
        supportConfigBuildStage = STAGE_INITIALIZED;
      }
      return this.supportConfig;
    }

    void supportConfig(Transition<SupportConfig> supportConfig) {
      this.supportConfig = supportConfig;
      supportConfigBuildStage = STAGE_INITIALIZED;
    }

    private byte downloadCacheBuildStage = STAGE_UNINITIALIZED;
    private Transition<DownloadCache> downloadCache;

    Transition<DownloadCache> downloadCache() {
      if (downloadCacheBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (downloadCacheBuildStage == STAGE_UNINITIALIZED) {
        downloadCacheBuildStage = STAGE_INITIALIZING;
        this.downloadCache = Objects.requireNonNull(ImmutableProcessFactory.super.downloadCache(), "downloadCache");
        downloadCacheBuildStage = STAGE_INITIALIZED;
      }
      return this.downloadCache;
    }

    void downloadCache(Transition<DownloadCache> downloadCache) {
      this.downloadCache = downloadCache;
      downloadCacheBuildStage = STAGE_INITIALIZED;
    }

    private byte extractedFileSetStoreBuildStage = STAGE_UNINITIALIZED;
    private Transition<ExtractedFileSetStore> extractedFileSetStore;

    Transition<ExtractedFileSetStore> extractedFileSetStore() {
      if (extractedFileSetStoreBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (extractedFileSetStoreBuildStage == STAGE_UNINITIALIZED) {
        extractedFileSetStoreBuildStage = STAGE_INITIALIZING;
        this.extractedFileSetStore = Objects.requireNonNull(ImmutableProcessFactory.super.extractedFileSetStore(), "extractedFileSetStore");
        extractedFileSetStoreBuildStage = STAGE_INITIALIZED;
      }
      return this.extractedFileSetStore;
    }

    void extractedFileSetStore(Transition<ExtractedFileSetStore> extractedFileSetStore) {
      this.extractedFileSetStore = extractedFileSetStore;
      extractedFileSetStoreBuildStage = STAGE_INITIALIZED;
    }

    private byte extractPackageBuildStage = STAGE_UNINITIALIZED;
    private Transition<ExtractedFileSet> extractPackage;

    Transition<ExtractedFileSet> extractPackage() {
      if (extractPackageBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (extractPackageBuildStage == STAGE_UNINITIALIZED) {
        extractPackageBuildStage = STAGE_INITIALIZING;
        this.extractPackage = Objects.requireNonNull(ImmutableProcessFactory.super.extractPackage(), "extractPackage");
        extractPackageBuildStage = STAGE_INITIALIZED;
      }
      return this.extractPackage;
    }

    void extractPackage(Transition<ExtractedFileSet> extractPackage) {
      this.extractPackage = extractPackage;
      extractPackageBuildStage = STAGE_INITIALIZED;
    }

    private byte downloadPackageBuildStage = STAGE_UNINITIALIZED;
    private Transition<Archive> downloadPackage;

    Transition<Archive> downloadPackage() {
      if (downloadPackageBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (downloadPackageBuildStage == STAGE_UNINITIALIZED) {
        downloadPackageBuildStage = STAGE_INITIALIZING;
        this.downloadPackage = Objects.requireNonNull(ImmutableProcessFactory.super.downloadPackage(), "downloadPackage");
        downloadPackageBuildStage = STAGE_INITIALIZED;
      }
      return this.downloadPackage;
    }

    void downloadPackage(Transition<Archive> downloadPackage) {
      this.downloadPackage = downloadPackage;
      downloadPackageBuildStage = STAGE_INITIALIZED;
    }

    private byte distributionBuildStage = STAGE_UNINITIALIZED;
    private Transition<Distribution> distribution;

    Transition<Distribution> distribution() {
      if (distributionBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (distributionBuildStage == STAGE_UNINITIALIZED) {
        distributionBuildStage = STAGE_INITIALIZING;
        this.distribution = Objects.requireNonNull(ImmutableProcessFactory.super.distribution(), "distribution");
        distributionBuildStage = STAGE_INITIALIZED;
      }
      return this.distribution;
    }

    void distribution(Transition<Distribution> distribution) {
      this.distribution = distribution;
      distributionBuildStage = STAGE_INITIALIZED;
    }

    private byte executerBuildStage = STAGE_UNINITIALIZED;
    private Transition<ExecutedProcess> executer;

    Transition<ExecutedProcess> executer() {
      if (executerBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (executerBuildStage == STAGE_UNINITIALIZED) {
        executerBuildStage = STAGE_INITIALIZING;
        this.executer = Objects.requireNonNull(ImmutableProcessFactory.super.executer(), "executer");
        executerBuildStage = STAGE_INITIALIZED;
      }
      return this.executer;
    }

    void executer(Transition<ExecutedProcess> executer) {
      this.executer = executer;
      executerBuildStage = STAGE_INITIALIZED;
    }

    private String formatInitCycleMessage() {
      List<String> attributes = new ArrayList<>();
      if (initTempDirectoryBuildStage == STAGE_INITIALIZING) attributes.add("initTempDirectory");
      if (processWorkingDirBuildStage == STAGE_INITIALIZING) attributes.add("processWorkingDir");
      if (processConfigBuildStage == STAGE_INITIALIZING) attributes.add("processConfig");
      if (processEnvBuildStage == STAGE_INITIALIZING) attributes.add("processEnv");
      if (processOutputBuildStage == STAGE_INITIALIZING) attributes.add("processOutput");
      if (progressListenerBuildStage == STAGE_INITIALIZING) attributes.add("progressListener");
      if (supportConfigBuildStage == STAGE_INITIALIZING) attributes.add("supportConfig");
      if (downloadCacheBuildStage == STAGE_INITIALIZING) attributes.add("downloadCache");
      if (extractedFileSetStoreBuildStage == STAGE_INITIALIZING) attributes.add("extractedFileSetStore");
      if (extractPackageBuildStage == STAGE_INITIALIZING) attributes.add("extractPackage");
      if (downloadPackageBuildStage == STAGE_INITIALIZING) attributes.add("downloadPackage");
      if (distributionBuildStage == STAGE_INITIALIZING) attributes.add("distribution");
      if (executerBuildStage == STAGE_INITIALIZING) attributes.add("executer");
      return "Cannot build ProcessFactory, attribute initializers form cycle " + attributes;
    }
  }

  /**
   * @return The value of the {@code version} attribute
   */
  @Override
  public Version version() {
    return version;
  }

  /**
   * @return The value of the {@code name} attribute
   */
  @Override
  protected Transition<Name> name() {
    return name;
  }

  /**
   * @return The value of the {@code initTempDirectory} attribute
   */
  @Override
  protected Transition<TempDir> initTempDirectory() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.initTempDirectory()
        : this.initTempDirectory;
  }

  /**
   * @return The value of the {@code processWorkingDir} attribute
   */
  @Override
  protected Transition<ProcessWorkingDir> processWorkingDir() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.processWorkingDir()
        : this.processWorkingDir;
  }

  /**
   * @return The value of the {@code processConfig} attribute
   */
  @Override
  protected Start<ProcessConfig> processConfig() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.processConfig()
        : this.processConfig;
  }

  /**
   * @return The value of the {@code processEnv} attribute
   */
  @Override
  protected Transition<ProcessEnv> processEnv() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.processEnv()
        : this.processEnv;
  }

  /**
   * @return The value of the {@code processArguments} attribute
   */
  @Override
  protected Transition<ProcessArguments> processArguments() {
    return processArguments;
  }

  /**
   * @return The value of the {@code processOutput} attribute
   */
  @Override
  protected Transition<ProcessOutput> processOutput() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.processOutput()
        : this.processOutput;
  }

  /**
   * @return The value of the {@code progressListener} attribute
   */
  @Override
  protected Transition<ProgressListener> progressListener() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.progressListener()
        : this.progressListener;
  }

  /**
   * @return The value of the {@code supportConfig} attribute
   */
  @Override
  protected Transition<SupportConfig> supportConfig() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.supportConfig()
        : this.supportConfig;
  }

  /**
   * @return The value of the {@code persistentBaseDir} attribute
   */
  @Override
  protected Transition<PersistentDir> persistentBaseDir() {
    return persistentBaseDir;
  }

  /**
   * @return The value of the {@code downloadCache} attribute
   */
  @Override
  protected Transition<DownloadCache> downloadCache() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.downloadCache()
        : this.downloadCache;
  }

  /**
   * @return The value of the {@code extractedFileSetStore} attribute
   */
  @Override
  protected Transition<ExtractedFileSetStore> extractedFileSetStore() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.extractedFileSetStore()
        : this.extractedFileSetStore;
  }

  /**
   * @return The value of the {@code extractPackage} attribute
   */
  @Override
  protected Transition<ExtractedFileSet> extractPackage() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.extractPackage()
        : this.extractPackage;
  }

  /**
   * @return The value of the {@code downloadPackage} attribute
   */
  @Override
  protected Transition<Archive> downloadPackage() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.downloadPackage()
        : this.downloadPackage;
  }

  /**
   * @return The value of the {@code distribution} attribute
   */
  @Override
  protected Transition<Distribution> distribution() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.distribution()
        : this.distribution;
  }

  /**
   * @return The value of the {@code packageInformation} attribute
   */
  @Override
  protected Function<Distribution, Package> packageInformation() {
    return packageInformation;
  }

  /**
   * @return The value of the {@code osList} attribute
   */
  @Override
  protected Supplier<Collection<? extends OS>> osList() {
    return osList;
  }

  /**
   * @return The value of the {@code executer} attribute
   */
  @Override
  protected Transition<ExecutedProcess> executer() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.executer()
        : this.executer;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProcessFactory#version() version} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for version
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProcessFactory withVersion(Version value) {
    if (this.version == value) return this;
    Version newValue = Objects.requireNonNull(value, "version");
    return new ImmutableProcessFactory(
        newValue,
        this.name,
        this.initTempDirectory,
        this.processWorkingDir,
        this.processConfig,
        this.processEnv,
        this.processArguments,
        this.processOutput,
        this.progressListener,
        this.supportConfig,
        this.persistentBaseDir,
        this.downloadCache,
        this.extractedFileSetStore,
        this.extractPackage,
        this.downloadPackage,
        this.distribution,
        this.packageInformation,
        this.osList,
        this.executer);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProcessFactory#name() name} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for name
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProcessFactory withName(Transition<Name> value) {
    if (this.name == value) return this;
    Transition<Name> newValue = Objects.requireNonNull(value, "name");
    return new ImmutableProcessFactory(
        this.version,
        newValue,
        this.initTempDirectory,
        this.processWorkingDir,
        this.processConfig,
        this.processEnv,
        this.processArguments,
        this.processOutput,
        this.progressListener,
        this.supportConfig,
        this.persistentBaseDir,
        this.downloadCache,
        this.extractedFileSetStore,
        this.extractPackage,
        this.downloadPackage,
        this.distribution,
        this.packageInformation,
        this.osList,
        this.executer);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProcessFactory#initTempDirectory() initTempDirectory} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for initTempDirectory
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProcessFactory withInitTempDirectory(Transition<TempDir> value) {
    if (this.initTempDirectory == value) return this;
    Transition<TempDir> newValue = Objects.requireNonNull(value, "initTempDirectory");
    return new ImmutableProcessFactory(
        this.version,
        this.name,
        newValue,
        this.processWorkingDir,
        this.processConfig,
        this.processEnv,
        this.processArguments,
        this.processOutput,
        this.progressListener,
        this.supportConfig,
        this.persistentBaseDir,
        this.downloadCache,
        this.extractedFileSetStore,
        this.extractPackage,
        this.downloadPackage,
        this.distribution,
        this.packageInformation,
        this.osList,
        this.executer);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProcessFactory#processWorkingDir() processWorkingDir} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for processWorkingDir
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProcessFactory withProcessWorkingDir(Transition<ProcessWorkingDir> value) {
    if (this.processWorkingDir == value) return this;
    Transition<ProcessWorkingDir> newValue = Objects.requireNonNull(value, "processWorkingDir");
    return new ImmutableProcessFactory(
        this.version,
        this.name,
        this.initTempDirectory,
        newValue,
        this.processConfig,
        this.processEnv,
        this.processArguments,
        this.processOutput,
        this.progressListener,
        this.supportConfig,
        this.persistentBaseDir,
        this.downloadCache,
        this.extractedFileSetStore,
        this.extractPackage,
        this.downloadPackage,
        this.distribution,
        this.packageInformation,
        this.osList,
        this.executer);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProcessFactory#processConfig() processConfig} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for processConfig
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProcessFactory withProcessConfig(Start<ProcessConfig> value) {
    if (this.processConfig == value) return this;
    Start<ProcessConfig> newValue = Objects.requireNonNull(value, "processConfig");
    return new ImmutableProcessFactory(
        this.version,
        this.name,
        this.initTempDirectory,
        this.processWorkingDir,
        newValue,
        this.processEnv,
        this.processArguments,
        this.processOutput,
        this.progressListener,
        this.supportConfig,
        this.persistentBaseDir,
        this.downloadCache,
        this.extractedFileSetStore,
        this.extractPackage,
        this.downloadPackage,
        this.distribution,
        this.packageInformation,
        this.osList,
        this.executer);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProcessFactory#processEnv() processEnv} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for processEnv
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProcessFactory withProcessEnv(Transition<ProcessEnv> value) {
    if (this.processEnv == value) return this;
    Transition<ProcessEnv> newValue = Objects.requireNonNull(value, "processEnv");
    return new ImmutableProcessFactory(
        this.version,
        this.name,
        this.initTempDirectory,
        this.processWorkingDir,
        this.processConfig,
        newValue,
        this.processArguments,
        this.processOutput,
        this.progressListener,
        this.supportConfig,
        this.persistentBaseDir,
        this.downloadCache,
        this.extractedFileSetStore,
        this.extractPackage,
        this.downloadPackage,
        this.distribution,
        this.packageInformation,
        this.osList,
        this.executer);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProcessFactory#processArguments() processArguments} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for processArguments
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProcessFactory withProcessArguments(Transition<ProcessArguments> value) {
    if (this.processArguments == value) return this;
    Transition<ProcessArguments> newValue = Objects.requireNonNull(value, "processArguments");
    return new ImmutableProcessFactory(
        this.version,
        this.name,
        this.initTempDirectory,
        this.processWorkingDir,
        this.processConfig,
        this.processEnv,
        newValue,
        this.processOutput,
        this.progressListener,
        this.supportConfig,
        this.persistentBaseDir,
        this.downloadCache,
        this.extractedFileSetStore,
        this.extractPackage,
        this.downloadPackage,
        this.distribution,
        this.packageInformation,
        this.osList,
        this.executer);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProcessFactory#processOutput() processOutput} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for processOutput
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProcessFactory withProcessOutput(Transition<ProcessOutput> value) {
    if (this.processOutput == value) return this;
    Transition<ProcessOutput> newValue = Objects.requireNonNull(value, "processOutput");
    return new ImmutableProcessFactory(
        this.version,
        this.name,
        this.initTempDirectory,
        this.processWorkingDir,
        this.processConfig,
        this.processEnv,
        this.processArguments,
        newValue,
        this.progressListener,
        this.supportConfig,
        this.persistentBaseDir,
        this.downloadCache,
        this.extractedFileSetStore,
        this.extractPackage,
        this.downloadPackage,
        this.distribution,
        this.packageInformation,
        this.osList,
        this.executer);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProcessFactory#progressListener() progressListener} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for progressListener
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProcessFactory withProgressListener(Transition<ProgressListener> value) {
    if (this.progressListener == value) return this;
    Transition<ProgressListener> newValue = Objects.requireNonNull(value, "progressListener");
    return new ImmutableProcessFactory(
        this.version,
        this.name,
        this.initTempDirectory,
        this.processWorkingDir,
        this.processConfig,
        this.processEnv,
        this.processArguments,
        this.processOutput,
        newValue,
        this.supportConfig,
        this.persistentBaseDir,
        this.downloadCache,
        this.extractedFileSetStore,
        this.extractPackage,
        this.downloadPackage,
        this.distribution,
        this.packageInformation,
        this.osList,
        this.executer);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProcessFactory#supportConfig() supportConfig} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for supportConfig
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProcessFactory withSupportConfig(Transition<SupportConfig> value) {
    if (this.supportConfig == value) return this;
    Transition<SupportConfig> newValue = Objects.requireNonNull(value, "supportConfig");
    return new ImmutableProcessFactory(
        this.version,
        this.name,
        this.initTempDirectory,
        this.processWorkingDir,
        this.processConfig,
        this.processEnv,
        this.processArguments,
        this.processOutput,
        this.progressListener,
        newValue,
        this.persistentBaseDir,
        this.downloadCache,
        this.extractedFileSetStore,
        this.extractPackage,
        this.downloadPackage,
        this.distribution,
        this.packageInformation,
        this.osList,
        this.executer);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProcessFactory#persistentBaseDir() persistentBaseDir} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for persistentBaseDir
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProcessFactory withPersistentBaseDir(Transition<PersistentDir> value) {
    if (this.persistentBaseDir == value) return this;
    Transition<PersistentDir> newValue = Objects.requireNonNull(value, "persistentBaseDir");
    return new ImmutableProcessFactory(
        this.version,
        this.name,
        this.initTempDirectory,
        this.processWorkingDir,
        this.processConfig,
        this.processEnv,
        this.processArguments,
        this.processOutput,
        this.progressListener,
        this.supportConfig,
        newValue,
        this.downloadCache,
        this.extractedFileSetStore,
        this.extractPackage,
        this.downloadPackage,
        this.distribution,
        this.packageInformation,
        this.osList,
        this.executer);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProcessFactory#downloadCache() downloadCache} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for downloadCache
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProcessFactory withDownloadCache(Transition<DownloadCache> value) {
    if (this.downloadCache == value) return this;
    Transition<DownloadCache> newValue = Objects.requireNonNull(value, "downloadCache");
    return new ImmutableProcessFactory(
        this.version,
        this.name,
        this.initTempDirectory,
        this.processWorkingDir,
        this.processConfig,
        this.processEnv,
        this.processArguments,
        this.processOutput,
        this.progressListener,
        this.supportConfig,
        this.persistentBaseDir,
        newValue,
        this.extractedFileSetStore,
        this.extractPackage,
        this.downloadPackage,
        this.distribution,
        this.packageInformation,
        this.osList,
        this.executer);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProcessFactory#extractedFileSetStore() extractedFileSetStore} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for extractedFileSetStore
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProcessFactory withExtractedFileSetStore(Transition<ExtractedFileSetStore> value) {
    if (this.extractedFileSetStore == value) return this;
    Transition<ExtractedFileSetStore> newValue = Objects.requireNonNull(value, "extractedFileSetStore");
    return new ImmutableProcessFactory(
        this.version,
        this.name,
        this.initTempDirectory,
        this.processWorkingDir,
        this.processConfig,
        this.processEnv,
        this.processArguments,
        this.processOutput,
        this.progressListener,
        this.supportConfig,
        this.persistentBaseDir,
        this.downloadCache,
        newValue,
        this.extractPackage,
        this.downloadPackage,
        this.distribution,
        this.packageInformation,
        this.osList,
        this.executer);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProcessFactory#extractPackage() extractPackage} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for extractPackage
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProcessFactory withExtractPackage(Transition<ExtractedFileSet> value) {
    if (this.extractPackage == value) return this;
    Transition<ExtractedFileSet> newValue = Objects.requireNonNull(value, "extractPackage");
    return new ImmutableProcessFactory(
        this.version,
        this.name,
        this.initTempDirectory,
        this.processWorkingDir,
        this.processConfig,
        this.processEnv,
        this.processArguments,
        this.processOutput,
        this.progressListener,
        this.supportConfig,
        this.persistentBaseDir,
        this.downloadCache,
        this.extractedFileSetStore,
        newValue,
        this.downloadPackage,
        this.distribution,
        this.packageInformation,
        this.osList,
        this.executer);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProcessFactory#downloadPackage() downloadPackage} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for downloadPackage
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProcessFactory withDownloadPackage(Transition<Archive> value) {
    if (this.downloadPackage == value) return this;
    Transition<Archive> newValue = Objects.requireNonNull(value, "downloadPackage");
    return new ImmutableProcessFactory(
        this.version,
        this.name,
        this.initTempDirectory,
        this.processWorkingDir,
        this.processConfig,
        this.processEnv,
        this.processArguments,
        this.processOutput,
        this.progressListener,
        this.supportConfig,
        this.persistentBaseDir,
        this.downloadCache,
        this.extractedFileSetStore,
        this.extractPackage,
        newValue,
        this.distribution,
        this.packageInformation,
        this.osList,
        this.executer);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProcessFactory#distribution() distribution} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for distribution
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProcessFactory withDistribution(Transition<Distribution> value) {
    if (this.distribution == value) return this;
    Transition<Distribution> newValue = Objects.requireNonNull(value, "distribution");
    return new ImmutableProcessFactory(
        this.version,
        this.name,
        this.initTempDirectory,
        this.processWorkingDir,
        this.processConfig,
        this.processEnv,
        this.processArguments,
        this.processOutput,
        this.progressListener,
        this.supportConfig,
        this.persistentBaseDir,
        this.downloadCache,
        this.extractedFileSetStore,
        this.extractPackage,
        this.downloadPackage,
        newValue,
        this.packageInformation,
        this.osList,
        this.executer);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProcessFactory#packageInformation() packageInformation} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for packageInformation
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProcessFactory withPackageInformation(Function<Distribution, Package> value) {
    if (this.packageInformation == value) return this;
    Function<Distribution, Package> newValue = Objects.requireNonNull(value, "packageInformation");
    return new ImmutableProcessFactory(
        this.version,
        this.name,
        this.initTempDirectory,
        this.processWorkingDir,
        this.processConfig,
        this.processEnv,
        this.processArguments,
        this.processOutput,
        this.progressListener,
        this.supportConfig,
        this.persistentBaseDir,
        this.downloadCache,
        this.extractedFileSetStore,
        this.extractPackage,
        this.downloadPackage,
        this.distribution,
        newValue,
        this.osList,
        this.executer);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProcessFactory#osList() osList} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for osList
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProcessFactory withOsList(Supplier<Collection<? extends OS>> value) {
    if (this.osList == value) return this;
    Supplier<Collection<? extends OS>> newValue = Objects.requireNonNull(value, "osList");
    return new ImmutableProcessFactory(
        this.version,
        this.name,
        this.initTempDirectory,
        this.processWorkingDir,
        this.processConfig,
        this.processEnv,
        this.processArguments,
        this.processOutput,
        this.progressListener,
        this.supportConfig,
        this.persistentBaseDir,
        this.downloadCache,
        this.extractedFileSetStore,
        this.extractPackage,
        this.downloadPackage,
        this.distribution,
        this.packageInformation,
        newValue,
        this.executer);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProcessFactory#executer() executer} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for executer
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProcessFactory withExecuter(Transition<ExecutedProcess> value) {
    if (this.executer == value) return this;
    Transition<ExecutedProcess> newValue = Objects.requireNonNull(value, "executer");
    return new ImmutableProcessFactory(
        this.version,
        this.name,
        this.initTempDirectory,
        this.processWorkingDir,
        this.processConfig,
        this.processEnv,
        this.processArguments,
        this.processOutput,
        this.progressListener,
        this.supportConfig,
        this.persistentBaseDir,
        this.downloadCache,
        this.extractedFileSetStore,
        this.extractPackage,
        this.downloadPackage,
        this.distribution,
        this.packageInformation,
        this.osList,
        newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableProcessFactory} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableProcessFactory
        && equalTo(0, (ImmutableProcessFactory) another);
  }

  private boolean equalTo(int synthetic, ImmutableProcessFactory another) {
    return version.equals(another.version)
        && name.equals(another.name)
        && initTempDirectory.equals(another.initTempDirectory)
        && processWorkingDir.equals(another.processWorkingDir)
        && processConfig.equals(another.processConfig)
        && processEnv.equals(another.processEnv)
        && processArguments.equals(another.processArguments)
        && processOutput.equals(another.processOutput)
        && progressListener.equals(another.progressListener)
        && supportConfig.equals(another.supportConfig)
        && persistentBaseDir.equals(another.persistentBaseDir)
        && downloadCache.equals(another.downloadCache)
        && extractedFileSetStore.equals(another.extractedFileSetStore)
        && extractPackage.equals(another.extractPackage)
        && downloadPackage.equals(another.downloadPackage)
        && distribution.equals(another.distribution)
        && packageInformation.equals(another.packageInformation)
        && osList.equals(another.osList)
        && executer.equals(another.executer);
  }

  /**
   * Computes a hash code from attributes: {@code version}, {@code name}, {@code initTempDirectory}, {@code processWorkingDir}, {@code processConfig}, {@code processEnv}, {@code processArguments}, {@code processOutput}, {@code progressListener}, {@code supportConfig}, {@code persistentBaseDir}, {@code downloadCache}, {@code extractedFileSetStore}, {@code extractPackage}, {@code downloadPackage}, {@code distribution}, {@code packageInformation}, {@code osList}, {@code executer}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + version.hashCode();
    h += (h << 5) + name.hashCode();
    h += (h << 5) + initTempDirectory.hashCode();
    h += (h << 5) + processWorkingDir.hashCode();
    h += (h << 5) + processConfig.hashCode();
    h += (h << 5) + processEnv.hashCode();
    h += (h << 5) + processArguments.hashCode();
    h += (h << 5) + processOutput.hashCode();
    h += (h << 5) + progressListener.hashCode();
    h += (h << 5) + supportConfig.hashCode();
    h += (h << 5) + persistentBaseDir.hashCode();
    h += (h << 5) + downloadCache.hashCode();
    h += (h << 5) + extractedFileSetStore.hashCode();
    h += (h << 5) + extractPackage.hashCode();
    h += (h << 5) + downloadPackage.hashCode();
    h += (h << 5) + distribution.hashCode();
    h += (h << 5) + packageInformation.hashCode();
    h += (h << 5) + osList.hashCode();
    h += (h << 5) + executer.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code ProcessFactory} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "ProcessFactory{"
        + "version=" + version
        + ", name=" + name
        + ", initTempDirectory=" + initTempDirectory
        + ", processWorkingDir=" + processWorkingDir
        + ", processConfig=" + processConfig
        + ", processEnv=" + processEnv
        + ", processArguments=" + processArguments
        + ", processOutput=" + processOutput
        + ", progressListener=" + progressListener
        + ", supportConfig=" + supportConfig
        + ", persistentBaseDir=" + persistentBaseDir
        + ", downloadCache=" + downloadCache
        + ", extractedFileSetStore=" + extractedFileSetStore
        + ", extractPackage=" + extractPackage
        + ", downloadPackage=" + downloadPackage
        + ", distribution=" + distribution
        + ", packageInformation=" + packageInformation
        + ", osList=" + osList
        + ", executer=" + executer
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link ProcessFactory} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable ProcessFactory instance
   */
  public static ImmutableProcessFactory copyOf(ProcessFactory instance) {
    if (instance instanceof ImmutableProcessFactory) {
      return (ImmutableProcessFactory) instance;
    }
    return ImmutableProcessFactory.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableProcessFactory ImmutableProcessFactory}.
   * <pre>
   * ImmutableProcessFactory.builder()
   *    .version(de.flapdoodle.embed.process.distribution.Version) // required {@link ProcessFactory#version() version}
   *    .name(de.flapdoodle.reverse.Transition&amp;lt;Name&amp;gt;) // required {@link ProcessFactory#name() name}
   *    .initTempDirectory(de.flapdoodle.reverse.Transition&amp;lt;de.flapdoodle.embed.process.io.directories.TempDir&amp;gt;) // optional {@link ProcessFactory#initTempDirectory() initTempDirectory}
   *    .processWorkingDir(de.flapdoodle.reverse.Transition&amp;lt;ProcessWorkingDir&amp;gt;) // optional {@link ProcessFactory#processWorkingDir() processWorkingDir}
   *    .processConfig(de.flapdoodle.reverse.transitions.Start&amp;lt;de.flapdoodle.embed.process.types.ProcessConfig&amp;gt;) // optional {@link ProcessFactory#processConfig() processConfig}
   *    .processEnv(de.flapdoodle.reverse.Transition&amp;lt;ProcessEnv&amp;gt;) // optional {@link ProcessFactory#processEnv() processEnv}
   *    .processArguments(de.flapdoodle.reverse.Transition&amp;lt;ProcessArguments&amp;gt;) // required {@link ProcessFactory#processArguments() processArguments}
   *    .processOutput(de.flapdoodle.reverse.Transition&amp;lt;de.flapdoodle.embed.process.io.ProcessOutput&amp;gt;) // optional {@link ProcessFactory#processOutput() processOutput}
   *    .progressListener(de.flapdoodle.reverse.Transition&amp;lt;de.flapdoodle.embed.process.io.progress.ProgressListener&amp;gt;) // optional {@link ProcessFactory#progressListener() progressListener}
   *    .supportConfig(de.flapdoodle.reverse.Transition&amp;lt;de.flapdoodle.embed.process.config.SupportConfig&amp;gt;) // optional {@link ProcessFactory#supportConfig() supportConfig}
   *    .persistentBaseDir(de.flapdoodle.reverse.Transition&amp;lt;de.flapdoodle.embed.process.io.directories.PersistentDir&amp;gt;) // required {@link ProcessFactory#persistentBaseDir() persistentBaseDir}
   *    .downloadCache(de.flapdoodle.reverse.Transition&amp;lt;de.flapdoodle.embed.process.store.DownloadCache&amp;gt;) // optional {@link ProcessFactory#downloadCache() downloadCache}
   *    .extractedFileSetStore(de.flapdoodle.reverse.Transition&amp;lt;de.flapdoodle.embed.process.store.ExtractedFileSetStore&amp;gt;) // optional {@link ProcessFactory#extractedFileSetStore() extractedFileSetStore}
   *    .extractPackage(de.flapdoodle.reverse.Transition&amp;lt;de.flapdoodle.embed.process.archives.ExtractedFileSet&amp;gt;) // optional {@link ProcessFactory#extractPackage() extractPackage}
   *    .downloadPackage(de.flapdoodle.reverse.Transition&amp;lt;Archive&amp;gt;) // optional {@link ProcessFactory#downloadPackage() downloadPackage}
   *    .distribution(de.flapdoodle.reverse.Transition&amp;lt;de.flapdoodle.embed.process.distribution.Distribution&amp;gt;) // optional {@link ProcessFactory#distribution() distribution}
   *    .packageInformation(function.Function&amp;lt;de.flapdoodle.embed.process.distribution.Distribution, de.flapdoodle.embed.process.config.store.Package&amp;gt;) // required {@link ProcessFactory#packageInformation() packageInformation}
   *    .osList(function.Supplier&amp;lt;Collection&amp;lt;? extends de.flapdoodle.os.OS&amp;gt;&amp;gt;) // required {@link ProcessFactory#osList() osList}
   *    .executer(de.flapdoodle.reverse.Transition&amp;lt;de.flapdoodle.embed.process.types.ExecutedProcess&amp;gt;) // optional {@link ProcessFactory#executer() executer}
   *    .build();
   * </pre>
   * @return A new ImmutableProcessFactory builder
   */
  public static ImmutableProcessFactory.Builder builder() {
    return new ImmutableProcessFactory.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableProcessFactory ImmutableProcessFactory}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private static final long INIT_BIT_VERSION = 0x1L;
    private static final long INIT_BIT_NAME = 0x2L;
    private static final long INIT_BIT_PROCESS_ARGUMENTS = 0x4L;
    private static final long INIT_BIT_PERSISTENT_BASE_DIR = 0x8L;
    private static final long INIT_BIT_PACKAGE_INFORMATION = 0x10L;
    private static final long INIT_BIT_OS_LIST = 0x20L;
    private long initBits = 0x3fL;

    private Version version;
    private Transition<Name> name;
    private Transition<TempDir> initTempDirectory;
    private Transition<ProcessWorkingDir> processWorkingDir;
    private Start<ProcessConfig> processConfig;
    private Transition<ProcessEnv> processEnv;
    private Transition<ProcessArguments> processArguments;
    private Transition<ProcessOutput> processOutput;
    private Transition<ProgressListener> progressListener;
    private Transition<SupportConfig> supportConfig;
    private Transition<PersistentDir> persistentBaseDir;
    private Transition<DownloadCache> downloadCache;
    private Transition<ExtractedFileSetStore> extractedFileSetStore;
    private Transition<ExtractedFileSet> extractPackage;
    private Transition<Archive> downloadPackage;
    private Transition<Distribution> distribution;
    private Function<Distribution, Package> packageInformation;
    private Supplier<Collection<? extends OS>> osList;
    private Transition<ExecutedProcess> executer;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code ProcessFactory} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(ProcessFactory instance) {
      Objects.requireNonNull(instance, "instance");
      this.version(instance.version());
      this.name(instance.name());
      this.initTempDirectory(instance.initTempDirectory());
      this.processWorkingDir(instance.processWorkingDir());
      this.processConfig(instance.processConfig());
      this.processEnv(instance.processEnv());
      this.processArguments(instance.processArguments());
      this.processOutput(instance.processOutput());
      this.progressListener(instance.progressListener());
      this.supportConfig(instance.supportConfig());
      this.persistentBaseDir(instance.persistentBaseDir());
      this.downloadCache(instance.downloadCache());
      this.extractedFileSetStore(instance.extractedFileSetStore());
      this.extractPackage(instance.extractPackage());
      this.downloadPackage(instance.downloadPackage());
      this.distribution(instance.distribution());
      this.packageInformation(instance.packageInformation());
      this.osList(instance.osList());
      this.executer(instance.executer());
      return this;
    }

    /**
     * Initializes the value for the {@link ProcessFactory#version() version} attribute.
     * @param version The value for version 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder version(Version version) {
      this.version = Objects.requireNonNull(version, "version");
      initBits &= ~INIT_BIT_VERSION;
      return this;
    }

    /**
     * Initializes the value for the {@link ProcessFactory#name() name} attribute.
     * @param name The value for name 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder name(Transition<Name> name) {
      this.name = Objects.requireNonNull(name, "name");
      initBits &= ~INIT_BIT_NAME;
      return this;
    }

    /**
     * Initializes the value for the {@link ProcessFactory#initTempDirectory() initTempDirectory} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link ProcessFactory#initTempDirectory() initTempDirectory}.</em>
     * @param initTempDirectory The value for initTempDirectory 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder initTempDirectory(Transition<TempDir> initTempDirectory) {
      this.initTempDirectory = Objects.requireNonNull(initTempDirectory, "initTempDirectory");
      return this;
    }

    /**
     * Initializes the value for the {@link ProcessFactory#processWorkingDir() processWorkingDir} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link ProcessFactory#processWorkingDir() processWorkingDir}.</em>
     * @param processWorkingDir The value for processWorkingDir 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder processWorkingDir(Transition<ProcessWorkingDir> processWorkingDir) {
      this.processWorkingDir = Objects.requireNonNull(processWorkingDir, "processWorkingDir");
      return this;
    }

    /**
     * Initializes the value for the {@link ProcessFactory#processConfig() processConfig} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link ProcessFactory#processConfig() processConfig}.</em>
     * @param processConfig The value for processConfig 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder processConfig(Start<ProcessConfig> processConfig) {
      this.processConfig = Objects.requireNonNull(processConfig, "processConfig");
      return this;
    }

    /**
     * Initializes the value for the {@link ProcessFactory#processEnv() processEnv} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link ProcessFactory#processEnv() processEnv}.</em>
     * @param processEnv The value for processEnv 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder processEnv(Transition<ProcessEnv> processEnv) {
      this.processEnv = Objects.requireNonNull(processEnv, "processEnv");
      return this;
    }

    /**
     * Initializes the value for the {@link ProcessFactory#processArguments() processArguments} attribute.
     * @param processArguments The value for processArguments 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder processArguments(Transition<ProcessArguments> processArguments) {
      this.processArguments = Objects.requireNonNull(processArguments, "processArguments");
      initBits &= ~INIT_BIT_PROCESS_ARGUMENTS;
      return this;
    }

    /**
     * Initializes the value for the {@link ProcessFactory#processOutput() processOutput} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link ProcessFactory#processOutput() processOutput}.</em>
     * @param processOutput The value for processOutput 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder processOutput(Transition<ProcessOutput> processOutput) {
      this.processOutput = Objects.requireNonNull(processOutput, "processOutput");
      return this;
    }

    /**
     * Initializes the value for the {@link ProcessFactory#progressListener() progressListener} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link ProcessFactory#progressListener() progressListener}.</em>
     * @param progressListener The value for progressListener 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder progressListener(Transition<ProgressListener> progressListener) {
      this.progressListener = Objects.requireNonNull(progressListener, "progressListener");
      return this;
    }

    /**
     * Initializes the value for the {@link ProcessFactory#supportConfig() supportConfig} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link ProcessFactory#supportConfig() supportConfig}.</em>
     * @param supportConfig The value for supportConfig 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder supportConfig(Transition<SupportConfig> supportConfig) {
      this.supportConfig = Objects.requireNonNull(supportConfig, "supportConfig");
      return this;
    }

    /**
     * Initializes the value for the {@link ProcessFactory#persistentBaseDir() persistentBaseDir} attribute.
     * @param persistentBaseDir The value for persistentBaseDir 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder persistentBaseDir(Transition<PersistentDir> persistentBaseDir) {
      this.persistentBaseDir = Objects.requireNonNull(persistentBaseDir, "persistentBaseDir");
      initBits &= ~INIT_BIT_PERSISTENT_BASE_DIR;
      return this;
    }

    /**
     * Initializes the value for the {@link ProcessFactory#downloadCache() downloadCache} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link ProcessFactory#downloadCache() downloadCache}.</em>
     * @param downloadCache The value for downloadCache 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder downloadCache(Transition<DownloadCache> downloadCache) {
      this.downloadCache = Objects.requireNonNull(downloadCache, "downloadCache");
      return this;
    }

    /**
     * Initializes the value for the {@link ProcessFactory#extractedFileSetStore() extractedFileSetStore} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link ProcessFactory#extractedFileSetStore() extractedFileSetStore}.</em>
     * @param extractedFileSetStore The value for extractedFileSetStore 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder extractedFileSetStore(Transition<ExtractedFileSetStore> extractedFileSetStore) {
      this.extractedFileSetStore = Objects.requireNonNull(extractedFileSetStore, "extractedFileSetStore");
      return this;
    }

    /**
     * Initializes the value for the {@link ProcessFactory#extractPackage() extractPackage} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link ProcessFactory#extractPackage() extractPackage}.</em>
     * @param extractPackage The value for extractPackage 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder extractPackage(Transition<ExtractedFileSet> extractPackage) {
      this.extractPackage = Objects.requireNonNull(extractPackage, "extractPackage");
      return this;
    }

    /**
     * Initializes the value for the {@link ProcessFactory#downloadPackage() downloadPackage} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link ProcessFactory#downloadPackage() downloadPackage}.</em>
     * @param downloadPackage The value for downloadPackage 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder downloadPackage(Transition<Archive> downloadPackage) {
      this.downloadPackage = Objects.requireNonNull(downloadPackage, "downloadPackage");
      return this;
    }

    /**
     * Initializes the value for the {@link ProcessFactory#distribution() distribution} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link ProcessFactory#distribution() distribution}.</em>
     * @param distribution The value for distribution 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder distribution(Transition<Distribution> distribution) {
      this.distribution = Objects.requireNonNull(distribution, "distribution");
      return this;
    }

    /**
     * Initializes the value for the {@link ProcessFactory#packageInformation() packageInformation} attribute.
     * @param packageInformation The value for packageInformation 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder packageInformation(Function<Distribution, Package> packageInformation) {
      this.packageInformation = Objects.requireNonNull(packageInformation, "packageInformation");
      initBits &= ~INIT_BIT_PACKAGE_INFORMATION;
      return this;
    }

    /**
     * Initializes the value for the {@link ProcessFactory#osList() osList} attribute.
     * @param osList The value for osList 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder osList(Supplier<Collection<? extends OS>> osList) {
      this.osList = Objects.requireNonNull(osList, "osList");
      initBits &= ~INIT_BIT_OS_LIST;
      return this;
    }

    /**
     * Initializes the value for the {@link ProcessFactory#executer() executer} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link ProcessFactory#executer() executer}.</em>
     * @param executer The value for executer 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder executer(Transition<ExecutedProcess> executer) {
      this.executer = Objects.requireNonNull(executer, "executer");
      return this;
    }

    /**
     * Builds a new {@link ImmutableProcessFactory ImmutableProcessFactory}.
     * @return An immutable instance of ProcessFactory
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableProcessFactory build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableProcessFactory(this);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_VERSION) != 0) attributes.add("version");
      if ((initBits & INIT_BIT_NAME) != 0) attributes.add("name");
      if ((initBits & INIT_BIT_PROCESS_ARGUMENTS) != 0) attributes.add("processArguments");
      if ((initBits & INIT_BIT_PERSISTENT_BASE_DIR) != 0) attributes.add("persistentBaseDir");
      if ((initBits & INIT_BIT_PACKAGE_INFORMATION) != 0) attributes.add("packageInformation");
      if ((initBits & INIT_BIT_OS_LIST) != 0) attributes.add("osList");
      return "Cannot build ProcessFactory, some of required attributes are not set " + attributes;
    }
  }
}
