/*
 * Copyright © 2017 camunda services GmbH (info@camunda.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.camunda.client.api.command;

import io.camunda.client.CamundaClientBuilder;
import io.camunda.client.api.CamundaFuture;
import java.time.Duration;

public interface FinalCommandStep<T> {

  /**
   * Sets the request timeout for the command. The default request timeout can be configured using
   * {@link CamundaClientBuilder#defaultRequestTimeout(Duration)}.
   *
   * @param requestTimeout the request timeout
   * @return the configured command
   */
  FinalCommandStep<T> requestTimeout(Duration requestTimeout);

  /**
   * Sends the command to the Camunda gateway. This operation is asynchronous. In case of success,
   * the future returns the event that was generated by the Camunda gateway in response to the
   * command.
   *
   * <p>Call {@link CamundaFuture#join()} to wait until the response is available.
   *
   * <pre>
   * Future&#60;JobEvent&#62 future = command.send();
   * JobEvent event = future.join();
   * </pre>
   *
   * @return a future tracking state of success/failure of the command.
   */
  CamundaFuture<T> send();

  /**
   * Sends the command to the Camunda gateway and returns the response. This operation is
   * synchronous.
   *
   * <pre>
   * JobEvent event = command.execute();
   * </pre>
   *
   * @return the response of the command.
   */
  default T execute() {
    return send().join();
  }
}
