/*
 * Copyright Camunda Services GmbH and/or licensed to Camunda Services GmbH under
 * one or more contributor license agreements. Licensed under a proprietary license. See the
 * License.txt file for more information. You may not use this file except in compliance with the
 * proprietary license.
 */

package io.camunda.identity.sdk.impl;

import static io.camunda.identity.sdk.IdentityConfiguration.Type.KEYCLOAK;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import io.camunda.identity.sdk.IdentityConfiguration;
import io.camunda.identity.sdk.authentication.Authentication;
import io.camunda.identity.sdk.impl.rest.RestClient;
import io.camunda.identity.sdk.impl.rest.request.UsersRequest;
import io.camunda.identity.sdk.users.Users;
import io.camunda.identity.sdk.users.dto.User;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;
import org.apache.commons.collections4.ListUtils;
import org.apache.commons.lang3.Validate;

public class UsersImpl implements Users {
  public static final String USERS_PATH = "/api/users";
  private static final String IDENTITY_AUDIENCE = "camunda-identity-resource-server";
  private static final int BATCH_SIZE = 100;

  private final RestClient restClient;
  private final Authentication authentication;
  private final String accessToken;
  private final IdentityConfiguration identityConfiguration;

  public UsersImpl(
      final IdentityConfiguration identityConfiguration,
      final Authentication authentication
  ) {
    this(identityConfiguration, authentication, null);
  }

  private UsersImpl(
      final IdentityConfiguration identityConfiguration,
      final Authentication authentication, final String accessToken
  ) {
    this.identityConfiguration = identityConfiguration;
    this.restClient = new RestClient();
    this.authentication = authentication;
    this.accessToken = accessToken;
  }

  public Users withAccessToken(final String accessToken) {
    Validate.notNull(accessToken, "access token must not be null");
    return new UsersImpl(identityConfiguration, authentication, accessToken);
  }

  public List<User> search(
      final String search
  ) {
    return search(search, null, null);
  }

  public List<User> search(
      final String search,
      final Integer page
  ) {
    return search(search, page, null);
  }

  public List<User> search(
      final String search,
      final Integer page,
      final Integer resultSize
  ) {
    return restClient.request(
        new UsersRequest(
            identityConfiguration.getBaseUrl(),
            accessToken(),
            search,
            page,
            resultSize
        )
    );
  }

  @Override
  public List<User> get(final List<String> userIds) {
    return ListUtils.partition(userIds, BATCH_SIZE)
        .stream()
        .map(ids -> requestUsers(accessToken(), ids))
        .flatMap(Collection::stream)
        .collect(Collectors.toList());
  }

  @Override
  public boolean isAvailable() {
    return isNotBlank(identityConfiguration.getBaseUrl())
        && KEYCLOAK.equals(identityConfiguration.getType());
  }

  private List<User> requestUsers(final String accessToken, final List<String> userIds) {
    return restClient.request(
        new UsersRequest(
            identityConfiguration.getBaseUrl(),
            accessToken,
            userIds
        )
    );
  }

  private String accessToken() {
    if (accessToken != null) {
      return accessToken;
    }

    return authentication.requestToken(IDENTITY_AUDIENCE).getAccessToken();
  }
}
