/*
 * Copyright Camunda Services GmbH and/or licensed to Camunda Services GmbH under
 * one or more contributor license agreements. Licensed under a proprietary license. See the
 * License.txt file for more information. You may not use this file except in compliance with the
 * proprietary license.
 */
package io.camunda.identity.sdk.impl.microsoft;

import com.auth0.jwt.interfaces.DecodedJWT;
import io.camunda.identity.sdk.IdentityConfiguration;
import io.camunda.identity.sdk.authentication.AuthorizeUriBuilder;
import io.camunda.identity.sdk.authentication.Tokens;
import io.camunda.identity.sdk.authentication.UserDetails;
import io.camunda.identity.sdk.impl.dto.AccessTokenDto;
import io.camunda.identity.sdk.impl.generic.GenericAuthentication;
import io.camunda.identity.sdk.impl.rest.RestClient;
import io.camunda.identity.sdk.impl.rest.request.ClientTokenRequest;
import io.camunda.identity.sdk.utility.UrlQueryString;
import java.net.URI;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MicrosoftAuthentication extends GenericAuthentication {

  private static final Logger LOGGER = LoggerFactory.getLogger(MicrosoftAuthentication.class);

  private static final String SIGN_OUT_CALLBACK_PARAM_NAME = "post_logout_redirect_uri";

  public MicrosoftAuthentication(
      final IdentityConfiguration configuration,
      final RestClient restClient
  ) {
    super(configuration, restClient);
  }

  @Override
  public AuthorizeUriBuilder authorizeUriBuilder(final String redirectUri) {
    return new MicrosoftAuthorizeUriBuilder(
        configuration,
        wellKnownConfiguration().getAuthorizationEndpoint(),
        redirectUri
    );
  }

  @Override
  public void revokeToken(final String refreshToken) {
    LOGGER.info("Revoke token not required for Microsoft authentication. Skipping...");
  }

  @Override
  protected Tokens requestFreshToken(final String audience) {
    final ClientTokenRequest request = new ClientTokenRequest(
        configuration,
        wellKnownConfiguration().getTokenEndpoint(),
        audience,
        audience + "/.default"
    );
    final AccessTokenDto accessTokenDto = restClient.request(request);

    return fromAccessTokenDto(accessTokenDto);
  }

  @Override
  public boolean isM2MToken(final String token) {
    final DecodedJWT decodedJwt = decodeJWT(token);

    return decodedJwt.getClaim("email").isMissing();
  }

  @Override
  public String getClientId(final String token) {
    final DecodedJWT decodedJwt = decodeJWT(token);

    return decodedJwt.getClaim("appid").asString();
  }

  @Override
  protected UserDetails getUserDetails(final DecodedJWT token, final String organizationId) {
    return new UserDetails(
        token.getSubject(),
        token.getClaim("email").asString(),
        token.getClaim("unique_name").asString(),
        token.getClaim("name").asString(),
        getGroupsInOrganization(token, organizationId)
    );
  }

  @Override
  protected URI generateSingleSignOutUri(final String callbackUrl) {
    URI uri = super.generateSingleSignOutUri(callbackUrl);
    UrlQueryString query = new UrlQueryString().add(SIGN_OUT_CALLBACK_PARAM_NAME, callbackUrl);

    return URI.create(
        String.format("%s?%s", uri, query.toString())
    );
  }
}
