/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.client.impl.oauth;

import com.auth0.jwt.JWT;
import com.auth0.jwt.algorithms.Algorithm;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import io.camunda.zeebe.client.CredentialsProvider;
import io.camunda.zeebe.client.impl.ZeebeClientCredentials;
import io.camunda.zeebe.client.impl.oauth.OAuthCredentialsCache;
import io.camunda.zeebe.client.impl.oauth.OAuthCredentialsProviderBuilder;
import io.camunda.zeebe.client.impl.util.VersionUtil;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.Reader;
import java.io.UncheckedIOException;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.MessageDigest;
import java.security.cert.X509Certificate;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.time.Duration;
import java.util.Base64;
import java.util.Date;
import java.util.HashMap;
import java.util.UUID;
import java.util.stream.Collectors;
import net.jcip.annotations.ThreadSafe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public final class OAuthCredentialsProvider
implements CredentialsProvider {
    private static final String HEADER_AUTH_KEY = "Authorization";
    private static final String JWT_ASSERTION_TYPE = "urn:ietf:params:oauth:client-assertion-type:jwt-bearer";
    private static final ObjectMapper JSON_MAPPER = new ObjectMapper().configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    private static final ObjectReader CREDENTIALS_READER = JSON_MAPPER.readerFor(ZeebeClientCredentials.class);
    private static final Logger LOG = LoggerFactory.getLogger(OAuthCredentialsProvider.class);
    private final URL authorizationServerUrl;
    private final String clientId;
    private final String clientSecret;
    private final String audience;
    private final String scope;
    private final String resource;
    private final OAuthCredentialsCache credentialsCache;
    private final Duration connectionTimeout;
    private final Duration readTimeout;
    private final boolean clientAssertionEnabled;
    private final Path clientAssertionKeystorePath;
    private final String clientAssertionKeystorePassword;
    private final String clientAssertionKeystoreKeyAlias;
    private final String clientAssertionKeystoreKeyPassword;

    OAuthCredentialsProvider(OAuthCredentialsProviderBuilder builder) {
        this.authorizationServerUrl = builder.getAuthorizationServer();
        this.clientId = builder.getClientId();
        this.clientSecret = builder.getClientSecret();
        this.audience = builder.getAudience();
        this.scope = builder.getScope();
        this.resource = builder.getResource();
        this.credentialsCache = new OAuthCredentialsCache(builder.getCredentialsCache());
        this.connectionTimeout = builder.getConnectTimeout();
        this.readTimeout = builder.getReadTimeout();
        this.clientAssertionEnabled = builder.clientAssertionEnabled();
        this.clientAssertionKeystorePath = builder.getClientAssertionKeystorePath();
        this.clientAssertionKeystorePassword = builder.getClientAssertionKeystorePassword();
        this.clientAssertionKeystoreKeyAlias = builder.getClientAssertionKeystoreKeyAlias();
        this.clientAssertionKeystoreKeyPassword = builder.getClientAssertionKeystoreKeyPassword();
    }

    public boolean isClientAssertionEnabled() {
        return this.clientAssertionEnabled;
    }

    @Override
    public void applyCredentials(CredentialsProvider.CredentialsApplier applier) throws IOException {
        ZeebeClientCredentials zeebeClientCredentials = this.credentialsCache.computeIfMissingOrInvalid(this.clientId, this::fetchCredentials);
        String type = zeebeClientCredentials.getTokenType();
        if (type == null || type.isEmpty()) {
            throw new IOException(String.format("Expected valid token type but was absent or invalid '%s'", type));
        }
        type = Character.toUpperCase(type.charAt(0)) + type.substring(1);
        applier.put(HEADER_AUTH_KEY, String.format("%s %s", type, zeebeClientCredentials.getAccessToken()));
    }

    @Override
    public boolean shouldRetryRequest(CredentialsProvider.StatusCode statusCode) {
        try {
            return statusCode.isUnauthorized() && this.credentialsCache.withCache(this.clientId, value -> {
                ZeebeClientCredentials fetchedCredentials = this.fetchCredentials();
                this.credentialsCache.put(this.clientId, fetchedCredentials).writeCache();
                return !fetchedCredentials.equals(value) || !value.isValid();
            }).orElse(false) != false;
        }
        catch (IOException e) {
            LOG.error("Failed while fetching credentials: ", (Throwable)e);
            return false;
        }
    }

    private String createPayload() {
        HashMap<String, String> payload = new HashMap<String, String>();
        if (this.clientAssertionEnabled) {
            payload.put("client_assertion", this.getClientAssertion());
            payload.put("client_assertion_type", JWT_ASSERTION_TYPE);
        } else {
            payload.put("client_secret", this.clientSecret);
        }
        payload.put("client_id", this.clientId);
        payload.put("audience", this.audience);
        payload.put("grant_type", "client_credentials");
        if (this.scope != null) {
            payload.put("scope", this.scope);
        }
        if (this.resource != null) {
            payload.put("resource", this.resource);
        }
        return payload.entrySet().stream().map(e -> OAuthCredentialsProvider.encode((String)e.getKey()) + "=" + OAuthCredentialsProvider.encode((String)e.getValue())).collect(Collectors.joining("&"));
    }

    private String getClientAssertion() {
        Algorithm algorithm;
        X509Certificate certificate;
        try (FileInputStream stream = new FileInputStream(this.clientAssertionKeystorePath.toFile());){
            KeyStore keyStore = KeyStore.getInstance("PKCS12");
            keyStore.load(stream, this.clientAssertionKeystorePassword.toCharArray());
            RSAPrivateKey privateKey = (RSAPrivateKey)keyStore.getKey(this.clientAssertionKeystoreKeyAlias, this.clientAssertionKeystoreKeyPassword.toCharArray());
            X509Certificate keyStoreCertificate = (X509Certificate)keyStore.getCertificate(this.clientAssertionKeystoreKeyAlias);
            RSAPublicKey publicKey = (RSAPublicKey)keyStoreCertificate.getPublicKey();
            certificate = (X509Certificate)keyStore.getCertificate(this.clientAssertionKeystoreKeyAlias);
            algorithm = Algorithm.RSA256((RSAPublicKey)publicKey, (RSAPrivateKey)privateKey);
        }
        catch (IOException | GeneralSecurityException e) {
            throw new RuntimeException("Failed to create client assertion", e);
        }
        Date now = new Date();
        String x5t = OAuthCredentialsProvider.generateX5tThumbprint(certificate);
        HashMap<String, String> header = new HashMap<String, String>();
        header.put("alg", "RSA256");
        header.put("typ", "JWT");
        header.put("x5t", x5t);
        return JWT.create().withHeader(header).withIssuer(this.clientId).withSubject(this.clientId).withAudience(new String[]{this.authorizationServerUrl.toString()}).withIssuedAt(now).withNotBefore(now).withExpiresAt(new Date(now.getTime() + 300000L)).withJWTId(UUID.randomUUID().toString()).sign(algorithm);
    }

    private static String generateX5tThumbprint(X509Certificate certificate) {
        try {
            MessageDigest digest = MessageDigest.getInstance("SHA-1");
            byte[] encoded = digest.digest(certificate.getEncoded());
            return Base64.getUrlEncoder().withoutPadding().encodeToString(encoded);
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to generate x5t thumbprint", e);
        }
    }

    private static String encode(String param) {
        try {
            return URLEncoder.encode(param, StandardCharsets.UTF_8.name());
        }
        catch (UnsupportedEncodingException e) {
            throw new UncheckedIOException("Failed while encoding OAuth request parameters: ", e);
        }
    }

    private ZeebeClientCredentials fetchCredentials() throws IOException {
        HttpURLConnection connection = (HttpURLConnection)this.authorizationServerUrl.openConnection();
        connection.setRequestMethod("POST");
        connection.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");
        connection.setRequestProperty("Accept", "application/json");
        connection.setDoOutput(true);
        connection.setReadTimeout(Math.toIntExact(this.readTimeout.toMillis()));
        connection.setConnectTimeout(Math.toIntExact(this.connectionTimeout.toMillis()));
        connection.setRequestProperty("User-Agent", "zeebe-client-java/" + VersionUtil.getVersion());
        try (OutputStream os = connection.getOutputStream();){
            String payload = this.createPayload();
            byte[] input = payload.getBytes(StandardCharsets.UTF_8);
            os.write(input, 0, input.length);
        }
        if (connection.getResponseCode() != 200) {
            throw new IOException(String.format("Failed while requesting access token with status code %d and message %s.", connection.getResponseCode(), connection.getResponseMessage()));
        }
        try (InputStream in = connection.getInputStream();){
            ZeebeClientCredentials zeebeClientCredentials;
            try (InputStreamReader reader = new InputStreamReader(in, StandardCharsets.UTF_8);){
                ZeebeClientCredentials fetchedCredentials = (ZeebeClientCredentials)CREDENTIALS_READER.readValue((Reader)reader);
                if (fetchedCredentials == null) {
                    throw new IOException("Expected valid credentials but got null instead.");
                }
                zeebeClientCredentials = fetchedCredentials;
            }
            return zeebeClientCredentials;
        }
    }
}

