package com.querydsl.sql.types;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.sql.Types;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import org.jetbrains.annotations.Nullable;

/**
 * JSR310LocalDateTimeType maps {@linkplain java.time.LocalDateTime} to {@linkplain
 * java.sql.Timestamp} on the JDBC level
 */
public class JSR310LocalDateTimeType extends AbstractJSR310DateTimeType<LocalDateTime> {

  public JSR310LocalDateTimeType() {
    super(Types.TIMESTAMP);
  }

  public JSR310LocalDateTimeType(int type) {
    super(type);
  }

  @Override
  public String getLiteral(LocalDateTime value) {
    return dateTimeFormatter.format(value);
  }

  @Override
  public Class<LocalDateTime> getReturnedClass() {
    return LocalDateTime.class;
  }

  @Nullable
  @Override
  public LocalDateTime getValue(ResultSet rs, int startIndex) throws SQLException {
    Timestamp ts = rs.getTimestamp(startIndex, utc());
    return ts != null ? LocalDateTime.ofInstant(ts.toInstant(), ZoneOffset.UTC) : null;
  }

  @Override
  public void setValue(PreparedStatement st, int startIndex, LocalDateTime value)
      throws SQLException {
    st.setTimestamp(startIndex, Timestamp.from(value.toInstant(ZoneOffset.UTC)), utc());
  }
}
