
/*******************************************************************************

 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ******************************************************************************/
package org.apache.arrow.vector;

import org.apache.arrow.vector.schema.ArrowFieldNode;
import java.util.Collections;


import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;

import com.google.common.collect.Lists;
import com.google.common.collect.ObjectArrays;
import com.google.common.base.Charsets;
import com.google.common.collect.ObjectArrays;

import com.google.flatbuffers.FlatBufferBuilder;

import com.google.common.base.Preconditions;
import io.netty.buffer.*;

import org.apache.arrow.memory.*;
import org.apache.arrow.vector.types.Types;
import org.apache.arrow.vector.types.Types.*;
import org.apache.arrow.vector.types.pojo.*;
import org.apache.arrow.vector.types.pojo.ArrowType.*;
import org.apache.arrow.vector.types.*;
import org.apache.arrow.vector.*;
import org.apache.arrow.vector.holders.*;
import org.apache.arrow.vector.util.*;
import org.apache.arrow.vector.complex.*;
import org.apache.arrow.vector.complex.reader.*;
import org.apache.arrow.vector.complex.impl.*;
import org.apache.arrow.vector.complex.writer.*;
import org.apache.arrow.vector.complex.writer.BaseWriter.MapWriter;
import org.apache.arrow.vector.complex.writer.BaseWriter.ListWriter;
import org.apache.arrow.vector.util.JsonStringArrayList;

import java.util.Arrays;
import java.util.Random;
import java.util.List;

import java.io.Closeable;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.ByteBuffer;

import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.math.BigDecimal;
import java.math.BigInteger;

import org.joda.time.DateTime;
import org.joda.time.LocalDateTime;
import org.joda.time.Period;







import org.apache.arrow.flatbuf.Precision;

/**
 * NullableUInt8Vector implements a vector of values which could be null.  Elements in the vector
 * are first checked against a fixed length vector of boolean values.  Then the element is retrieved
 * from the base class (if not null).
 *
 * NB: this class is automatically generated from NullableValueVectors.java and ValueVectorTypes.tdd using FreeMarker.
 */
@SuppressWarnings("unused")
public final class NullableUInt8Vector extends BaseValueVector implements FixedWidthVector, NullableVector, FieldVector {
  private static final org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(NullableUInt8Vector.class);

protected final static byte[] emptyByteArray = new byte[]{};
  private final FieldReader reader = new UInt8ReaderImpl(NullableUInt8Vector.this);

  private final String bitsField = "$bits$";
  private final String valuesField = "$values$";
  private final Field field;

  final BitVector bits = new BitVector(bitsField, allocator);
  final UInt8Vector values;

  private final Mutator mutator;
  private final Accessor accessor;

  private final List<BufferBacked> innerVectors;

  public NullableUInt8Vector(String name, BufferAllocator allocator) {
    this(name, FieldType.nullable(org.apache.arrow.vector.types.Types.MinorType.UINT8.getType()), allocator);
  }

  public NullableUInt8Vector(String name, FieldType fieldType, BufferAllocator allocator) {
    super(name, allocator);
    this.values = new UInt8Vector(valuesField, allocator);
    this.mutator = new Mutator();
    this.accessor = new Accessor();
    this.field = new Field(name, fieldType, null);
    innerVectors = Collections.unmodifiableList(Arrays.<BufferBacked>asList(
        bits,
        values
    ));
  }

  @Override
  public BitVector getValidityVector() {
    return bits;
  }

  @Override
  public List<BufferBacked> getFieldInnerVectors() {
    return innerVectors;
  }

  @Override
  public void initializeChildrenFromFields(List<Field> children) {
    if (!children.isEmpty()) {
      throw new IllegalArgumentException("primitive type vector NullableUInt8Vector can not have children: " + children);
    }
  }

  @Override
  public List<FieldVector> getChildrenFromFields() {
    return Collections.emptyList();
  }

  @Override
  public void loadFieldBuffers(ArrowFieldNode fieldNode, List<ArrowBuf> ownBuffers) {
    // fixed width values truncate value vector to size (#1)
    org.apache.arrow.vector.BaseDataValueVector.truncateBufferBasedOnSize(ownBuffers, 1, values.getBufferSizeFor(fieldNode.getLength()));
    org.apache.arrow.vector.BaseDataValueVector.load(fieldNode, getFieldInnerVectors(), ownBuffers);
    bits.valueCount = fieldNode.getLength();
  }

  public List<ArrowBuf> getFieldBuffers() {
    return org.apache.arrow.vector.BaseDataValueVector.unload(getFieldInnerVectors());
  }

  @Override
  public Field getField() {
    return field;
  }

  @Override
  public MinorType getMinorType() {
    return MinorType.UINT8;
  }

  @Override
  public FieldReader getReader(){
    return reader;
  }

  @Override
  public int getValueCapacity(){
    return Math.min(bits.getValueCapacity(), values.getValueCapacity());
  }

  @Override
  public ArrowBuf[] getBuffers(boolean clear) {
    final ArrowBuf[] buffers = ObjectArrays.concat(bits.getBuffers(false), values.getBuffers(false), ArrowBuf.class);
    if (clear) {
      for (final ArrowBuf buffer:buffers) {
        buffer.retain(1);
      }
      clear();
    }
    return buffers;
  }

  @Override
  public void close() {
    bits.close();
    values.close();
    super.close();
  }

  @Override
  public void clear() {
    bits.clear();
    values.clear();
    super.clear();
  }

  @Override
  public int getBufferSize(){
    return values.getBufferSize() + bits.getBufferSize();
  }

  @Override
  public int getBufferSizeFor(final int valueCount) {
    if (valueCount == 0) {
      return 0;
    }

    return values.getBufferSizeFor(valueCount)
        + bits.getBufferSizeFor(valueCount);
  }

  public ArrowBuf getBuffer() {
    return values.getDataBuffer();
  }

  @Override
  public UInt8Vector getValuesVector() {
    return values;
  }

  @Override
  public void setInitialCapacity(int numRecords) {
    bits.setInitialCapacity(numRecords);
    values.setInitialCapacity(numRecords);
  }

  @Override
  public void allocateNew() {
    if(!allocateNewSafe()){
      throw new OutOfMemoryException("Failure while allocating buffer.");
    }
  }

  @Override
  public boolean allocateNewSafe() {
    /* Boolean to keep track if all the memory allocations were successful
     * Used in the case of composite vectors when we need to allocate multiple
     * buffers for multiple vectors. If one of the allocations failed we need to
     * clear all the memory that we allocated
     */
    boolean success = false;
    try {
      success = values.allocateNewSafe() && bits.allocateNewSafe();
    } finally {
      if (!success) {
        clear();
      }
    }
    bits.zeroVector();
    mutator.reset();
    accessor.reset();
    return success;
  }

  @Override
  public void reAlloc() {
    bits.reAlloc();
    values.reAlloc();
  }

  public void reset() {
    bits.zeroVector();
    mutator.reset();
    accessor.reset();
  }

  @Override
  public void allocateNew(int valueCount) {
    try {
      values.allocateNew(valueCount);
      bits.allocateNew(valueCount);
    } catch(OutOfMemoryException e) {
      clear();
      throw e;
    }
    bits.zeroVector();
    mutator.reset();
    accessor.reset();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void zeroVector() {
    bits.zeroVector();
    values.zeroVector();
  }

  @Override
  public TransferPair getTransferPair(String ref, BufferAllocator allocator, CallBack callBack) {
        return getTransferPair(ref, allocator);
  }

  @Override
  public TransferPair getTransferPair(BufferAllocator allocator){
    return new TransferImpl(name, allocator);

  }

  @Override
  public TransferPair getTransferPair(String ref, BufferAllocator allocator){
    return new TransferImpl(ref, allocator);
  }

  @Override
  public TransferPair makeTransferPair(ValueVector to) {
    return new TransferImpl((NullableUInt8Vector) to);
  }

  public void transferTo(NullableUInt8Vector target){
    bits.transferTo(target.bits);
    values.transferTo(target.values);
    clear();
  }

  public void splitAndTransferTo(int startIndex, int length, NullableUInt8Vector target) {
    bits.splitAndTransferTo(startIndex, length, target.bits);
    values.splitAndTransferTo(startIndex, length, target.values);
  }

  private class TransferImpl implements TransferPair {
    NullableUInt8Vector to;

    public TransferImpl(String ref, BufferAllocator allocator){
      to = new NullableUInt8Vector(ref, field.getFieldType(), allocator);
    }

    public TransferImpl(NullableUInt8Vector to){
      this.to = to;
    }

    @Override
    public NullableUInt8Vector getTo(){
      return to;
    }

    @Override
    public void transfer(){
      transferTo(to);
    }

    @Override
    public void splitAndTransfer(int startIndex, int length) {
      splitAndTransferTo(startIndex, length, to);
    }

    @Override
    public void copyValueSafe(int fromIndex, int toIndex) {
      to.copyFromSafe(fromIndex, toIndex, NullableUInt8Vector.this);
    }
  }

  @Override
  public Accessor getAccessor(){
    return accessor;
  }

  @Override
  public Mutator getMutator(){
    return mutator;
  }

  public void copyFrom(int fromIndex, int thisIndex, NullableUInt8Vector from){
    final Accessor fromAccessor = from.getAccessor();
    if (!fromAccessor.isNull(fromIndex)) {
      mutator.set(thisIndex, fromAccessor.get(fromIndex));
    }
    
  }

  public void copyFromSafe(int fromIndex, int thisIndex, UInt8Vector from){
    values.copyFromSafe(fromIndex, thisIndex, from);
    bits.getMutator().setSafeToOne(thisIndex);
    
  }

  public void copyFromSafe(int fromIndex, int thisIndex, NullableUInt8Vector from){
    bits.copyFromSafe(fromIndex, thisIndex, from.bits);
    values.copyFromSafe(fromIndex, thisIndex, from.values);
    
  }

  @Override
  public long getValidityBufferAddress() {
    /* address of the databuffer associated with the bitVector */
    return (bits.getDataBuffer().memoryAddress());
  }

  @Override
  public long getDataBufferAddress() {
    /* address of the dataBuffer associated with the valueVector */
    return (values.getDataBuffer().memoryAddress());
  }

  @Override
  public long getOffsetBufferAddress() {
    /* address of the dataBuffer associated with the offsetVector
     * this operation is not supported for fixed-width vector types.
     */
        throw new UnsupportedOperationException();
  }

  @Override
  public ArrowBuf getValidityBuffer() {
    /* dataBuffer associated with the bitVector */
    return (bits.getDataBuffer());
  }

  @Override
  public ArrowBuf getDataBuffer() {
    /* dataBuffer associated with the valueVector */
    return (values.getDataBuffer());
  }

  @Override
  public ArrowBuf getOffsetBuffer() {
    /* dataBuffer associated with the offsetVector of the valueVector */
        throw new UnsupportedOperationException();
  }

  public final class Accessor extends BaseDataValueVector.BaseAccessor  {
    final BitVector.Accessor bAccessor = bits.getAccessor();
    final UInt8Vector.Accessor vAccessor = values.getAccessor();

    /**
     * Get the element at the specified position.
     *
     * @param  index   position of the value
     * @return value of the element, if not null
     */
    public long get(int index) {
      if (isNull(index)) {
          throw new IllegalStateException("Can't get a null value");
      }
      return vAccessor.get(index);
    }

    @Override
    public boolean isNull(int index) {
      return isSet(index) == 0;
    }

    public int isSet(int index){
      return bAccessor.get(index);
    }


    public void get(int index, NullableUInt8Holder holder){
      vAccessor.get(index, holder);
      holder.isSet = bAccessor.get(index);
    }

    @Override
    public Long getObject(int index) {
      if (isNull(index)) {
          return null;
      }else{
        return vAccessor.getObject(index);
      }
    }


    @Override
    public int getValueCount(){
      return bits.getAccessor().getValueCount();
    }

    public void reset(){}
  }

  public final class Mutator extends BaseDataValueVector.BaseMutator implements NullableVectorDefinitionSetter {
    private int setCount;
    

    private Mutator(){
    }

    public UInt8Vector getVectorWithValues(){
      return values;
    }

    @Override
    public void setIndexDefined(int index){
      bits.getMutator().setToOne(index);
    }

    /**
     * Set the variable length element at the specified index to the supplied byte array.
     *
     * @param index   position of the bit to set
     * @param value   array of bytes (or int if smaller than 4 bytes) to write
     */
    public void set(int index, long value) {
      setCount++;
      final UInt8Vector.Mutator valuesMutator = values.getMutator();
      final BitVector.Mutator bitsMutator = bits.getMutator();
      bitsMutator.setToOne(index);
      valuesMutator.set(index, value);
      
    }


    public void setSafe(int index, byte[] value, int start, int length) {
      throw new UnsupportedOperationException();
    }

    public void setSafe(int index, ByteBuffer value, int start, int length) {
      throw new UnsupportedOperationException();
    }

    public void setNull(int index){
      bits.getMutator().setSafe(index, 0);
    }

    public void setSkipNull(int index, UInt8Holder holder){
      values.getMutator().set(index, holder);
    }

    public void setSkipNull(int index, NullableUInt8Holder holder){
      values.getMutator().set(index, holder);
    }

    public void set(int index, NullableUInt8Holder holder){
      final UInt8Vector.Mutator valuesMutator = values.getMutator();
      bits.getMutator().set(index, holder.isSet);
      valuesMutator.set(index, holder);
      
    }

    public void set(int index, UInt8Holder holder){
      final UInt8Vector.Mutator valuesMutator = values.getMutator();
      bits.getMutator().setToOne(index);
      valuesMutator.set(index, holder);
      
    }

    public boolean isSafe(int outIndex) {
      return outIndex < NullableUInt8Vector.this.getValueCapacity();
    }

    public void set(int index, int isSet, long valueField ){
      final UInt8Vector.Mutator valuesMutator = values.getMutator();
      bits.getMutator().set(index, isSet);
      valuesMutator.set(index, valueField);
      
    }

    public void setSafe(int index, int isSet, long valueField ) {
      bits.getMutator().setSafe(index, isSet);
      values.getMutator().setSafe(index, valueField);
      setCount++;
      
    }


    public void setSafe(int index, NullableUInt8Holder value) {
      bits.getMutator().setSafe(index, value.isSet);
      values.getMutator().setSafe(index, value);
      setCount++;
      
    }

    public void setSafe(int index, UInt8Holder value) {
      bits.getMutator().setSafeToOne(index);
      values.getMutator().setSafe(index, value);
      setCount++;
      
    }

    public void setSafe(int index, long value) {
      bits.getMutator().setSafeToOne(index);
      values.getMutator().setSafe(index, value);
      setCount++;
    }

    @Override
    public void setValueCount(int valueCount) {
      assert valueCount >= 0;
      values.getMutator().setValueCount(valueCount);
      bits.getMutator().setValueCount(valueCount);
    }

    @Override
    public void generateTestData(int valueCount){
      bits.getMutator().generateTestDataAlt(valueCount);
      values.getMutator().generateTestData(valueCount);
      
      setValueCount(valueCount);
    }

    @Override
    public void reset(){
      setCount = 0;
      
    }

    public void setLastSet(int value) {
        throw new UnsupportedOperationException();
    }

    public int getLastSet() {
        throw new UnsupportedOperationException();
    }
  }
}


