/*******************************************************************************

 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ******************************************************************************/
package org.apache.arrow.vector;


import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;

import com.google.common.collect.Lists;
import com.google.common.collect.ObjectArrays;
import com.google.common.base.Charsets;
import com.google.common.collect.ObjectArrays;

import com.google.flatbuffers.FlatBufferBuilder;

import com.google.common.base.Preconditions;
import io.netty.buffer.*;

import org.apache.arrow.memory.*;
import org.apache.arrow.vector.types.Types;
import org.apache.arrow.vector.types.Types.*;
import org.apache.arrow.vector.types.pojo.*;
import org.apache.arrow.vector.types.pojo.ArrowType.*;
import org.apache.arrow.vector.types.*;
import org.apache.arrow.vector.*;
import org.apache.arrow.vector.holders.*;
import org.apache.arrow.vector.util.*;
import org.apache.arrow.vector.complex.*;
import org.apache.arrow.vector.complex.reader.*;
import org.apache.arrow.vector.complex.impl.*;
import org.apache.arrow.vector.complex.writer.*;
import org.apache.arrow.vector.complex.writer.BaseWriter.MapWriter;
import org.apache.arrow.vector.complex.writer.BaseWriter.ListWriter;
import org.apache.arrow.vector.util.JsonStringArrayList;

import java.util.Arrays;
import java.util.Random;
import java.util.List;

import java.io.Closeable;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.ByteBuffer;

import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.math.BigDecimal;
import java.math.BigInteger;

import org.joda.time.DateTime;
import org.joda.time.LocalDateTime;
import org.joda.time.Period;







/**
 * TimeStampSecTZ implements a vector of fixed width values.  Elements in the vector are accessed
 * by position, starting from the logical start of the vector.  Values should be pushed onto the
 * vector sequentially, but may be randomly accessed.
 *   The width of each element is 8 byte(s)
 *   The equivalent Java primitive is 'long'
 *
 * NB: this class is automatically generated from FixedValueVectors.java and ValueVectorTypes.tdd using FreeMarker.
 */
public final class TimeStampSecTZVector extends BaseDataValueVector implements FixedWidthVector{
  private static final org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(TimeStampSecTZVector.class);

  public static final int TYPE_WIDTH = 8;

  private final Accessor accessor = new Accessor();
  private final Mutator mutator = new Mutator();

  private int allocationSizeInBytes = INITIAL_VALUE_ALLOCATION * 8;
  private int allocationMonitor = 0;

  private final String timezone;

  public TimeStampSecTZVector(String name, BufferAllocator allocator, String timezone) {
    super(name, allocator);
    this.timezone = timezone;
  }

  @Override
  public MinorType getMinorType() {
    return MinorType.TIMESTAMPSECTZ;
  }

  @Override
  public Field getField() {
        throw new UnsupportedOperationException("internal vector");
  }

  @Override
  public FieldReader getReader(){
        throw new UnsupportedOperationException("non-nullable vectors cannot be used in readers");
  }

  @Override
  public int getBufferSizeFor(final int valueCount) {
    if (valueCount == 0) {
      return 0;
    }
    return valueCount * 8;
  }

  @Override
  public ArrowBuf getValidityBuffer() {
    /* this operation is not supported for non-nullable vectors */
    throw new UnsupportedOperationException();
  }

  @Override
  public ArrowBuf getDataBuffer() {
    /* we are not throwing away getBuffer() of BaseDataValueVector so use it wherever applicable */
    return getBuffer();
  }

  @Override
  public ArrowBuf getOffsetBuffer() {
    /* this operation is not supported for fixed-width vectors */
    throw new UnsupportedOperationException();
  }

  @Override
  public int getValueCapacity(){
    return (int) (data.capacity() *1.0 / 8);
  }

  @Override
  public Accessor getAccessor(){
    return accessor;
  }

  @Override
  public Mutator getMutator(){
    return mutator;
  }

  int getAllocationSize() {
    return allocationSizeInBytes;
  }

  @Override
  public void setInitialCapacity(final int valueCount) {
    final long size = 1L * valueCount * 8;
    if (size > MAX_ALLOCATION_SIZE) {
      throw new OversizedAllocationException("Requested amount of memory is more than max allowed allocation size");
    }
    allocationSizeInBytes = (int)size;
  }

  @Override
  public void allocateNew() {
    if(!allocateNewSafe()){
      throw new OutOfMemoryException("Failure while allocating buffer.");
    }
  }

  @Override
  public boolean allocateNewSafe() {
    long curAllocationSize = allocationSizeInBytes;
    if (allocationMonitor > 10) {
      curAllocationSize = Math.max(8, curAllocationSize / 2);
      allocationMonitor = 0;
    } else if (allocationMonitor < -2) {
      curAllocationSize = allocationSizeInBytes * 2L;
      allocationMonitor = 0;
    }

    try{
      allocateBytes(curAllocationSize);
    } catch (RuntimeException ex) {
      return false;
    }
    return true;
  }

  /**
   * Allocate a new buffer that supports setting at least the provided number of values. May actually be sized bigger
   * depending on underlying buffer rounding size. Must be called prior to using the ValueVector.
   *
   * Note that the maximum number of values a vector can allocate is Integer.MAX_VALUE / value width.
   *
   * @param valueCount the number of values to allocate for
   * @throws org.apache.arrow.memory.OutOfMemoryException if it can't allocate the new buffer
   */
  @Override
  public void allocateNew(final int valueCount) {
    allocateBytes(valueCount * 8);
  }

  @Override
  public void reset() {
    allocationSizeInBytes = INITIAL_VALUE_ALLOCATION * 8;
    allocationMonitor = 0;
    zeroVector();
    super.reset();
  }

  private void allocateBytes(final long size) {
    if (size > MAX_ALLOCATION_SIZE) {
      throw new OversizedAllocationException("Requested amount of memory is more than max allowed allocation size");
    }

    final int curSize = (int)size;
    clear();
    data = allocator.buffer(curSize);
    data.readerIndex(0);
    allocationSizeInBytes = curSize;
  }

  /**
   * Allocate new buffer with double capacity, and copy data into the new buffer. Replace vector's buffer with new buffer, and release old one
   *
   * @throws org.apache.arrow.memory.OutOfMemoryException if it can't allocate the new buffer
   */
  public void reAlloc() {
    long baseSize  = allocationSizeInBytes;
    final int currentBufferCapacity = data.capacity();
    if (baseSize < (long)currentBufferCapacity) {
        baseSize = (long)currentBufferCapacity;
    }
    long newAllocationSize = baseSize * 2L;
    newAllocationSize = BaseAllocator.nextPowerOfTwo(newAllocationSize);

    if (newAllocationSize > MAX_ALLOCATION_SIZE) {
      throw new OversizedAllocationException("Unable to expand the buffer. Max allowed buffer size is reached.");
    }

    logger.debug("Reallocating vector [{}]. # of bytes: [{}] -> [{}]", name, allocationSizeInBytes, newAllocationSize);
    final ArrowBuf newBuf = allocator.buffer((int)newAllocationSize);
    newBuf.setBytes(0, data, 0, currentBufferCapacity);
    final int halfNewCapacity = newBuf.capacity() / 2;
    newBuf.setZero(halfNewCapacity, halfNewCapacity);
    newBuf.writerIndex(data.writerIndex());
    data.release(1);
    data = newBuf;
    allocationSizeInBytes = (int)newAllocationSize;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void zeroVector() {
    data.setZero(0, data.capacity());
  }

  public TransferPair getTransferPair(BufferAllocator allocator){
    return new TransferImpl(name, allocator);
  }

  @Override
  public TransferPair getTransferPair(String ref, BufferAllocator allocator){
    return new TransferImpl(ref, allocator);
  }

  @Override
  public TransferPair makeTransferPair(ValueVector to) {
    return new TransferImpl((TimeStampSecTZVector) to);
  }

  public void transferTo(TimeStampSecTZVector target){
    target.clear();
    target.data = data.transferOwnership(target.allocator).buffer;
    target.data.writerIndex(data.writerIndex());
    clear();
  }

  public void splitAndTransferTo(int startIndex, int length, TimeStampSecTZVector target) {
    final int startPoint = startIndex * 8;
    final int sliceLength = length * 8;
    target.clear();
    target.data = data.slice(startPoint, sliceLength).transferOwnership(target.allocator).buffer;
    target.data.writerIndex(sliceLength);
  }

  private class TransferImpl implements TransferPair{
    private TimeStampSecTZVector to;

    public TransferImpl(String name, BufferAllocator allocator){
      to = new TimeStampSecTZVector(name, allocator,  TimeStampSecTZVector.this.timezone);
    }

    public TransferImpl(TimeStampSecTZVector to) {
      this.to = to;
    }

    @Override
    public TimeStampSecTZVector getTo(){
      return to;
    }

    @Override
    public void transfer(){
      transferTo(to);
    }

    @Override
    public void splitAndTransfer(int startIndex, int length) {
      splitAndTransferTo(startIndex, length, to);
    }

    @Override
    public void copyValueSafe(int fromIndex, int toIndex) {
      to.copyFromSafe(fromIndex, toIndex, TimeStampSecTZVector.this);
    }
  }

  public void copyFrom(int fromIndex, int thisIndex, TimeStampSecTZVector from){
 
    data.setLong(thisIndex * 8,
        from.data.getLong(fromIndex * 8)
    );
     
  }

  public void copyFromSafe(int fromIndex, int thisIndex, TimeStampSecTZVector from){
    while(thisIndex >= getValueCapacity()) {
        reAlloc();
    }
    copyFrom(fromIndex, thisIndex, from);
  }

  public void decrementAllocationMonitor() {
    if (allocationMonitor > 0) {
      allocationMonitor = 0;
    }
    --allocationMonitor;
  }

  private void incrementAllocationMonitor() {
    ++allocationMonitor;
  }

  public final class Accessor extends BaseDataValueVector.BaseAccessor {
    @Override
    public int getValueCount() {
      return data.writerIndex() / 8;
    }

    @Override
    public boolean isNull(int index){
      return false;
    }

 

    public long get(int index) {
      return data.getLong(index * 8);
    }
    @Override
    public Long getObject(int index) {
      return get(index);
    }
    public long getPrimitiveObject(int index) {
      return get(index);
    }

    public void get(int index, TimeStampSecTZHolder holder){
      holder.value = data.getLong(index * 8);
    }

    public void get(int index, NullableTimeStampSecTZHolder holder){
      holder.isSet = 1;
      holder.value = data.getLong(index * 8);
    }

     
  }

  /**
   * TimeStampSecTZ.Mutator implements a mutable vector of fixed width values.  Elements in the
   * vector are accessed by position from the logical start of the vector.  Values should be pushed
   * onto the vector sequentially, but may be randomly accessed.
   *   The width of each element is 8 byte(s)
   *   The equivalent Java primitive is 'long'
   *
   * NB: this class is automatically generated from FixedValueVectorTypes.tdd using FreeMarker.
   */
  public final class Mutator extends BaseDataValueVector.BaseMutator {

    private Mutator(){};

   /**
    * Set the element at the given index to the given value.  Note that widths smaller than
    * 32 bits are handled by the ArrowBuf interface.
    *
    * @param index   position of the bit to set
    * @param value   value to set
    */
 
   public void set(int index, long value) {
     data.setLong(index * 8, value);
   }

   public void setSafe(int index, long value) {
     while(index >= getValueCapacity()) {
       reAlloc();
     }
     set(index, value);
   }

   protected void set(int index, TimeStampSecTZHolder holder){
     data.setLong(index * 8, holder.value);
   }

   public void setSafe(int index, TimeStampSecTZHolder holder){
     while(index >= getValueCapacity()) {
       reAlloc();
     }
     set(index, holder);
   }

   protected void set(int index, NullableTimeStampSecTZHolder holder){
     data.setLong(index * 8, holder.value);
   }

   public void setSafe(int index, NullableTimeStampSecTZHolder holder){
     while(index >= getValueCapacity()) {
       reAlloc();
     }
     set(index, holder);
   }

   @Override
   public void generateTestData(int size) {
     setValueCount(size);
     boolean even = true;
     final int valueCount = getAccessor().getValueCount();
     for(int i = 0; i < valueCount; i++, even = !even) {
       if(even){
         set(i, Long.MIN_VALUE);
       }else{
         set(i, Long.MAX_VALUE);
       }
     }
   }

   public void generateTestDataAlt(int size) {
     setValueCount(size);
     boolean even = true;
     final int valueCount = getAccessor().getValueCount();
     for(int i = 0; i < valueCount; i++, even = !even) {
       if(even){
         set(i, (long) 1);
       }else{
         set(i, (long) 0);
       }
     }
   }

     

   @Override
   public void setValueCount(int valueCount) {
     final int currentValueCapacity = getValueCapacity();
     final int idx = (8 * valueCount);
     while(valueCount > getValueCapacity()) {
       reAlloc();
     }
     if (valueCount > 0 && currentValueCapacity > valueCount * 2) {
       incrementAllocationMonitor();
     } else if (allocationMonitor > 0) {
       allocationMonitor = 0;
     }
     VectorTrimmer.trim(data, idx);
     data.writerIndex(valueCount * 8);
   }
 }
}

   

