/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.impl.prefetch;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.hadoop.fs.impl.prefetch.BlockCache;
import org.apache.hadoop.fs.impl.prefetch.BufferData;
import org.apache.hadoop.fs.impl.prefetch.PrefetchingStatistics;
import org.apache.hadoop.fs.impl.prefetch.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SingleFilePerBlockCache
implements BlockCache {
    private static final Logger LOG = LoggerFactory.getLogger(SingleFilePerBlockCache.class);
    private final Map<Integer, Entry> blocks = new ConcurrentHashMap<Integer, Entry>();
    private int numGets = 0;
    private boolean closed;
    private final PrefetchingStatistics prefetchingStatistics;
    private static final Set<? extends OpenOption> CREATE_OPTIONS = EnumSet.of(StandardOpenOption.WRITE, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
    private static final String CACHE_FILE_PREFIX = "fs-cache-";
    private static final String BINARY_FILE_SUFFIX = ".bin";
    private static final FileAttribute<Set<PosixFilePermission>> TEMP_FILE_ATTRS = PosixFilePermissions.asFileAttribute(EnumSet.of(PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE));

    public SingleFilePerBlockCache(PrefetchingStatistics prefetchingStatistics) {
        this.prefetchingStatistics = Objects.requireNonNull(prefetchingStatistics);
    }

    @Override
    public boolean containsBlock(int blockNumber) {
        return this.blocks.containsKey(blockNumber);
    }

    @Override
    public Iterable<Integer> blocks() {
        return Collections.unmodifiableList(new ArrayList<Integer>(this.blocks.keySet()));
    }

    @Override
    public int size() {
        return this.blocks.size();
    }

    @Override
    public void get(int blockNumber, ByteBuffer buffer) throws IOException {
        if (this.closed) {
            return;
        }
        Validate.checkNotNull(buffer, "buffer");
        Entry entry = this.getEntry(blockNumber);
        buffer.clear();
        this.readFile(entry.path, buffer);
        buffer.rewind();
        this.validateEntry(entry, buffer);
    }

    protected int readFile(Path path, ByteBuffer buffer) throws IOException {
        int numBytes;
        int numBytesRead = 0;
        FileChannel channel = FileChannel.open(path, StandardOpenOption.READ);
        while ((numBytes = channel.read(buffer)) > 0) {
            numBytesRead += numBytes;
        }
        buffer.limit(buffer.position());
        channel.close();
        return numBytesRead;
    }

    private Entry getEntry(int blockNumber) {
        Validate.checkNotNegative(blockNumber, "blockNumber");
        Entry entry = this.blocks.get(blockNumber);
        if (entry == null) {
            throw new IllegalStateException(String.format("block %d not found in cache", blockNumber));
        }
        ++this.numGets;
        return entry;
    }

    @Override
    public void put(int blockNumber, ByteBuffer buffer) throws IOException {
        if (this.closed) {
            return;
        }
        Validate.checkNotNull(buffer, "buffer");
        if (this.blocks.containsKey(blockNumber)) {
            Entry entry = this.blocks.get(blockNumber);
            this.validateEntry(entry, buffer);
            return;
        }
        Validate.checkPositiveInteger(buffer.limit(), "buffer.limit()");
        Path blockFilePath = this.getCacheFilePath();
        long size = Files.size(blockFilePath);
        if (size != 0L) {
            String message = String.format("[%d] temp file already has data. %s (%d)", blockNumber, blockFilePath, size);
            throw new IllegalStateException(message);
        }
        this.writeFile(blockFilePath, buffer);
        this.prefetchingStatistics.blockAddedToFileCache();
        long checksum = BufferData.getChecksum(buffer);
        Entry entry = new Entry(blockNumber, blockFilePath, buffer.limit(), checksum);
        this.blocks.put(blockNumber, entry);
    }

    protected void writeFile(Path path, ByteBuffer buffer) throws IOException {
        buffer.rewind();
        SeekableByteChannel writeChannel = Files.newByteChannel(path, CREATE_OPTIONS, new FileAttribute[0]);
        while (buffer.hasRemaining()) {
            writeChannel.write(buffer);
        }
        writeChannel.close();
    }

    protected Path getCacheFilePath() throws IOException {
        return SingleFilePerBlockCache.getTempFilePath();
    }

    @Override
    public void close() throws IOException {
        if (this.closed) {
            return;
        }
        this.closed = true;
        LOG.info(this.getStats());
        int numFilesDeleted = 0;
        for (Entry entry : this.blocks.values()) {
            try {
                Files.deleteIfExists(entry.path);
                this.prefetchingStatistics.blockRemovedFromFileCache();
                ++numFilesDeleted;
            }
            catch (IOException iOException) {}
        }
        if (numFilesDeleted > 0) {
            LOG.info("Deleted {} cache files", (Object)numFilesDeleted);
        }
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("stats: ");
        sb.append(this.getStats());
        sb.append(", blocks:[");
        sb.append(this.getIntList(this.blocks()));
        sb.append("]");
        return sb.toString();
    }

    private void validateEntry(Entry entry, ByteBuffer buffer) {
        if (entry.size != buffer.limit()) {
            String message = String.format("[%d] entry.size(%d) != buffer.limit(%d)", entry.blockNumber, entry.size, buffer.limit());
            throw new IllegalStateException(message);
        }
        long checksum = BufferData.getChecksum(buffer);
        if (entry.checksum != checksum) {
            String message = String.format("[%d] entry.checksum(%d) != buffer checksum(%d)", entry.blockNumber, entry.checksum, checksum);
            throw new IllegalStateException(message);
        }
    }

    private String getIntList(Iterable<Integer> nums) {
        ArrayList<String> numList = new ArrayList<String>();
        ArrayList<Integer> numbers = new ArrayList<Integer>();
        for (Integer n : nums) {
            numbers.add(n);
        }
        Collections.sort(numbers);
        int index = 0;
        while (index < numbers.size()) {
            int start;
            int prev = start = ((Integer)numbers.get(index)).intValue();
            int end = start;
            while (++index < numbers.size() && (end = ((Integer)numbers.get(index)).intValue()) == prev + 1) {
                prev = end;
            }
            if (start == prev) {
                numList.add(Integer.toString(start));
                continue;
            }
            numList.add(String.format("%d~%d", start, prev));
        }
        return String.join((CharSequence)", ", numList);
    }

    private String getStats() {
        StringBuilder sb = new StringBuilder();
        sb.append(String.format("#entries = %d, #gets = %d", this.blocks.size(), this.numGets));
        return sb.toString();
    }

    public static boolean isCacheSpaceAvailable(long fileSize) {
        try {
            Path cacheFilePath = SingleFilePerBlockCache.getTempFilePath();
            long freeSpace = new File(cacheFilePath.toString()).getUsableSpace();
            LOG.info("fileSize = {}, freeSpace = {}", (Object)fileSize, (Object)freeSpace);
            Files.deleteIfExists(cacheFilePath);
            return fileSize < freeSpace;
        }
        catch (IOException e) {
            LOG.error("isCacheSpaceAvailable", (Throwable)e);
            return false;
        }
    }

    private static Path getTempFilePath() throws IOException {
        return Files.createTempFile(CACHE_FILE_PREFIX, BINARY_FILE_SUFFIX, TEMP_FILE_ATTRS);
    }

    private static final class Entry {
        private final int blockNumber;
        private final Path path;
        private final int size;
        private final long checksum;

        Entry(int blockNumber, Path path, int size, long checksum) {
            this.blockNumber = blockNumber;
            this.path = path;
            this.size = size;
            this.checksum = checksum;
        }

        public String toString() {
            return String.format("([%03d] %s: size = %d, checksum = %d)", this.blockNumber, this.path, this.size, this.checksum);
        }
    }
}

