/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.datanode.fsdataset.impl;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.RandomAccessFile;
import java.io.Writer;
import java.net.URI;
import java.nio.channels.ClosedChannelException;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.VisibleForTesting;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.DF;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdfs.DFSUtilClient;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.BlockListAsLongs;
import org.apache.hadoop.hdfs.protocol.ExtendedBlock;
import org.apache.hadoop.hdfs.server.common.HdfsServerConstants;
import org.apache.hadoop.hdfs.server.common.Storage;
import org.apache.hadoop.hdfs.server.datanode.BlockMetadataHeader;
import org.apache.hadoop.hdfs.server.datanode.DatanodeUtil;
import org.apache.hadoop.hdfs.server.datanode.DirectoryScanner;
import org.apache.hadoop.hdfs.server.datanode.FileIoProvider;
import org.apache.hadoop.hdfs.server.datanode.FinalizedReplica;
import org.apache.hadoop.hdfs.server.datanode.LocalReplica;
import org.apache.hadoop.hdfs.server.datanode.LocalReplicaInPipeline;
import org.apache.hadoop.hdfs.server.datanode.ReplicaBeingWritten;
import org.apache.hadoop.hdfs.server.datanode.ReplicaBuilder;
import org.apache.hadoop.hdfs.server.datanode.ReplicaInPipeline;
import org.apache.hadoop.hdfs.server.datanode.ReplicaInfo;
import org.apache.hadoop.hdfs.server.datanode.StorageLocation;
import org.apache.hadoop.hdfs.server.datanode.checker.VolumeCheckResult;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.DataNodeVolumeMetrics;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.FsDatasetSpi;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.FsVolumeReference;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.FsVolumeSpi;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.BlockPoolSlice;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.FsDatasetImpl;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.FsDatasetUtil;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.RamDiskReplicaTracker;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.ReplicaMap;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.ReservedSpaceCalculator;
import org.apache.hadoop.hdfs.server.protocol.DatanodeStorage;
import org.apache.hadoop.thirdparty.com.google.common.base.Joiner;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.apache.hadoop.util.CloseableReferenceCount;
import org.apache.hadoop.util.DataChecksum;
import org.apache.hadoop.util.DiskChecker;
import org.apache.hadoop.util.Preconditions;
import org.apache.hadoop.util.Time;
import org.apache.hadoop.util.Timer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@VisibleForTesting
public class FsVolumeImpl
implements FsVolumeSpi {
    public static final Logger LOG = LoggerFactory.getLogger(FsVolumeImpl.class);
    private static final ObjectWriter WRITER = new ObjectMapper().writerWithDefaultPrettyPrinter();
    private static final ObjectReader READER = new ObjectMapper().readerFor(BlockIteratorState.class);
    private final FsDatasetImpl dataset;
    private final String storageID;
    private final StorageType storageType;
    private final Map<String, BlockPoolSlice> bpSlices = new ConcurrentHashMap<String, BlockPoolSlice>();
    private final StorageLocation storageLocation;
    private final File currentDir;
    private final DF usage;
    private final ReservedSpaceCalculator reserved;
    private long cachedCapacity;
    private CloseableReferenceCount reference = new CloseableReferenceCount();
    private AtomicLong reservedForReplicas;
    private long recentReserved = 0L;
    private final Configuration conf;
    protected volatile long configuredCapacity;
    private final FileIoProvider fileIoProvider;
    private final DataNodeVolumeMetrics metrics;
    private URI baseURI;
    private boolean enableSameDiskTiering;
    private final String mount;
    private double reservedForArchive;
    protected ThreadPoolExecutor cacheExecutor;

    FsVolumeImpl(FsDatasetImpl dataset, String storageID, Storage.StorageDirectory sd, FileIoProvider fileIoProvider, Configuration conf) throws IOException {
        this(dataset, storageID, sd, fileIoProvider, conf, null);
    }

    FsVolumeImpl(FsDatasetImpl dataset, String storageID, Storage.StorageDirectory sd, FileIoProvider fileIoProvider, Configuration conf, DF usage) throws IOException {
        if (sd.getStorageLocation() == null) {
            throw new IOException("StorageLocation specified for storage directory " + sd + " is null");
        }
        this.dataset = dataset;
        this.storageID = storageID;
        this.reservedForReplicas = new AtomicLong(0L);
        this.storageLocation = sd.getStorageLocation();
        this.currentDir = sd.getCurrentDir();
        this.storageType = this.storageLocation.getStorageType();
        this.configuredCapacity = -1L;
        this.usage = usage;
        if (this.usage != null) {
            this.reserved = new ReservedSpaceCalculator.Builder(conf).setUsage(this.usage).setStorageType(this.storageType).build();
            boolean fixedSizeVolume = conf.getBoolean("dfs.datanode.fixed.volume.size", false);
            if (fixedSizeVolume) {
                this.cachedCapacity = this.usage.getCapacity();
            }
        } else {
            this.reserved = null;
            LOG.warn("Setting reserved to null as usage is null");
            this.cachedCapacity = -1L;
        }
        if (this.currentDir != null) {
            File parent = this.currentDir.getParentFile();
            this.cacheExecutor = this.initializeCacheExecutor(parent);
            this.metrics = DataNodeVolumeMetrics.create(conf, parent.getPath());
            this.baseURI = new File(this.currentDir.getParent()).toURI();
        } else {
            this.cacheExecutor = null;
            this.metrics = null;
        }
        this.conf = conf;
        this.fileIoProvider = fileIoProvider;
        this.enableSameDiskTiering = conf.getBoolean("dfs.datanode.same-disk-tiering.enabled", false);
        this.mount = this.enableSameDiskTiering && usage != null ? usage.getMount() : "";
    }

    String getMount() {
        return this.mount;
    }

    protected ThreadPoolExecutor initializeCacheExecutor(File parent) {
        if (this.storageType.isRAM()) {
            return null;
        }
        if (this.dataset.datanode == null) {
            return null;
        }
        int maxNumThreads = this.dataset.datanode.getConf().getInt("dfs.datanode.fsdatasetcache.max.threads.per.volume", 4);
        String escapedPath = parent.toString().replaceAll("%", "%%");
        ThreadFactory workerFactory = new ThreadFactoryBuilder().setDaemon(true).setNameFormat("FsVolumeImplWorker-" + escapedPath + "-%d").build();
        ThreadPoolExecutor executor = new ThreadPoolExecutor(1, maxNumThreads, 60L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(), workerFactory);
        executor.allowCoreThreadTimeOut(true);
        return executor;
    }

    private void printReferenceTraceInfo(String op) {
        StackTraceElement[] stack;
        for (StackTraceElement ste : stack = Thread.currentThread().getStackTrace()) {
            switch (ste.getMethodName()) {
                case "getDfsUsed": 
                case "getBlockPoolUsed": 
                case "getAvailable": 
                case "getVolumeMap": {
                    return;
                }
            }
        }
        FsDatasetImpl.LOG.trace("Reference count: " + op + " " + this + ": " + this.reference.getReferenceCount());
        FsDatasetImpl.LOG.trace(Joiner.on((String)"\n").join((Object[])Thread.currentThread().getStackTrace()));
    }

    private void reference() throws ClosedChannelException {
        this.reference.reference();
        if (FsDatasetImpl.LOG.isTraceEnabled()) {
            this.printReferenceTraceInfo("incr");
        }
    }

    private void unreference() {
        if (FsDatasetImpl.LOG.isTraceEnabled()) {
            this.printReferenceTraceInfo("desc");
        }
        if (FsDatasetImpl.LOG.isDebugEnabled() && this.reference.getReferenceCount() <= 0) {
            FsDatasetImpl.LOG.debug("Decrease reference count <= 0 on " + this + Joiner.on((String)"\n").join((Object[])Thread.currentThread().getStackTrace()));
        }
        this.checkReference();
        this.reference.unreference();
    }

    @Override
    public FsVolumeReference obtainReference() throws ClosedChannelException {
        return new FsVolumeReferenceImpl(this);
    }

    private void checkReference() {
        Preconditions.checkState((this.reference.getReferenceCount() > 0 ? 1 : 0) != 0);
    }

    @VisibleForTesting
    public int getReferenceCount() {
        return this.reference.getReferenceCount();
    }

    void setClosed() throws IOException {
        try {
            this.reference.setClosed();
            this.dataset.stopAllDataxceiverThreads(this);
        }
        catch (ClosedChannelException e) {
            throw new IOException("The volume has already closed.", e);
        }
    }

    boolean checkClosed() {
        if (this.reference.getReferenceCount() > 0) {
            FsDatasetImpl.LOG.debug("The reference count for {} is {}, wait to be 0.", (Object)this, (Object)this.reference.getReferenceCount());
            return false;
        }
        return true;
    }

    @VisibleForTesting
    File getCurrentDir() {
        return this.currentDir;
    }

    protected File getRbwDir(String bpid) throws IOException {
        return this.getBlockPoolSlice(bpid).getRbwDir();
    }

    protected File getLazyPersistDir(String bpid) throws IOException {
        return this.getBlockPoolSlice(bpid).getLazypersistDir();
    }

    protected File getTmpDir(String bpid) throws IOException {
        return this.getBlockPoolSlice(bpid).getTmpDir();
    }

    void onBlockFileDeletion(String bpid, long value) {
        this.decDfsUsedAndNumBlocks(bpid, value, true);
        if (this.isTransientStorage()) {
            this.dataset.releaseLockedMemory(value, true);
        }
    }

    void onMetaFileDeletion(String bpid, long value) {
        this.decDfsUsedAndNumBlocks(bpid, value, false);
    }

    private void decDfsUsedAndNumBlocks(String bpid, long value, boolean blockFileDeleted) {
        BlockPoolSlice bp = this.bpSlices.get(bpid);
        if (bp != null) {
            bp.decDfsUsed(value);
            if (blockFileDeleted) {
                bp.decrNumBlocks();
            }
        }
    }

    void incDfsUsedAndNumBlocks(String bpid, long value) {
        BlockPoolSlice bp = this.bpSlices.get(bpid);
        if (bp != null) {
            bp.incDfsUsed(value);
            bp.incrNumBlocks();
        }
    }

    void incDfsUsed(String bpid, long value) {
        BlockPoolSlice bp = this.bpSlices.get(bpid);
        if (bp != null) {
            bp.incDfsUsed(value);
        }
    }

    @VisibleForTesting
    public long getDfsUsed() throws IOException {
        long dfsUsed = 0L;
        for (BlockPoolSlice s : this.bpSlices.values()) {
            dfsUsed += s.getDfsUsed();
        }
        return dfsUsed;
    }

    long getBlockPoolUsed(String bpid) throws IOException {
        return this.getBlockPoolSlice(bpid).getDfsUsed();
    }

    @VisibleForTesting
    public long getCapacity() {
        long capacity;
        if (this.configuredCapacity < 0L) {
            long remaining = this.cachedCapacity > 0L ? this.cachedCapacity - this.getReserved() : this.usage.getCapacity() - this.getReserved();
            capacity = Math.max(remaining, 0L);
        } else {
            capacity = this.configuredCapacity;
        }
        if (this.enableSameDiskTiering && this.dataset.getMountVolumeMap() != null) {
            double capacityRatio = this.dataset.getMountVolumeMap().getCapacityRatioByMountAndStorageType(this.mount, this.storageType);
            capacity = (long)((double)capacity * capacityRatio);
        }
        return capacity;
    }

    @VisibleForTesting
    public void setCapacityForTesting(long capacity) {
        this.configuredCapacity = capacity;
    }

    @Override
    public long getAvailable() throws IOException {
        long available;
        long remaining = this.getCapacity() - this.getDfsUsed() - this.getReservedForReplicas();
        if (remaining > (available = this.usage.getAvailable() - this.getRemainingReserved() - this.getReservedForReplicas())) {
            remaining = available;
        }
        return Math.max(remaining, 0L);
    }

    long getActualNonDfsUsed() throws IOException {
        if (this.enableSameDiskTiering && StorageType.allowSameDiskTiering((StorageType)this.storageType)) {
            StorageType counterpartStorageType = this.storageType == StorageType.DISK ? StorageType.ARCHIVE : StorageType.DISK;
            FsVolumeReference counterpartRef = this.dataset.getMountVolumeMap().getVolumeRefByMountAndStorageType(this.mount, counterpartStorageType);
            if (counterpartRef != null) {
                FsVolumeImpl counterpartVol = (FsVolumeImpl)counterpartRef.getVolume();
                long used = this.getDfUsed() - this.getDfsUsed() - counterpartVol.getDfsUsed();
                counterpartRef.close();
                return used;
            }
        }
        return this.getDfUsed() - this.getDfsUsed();
    }

    @VisibleForTesting
    public long getDfUsed() {
        return this.usage.getUsed();
    }

    private long getRemainingReserved() throws IOException {
        long actualReserved;
        long actualNonDfsUsed = this.getActualNonDfsUsed();
        if (actualNonDfsUsed < (actualReserved = this.getReserved())) {
            return actualReserved - actualNonDfsUsed;
        }
        return 0L;
    }

    public long getNonDfsUsed() throws IOException {
        long actualNonDfsUsed = this.getActualNonDfsUsed();
        long actualReserved = this.getReserved();
        long nonDfsUsed = actualNonDfsUsed - actualReserved;
        return Math.max(nonDfsUsed, 0L);
    }

    @VisibleForTesting
    long getDfAvailable() {
        return this.usage.getAvailable();
    }

    @VisibleForTesting
    public long getReservedForReplicas() {
        return this.reservedForReplicas.get();
    }

    @VisibleForTesting
    long getRecentReserved() {
        return this.recentReserved;
    }

    public Map<String, BlockPoolSlice> getBlockPoolSlices() {
        return this.bpSlices;
    }

    long getReserved() {
        return this.reserved != null ? this.reserved.getReserved() : 0L;
    }

    @VisibleForTesting
    BlockPoolSlice getBlockPoolSlice(String bpid) throws IOException {
        BlockPoolSlice bp = this.bpSlices.get(bpid);
        if (bp == null) {
            throw new IOException("block pool " + bpid + " is not found");
        }
        return bp;
    }

    @Override
    public URI getBaseURI() {
        return this.baseURI;
    }

    @Override
    public DF getUsageStats(Configuration conf) {
        if (this.currentDir != null) {
            try {
                return new DF(new File(this.currentDir.getParent()), conf);
            }
            catch (IOException e) {
                LOG.error("Unable to get disk statistics for volume {}", (Object)this, (Object)e);
            }
        }
        return null;
    }

    @Override
    public StorageLocation getStorageLocation() {
        return this.storageLocation;
    }

    @Override
    public boolean isTransientStorage() {
        return this.storageType.isTransient();
    }

    @Override
    public boolean isRAMStorage() {
        return this.storageType.isRAM();
    }

    @VisibleForTesting
    public File getFinalizedDir(String bpid) throws IOException {
        return this.getBlockPoolSlice(bpid).getFinalizedDir();
    }

    @Override
    public String[] getBlockPoolList() {
        return this.bpSlices.keySet().toArray(new String[0]);
    }

    File createTmpFile(String bpid, Block b) throws IOException {
        this.checkReference();
        this.reserveSpaceForReplica(b.getNumBytes());
        try {
            return this.getBlockPoolSlice(bpid).createTmpFile(b);
        }
        catch (IOException exception) {
            this.releaseReservedSpace(b.getNumBytes());
            throw exception;
        }
    }

    @Override
    public void reserveSpaceForReplica(long bytesToReserve) {
        if (bytesToReserve != 0L) {
            this.reservedForReplicas.addAndGet(bytesToReserve);
            this.recentReserved = bytesToReserve;
        }
    }

    @Override
    public void releaseReservedSpace(long bytesToRelease) {
        if (bytesToRelease != 0L) {
            long newReservation;
            long oldReservation;
            do {
                oldReservation = this.reservedForReplicas.get();
                newReservation = oldReservation - bytesToRelease;
            } while (!this.reservedForReplicas.compareAndSet(oldReservation, newReservation = Math.max(newReservation, 0L)));
        }
    }

    @Override
    public void releaseLockedMemory(long bytesToRelease) {
        if (this.isTransientStorage()) {
            this.dataset.releaseLockedMemory(bytesToRelease, false);
        }
    }

    @VisibleForTesting
    public static String nextSorted(List<String> arr, String prev) {
        int res = 0;
        if (prev != null) {
            res = Collections.binarySearch(arr, prev);
            res = res < 0 ? -1 - res : ++res;
        }
        if (res >= arr.size()) {
            return null;
        }
        return arr.get(res);
    }

    @Override
    public FsVolumeSpi.BlockIterator newBlockIterator(String bpid, String name) {
        return new BlockIteratorImpl(bpid, name);
    }

    @Override
    public FsVolumeSpi.BlockIterator loadBlockIterator(String bpid, String name) throws IOException {
        BlockIteratorImpl iter = new BlockIteratorImpl(bpid, name);
        iter.load();
        return iter;
    }

    @Override
    public FsDatasetSpi<? extends FsVolumeSpi> getDataset() {
        return this.dataset;
    }

    File createRbwFile(String bpid, Block b) throws IOException {
        this.checkReference();
        this.reserveSpaceForReplica(b.getNumBytes());
        try {
            return this.getBlockPoolSlice(bpid).createRbwFile(b);
        }
        catch (IOException exception) {
            this.releaseReservedSpace(b.getNumBytes());
            throw exception;
        }
    }

    ReplicaInfo addFinalizedBlock(String bpid, Block b, ReplicaInfo replicaInfo, long bytesReserved) throws IOException {
        byte[] checksum;
        this.releaseReservedSpace(bytesReserved);
        File dest = this.getBlockPoolSlice(bpid).addFinalizedBlock(b, replicaInfo);
        switch (replicaInfo.getState()) {
            case FINALIZED: {
                FinalizedReplica finalized = (FinalizedReplica)replicaInfo;
                checksum = finalized.getLastPartialChunkChecksum();
                break;
            }
            case RBW: {
                ReplicaBeingWritten rbw = (ReplicaBeingWritten)replicaInfo;
                checksum = rbw.getLastChecksumAndDataLen().getChecksum();
                break;
            }
            default: {
                checksum = null;
            }
        }
        return new ReplicaBuilder(HdfsServerConstants.ReplicaState.FINALIZED).setBlock(replicaInfo).setFsVolume(this).setDirectoryToUse(dest.getParentFile()).setLastPartialChunkChecksum(checksum).build();
    }

    Executor getCacheExecutor() {
        return this.cacheExecutor;
    }

    @Override
    public VolumeCheckResult check(FsVolumeSpi.VolumeCheckContext ignored) throws DiskChecker.DiskErrorException {
        for (BlockPoolSlice s : this.bpSlices.values()) {
            s.checkDirs();
        }
        return VolumeCheckResult.HEALTHY;
    }

    void getVolumeMap(ReplicaMap volumeMap, RamDiskReplicaTracker ramDiskReplicaMap) throws IOException {
        for (BlockPoolSlice s : this.bpSlices.values()) {
            s.getVolumeMap(volumeMap, ramDiskReplicaMap);
        }
    }

    void getVolumeMap(String bpid, ReplicaMap volumeMap, RamDiskReplicaTracker ramDiskReplicaMap) throws IOException {
        this.getBlockPoolSlice(bpid).getVolumeMap(volumeMap, ramDiskReplicaMap);
    }

    long getNumBlocks() {
        long numBlocks = 0L;
        for (BlockPoolSlice s : this.bpSlices.values()) {
            numBlocks += s.getNumOfBlocks();
        }
        return numBlocks;
    }

    public String toString() {
        return this.currentDir != null ? this.currentDir.getParent() : "NULL";
    }

    void shutdown() {
        if (this.cacheExecutor != null) {
            this.cacheExecutor.shutdown();
        }
        Set<Map.Entry<String, BlockPoolSlice>> set = this.bpSlices.entrySet();
        for (Map.Entry<String, BlockPoolSlice> entry : set) {
            entry.getValue().shutdown(null);
        }
        if (this.metrics != null) {
            this.metrics.unRegister();
        }
    }

    void addBlockPool(String bpid, Configuration c) throws IOException {
        this.addBlockPool(bpid, c, null);
    }

    void addBlockPool(String bpid, Configuration c, Timer timer) throws IOException {
        File bpdir = new File(this.currentDir, bpid);
        if (timer == null) {
            timer = new Timer();
        }
        BlockPoolSlice bp = new BlockPoolSlice(bpid, this, bpdir, c, timer);
        this.bpSlices.put(bpid, bp);
    }

    void shutdownBlockPool(String bpid, BlockListAsLongs blocksListsAsLongs) {
        BlockPoolSlice bp = this.bpSlices.get(bpid);
        if (bp != null) {
            bp.shutdown(blocksListsAsLongs);
        }
        this.bpSlices.remove(bpid);
    }

    boolean isBPDirEmpty(String bpid) throws IOException {
        File volumeCurrentDir = this.getCurrentDir();
        File bpDir = new File(volumeCurrentDir, bpid);
        File bpCurrentDir = new File(bpDir, "current");
        File finalizedDir = new File(bpCurrentDir, "finalized");
        File rbwDir = new File(bpCurrentDir, "rbw");
        if (this.fileIoProvider.exists(this, finalizedDir) && !DatanodeUtil.dirNoFilesRecursive(this, finalizedDir, this.fileIoProvider)) {
            return false;
        }
        return !this.fileIoProvider.exists(this, rbwDir) || this.fileIoProvider.list(this, rbwDir).length == 0;
    }

    void deleteBPDirectories(String bpid, boolean force) throws IOException {
        File volumeCurrentDir = this.getCurrentDir();
        File bpDir = new File(volumeCurrentDir, bpid);
        if (!bpDir.isDirectory()) {
            return;
        }
        File tmpDir = new File(bpDir, "tmp");
        File bpCurrentDir = new File(bpDir, "current");
        File finalizedDir = new File(bpCurrentDir, "finalized");
        File lazypersistDir = new File(bpCurrentDir, "lazypersist");
        File rbwDir = new File(bpCurrentDir, "rbw");
        if (force) {
            this.fileIoProvider.fullyDelete(this, bpDir);
        } else {
            if (!this.fileIoProvider.delete(this, rbwDir)) {
                throw new IOException("Failed to delete " + rbwDir);
            }
            if (!DatanodeUtil.dirNoFilesRecursive(this, finalizedDir, this.fileIoProvider) || !this.fileIoProvider.fullyDelete(this, finalizedDir)) {
                throw new IOException("Failed to delete " + finalizedDir);
            }
            if (!(!lazypersistDir.exists() || DatanodeUtil.dirNoFilesRecursive(this, lazypersistDir, this.fileIoProvider) && this.fileIoProvider.fullyDelete(this, lazypersistDir))) {
                throw new IOException("Failed to delete " + lazypersistDir);
            }
            this.fileIoProvider.fullyDelete(this, tmpDir);
            for (File f : this.fileIoProvider.listFiles(this, bpCurrentDir)) {
                if (this.fileIoProvider.delete(this, f)) continue;
                throw new IOException("Failed to delete " + f);
            }
            if (!this.fileIoProvider.delete(this, bpCurrentDir)) {
                throw new IOException("Failed to delete " + bpCurrentDir);
            }
            for (File f : this.fileIoProvider.listFiles(this, bpDir)) {
                if (this.fileIoProvider.delete(this, f)) continue;
                throw new IOException("Failed to delete " + f);
            }
            if (!this.fileIoProvider.delete(this, bpDir)) {
                throw new IOException("Failed to delete " + bpDir);
            }
        }
    }

    @Override
    public String getStorageID() {
        return this.storageID;
    }

    @Override
    public StorageType getStorageType() {
        return this.storageType;
    }

    DatanodeStorage toDatanodeStorage() {
        return new DatanodeStorage(this.storageID, DatanodeStorage.State.NORMAL, this.storageType);
    }

    @Override
    public byte[] loadLastPartialChunkChecksum(File blockFile, File metaFile) throws IOException {
        DataChecksum dcs;
        try (FileInputStream fis = this.fileIoProvider.getFileInputStream(this, metaFile);){
            dcs = BlockMetadataHeader.readHeader((FileInputStream)fis).getChecksum();
        }
        int checksumSize = dcs.getChecksumSize();
        long onDiskLen = blockFile.length();
        int bytesPerChecksum = dcs.getBytesPerChecksum();
        if (onDiskLen % (long)bytesPerChecksum == 0L) {
            return null;
        }
        long offsetInChecksum = (long)BlockMetadataHeader.getHeaderSize() + onDiskLen / (long)bytesPerChecksum * (long)checksumSize;
        byte[] lastChecksum = new byte[checksumSize];
        try (RandomAccessFile raf = this.fileIoProvider.getRandomAccessFile(this, metaFile, "r");){
            raf.seek(offsetInChecksum);
            int readBytes = raf.read(lastChecksum, 0, checksumSize);
            if (readBytes == -1) {
                throw new IOException("Expected to read " + checksumSize + " bytes from offset " + offsetInChecksum + " but reached end of file.");
            }
            if (readBytes != checksumSize) {
                throw new IOException("Expected to read " + checksumSize + " bytes from offset " + offsetInChecksum + " but read " + readBytes + " bytes.");
            }
        }
        return lastChecksum;
    }

    public ReplicaInPipeline append(String bpid, ReplicaInfo replicaInfo, long newGS, long estimateBlockLen) throws IOException {
        long bytesReserved = estimateBlockLen - replicaInfo.getNumBytes();
        if (this.getAvailable() < bytesReserved) {
            throw new DiskChecker.DiskOutOfSpaceException("Insufficient space for appending to " + replicaInfo);
        }
        assert (replicaInfo.getVolume() == this) : "The volume of the replica should be the same as this volume";
        File newBlkFile = new File(this.getRbwDir(bpid), replicaInfo.getBlockName());
        LocalReplicaInPipeline newReplicaInfo = new ReplicaBuilder(HdfsServerConstants.ReplicaState.RBW).setBlockId(replicaInfo.getBlockId()).setLength(replicaInfo.getNumBytes()).setGenerationStamp(newGS).setFsVolume(this).setDirectoryToUse(newBlkFile.getParentFile()).setWriterThread(Thread.currentThread()).setBytesToReserve(bytesReserved).buildLocalReplicaInPipeline();
        FinalizedReplica finalized = (FinalizedReplica)replicaInfo;
        newReplicaInfo.setLastChecksumAndDataLen(finalized.getVisibleLength(), finalized.getLastPartialChunkChecksum());
        newReplicaInfo.moveReplicaFrom(replicaInfo, newBlkFile);
        this.reserveSpaceForReplica(bytesReserved);
        return newReplicaInfo;
    }

    public ReplicaInPipeline createRbw(ExtendedBlock b) throws IOException {
        File f = this.createRbwFile(b.getBlockPoolId(), b.getLocalBlock());
        LocalReplicaInPipeline newReplicaInfo = new ReplicaBuilder(HdfsServerConstants.ReplicaState.RBW).setBlockId(b.getBlockId()).setGenerationStamp(b.getGenerationStamp()).setFsVolume(this).setDirectoryToUse(f.getParentFile()).setBytesToReserve(b.getNumBytes()).buildLocalReplicaInPipeline();
        return newReplicaInfo;
    }

    public ReplicaInPipeline convertTemporaryToRbw(ExtendedBlock b, ReplicaInfo temp) throws IOException {
        long blockId = b.getBlockId();
        long expectedGs = b.getGenerationStamp();
        long visible = b.getNumBytes();
        long numBytes = temp.getNumBytes();
        BlockPoolSlice bpslice = this.getBlockPoolSlice(b.getBlockPoolId());
        File dest = FsDatasetImpl.moveBlockFiles(b.getLocalBlock(), temp, bpslice.getRbwDir());
        LocalReplicaInPipeline rbw = new ReplicaBuilder(HdfsServerConstants.ReplicaState.RBW).setBlockId(blockId).setLength(numBytes).setGenerationStamp(expectedGs).setFsVolume(this).setDirectoryToUse(dest.getParentFile()).setWriterThread(Thread.currentThread()).setBytesToReserve(0L).buildLocalReplicaInPipeline();
        rbw.setBytesAcked(visible);
        File destMeta = FsDatasetUtil.getMetaFile(dest, b.getGenerationStamp());
        byte[] lastChunkChecksum = this.loadLastPartialChunkChecksum(dest, destMeta);
        rbw.setLastChecksumAndDataLen(numBytes, lastChunkChecksum);
        return rbw;
    }

    public ReplicaInPipeline createTemporary(ExtendedBlock b) throws IOException {
        File f = this.createTmpFile(b.getBlockPoolId(), b.getLocalBlock());
        LocalReplicaInPipeline newReplicaInfo = new ReplicaBuilder(HdfsServerConstants.ReplicaState.TEMPORARY).setBlockId(b.getBlockId()).setGenerationStamp(b.getGenerationStamp()).setDirectoryToUse(f.getParentFile()).setBytesToReserve(b.getLocalBlock().getNumBytes()).setFsVolume(this).buildLocalReplicaInPipeline();
        return newReplicaInfo;
    }

    public ReplicaInPipeline updateRURCopyOnTruncate(ReplicaInfo rur, String bpid, long newBlockId, long recoveryId, long newlength) throws IOException {
        rur.breakHardLinksIfNeeded();
        File[] copiedReplicaFiles = this.copyReplicaWithNewBlockIdAndGS(rur, bpid, newBlockId, recoveryId);
        File blockFile = copiedReplicaFiles[1];
        File metaFile = copiedReplicaFiles[0];
        LocalReplica.truncateBlock(rur.getVolume(), blockFile, metaFile, rur.getNumBytes(), newlength, this.fileIoProvider);
        LocalReplicaInPipeline newReplicaInfo = new ReplicaBuilder(HdfsServerConstants.ReplicaState.RBW).setBlockId(newBlockId).setGenerationStamp(recoveryId).setFsVolume(this).setDirectoryToUse(blockFile.getParentFile()).setBytesToReserve(newlength).buildLocalReplicaInPipeline();
        return newReplicaInfo;
    }

    private File[] copyReplicaWithNewBlockIdAndGS(ReplicaInfo replicaInfo, String bpid, long newBlkId, long newGS) throws IOException {
        String blockFileName = "blk_" + newBlkId;
        FsVolumeImpl v = (FsVolumeImpl)replicaInfo.getVolume();
        File tmpDir = v.getBlockPoolSlice(bpid).getTmpDir();
        File destDir = DatanodeUtil.idToBlockDir(tmpDir, newBlkId);
        File dstBlockFile = new File(destDir, blockFileName);
        File dstMetaFile = FsDatasetUtil.getMetaFile(dstBlockFile, newGS);
        return FsDatasetImpl.copyBlockFiles(replicaInfo, dstMetaFile, dstBlockFile, true, DFSUtilClient.getSmallBufferSize((Configuration)this.conf), this.conf);
    }

    @Override
    public void compileReport(String bpid, Collection<FsVolumeSpi.ScanInfo> report, DirectoryScanner.ReportCompiler reportCompiler) throws InterruptedException, IOException {
        this.compileReport(this.getFinalizedDir(bpid), this.getFinalizedDir(bpid), report, reportCompiler);
    }

    @Override
    public FileIoProvider getFileIoProvider() {
        return this.fileIoProvider;
    }

    @Override
    public DataNodeVolumeMetrics getMetrics() {
        return this.metrics;
    }

    private void compileReport(File bpFinalizedDir, File dir, Collection<FsVolumeSpi.ScanInfo> report, DirectoryScanner.ReportCompiler reportCompiler) throws InterruptedException {
        List<String> fileNames;
        reportCompiler.throttle();
        try {
            fileNames = this.fileIoProvider.listDirectory(this, dir, BlockDirFilter.INSTANCE);
        }
        catch (IOException ioe) {
            LOG.warn("Exception occurred while compiling report", (Throwable)ioe);
            return;
        }
        Collections.sort(fileNames);
        for (int i = 0; i < fileNames.size(); ++i) {
            File blkMetaFile;
            if (Thread.interrupted()) {
                throw new InterruptedException();
            }
            File file = new File(dir, fileNames.get(i));
            if (file.isDirectory()) {
                this.compileReport(bpFinalizedDir, file, report, reportCompiler);
                continue;
            }
            if (!Block.isBlockFilename((File)file)) {
                if (!FsVolumeImpl.isBlockMetaFile("blk_", file.getName())) continue;
                long blockId = Block.getBlockId((String)file.getName());
                this.verifyFileLocation(file, bpFinalizedDir, blockId);
                report.add(new FsVolumeSpi.ScanInfo(blockId, dir, null, fileNames.get(i), this));
                continue;
            }
            File blockFile = file;
            long blockId = Block.filename2id((String)file.getName());
            File metaFile = null;
            while (i + 1 < fileNames.size() && (blkMetaFile = new File(dir, fileNames.get(i + 1))).isFile() && blkMetaFile.getName().startsWith(blockFile.getName())) {
                ++i;
                if (!FsVolumeImpl.isBlockMetaFile(blockFile.getName(), blkMetaFile.getName())) continue;
                metaFile = blkMetaFile;
                break;
            }
            this.verifyFileLocation(blockFile, bpFinalizedDir, blockId);
            report.add(new FsVolumeSpi.ScanInfo(blockId, dir, blockFile.getName(), metaFile == null ? null : metaFile.getName(), this));
        }
    }

    private static boolean isBlockMetaFile(String blockId, String metaFile) {
        return metaFile.startsWith(blockId) && metaFile.endsWith(".meta");
    }

    private void verifyFileLocation(File actualBlockFile, File bpFinalizedDir, long blockId) {
        File expectedBlockDir = DatanodeUtil.idToBlockDir(bpFinalizedDir, blockId);
        File actualBlockDir = actualBlockFile.getParentFile();
        if (actualBlockDir.compareTo(expectedBlockDir) != 0) {
            LOG.warn("Block: " + blockId + " found in invalid directory.  Expected directory: " + expectedBlockDir + ".  Actual directory: " + actualBlockDir);
        }
    }

    public ReplicaInfo moveBlockToTmpLocation(ExtendedBlock block, ReplicaInfo replicaInfo, int smallBufferSize, Configuration conf) throws IOException {
        File[] blockFiles = FsDatasetImpl.copyBlockFiles(block.getBlockId(), block.getGenerationStamp(), replicaInfo, this.getTmpDir(block.getBlockPoolId()), replicaInfo.isOnTransientStorage(), smallBufferSize, conf);
        ReplicaInfo newReplicaInfo = new ReplicaBuilder(HdfsServerConstants.ReplicaState.TEMPORARY).setBlockId(replicaInfo.getBlockId()).setGenerationStamp(replicaInfo.getGenerationStamp()).setFsVolume(this).setDirectoryToUse(blockFiles[0].getParentFile()).setBytesToReserve(0L).build();
        newReplicaInfo.setNumBytes(blockFiles[1].length());
        return newReplicaInfo;
    }

    public ReplicaInfo hardLinkBlockToTmpLocation(ExtendedBlock block, ReplicaInfo replicaInfo) throws IOException {
        File[] blockFiles = FsDatasetImpl.hardLinkBlockFiles(block.getBlockId(), block.getGenerationStamp(), replicaInfo, this.getTmpDir(block.getBlockPoolId()));
        ReplicaInfo newReplicaInfo = new ReplicaBuilder(HdfsServerConstants.ReplicaState.TEMPORARY).setBlockId(replicaInfo.getBlockId()).setGenerationStamp(replicaInfo.getGenerationStamp()).setFsVolume(this).setDirectoryToUse(blockFiles[0].getParentFile()).setBytesToReserve(0L).build();
        newReplicaInfo.setNumBytes(blockFiles[1].length());
        return newReplicaInfo;
    }

    public File[] copyBlockToLazyPersistLocation(String bpId, long blockId, long genStamp, ReplicaInfo replicaInfo, int smallBufferSize, Configuration conf) throws IOException {
        File lazyPersistDir = this.getLazyPersistDir(bpId);
        if (!lazyPersistDir.exists() && !lazyPersistDir.mkdirs()) {
            FsDatasetImpl.LOG.warn("LazyWriter failed to create " + lazyPersistDir);
            throw new IOException("LazyWriter fail to find or create lazy persist dir: " + lazyPersistDir.toString());
        }
        File[] targetFiles = FsDatasetImpl.copyBlockFiles(blockId, genStamp, replicaInfo, lazyPersistDir, true, smallBufferSize, conf);
        return targetFiles;
    }

    public void incrNumBlocks(String bpid) throws IOException {
        this.getBlockPoolSlice(bpid).incrNumBlocks();
    }

    public void resolveDuplicateReplicas(String bpid, ReplicaInfo memBlockInfo, ReplicaInfo diskBlockInfo, ReplicaMap volumeMap) throws IOException {
        this.getBlockPoolSlice(bpid).resolveDuplicateReplicas(memBlockInfo, diskBlockInfo, volumeMap);
    }

    public ReplicaInfo activateSavedReplica(String bpid, ReplicaInfo replicaInfo, RamDiskReplicaTracker.RamDiskReplica replicaState) throws IOException {
        return this.getBlockPoolSlice(bpid).activateSavedReplica(replicaInfo, replicaState);
    }

    public static enum BlockDirFilter implements FilenameFilter
    {
        INSTANCE;


        @Override
        public boolean accept(File dir, String name) {
            return name.startsWith("subdir") || name.startsWith("finalized") || name.startsWith("blk_");
        }
    }

    private class BlockIteratorImpl
    implements FsVolumeSpi.BlockIterator {
        private final File bpidDir;
        private final String name;
        private final String bpid;
        private long maxStalenessMs = 0L;
        private List<String> cache;
        private long cacheMs;
        private BlockIteratorState state;

        BlockIteratorImpl(String bpid, String name) {
            this.bpidDir = new File(FsVolumeImpl.this.currentDir, bpid);
            this.name = name;
            this.bpid = bpid;
            this.rewind();
        }

        private String getNextSubDir(String prev, File dir) throws IOException {
            List<String> children = FsVolumeImpl.this.fileIoProvider.listDirectory(FsVolumeImpl.this, dir, SubdirFilter.INSTANCE);
            this.cache = null;
            this.cacheMs = 0L;
            if (children.isEmpty()) {
                LOG.trace("getNextSubDir({}, {}): no subdirectories found in {}", new Object[]{FsVolumeImpl.this.storageID, this.bpid, dir.getAbsolutePath()});
                return null;
            }
            Collections.sort(children);
            String nextSubDir = FsVolumeImpl.nextSorted(children, prev);
            LOG.trace("getNextSubDir({}, {}): picking next subdirectory {} within {}", new Object[]{FsVolumeImpl.this.storageID, this.bpid, nextSubDir, dir.getAbsolutePath()});
            return nextSubDir;
        }

        private String getNextFinalizedDir() throws IOException {
            File dir = Paths.get(this.bpidDir.getAbsolutePath(), "current", "finalized").toFile();
            return this.getNextSubDir(this.state.curFinalizedDir, dir);
        }

        private String getNextFinalizedSubDir() throws IOException {
            if (this.state.curFinalizedDir == null) {
                return null;
            }
            File dir = Paths.get(this.bpidDir.getAbsolutePath(), "current", "finalized", this.state.curFinalizedDir).toFile();
            return this.getNextSubDir(this.state.curFinalizedSubDir, dir);
        }

        private List<String> getSubdirEntries() throws IOException {
            if (this.state.curFinalizedSubDir == null) {
                return null;
            }
            long now = Time.monotonicNow();
            if (this.cache != null) {
                long delta = now - this.cacheMs;
                if (delta < this.maxStalenessMs) {
                    return this.cache;
                }
                LOG.trace("getSubdirEntries({}, {}): purging entries cache for {} after {} ms.", new Object[]{FsVolumeImpl.this.storageID, this.bpid, this.state.curFinalizedSubDir, delta});
                this.cache = null;
            }
            File dir = Paths.get(this.bpidDir.getAbsolutePath(), "current", "finalized", this.state.curFinalizedDir, this.state.curFinalizedSubDir).toFile();
            List<String> entries = FsVolumeImpl.this.fileIoProvider.listDirectory(FsVolumeImpl.this, dir, BlockFileFilter.INSTANCE);
            if (entries.isEmpty()) {
                entries = null;
                LOG.trace("getSubdirEntries({}, {}): no entries found in {}", new Object[]{FsVolumeImpl.this.storageID, this.bpid, dir.getAbsolutePath()});
            } else {
                Collections.sort(entries);
                LOG.trace("getSubdirEntries({}, {}): listed {} entries in {}", new Object[]{FsVolumeImpl.this.storageID, this.bpid, entries.size(), dir.getAbsolutePath()});
            }
            this.cache = entries;
            this.cacheMs = now;
            return this.cache;
        }

        @Override
        public ExtendedBlock nextBlock() throws IOException {
            if (this.state.atEnd) {
                return null;
            }
            try {
                while (true) {
                    List<String> entries;
                    if ((entries = this.getSubdirEntries()) != null) {
                        this.state.curEntry = FsVolumeImpl.nextSorted(entries, this.state.curEntry);
                        if (this.state.curEntry == null) {
                            LOG.trace("nextBlock({}, {}): advancing from {} to next subdirectory.", new Object[]{FsVolumeImpl.this.storageID, this.bpid, this.state.curFinalizedSubDir});
                        } else {
                            File metaFile;
                            ExtendedBlock block = new ExtendedBlock(this.bpid, Block.filename2id((String)this.state.curEntry));
                            File expectedBlockDir = DatanodeUtil.idToBlockDir(new File("."), block.getBlockId());
                            File actualBlockDir = Paths.get(".", this.state.curFinalizedDir, this.state.curFinalizedSubDir).toFile();
                            if (!expectedBlockDir.equals(actualBlockDir)) {
                                LOG.error("nextBlock({}, {}): block id {} found in invalid directory.  Expected directory: {}.  Actual directory: {}", new Object[]{FsVolumeImpl.this.storageID, this.bpid, block.getBlockId(), expectedBlockDir.getPath(), actualBlockDir.getPath()});
                                continue;
                            }
                            File blkFile = this.getBlockFile(this.bpid, block);
                            try {
                                metaFile = FsDatasetUtil.findMetaFile(blkFile);
                            }
                            catch (FileNotFoundException e) {
                                LOG.warn("nextBlock({}, {}): {}", new Object[]{FsVolumeImpl.this.storageID, this.bpid, e.getMessage()});
                                continue;
                            }
                            block.setGenerationStamp(Block.getGenerationStamp((String)metaFile.getName()));
                            block.setNumBytes(blkFile.length());
                            LOG.trace("nextBlock({}, {}): advancing to {}", new Object[]{FsVolumeImpl.this.storageID, this.bpid, block});
                            return block;
                        }
                    }
                    this.state.curFinalizedSubDir = this.getNextFinalizedSubDir();
                    if (this.state.curFinalizedSubDir != null) continue;
                    this.state.curFinalizedDir = this.getNextFinalizedDir();
                    if (this.state.curFinalizedDir == null) break;
                }
                this.state.atEnd = true;
                return null;
            }
            catch (IOException e) {
                this.state.atEnd = true;
                LOG.error("nextBlock({}, {}): I/O error", new Object[]{FsVolumeImpl.this.storageID, this.bpid, e});
                throw e;
            }
        }

        private File getBlockFile(String bpid, ExtendedBlock blk) throws IOException {
            return new File(DatanodeUtil.idToBlockDir(FsVolumeImpl.this.getFinalizedDir(bpid), blk.getBlockId()).toString() + "/" + blk.getBlockName());
        }

        @Override
        public boolean atEnd() {
            return this.state.atEnd;
        }

        @Override
        public void rewind() {
            this.cache = null;
            this.cacheMs = 0L;
            this.state = new BlockIteratorState();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void save() throws IOException {
            this.state.lastSavedMs = Time.now();
            boolean success = false;
            try (BufferedWriter writer = new BufferedWriter(new OutputStreamWriter((OutputStream)FsVolumeImpl.this.fileIoProvider.getFileOutputStream((FsVolumeSpi)FsVolumeImpl.this, this.getTempSaveFile()), "UTF-8"));){
                WRITER.writeValue((Writer)writer, (Object)this.state);
                success = true;
            }
            finally {
                if (!success) {
                    FsVolumeImpl.this.fileIoProvider.delete(FsVolumeImpl.this, this.getTempSaveFile());
                }
            }
            FsVolumeImpl.this.fileIoProvider.move(FsVolumeImpl.this, this.getTempSaveFile().toPath(), this.getSaveFile().toPath(), StandardCopyOption.ATOMIC_MOVE);
            if (LOG.isTraceEnabled()) {
                LOG.trace("save({}, {}): saved {}", new Object[]{FsVolumeImpl.this.storageID, this.bpid, WRITER.writeValueAsString((Object)this.state)});
            }
        }

        public void load() throws IOException {
            File file = this.getSaveFile();
            this.state = (BlockIteratorState)READER.readValue(file);
            if (LOG.isTraceEnabled()) {
                LOG.trace("load({}, {}): loaded iterator {} from {}: {}", new Object[]{FsVolumeImpl.this.storageID, this.bpid, this.name, file.getAbsoluteFile(), WRITER.writeValueAsString((Object)this.state)});
            }
        }

        File getSaveFile() {
            return new File(this.bpidDir, this.name + ".cursor");
        }

        File getTempSaveFile() {
            return new File(this.bpidDir, this.name + ".cursor.tmp");
        }

        @Override
        public void setMaxStalenessMs(long maxStalenessMs) {
            this.maxStalenessMs = maxStalenessMs;
        }

        @Override
        public void close() throws IOException {
        }

        @Override
        public long getIterStartMs() {
            return this.state.iterStartMs;
        }

        @Override
        public long getLastSavedMs() {
            return this.state.lastSavedMs;
        }

        @Override
        public String getBlockPoolId() {
            return this.bpid;
        }
    }

    private static class BlockIteratorState {
        @JsonProperty
        private long lastSavedMs;
        @JsonProperty
        private long iterStartMs;
        @JsonProperty
        private String curFinalizedDir;
        @JsonProperty
        private String curFinalizedSubDir;
        @JsonProperty
        private String curEntry;
        @JsonProperty
        private boolean atEnd;

        BlockIteratorState() {
            this.lastSavedMs = this.iterStartMs = Time.now();
            this.curFinalizedDir = null;
            this.curFinalizedSubDir = null;
            this.curEntry = null;
            this.atEnd = false;
        }
    }

    private static enum BlockFileFilter implements FilenameFilter
    {
        INSTANCE;


        @Override
        public boolean accept(File dir, String name) {
            return !name.endsWith(".meta") && name.startsWith("blk_");
        }
    }

    private static enum SubdirFilter implements FilenameFilter
    {
        INSTANCE;


        @Override
        public boolean accept(File dir, String name) {
            return name.startsWith("subdir");
        }
    }

    private static class FsVolumeReferenceImpl
    implements FsVolumeReference {
        private FsVolumeImpl volume;

        FsVolumeReferenceImpl(FsVolumeImpl volume) throws ClosedChannelException {
            this.volume = volume;
            volume.reference();
        }

        @Override
        public void close() throws IOException {
            if (this.volume != null) {
                this.volume.unreference();
                this.volume = null;
            }
        }

        @Override
        public FsVolumeSpi getVolume() {
            return this.volume;
        }
    }
}

