/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.document;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nonnull;
import org.apache.jackrabbit.oak.plugins.document.Revision;
import org.apache.jackrabbit.oak.plugins.document.RevisionContext;
import org.apache.jackrabbit.oak.plugins.document.RevisionVector;
import org.apache.jackrabbit.oak.plugins.document.StableRevisionComparator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class CommitQueue {
    static final Logger LOG = LoggerFactory.getLogger(CommitQueue.class);
    static final long DEFAULT_SUSPEND_TIMEOUT = TimeUnit.MINUTES.toMillis(1L);
    private final SortedMap<Revision, Entry> commits = new TreeMap<Revision, Entry>(StableRevisionComparator.INSTANCE);
    private final Map<Semaphore, SuspendedCommit> suspendedCommits = Maps.newIdentityHashMap();
    private final RevisionContext context;
    private long suspendTimeout = Long.getLong("oak.documentMK.suspendTimeoutMillis", DEFAULT_SUSPEND_TIMEOUT);

    CommitQueue(@Nonnull RevisionContext context) {
        this.context = Preconditions.checkNotNull(context);
    }

    @Nonnull
    Revision createRevision() {
        return this.createRevisions(1).first();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nonnull
    SortedSet<Revision> createRevisions(int num) {
        Preconditions.checkArgument(num > 0);
        TreeSet<Revision> revs = new TreeSet<Revision>(StableRevisionComparator.INSTANCE);
        Revision rev = null;
        CommitQueue commitQueue = this;
        synchronized (commitQueue) {
            for (int i = 0; i < num; ++i) {
                rev = this.context.newRevision();
                revs.add(rev);
            }
            this.commits.put(rev, new Entry(rev));
        }
        LOG.debug("created commit {}", (Object)rev);
        return revs;
    }

    void done(@Nonnull Revision revision, @Nonnull Callback c) {
        Preconditions.checkNotNull(revision);
        this.waitUntilHeadOfQueue(revision, c);
    }

    void canceled(@Nonnull Revision rev) {
        this.removeCommit(rev);
        this.notifySuspendedCommits(rev);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean contains(@Nonnull Revision revision) {
        CommitQueue commitQueue = this;
        synchronized (commitQueue) {
            return this.commits.containsKey(Preconditions.checkNotNull(revision));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void suspendUntilAll(@Nonnull Set<Revision> conflictRevisions) {
        int addedRevisions;
        Semaphore s;
        Map<Semaphore, SuspendedCommit> map = this.suspendedCommits;
        synchronized (map) {
            RevisionVector headRevision = this.context.getHeadRevision();
            HashSet<Revision> afterHead = new HashSet<Revision>(conflictRevisions.size());
            for (Revision r : conflictRevisions) {
                if (!headRevision.isRevisionNewer(r)) continue;
                afterHead.add(r);
            }
            s = new Semaphore(0);
            this.suspendedCommits.put(s, new SuspendedCommit(s, afterHead));
            addedRevisions = afterHead.size();
        }
        try {
            s.tryAcquire(addedRevisions, this.suspendTimeout, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException e) {
            LOG.debug("The suspended thread has been interrupted", e);
        }
        finally {
            Map<Semaphore, SuspendedCommit> map2 = this.suspendedCommits;
            synchronized (map2) {
                this.suspendedCommits.remove(s);
            }
        }
    }

    void headRevisionChanged() {
        this.notifySuspendedCommits();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    int numSuspendedThreads() {
        Map<Semaphore, SuspendedCommit> map = this.suspendedCommits;
        synchronized (map) {
            return this.suspendedCommits.size();
        }
    }

    void setSuspendTimeoutMillis(long timeout) {
        this.suspendTimeout = timeout;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void notifySuspendedCommits() {
        Map<Semaphore, SuspendedCommit> map = this.suspendedCommits;
        synchronized (map) {
            if (this.suspendedCommits.isEmpty()) {
                return;
            }
            RevisionVector headRevision = this.context.getHeadRevision();
            Iterator<SuspendedCommit> it = this.suspendedCommits.values().iterator();
            while (it.hasNext()) {
                SuspendedCommit suspended = it.next();
                if (!suspended.removeRevisionsVisibleFrom(headRevision) || !suspended.revisions.isEmpty()) continue;
                it.remove();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void notifySuspendedCommits(@Nonnull Revision revision) {
        Preconditions.checkNotNull(revision);
        Map<Semaphore, SuspendedCommit> map = this.suspendedCommits;
        synchronized (map) {
            if (this.suspendedCommits.isEmpty()) {
                return;
            }
            Iterator<SuspendedCommit> it = this.suspendedCommits.values().iterator();
            while (it.hasNext()) {
                SuspendedCommit suspended = it.next();
                if (!suspended.removeRevision(revision) || !suspended.revisions.isEmpty()) continue;
                it.remove();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void removeCommit(@Nonnull Revision rev) {
        CommitQueue commitQueue = this;
        synchronized (commitQueue) {
            boolean wasHead = this.commits.firstKey().equals(rev);
            this.commits.remove(rev);
            LOG.debug("removed commit {}, wasHead={}", (Object)rev, (Object)wasHead);
            if (wasHead) {
                this.notifyHead();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void waitUntilHeadOfQueue(@Nonnull Revision rev, @Nonnull Callback c) {
        Entry commitEntry;
        boolean isHead;
        assert (!this.commits.isEmpty());
        CommitQueue commitQueue = this;
        synchronized (commitQueue) {
            isHead = this.commits.firstKey().equals(rev);
            commitEntry = (Entry)this.commits.get(rev);
        }
        if (!isHead) {
            LOG.debug("not head: {}, waiting...", (Object)rev);
            commitEntry.await();
        }
        commitQueue = this;
        synchronized (commitQueue) {
            this.commits.remove(rev);
            try {
                LOG.debug("removed {}, head is now {}", (Object)rev, (Object)(this.commits.isEmpty() ? null : this.commits.firstKey()));
                c.headOfQueue(rev);
            }
            finally {
                this.notifyHead();
            }
        }
    }

    private void notifyHead() {
        if (!this.commits.isEmpty()) {
            LOG.debug("release {}", (Object)this.commits.firstKey());
            ((Entry)this.commits.get(this.commits.firstKey())).release();
        }
    }

    private class SuspendedCommit {
        private final Semaphore semaphore;
        private final Set<Revision> revisions;

        private SuspendedCommit(Semaphore semaphore, Set<Revision> revisions) {
            this.semaphore = semaphore;
            this.revisions = revisions;
        }

        private boolean removeRevisionsVisibleFrom(RevisionVector revision) {
            Iterator<Revision> it = this.revisions.iterator();
            boolean removed = false;
            while (it.hasNext()) {
                if (revision.isRevisionNewer(it.next())) continue;
                it.remove();
                this.semaphore.release();
                removed = true;
            }
            return removed;
        }

        private boolean removeRevision(Revision r) {
            if (this.revisions.remove(r)) {
                this.semaphore.release();
                return true;
            }
            return false;
        }
    }

    private static final class Entry {
        private final Revision revision;
        private final CountDownLatch latch = new CountDownLatch(1);

        Entry(Revision revision) {
            this.revision = revision;
        }

        void release() {
            this.latch.countDown();
        }

        void await() {
            while (true) {
                try {
                    LOG.debug("awaiting {}", (Object)this.revision);
                    this.latch.await();
                }
                catch (InterruptedException interruptedException) {
                    continue;
                }
                break;
            }
        }
    }

    static interface Callback {
        public void headOfQueue(@Nonnull Revision var1);
    }
}

