/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.document;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import org.apache.jackrabbit.oak.plugins.document.DocumentStore;
import org.apache.jackrabbit.oak.plugins.document.NodeDocument;
import org.apache.jackrabbit.oak.plugins.document.Range;
import org.apache.jackrabbit.oak.plugins.document.Revision;
import org.apache.jackrabbit.oak.plugins.document.RevisionContext;
import org.apache.jackrabbit.oak.plugins.document.RevisionVector;
import org.apache.jackrabbit.oak.plugins.document.StableRevisionComparator;
import org.apache.jackrabbit.oak.plugins.document.UpdateOp;
import org.apache.jackrabbit.oak.plugins.document.UpdateUtils;
import org.apache.jackrabbit.oak.plugins.document.memory.MemoryDocumentStore;
import org.apache.jackrabbit.oak.plugins.document.util.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class SplitOperations {
    private static final Logger LOG = LoggerFactory.getLogger(SplitOperations.class);
    private static final int GARBAGE_LIMIT = Integer.getInteger("oak.documentMK.garbage.limit", 1000);
    private static final DocumentStore STORE = new MemoryDocumentStore();
    private final NodeDocument doc;
    private final String path;
    private final String id;
    private final Revision headRevision;
    private final RevisionContext context;
    private final int numRevsThreshold;
    private Revision high;
    private Revision low;
    private int numValues;
    private Map<String, NavigableMap<Revision, String>> committedChanges;
    private Set<Revision> changes;
    private Map<String, Set<Revision>> garbage;
    private int garbageCount = 0;
    private Set<Revision> mostRecentRevs;
    private Set<Revision> splitRevs;
    private List<UpdateOp> splitOps;
    private UpdateOp main;

    private SplitOperations(@Nonnull NodeDocument doc, @Nonnull RevisionContext context, @Nonnull RevisionVector headRevision, int numRevsThreshold) {
        this.doc = Preconditions.checkNotNull(doc);
        this.context = Preconditions.checkNotNull(context);
        this.path = doc.getPath();
        this.id = doc.getId();
        this.headRevision = Preconditions.checkNotNull(headRevision).getRevision(context.getClusterId());
        this.numRevsThreshold = numRevsThreshold;
    }

    @Nonnull
    static List<UpdateOp> forDocument(@Nonnull NodeDocument doc, @Nonnull RevisionContext context, @Nonnull RevisionVector headRevision, int numRevsThreshold) {
        if (doc.isSplitDocument()) {
            throw new IllegalArgumentException("Not a main document: " + doc.getId());
        }
        return new SplitOperations(doc, context, headRevision, numRevsThreshold).create();
    }

    private List<UpdateOp> create() {
        if (!this.considerSplit()) {
            return Collections.emptyList();
        }
        this.splitOps = Lists.newArrayList();
        this.mostRecentRevs = Sets.newHashSet();
        this.splitRevs = Sets.newHashSet();
        this.garbage = Maps.newHashMap();
        this.changes = Sets.newHashSet();
        this.committedChanges = Maps.newHashMap();
        this.collectLocalChanges(this.committedChanges, this.changes);
        this.populateSplitRevs();
        this.collectRevisionsAndCommitRoot();
        this.main = this.createSplitOps();
        this.createIntermediateDocs();
        this.disconnectStalePrevDocs();
        this.removeGarbage();
        if (this.main != null) {
            this.splitOps.add(this.main);
        }
        return this.splitOps;
    }

    private boolean considerSplit() {
        NavigableMap<Revision, Range> previous = this.doc.getPreviousRanges();
        return this.doc.getLocalRevisions().size() + this.doc.getLocalCommitRoot().size() > this.numRevsThreshold || this.doc.getMemory() >= 262144 || previous.size() >= 10 || !this.doc.getStalePrev().isEmpty();
    }

    private void populateSplitRevs() {
        for (NavigableMap<Revision, String> splitMap : this.committedChanges.values()) {
            if (!splitMap.isEmpty()) {
                Revision r = (Revision)splitMap.lastKey();
                splitMap.remove(r);
                this.splitRevs.addAll(splitMap.keySet());
                this.mostRecentRevs.add(r);
            }
            if (splitMap.isEmpty()) continue;
            this.trackHigh((Revision)splitMap.lastKey());
            this.trackLow((Revision)splitMap.firstKey());
            this.numValues += splitMap.size();
        }
    }

    private void collectRevisionsAndCommitRoot() {
        TreeMap<Revision, String> revisions = new TreeMap<Revision, String>(StableRevisionComparator.INSTANCE);
        for (Map.Entry<Revision, String> entry : this.doc.getLocalRevisions().entrySet()) {
            if (this.splitRevs.contains(entry.getKey())) {
                revisions.put(entry.getKey(), entry.getValue());
                ++this.numValues;
                continue;
            }
            if (this.context.getClusterId() != entry.getKey().getClusterId() || !this.doc.isCommitted(entry.getKey()) || this.mostRecentRevs.contains(entry.getKey())) continue;
            revisions.put(entry.getKey(), entry.getValue());
            ++this.numValues;
            this.trackHigh(entry.getKey());
            this.trackLow(entry.getKey());
        }
        this.committedChanges.put("_revisions", revisions);
        TreeMap<Revision, String> commitRoot = new TreeMap<Revision, String>(StableRevisionComparator.INSTANCE);
        boolean mostRecent = true;
        for (Map.Entry<Revision, String> entry : this.doc.getLocalCommitRoot().entrySet()) {
            Revision r = entry.getKey();
            if (this.splitRevs.contains(r)) {
                commitRoot.put(r, entry.getValue());
                ++this.numValues;
                continue;
            }
            if (r.getClusterId() != this.context.getClusterId() || this.changes.contains(r)) continue;
            if (mostRecent && this.doc.isCommitted(r)) {
                mostRecent = false;
                continue;
            }
            if (!this.isGarbage(r)) continue;
            this.addGarbage(r, "_commitRoot");
        }
        this.committedChanges.put("_commitRoot", commitRoot);
    }

    private void createIntermediateDocs() {
        Map<Integer, List<Range>> prevHisto = this.getPreviousDocsHistogram();
        for (Map.Entry<Integer, List<Range>> entry : prevHisto.entrySet()) {
            if (entry.getValue().size() < 10) continue;
            if (this.main == null) {
                this.main = new UpdateOp(this.id, false);
            }
            Revision h = null;
            Revision l = null;
            for (Range r : entry.getValue()) {
                if (h == null || r.high.compareRevisionTime(h) > 0) {
                    h = r.high;
                }
                if (l == null || l.compareRevisionTime(r.low) > 0) {
                    l = r.low;
                }
                NodeDocument.removePrevious(this.main, r);
            }
            if (h == null || l == null) {
                throw new IllegalStateException();
            }
            String prevPath = Utils.getPreviousPathFor(this.path, h, entry.getKey() + 1);
            String prevId = Utils.getIdFromPath(prevPath);
            UpdateOp intermediate = new UpdateOp(prevId, true);
            intermediate.set("_id", prevId);
            if (Utils.isLongPath(prevPath)) {
                intermediate.set("_path", prevPath);
            }
            NodeDocument.setPrevious(this.main, new Range(h, l, entry.getKey() + 1));
            for (Range r : entry.getValue()) {
                NodeDocument.setPrevious(intermediate, r);
            }
            SplitOperations.setIntermediateDocProps(intermediate, h);
            this.splitOps.add(intermediate);
        }
    }

    @CheckForNull
    private UpdateOp createSplitOps() {
        UpdateOp main = null;
        if (this.high != null && this.low != null && (this.numValues >= this.numRevsThreshold || this.doc.getMemory() > 262144)) {
            main = new UpdateOp(this.id, false);
            NodeDocument.setPrevious(main, new Range(this.high, this.low, 0));
            String oldPath = Utils.getPreviousPathFor(this.path, this.high, 0);
            UpdateOp old = new UpdateOp(Utils.getIdFromPath(oldPath), true);
            old.set("_id", old.getId());
            if (Utils.isLongPath(oldPath)) {
                old.set("_path", oldPath);
            }
            for (String property : this.committedChanges.keySet()) {
                NavigableMap<Revision, String> splitMap = this.committedChanges.get(property);
                for (Map.Entry entry : splitMap.entrySet()) {
                    Revision r = (Revision)entry.getKey();
                    if (NodeDocument.isRevisionsEntry(property) || NodeDocument.isCommitRootEntry(property)) {
                        if (!this.mostRecentRevs.contains(r)) {
                            main.removeMapEntry(property, r);
                        }
                    } else {
                        main.removeMapEntry(property, r);
                    }
                    old.setMapEntry(property, r, (String)entry.getValue());
                }
            }
            NodeDocument oldDoc = new NodeDocument(STORE);
            UpdateUtils.applyChanges(oldDoc, old);
            SplitOperations.setSplitDocProps(this.doc, oldDoc, old, this.high);
            if ((float)oldDoc.getMemory() > (float)this.doc.getMemory() * 0.3f || this.numValues >= this.numRevsThreshold) {
                this.splitOps.add(old);
            } else {
                main = null;
            }
        }
        return main;
    }

    private Map<Integer, List<Range>> getPreviousDocsHistogram() {
        HashMap<Integer, List<Range>> prevHisto = Maps.newHashMap();
        for (Map.Entry entry : this.doc.getPreviousRanges().entrySet()) {
            Revision rev = (Revision)entry.getKey();
            if (rev.getClusterId() != this.context.getClusterId()) continue;
            Range r = (Range)entry.getValue();
            ArrayList<Range> list = (ArrayList<Range>)prevHisto.get(r.getHeight());
            if (list == null) {
                list = new ArrayList<Range>();
                prevHisto.put(r.getHeight(), list);
            }
            list.add(r);
        }
        return prevHisto;
    }

    private void collectLocalChanges(Map<String, NavigableMap<Revision, String>> committedLocally, Set<Revision> changes) {
        for (String property : Sets.filter(this.doc.keySet(), Utils.PROPERTY_OR_DELETED)) {
            TreeMap splitMap = new TreeMap(StableRevisionComparator.INSTANCE);
            committedLocally.put(property, splitMap);
            SortedMap<Revision, String> valueMap = this.doc.getLocalMap(property);
            for (Map.Entry entry : valueMap.entrySet()) {
                Revision rev = (Revision)entry.getKey();
                if (rev.getClusterId() != this.context.getClusterId()) continue;
                changes.add(rev);
                if (this.doc.isCommitted(rev)) {
                    splitMap.put(rev, entry.getValue());
                    continue;
                }
                if (!this.isGarbage(rev)) continue;
                this.addGarbage(rev, property);
            }
        }
    }

    private boolean isGarbage(Revision rev) {
        if (this.headRevision.compareRevisionTime(rev) <= 0) {
            return false;
        }
        return this.context.getBranches().getBranchCommit(rev) == null;
    }

    private void addGarbage(Revision rev, String property) {
        if (this.garbageCount > GARBAGE_LIMIT) {
            return;
        }
        Set<Revision> revisions = this.garbage.get(property);
        if (revisions == null) {
            revisions = Sets.newHashSet();
            this.garbage.put(property, revisions);
        }
        if (revisions.add(rev)) {
            ++this.garbageCount;
        }
    }

    private void disconnectStalePrevDocs() {
        NavigableMap<Revision, Range> ranges = this.doc.getPreviousRanges(true);
        for (Map.Entry<Revision, String> entry : this.doc.getStalePrev().entrySet()) {
            Revision r = entry.getKey();
            if (r.getClusterId() != this.context.getClusterId()) continue;
            if (this.main == null) {
                this.main = new UpdateOp(this.id, false);
            }
            NodeDocument.removeStalePrevious(this.main, r);
            if (ranges.containsKey(r) && entry.getValue().equals(String.valueOf(((Range)ranges.get((Object)r)).height))) {
                NodeDocument.removePrevious(this.main, r);
                continue;
            }
            int height = Integer.parseInt(entry.getValue());
            NodeDocument intermediate = this.doc.findPrevReferencingDoc(r, height);
            if (intermediate == null) {
                LOG.warn("Split document {} not referenced anymore. Main document is {}", (Object)Utils.getPreviousIdFor(this.doc.getPath(), r, height), (Object)this.id);
                continue;
            }
            UpdateOp op = new UpdateOp(intermediate.getId(), false);
            NodeDocument.removePrevious(op, r);
            this.splitOps.add(op);
        }
    }

    private void removeGarbage() {
        if (this.garbage.isEmpty()) {
            return;
        }
        if (this.main == null) {
            this.main = new UpdateOp(this.id, false);
        }
        for (Map.Entry<String, Set<Revision>> entry : this.garbage.entrySet()) {
            for (Revision r : entry.getValue()) {
                this.main.removeMapEntry(entry.getKey(), r);
                if (!Utils.PROPERTY_OR_DELETED.apply(entry.getKey())) continue;
                NodeDocument.removeCommitRoot(this.main, r);
                NodeDocument.removeRevision(this.main, r);
            }
        }
    }

    private void trackHigh(Revision r) {
        if (this.high == null || r.compareRevisionTime(this.high) > 0) {
            this.high = r;
        }
    }

    private void trackLow(Revision r) {
        if (this.low == null || this.low.compareRevisionTime(r) > 0) {
            this.low = r;
        }
    }

    private static void setSplitDocProps(NodeDocument mainDoc, NodeDocument oldDoc, UpdateOp old, Revision maxRev) {
        SplitOperations.setSplitDocMaxRev(old, maxRev);
        NodeDocument.SplitDocType type = NodeDocument.SplitDocType.DEFAULT;
        if (!mainDoc.hasChildren()) {
            type = NodeDocument.SplitDocType.DEFAULT_LEAF;
        } else if (oldDoc.getLocalRevisions().isEmpty()) {
            type = NodeDocument.SplitDocType.COMMIT_ROOT_ONLY;
        }
        if (mainDoc.hasBinary()) {
            NodeDocument.setHasBinary(old);
        }
        SplitOperations.setSplitDocType(old, type);
    }

    private static void setIntermediateDocProps(UpdateOp intermediate, Revision maxRev) {
        SplitOperations.setSplitDocMaxRev(intermediate, maxRev);
        SplitOperations.setSplitDocType(intermediate, NodeDocument.SplitDocType.INTERMEDIATE);
    }

    private static void setSplitDocType(@Nonnull UpdateOp op, @Nonnull NodeDocument.SplitDocType type) {
        Preconditions.checkNotNull(op).set("_sdType", type.type);
    }

    private static void setSplitDocMaxRev(@Nonnull UpdateOp op, @Nonnull Revision maxRev) {
        Preconditions.checkNotNull(op).set("_sdMaxRevTime", NodeDocument.getModifiedInSecs(maxRev.getTimestamp()));
    }
}

