/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.segment;

import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import org.apache.jackrabbit.oak.plugins.segment.RecordId;
import org.apache.jackrabbit.oak.plugins.segment.RecordType;
import org.apache.jackrabbit.oak.plugins.segment.RecordWriters;
import org.apache.jackrabbit.oak.plugins.segment.Segment;
import org.apache.jackrabbit.oak.plugins.segment.SegmentId;
import org.apache.jackrabbit.oak.plugins.segment.SegmentOverflowException;
import org.apache.jackrabbit.oak.plugins.segment.SegmentStore;
import org.apache.jackrabbit.oak.plugins.segment.SegmentTracker;
import org.apache.jackrabbit.oak.plugins.segment.SegmentVersion;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class SegmentBufferWriter {
    private static final Logger LOG = LoggerFactory.getLogger(SegmentBufferWriter.class);
    private final Map<RecordId, RecordType> roots = Maps.newLinkedHashMap();
    private final List<RecordId> blobrefs = Lists.newArrayList();
    private final SegmentStore store;
    private final SegmentVersion version;
    private final String wid;
    private final SegmentTracker tracker;
    private byte[] buffer;
    private Segment segment;
    private int length;
    private int position;

    public SegmentBufferWriter(SegmentStore store, SegmentVersion version, String wid) throws IOException {
        this.store = store;
        this.version = version;
        this.wid = wid == null ? "w-" + System.identityHashCode(this) : wid;
        this.tracker = store.getTracker();
        this.buffer = SegmentBufferWriter.createNewBuffer(version);
        this.newSegment(this.wid);
    }

    private void newSegment(String wid) throws IOException {
        this.segment = new Segment(this.tracker, this.buffer);
        String metaInfo = "{\"wid\":\"" + wid + '\"' + ",\"sno\":" + this.tracker.getNextSegmentNo() + ",\"gc\":" + this.tracker.getCompactionMap().getGeneration() + ",\"t\":" + System.currentTimeMillis() + "}";
        byte[] data = metaInfo.getBytes(Charsets.UTF_8);
        RecordWriters.newValueWriter(data.length, data).write(this);
    }

    static byte[] createNewBuffer(SegmentVersion v) {
        byte[] buffer = new byte[262144];
        buffer[0] = 48;
        buffer[1] = 97;
        buffer[2] = 75;
        buffer[3] = SegmentVersion.asByte(v);
        buffer[4] = 0;
        buffer[5] = 0;
        return buffer;
    }

    public void writeByte(byte value) {
        this.buffer[this.position++] = value;
    }

    public void writeShort(short value) {
        this.buffer[this.position++] = (byte)(value >> 8);
        this.buffer[this.position++] = (byte)value;
    }

    public void writeInt(int value) {
        this.buffer[this.position++] = (byte)(value >> 24);
        this.buffer[this.position++] = (byte)(value >> 16);
        this.buffer[this.position++] = (byte)(value >> 8);
        this.buffer[this.position++] = (byte)value;
    }

    public void writeLong(long value) {
        this.writeInt((int)(value >> 32));
        this.writeInt((int)value);
    }

    public void writeRecordId(RecordId listId) {
        Preconditions.checkNotNull(listId);
        this.roots.remove(listId);
        int offset = listId.getOffset();
        Preconditions.checkState(0 <= offset && offset < 262144);
        Preconditions.checkState(offset == Segment.align(offset, 4));
        this.buffer[this.position++] = (byte)this.getSegmentRef(listId.getSegmentId());
        this.buffer[this.position++] = (byte)(offset >> 10);
        this.buffer[this.position++] = (byte)(offset >> 2);
    }

    private int getSegmentRef(SegmentId segmentId) {
        int refCount = this.segment.getRefCount();
        if (refCount > 255) {
            throw new SegmentOverflowException("Segment cannot have more than 255 references " + this.segment.getSegmentId());
        }
        for (int index = 0; index < refCount; ++index) {
            if (!segmentId.equals(this.segment.getRefId(index))) continue;
            return index;
        }
        ByteBuffer.wrap(this.buffer, refCount * 16, 16).putLong(segmentId.getMostSignificantBits()).putLong(segmentId.getLeastSignificantBits());
        this.buffer[5] = (byte)refCount;
        return refCount;
    }

    public void writeBytes(byte[] data, int offset, int length) {
        System.arraycopy(data, offset, this.buffer, this.position, length);
        this.position += length;
    }

    public void addBlobRef(RecordId blobId) {
        this.blobrefs.add(blobId);
    }

    public void flush() throws IOException {
        if (this.length > 0) {
            ByteBuffer data;
            int offset;
            int refcount = this.segment.getRefCount();
            int rootcount = this.roots.size();
            this.buffer[6] = (byte)(rootcount >> 8);
            this.buffer[7] = (byte)rootcount;
            int blobrefcount = this.blobrefs.size();
            this.buffer[8] = (byte)(blobrefcount >> 8);
            this.buffer[9] = (byte)blobrefcount;
            this.length = Segment.align(refcount * 16 + rootcount * 3 + blobrefcount * 2 + this.length, 16);
            Preconditions.checkState(this.length <= this.buffer.length);
            int pos = refcount * 16;
            if (pos + this.length <= this.buffer.length) {
                System.arraycopy(this.buffer, 0, this.buffer, this.buffer.length - this.length, pos);
                pos += this.buffer.length - this.length;
            } else {
                this.length = this.buffer.length;
            }
            for (Map.Entry<RecordId, RecordType> entry : this.roots.entrySet()) {
                offset = entry.getKey().getOffset();
                this.buffer[pos++] = (byte)entry.getValue().ordinal();
                this.buffer[pos++] = (byte)(offset >> 10);
                this.buffer[pos++] = (byte)(offset >> 2);
            }
            for (RecordId blobref : this.blobrefs) {
                offset = blobref.getOffset();
                this.buffer[pos++] = (byte)(offset >> 10);
                this.buffer[pos++] = (byte)(offset >> 2);
            }
            SegmentId segmentId = this.segment.getSegmentId();
            int segmentOffset = this.buffer.length - this.length;
            LOG.debug("Writing data segment {} ({} bytes)", (Object)segmentId, (Object)this.length);
            this.store.writeSegment(segmentId, this.buffer, segmentOffset, this.length);
            if (segmentOffset > 4096) {
                data = ByteBuffer.allocate(this.length);
                data.put(this.buffer, segmentOffset, this.length);
                data.rewind();
            } else {
                data = ByteBuffer.wrap(this.buffer, segmentOffset, this.length);
            }
            this.tracker.setSegment(segmentId, new Segment(this.tracker, segmentId, data));
            this.buffer = SegmentBufferWriter.createNewBuffer(this.version);
            this.roots.clear();
            this.blobrefs.clear();
            this.length = 0;
            this.position = this.buffer.length;
            this.newSegment(this.wid);
        }
    }

    public RecordId prepare(RecordType type, int size, Collection<RecordId> ids) throws IOException {
        Preconditions.checkArgument(size >= 0);
        Preconditions.checkNotNull(ids);
        int idCount = ids.size();
        int recordSize = Segment.align(size + idCount * 3, 4);
        int refCount = this.segment.getRefCount() + idCount;
        int blobRefCount = this.blobrefs.size() + 1;
        int rootCount = this.roots.size() + 1;
        int headerSize = refCount * 16 + rootCount * 3 + blobRefCount * 2;
        int segmentSize = Segment.align(headerSize + recordSize + this.length, 16);
        if (segmentSize > this.buffer.length - 1 || refCount > 255) {
            refCount -= idCount;
            HashSet<SegmentId> segmentIds = Sets.newHashSet();
            HashSet<RecordId> notRoots = new HashSet<RecordId>();
            for (RecordId recordId : ids) {
                SegmentId segmentId = recordId.getSegmentId();
                if (!segmentId.equals(this.segment.getSegmentId())) {
                    segmentIds.add(segmentId);
                    continue;
                }
                if (!this.roots.containsKey(recordId)) continue;
                notRoots.add(recordId);
            }
            rootCount -= notRoots.size();
            if (!segmentIds.isEmpty()) {
                for (int refid = 1; refid < refCount; ++refid) {
                    segmentIds.remove(this.segment.getRefId(refid));
                }
                refCount += segmentIds.size();
            }
            headerSize = refCount * 16 + rootCount * 3 + blobRefCount * 2;
            segmentSize = Segment.align(headerSize + recordSize + this.length, 16);
        }
        if (segmentSize > this.buffer.length - 1 || blobRefCount > 65535 || rootCount > 65535 || refCount > 255) {
            this.flush();
        }
        this.length += recordSize;
        this.position = this.buffer.length - this.length;
        Preconditions.checkState(this.position >= 0);
        RecordId id = new RecordId(this.segment.getSegmentId(), this.position);
        this.roots.put(id, type);
        return id;
    }
}

