/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.segment;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.apache.jackrabbit.oak.api.Type;
import org.apache.jackrabbit.oak.plugins.segment.ListRecord;
import org.apache.jackrabbit.oak.plugins.segment.MapEntry;
import org.apache.jackrabbit.oak.plugins.segment.MapRecord;
import org.apache.jackrabbit.oak.plugins.segment.PropertyTemplate;
import org.apache.jackrabbit.oak.plugins.segment.RecordId;
import org.apache.jackrabbit.oak.plugins.segment.Segment;
import org.apache.jackrabbit.oak.plugins.segment.SegmentNodeState;
import org.apache.jackrabbit.oak.plugins.segment.SegmentVersion;
import org.apache.jackrabbit.oak.plugins.segment.Template;
import org.apache.jackrabbit.oak.spi.state.ChildNodeEntry;
import org.apache.jackrabbit.oak.spi.state.NodeState;

public class SegmentParser {
    protected void onNode(RecordId parentId, RecordId nodeId) {
        this.parseNode(nodeId);
    }

    protected void onTemplate(RecordId parentId, RecordId templateId) {
        this.parseTemplate(templateId);
    }

    protected void onMap(RecordId parentId, RecordId mapId, MapRecord map) {
        this.parseMap(parentId, mapId, map);
    }

    protected void onMapDiff(RecordId parentId, RecordId mapId, MapRecord map) {
        this.parseMapDiff(mapId, map);
    }

    protected void onMapLeaf(RecordId parentId, RecordId mapId, MapRecord map) {
        this.parseMapLeaf(mapId, map);
    }

    protected void onMapBranch(RecordId parentId, RecordId mapId, MapRecord map) {
        this.parseMapBranch(mapId, map);
    }

    protected void onProperty(RecordId parentId, RecordId propertyId, PropertyTemplate template) {
        this.parseProperty(parentId, propertyId, template);
    }

    protected void onValue(RecordId parentId, RecordId valueId, Type<?> type) {
        this.parseValue(parentId, valueId, type);
    }

    protected void onBlob(RecordId parentId, RecordId blobId) {
        this.parseBlob(blobId);
    }

    protected void onString(RecordId parentId, RecordId stringId) {
        this.parseString(stringId);
    }

    protected void onList(RecordId parentId, RecordId listId, int count) {
        this.parseList(parentId, listId, count);
    }

    protected void onListBucket(RecordId parentId, RecordId listId, int index, int count, int capacity) {
        this.parseListBucket(listId, index, count, capacity);
    }

    public NodeInfo parseNode(RecordId nodeId) {
        int size = 0;
        int nodeCount = 0;
        int propertyCount = 0;
        Segment segment = nodeId.getSegment();
        int offset = nodeId.getOffset();
        RecordId templateId = segment.readRecordId(offset);
        this.onTemplate(nodeId, templateId);
        Template template = segment.readTemplate(templateId);
        if (template.getChildName() == "") {
            RecordId childMapId = segment.readRecordId(offset + 3);
            MapRecord childMap = segment.readMap(childMapId);
            this.onMap(nodeId, childMapId, childMap);
            for (ChildNodeEntry childNodeEntry : childMap.getEntries()) {
                NodeState child = childNodeEntry.getNodeState();
                if (!(child instanceof SegmentNodeState)) continue;
                RecordId childId = ((SegmentNodeState)child).getRecordId();
                this.onNode(nodeId, childId);
                ++nodeCount;
            }
        } else if (template.getChildName() != Template.ZERO_CHILD_NODES) {
            RecordId childId = segment.readRecordId(offset + 3);
            this.onNode(nodeId, childId);
            ++nodeCount;
        }
        int ids = template.getChildName() == Template.ZERO_CHILD_NODES ? 1 : 2;
        size += ids * 3;
        PropertyTemplate[] propertyTemplates = template.getPropertyTemplates();
        if (segment.getSegmentVersion().onOrAfter(SegmentVersion.V_11)) {
            if (propertyTemplates.length > 0) {
                size += 3;
                RecordId id = segment.readRecordId(offset + ids * 3);
                ListRecord listRecord = new ListRecord(id, propertyTemplates.length);
                for (int i = 0; i < propertyTemplates.length; ++i) {
                    RecordId propertyId = listRecord.getEntry(i);
                    this.onProperty(nodeId, propertyId, propertyTemplates[i]);
                    ++propertyCount;
                }
                this.onList(nodeId, id, propertyTemplates.length);
            }
        } else {
            for (PropertyTemplate propertyTemplate : propertyTemplates) {
                size += 3;
                RecordId propertyId = segment.readRecordId(offset + ids++ * 3);
                this.onProperty(nodeId, propertyId, propertyTemplate);
                ++propertyCount;
            }
        }
        return new NodeInfo(nodeId, nodeCount, propertyCount, size);
    }

    public TemplateInfo parseTemplate(RecordId templateId) {
        int offset;
        int size = 0;
        Segment segment = templateId.getSegment();
        int head = segment.readInt((offset = templateId.getOffset()) + size);
        boolean hasPrimaryType = (head & Integer.MIN_VALUE) != 0;
        boolean hasMixinTypes = (head & 0x40000000) != 0;
        boolean zeroChildNodes = (head & 0x20000000) != 0;
        boolean manyChildNodes = (head & 0x10000000) != 0;
        int mixinCount = head >> 18 & 0x3FF;
        int propertyCount = head & 0x3FFFF;
        size += 4;
        if (hasPrimaryType) {
            RecordId primaryId = segment.readRecordId(offset + size);
            this.onString(templateId, primaryId);
            size += 3;
        }
        if (hasMixinTypes) {
            for (int i = 0; i < mixinCount; ++i) {
                RecordId mixinId = segment.readRecordId(offset + size);
                this.onString(templateId, mixinId);
                size += 3;
            }
        }
        if (!zeroChildNodes && !manyChildNodes) {
            RecordId childNameId = segment.readRecordId(offset + size);
            this.onString(templateId, childNameId);
            size += 3;
        }
        if (segment.getSegmentVersion().onOrAfter(SegmentVersion.V_11)) {
            if (propertyCount > 0) {
                RecordId listId = segment.readRecordId(offset + size);
                size += 3;
                ListRecord propertyNames = new ListRecord(listId, propertyCount);
                for (int i = 0; i < propertyCount; ++i) {
                    RecordId propertyNameId = propertyNames.getEntry(i);
                    ++size;
                    this.onString(templateId, propertyNameId);
                }
                this.onList(templateId, listId, propertyCount);
            }
        } else {
            for (int i = 0; i < propertyCount; ++i) {
                RecordId propertyNameId = segment.readRecordId(offset + size);
                size += 3;
                ++size;
                this.onString(templateId, propertyNameId);
            }
        }
        return new TemplateInfo(templateId, hasPrimaryType, hasMixinTypes, zeroChildNodes, manyChildNodes, mixinCount, propertyCount, size);
    }

    public MapInfo parseMap(RecordId parentId, RecordId mapId, MapRecord map) {
        if (map.isDiff()) {
            this.onMapDiff(parentId, mapId, map);
        } else if (map.isLeaf()) {
            this.onMapLeaf(parentId, mapId, map);
        } else {
            this.onMapBranch(parentId, mapId, map);
        }
        return new MapInfo(mapId, -1);
    }

    public MapInfo parseMapDiff(RecordId mapId, MapRecord map) {
        int size = 4;
        size += 4;
        size += 3;
        size += 3;
        RecordId baseId = mapId.getSegment().readRecordId(mapId.getOffset() + 8 + 6);
        this.onMap(mapId, baseId, new MapRecord(baseId));
        return new MapInfo(mapId, size += 3);
    }

    public MapInfo parseMapLeaf(RecordId mapId, MapRecord map) {
        int size = 4;
        size += map.size() * 4;
        for (MapEntry entry : map.getEntries()) {
            size += 6;
            this.onString(mapId, entry.getKey());
        }
        return new MapInfo(mapId, size);
    }

    public MapInfo parseMapBranch(RecordId mapId, MapRecord map) {
        int size = 4;
        size += 4;
        for (MapRecord bucket : map.getBuckets()) {
            if (bucket == null) continue;
            size += 3;
            this.onMap(map.getRecordId(), bucket.getRecordId(), bucket);
        }
        return new MapInfo(mapId, size);
    }

    public PropertyInfo parseProperty(RecordId parentId, RecordId propertyId, PropertyTemplate template) {
        int size = 0;
        int count = -1;
        Segment segment = propertyId.getSegment();
        int offset = propertyId.getOffset();
        Type<?> type = template.getType();
        if (type.isArray()) {
            count = segment.readInt(offset);
            size += 4;
            if (count > 0) {
                RecordId listId = segment.readRecordId(offset + 4);
                size += 3;
                for (RecordId valueId : new ListRecord(listId, count).getEntries()) {
                    this.onValue(propertyId, valueId, type.getBaseType());
                }
                this.onList(propertyId, listId, count);
            }
        } else {
            this.onValue(parentId, propertyId, type);
        }
        return new PropertyInfo(propertyId, count, size);
    }

    public ValueInfo parseValue(RecordId parentId, RecordId valueId, Type<?> type) {
        Preconditions.checkArgument(!type.isArray());
        if (type == Type.BINARY) {
            this.onBlob(parentId, valueId);
        } else {
            this.onString(parentId, valueId);
        }
        return new ValueInfo(valueId, type);
    }

    public BlobInfo parseBlob(RecordId blobId) {
        BlobType blobType;
        int offset;
        int size = 0;
        Segment segment = blobId.getSegment();
        byte head = segment.readByte(offset = blobId.getOffset());
        if ((head & 0x80) == 0) {
            size += 1 + head;
            blobType = BlobType.SMALL;
        } else if ((head & 0xC0) == 128) {
            int length = (segment.readShort(offset) & 0x3FFF) + 128;
            size += 2 + length;
            blobType = BlobType.MEDIUM;
        } else if ((head & 0xE0) == 192) {
            long length = (segment.readLong(offset) & 0x1FFFFFFFFFFFFFFFL) + 16512L;
            int count = (int)((length + 4096L - 1L) / 4096L);
            RecordId listId = segment.readRecordId(offset + 8);
            this.onList(blobId, listId, count);
            size = (int)((long)size + (11L + length));
            blobType = BlobType.LONG;
        } else if ((head & 0xF0) == 224) {
            int length = (head & 0xF) << 8 | segment.readByte(offset + 1) & 0xFF;
            size += 2 + length;
            blobType = BlobType.EXTERNAL;
        } else {
            throw new IllegalStateException(String.format("Unexpected value record type: %02x", head & 0xFF));
        }
        return new BlobInfo(blobId, blobType, size);
    }

    public BlobInfo parseString(RecordId stringId) {
        BlobType blobType;
        int offset;
        int size = 0;
        Segment segment = stringId.getSegment();
        long length = segment.readLength(offset = stringId.getOffset());
        if (length < 128L) {
            size = (int)((long)size + (1L + length));
            blobType = BlobType.SMALL;
        } else if (length < 16512L) {
            size = (int)((long)size + (2L + length));
            blobType = BlobType.MEDIUM;
        } else if (length < Integer.MAX_VALUE) {
            int count = (int)((length + 4096L - 1L) / 4096L);
            RecordId listId = segment.readRecordId(offset + 8);
            this.onList(stringId, listId, count);
            size = (int)((long)size + (11L + length));
            blobType = BlobType.LONG;
        } else {
            throw new IllegalStateException("String is too long: " + length);
        }
        return new BlobInfo(stringId, blobType, size);
    }

    public ListInfo parseList(RecordId parentId, RecordId listId, int count) {
        if (count != 0) {
            this.onListBucket(parentId, listId, 0, count, count);
        }
        return new ListInfo(listId, count, SegmentParser.noOfListSlots(count) * 3);
    }

    public ListBucketInfo parseListBucket(RecordId listId, int index, int count, int capacity) {
        Segment segment = listId.getSegment();
        int bucketSize = 1;
        while (bucketSize * 255 < capacity) {
            bucketSize *= 255;
        }
        if (capacity == 1) {
            List<RecordId> entries = Collections.singletonList(listId);
            return new ListBucketInfo(listId, true, entries, 3);
        }
        if (bucketSize == 1) {
            ArrayList<RecordId> entries = Lists.newArrayListWithCapacity(count);
            for (int i = 0; i < count; ++i) {
                entries.add(segment.readRecordId(SegmentParser.getOffset(listId, index + i)));
            }
            return new ListBucketInfo(listId, true, entries, count * 3);
        }
        ArrayList<RecordId> entries = Lists.newArrayList();
        while (count > 0) {
            int bucketIndex = index / bucketSize;
            int bucketOffset = index % bucketSize;
            RecordId bucketId = segment.readRecordId(SegmentParser.getOffset(listId, bucketIndex));
            entries.add(bucketId);
            int c = Math.min(bucketSize, capacity - bucketIndex * bucketSize);
            int n = Math.min(c - bucketOffset, count);
            this.onListBucket(listId, bucketId, bucketOffset, n, c);
            index += n;
            count -= n;
        }
        return new ListBucketInfo(listId, false, entries, entries.size() * 3);
    }

    private static int getOffset(RecordId id, int ids) {
        return id.getOffset() + ids * 3;
    }

    private static int noOfListSlots(int size) {
        if (size <= 255) {
            return size;
        }
        int fullBuckets = size / 255;
        if (size % 255 > 1) {
            return size + SegmentParser.noOfListSlots(fullBuckets + 1);
        }
        return size + SegmentParser.noOfListSlots(fullBuckets);
    }

    public static class ListBucketInfo {
        public final RecordId listId;
        public final boolean leaf;
        public final List<RecordId> entries;
        public final int size;

        public ListBucketInfo(RecordId listId, boolean leaf, List<RecordId> entries, int size) {
            this.listId = listId;
            this.leaf = leaf;
            this.entries = entries;
            this.size = size;
        }
    }

    public static class ListInfo {
        public final RecordId listId;
        public final int count;
        public final int size;

        public ListInfo(RecordId listId, int count, int size) {
            this.listId = listId;
            this.count = count;
            this.size = size;
        }
    }

    public static class BlobInfo {
        public final RecordId blobId;
        public final BlobType blobType;
        public final int size;

        public BlobInfo(RecordId blobId, BlobType blobType, int size) {
            this.blobId = blobId;
            this.blobType = blobType;
            this.size = size;
        }
    }

    public static class ValueInfo {
        public final RecordId valueId;
        public final Type<?> type;

        public ValueInfo(RecordId valueId, Type<?> type) {
            this.valueId = valueId;
            this.type = type;
        }
    }

    public static class PropertyInfo {
        public final RecordId propertyId;
        public final int count;
        public final int size;

        public PropertyInfo(RecordId propertyId, int count, int size) {
            this.propertyId = propertyId;
            this.count = count;
            this.size = size;
        }
    }

    public static class MapInfo {
        public final RecordId mapId;
        public final int size;

        public MapInfo(RecordId mapId, int size) {
            this.mapId = mapId;
            this.size = size;
        }
    }

    public static class TemplateInfo {
        public final RecordId templateId;
        public final boolean hasPrimaryType;
        public final boolean hasMixinType;
        public final boolean zeroChildNodes;
        public final boolean manyChildNodes;
        public final int mixinCount;
        public final int propertyCount;
        public final int size;

        public TemplateInfo(RecordId templateId, boolean hasPrimaryType, boolean hasMixinType, boolean zeroChildNodes, boolean manyChildNodes, int mixinCount, int propertyCount, int size) {
            this.templateId = templateId;
            this.hasPrimaryType = hasPrimaryType;
            this.hasMixinType = hasMixinType;
            this.zeroChildNodes = zeroChildNodes;
            this.manyChildNodes = manyChildNodes;
            this.mixinCount = mixinCount;
            this.propertyCount = propertyCount;
            this.size = size;
        }
    }

    public static class NodeInfo {
        public final RecordId nodeId;
        public final int nodeCount;
        public final int propertyCount;
        public final int size;

        public NodeInfo(RecordId nodeId, int nodeCount, int propertyCount, int size) {
            this.nodeId = nodeId;
            this.nodeCount = nodeCount;
            this.propertyCount = propertyCount;
            this.size = size;
        }
    }

    public static enum BlobType {
        SMALL,
        MEDIUM,
        LONG,
        EXTERNAL;

    }
}

