/**********************************************************************
Copyright (c) 2003 Andy Jefferson and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 
 
Contributors:
2004 Joerg Van Frantzius - changes to support a form of DDL output
2004 Erik Bengtson - dbinfo() mode
2004 Andy Jefferson - added "mapping" property to allow ORM files
2010 Andy Jefferson - rewritten the commandline interface to not need OMFContext etc
    ...
**********************************************************************/
package org.datanucleus.store.rdbms;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.TreeSet;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ObjectManagerFactoryImpl;
import org.datanucleus.UserTransaction;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.jdo.JDOPersistenceManager;
import org.datanucleus.jdo.JDOPersistenceManagerFactory;
import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.metadata.FileMetaData;
import org.datanucleus.metadata.IdentityStrategy;
import org.datanucleus.metadata.MetaData;
import org.datanucleus.metadata.MetaDataManager;
import org.datanucleus.metadata.SequenceMetaData;
import org.datanucleus.store.ExecutionContext;
import org.datanucleus.store.StoreManager;
import org.datanucleus.store.connection.ManagedConnection;
import org.datanucleus.store.mapped.DatastoreIdentifier;
import org.datanucleus.store.rdbms.adapter.RDBMSAdapter;
import org.datanucleus.store.rdbms.valuegenerator.SequenceTable;
import org.datanucleus.store.rdbms.valuegenerator.TableGenerator;
import org.datanucleus.util.ClassUtils;
import org.datanucleus.util.CommandLine;
import org.datanucleus.util.Localiser;
import org.datanucleus.util.NucleusLogger;
import org.datanucleus.util.PersistenceUtils;
import org.datanucleus.util.StringUtils;

/**
 * SchemaTool providing an interface for the maintenance of schemas.
 * These utilities include:-
 * <ul>
 * <li>creation of tables representing classes specified in input data</li>
 * <li>deletion of tables representing classes specified in input data</li>
 * <li>validation of tables representing classes specified in input data</li>
 * <li>details about the datastore</li>
 * </ul>
 */
public class SchemaTool
{
    /** Localiser for messages. */
    private static final Localiser LOCALISER = Localiser.getInstance(
        "org.datanucleus.store.rdbms.Localisation", RDBMSStoreManager.class.getClassLoader());

    /** Name of the persistence API to use. */
    private String apiName = "JDO";

    /** Name of a file in which to put the DDL (or null if wanting to execute in the datastore). */
    private String ddlFilename = null;

    /** When generating DDL to a file, whether to generate complete DDL, or just for missing components. */
    private boolean completeDdl = false;

    /** Whether to operate in verbose mode. */
    private boolean verbose = false;

    /** create mode **/
    public static final int SCHEMATOOL_CREATE_MODE = 1;

    /** delete mode **/
    public static final int SCHEMATOOL_DELETE_MODE = 2;

    /** validate mode **/
    public static final int SCHEMATOOL_VALIDATE_MODE = 3;

    /** database info mode **/
    public static final int SCHEMATOOL_DATABASE_INFO_MODE = 4;

    /** schema info mode **/
    public static final int SCHEMATOOL_SCHEMA_INFO_MODE = 5;

    public static NucleusLogger LOGGER = NucleusLogger.getLoggerInstance("DataNucleus.SchemaTool");

    /**
     * Entry method when invoked from the command line.
     * @param args List of options for processing by the available methods in this class.
     * @throws Exception
     */
    public static void main(String[] args) throws Exception
    {
        SchemaTool tool = new SchemaTool();

        CommandLine cmd = new CommandLine();
        cmd.addOption("create", "create", null, LOCALISER.msg(false, "014026"));
        cmd.addOption("delete", "delete", null, LOCALISER.msg(false, "014027"));
        cmd.addOption("validate", "validate", null, LOCALISER.msg(false, "014028"));
        cmd.addOption("dbinfo", "dbinfo", null, LOCALISER.msg(false, "014029"));
        cmd.addOption("schemainfo", "schemainfo", null, LOCALISER.msg(false, "014030"));
        cmd.addOption("help", "help", null, LOCALISER.msg(false, "014033"));

        cmd.addOption("ddlFile", "ddlFile", "ddlFile", LOCALISER.msg(false, "014031"));
        cmd.addOption("completeDdl", "completeDdl", null, LOCALISER.msg(false, "014032"));
        cmd.addOption("api", "api", "api", "API Adapter (JDO, JPA, etc)");
        cmd.addOption("v", "verbose", null, "verbose output");

        cmd.addOption("pu", "persistenceUnit", "<persistence-unit>", 
            "name of the persistence unit to handle the schema for");
        cmd.addOption("props", "properties", "props", "path to a properties file");
        cmd.parse(args);

        // Remaining command line args are filenames (class files, metadata files)
        String[] filenames = cmd.getDefaultArgs();

        if (cmd.hasOption("api"))
        {
            tool.setApi(cmd.getOptionArg("api"));
        }

        // Determine the mode of operation required
        String msg = null;
        int mode = SCHEMATOOL_CREATE_MODE;
        if (cmd.hasOption("create"))
        {
            mode = SCHEMATOOL_CREATE_MODE;
            msg = LOCALISER.msg(false, "014000", ObjectManagerFactoryImpl.getVersionNumber());
        }
        else if (cmd.hasOption("delete"))
        {
            mode = SCHEMATOOL_DELETE_MODE;
            msg = LOCALISER.msg(false, "014001", ObjectManagerFactoryImpl.getVersionNumber());
        }
        else if (cmd.hasOption("validate"))
        {
            mode = SCHEMATOOL_VALIDATE_MODE;
            msg = LOCALISER.msg(false, "014002", ObjectManagerFactoryImpl.getVersionNumber());
        }
        else if (cmd.hasOption("dbinfo"))
        {
            mode = SCHEMATOOL_DATABASE_INFO_MODE;
            msg = LOCALISER.msg(false, "014003", ObjectManagerFactoryImpl.getVersionNumber());
        }
        else if (cmd.hasOption("schemainfo"))
        {
            mode = SCHEMATOOL_SCHEMA_INFO_MODE;
            msg = LOCALISER.msg(false, "014004", ObjectManagerFactoryImpl.getVersionNumber());
        }
        else if (cmd.hasOption("help"))
        {
            System.out.println(LOCALISER.msg(false, "014023"));
            System.out.println(LOCALISER.msg(false, "014024"));
            System.out.println(LOCALISER.msg(false, "014025"));
            System.out.println(cmd.toString());
            System.out.println(LOCALISER.msg(false, "014034"));
            System.out.println(LOCALISER.msg(false, "014035"));
            System.exit(0);
        }
        LOGGER.info(msg);
        System.out.println(msg);

        // Extract the selected options
        String propsFileName = null;
        String persistenceUnitName = null;
        if (cmd.hasOption("ddlFile"))
        {
            tool.setDdlFile(cmd.getOptionArg("ddlFile"));
        }
        if (cmd.hasOption("completeDdl"))
        {
            tool.setCompleteDdl(true);
        }
        if (cmd.hasOption("v"))
        {
            tool.setVerbose(true);
        }

        if (cmd.hasOption("pu"))
        {
            persistenceUnitName = cmd.getOptionArg("pu");
        }
        if (cmd.hasOption("props"))
        {
            propsFileName = cmd.getOptionArg("props");
        }

        // Classpath
        msg = LOCALISER.msg(false, "014005");
        LOGGER.info(msg);
        if (tool.isVerbose())
        {
            System.out.println(msg);
        }
        StringTokenizer tokeniser = new StringTokenizer(System.getProperty("java.class.path"), File.pathSeparator);
        while (tokeniser.hasMoreTokens())
        {
            msg = LOCALISER.msg(false, "014006", tokeniser.nextToken());
            LOGGER.info(msg);
            if (tool.isVerbose())
            {
                System.out.println(msg);
            }
        }
        if (tool.isVerbose())
        {
            System.out.println();
        }

        // DDL file
        String ddlFilename = tool.getDdlFile();
        if (ddlFilename != null)
        {
            msg = LOCALISER.msg(false, tool.getCompleteDdl() ? "014018" : "014019", ddlFilename);
            LOGGER.info(msg);
            if (tool.isVerbose())
            {
                System.out.println(msg);
                System.out.println();
            }
        }

        // Create a PMF for use with this mode
        boolean requiresAutoStartTable = false;
        JDOPersistenceManagerFactory pmf = null;
        try
        {
            if (propsFileName != null)
            {
                Properties props = PersistenceUtils.setPropertiesUsingFile(propsFileName);

                // Currently only support creation of autoStart table when passing properties file
                // TODO Take in a commandline property for it
                String autostart = props.getProperty("datanucleus.autoStartMechanism");
                if (autostart != null && autostart.equalsIgnoreCase("SchemaTable"))
                {
                    requiresAutoStartTable = true;
                }

                pmf = getPMFForMode(mode, tool.getApi(), props, 
                    persistenceUnitName, ddlFilename, tool.isVerbose());
            }
            else
            {
                pmf = getPMFForMode(mode, tool.getApi(), null, 
                    persistenceUnitName, ddlFilename, tool.isVerbose());
            }
        }
        catch (Exception e)
        {
            // Unable to create a PMF so likely input errors
            LOGGER.error("Error creating PMF", e);
            System.out.println(LOCALISER.msg(false, "014008", e.getMessage()));
            System.exit(1);
            return;
        }

        Set<String> classNames = null;
        if (mode != SCHEMATOOL_SCHEMA_INFO_MODE && mode != SCHEMATOOL_DATABASE_INFO_MODE)
        {
            // Find the names of the classes to be processed
            // This will load up all MetaData for the specified input and throw exceptions where errors are found
            try
            {
                MetaDataManager metaDataMgr = pmf.getOMFContext().getMetaDataManager();
                ClassLoaderResolver clr = pmf.getOMFContext().getClassLoaderResolver(null);

                FileMetaData[] filemds = getFileMetaDataForInput(metaDataMgr, clr, tool.isVerbose(),
                    persistenceUnitName, filenames);
                classNames = new TreeSet<String>();
                if (filemds == null)
                {
                    msg = LOCALISER.msg(false, "014021");
                    LOGGER.error(msg);
                    System.out.println(msg);
                    System.exit(2);
                    return;
                }
                for (int i=0;i<filemds.length;i++)
                {
                    for (int j=0;j<filemds[i].getNoOfPackages();j++)
                    {
                        for (int k=0;k<filemds[i].getPackage(j).getNoOfClasses();k++)
                        {
                            String className = filemds[i].getPackage(j).getClass(k).getFullClassName();
                            if (!classNames.contains(className))
                            {
                                classNames.add(className);
                            }
                        }
                    }
                }
            }
            catch (Exception e)
            {
                // Exception will have been logged and sent to System.out in "getFileMetaDataForInput()"
                System.exit(2);
                return;
            }
        }

        // Run SchemaTool
        try
        {
            if (mode == SCHEMATOOL_CREATE_MODE)
            {
                tool.createSchema(pmf, classNames, requiresAutoStartTable);
            }
            else if (mode == SCHEMATOOL_DELETE_MODE)
            {
                tool.deleteSchema(pmf, classNames);
            }
            else if (mode == SCHEMATOOL_VALIDATE_MODE)
            {
                tool.validateSchema(pmf, classNames);
            }
            else if (mode == SCHEMATOOL_DATABASE_INFO_MODE)
            {
                StoreManager srm = ((JDOPersistenceManager) pmf.getPersistenceManager()).getObjectManager().getStoreManager();
                srm.printInformation("DATASTORE", System.out);
            }
            else if (mode == SCHEMATOOL_SCHEMA_INFO_MODE)
            {
                StoreManager srm = ((JDOPersistenceManager) pmf.getPersistenceManager()).getObjectManager().getStoreManager();
                srm.printInformation("SCHEMA", System.out);
            }

            msg = LOCALISER.msg(false, "014043");
            LOGGER.info(msg);
            System.out.println(msg);
        }
        catch (Exception e)
        {
            msg = LOCALISER.msg(false, "014037", e.getMessage());
            System.out.println(msg);
            LOGGER.error(msg, e);
            System.exit(2);
            return;
        }
    }

    /**
     * Constructor
     */
    public SchemaTool()
    {
    }

    /**
     * Method to create a PersistenceManagerFactory for the specified mode of SchemaTool
     * @param mode Mode of operation of SchemaTool
     * @param api Persistence API
     * @param userProps Map containing user provided properties (usually input via a file)
     * @param persistenceUnitName Name of the persistence-unit (if any)
     * @param ddlFile Name of a file to output DDL to
     * @param verbose Verbose mode
     * @return The PersistenceManagerFactory to use
     * @throws NucleusException Thrown if an error occurs in creating the required PMF
     */
    protected static JDOPersistenceManagerFactory getPMFForMode(int mode, String api, Map userProps, 
            String persistenceUnitName, String ddlFile, boolean verbose)
    {
        Map props = new HashMap();
        if (persistenceUnitName != null)
        {
            // persistence-unit specified so take the persistence properties from that
            // Note that this will create the persistence with the properties for the <persistence-unit>
            props.put("javax.jdo.option.PersistenceUnitName", persistenceUnitName);
        }

        if (userProps != null)
        {
            // Properties specified by the user in a file
            props.putAll(userProps);
        }
        else
        {
            // Properties specified via System properties (only support particular ones)
            String[] propNames = 
            {
                    "datanucleus.ConnectionURL",
                    "datanucleus.ConnectionDriverName",
                    "datanucleus.ConnectionUserName",
                    "datanucleus.ConnectionPassword",
                    "datanucleus.Mapping",
                    "javax.jdo.option.ConnectionURL",
                    "javax.jdo.option.ConnectionDriverName",
                    "javax.jdo.option.ConnectionUserName",
                    "javax.jdo.option.ConnectionPassword",
                    "javax.jdo.option.Mapping"
            };
            for (int i=0;i<propNames.length;i++)
            {
                if (System.getProperty(propNames[i]) != null)
                {
                    props.put(propNames[i], System.getProperty(propNames[i]));
                }
            }

            if (persistenceUnitName == null && 
                (props.get("javax.jdo.option.ConnectionURL") == null) &&
                (props.get("datanucleus.ConnectionURL") == null) &&
                (props.get("javax.persistence.jdbc.url") == null))
            {
                // No URL or persistence-unit defined, so fall back to some (undocumented) properties file
                File file = new File(System.getProperty("user.home") + "/datanucleus.properties");
                if (file.exists())
                {
                    try
                    {
                        InputStream is = new FileInputStream(file);
                        Properties fileProps = new Properties();
                        fileProps.load(is);
                        props.putAll(fileProps);
                        is.close();
                    }
                    catch (IOException ioe)
                    {
                        // Ignore this
                    }
                }
                else
                {
                    throw new NucleusException(LOCALISER.msg("014041"));
                }
            }
        }

        // Use our PMF
        props.put("javax.jdo.PersistenceManagerFactoryClass", JDOPersistenceManagerFactory.class.getName());

        // Set the API
        props.put("datanucleus.persistenceApiName", api);

        // Disable autostart mechanism
        props.put("datanucleus.autostartmechanism", "None");

        // Tag on the mandatory props that we must have for each mode
        if (mode == SCHEMATOOL_CREATE_MODE)
        {
            if (ddlFile != null)
            {
                // the tables must not be created in the DB, so do not validate (DDL is being output to a file)
                props.put("datanucleus.validateconstraints", "false");
                props.put("datanucleus.validatecolumns", "false");
                props.put("datanucleus.validatetables", "false");
            }
            props.put("datanucleus.autocreateschema", "true");
            props.put("datanucleus.autocreatetables", "true");
            props.put("datanucleus.autocreateconstraints", "true");
            props.put("datanucleus.fixeddatastore", "false");
            props.put("datanucleus.readOnlydatastore", "false");
            props.put("datanucleus.rdbms.checkexisttablesorviews", "true");
        }
        else if (mode == SCHEMATOOL_DELETE_MODE)
        {
            props.put("datanucleus.fixeddatastore", "false");
            props.put("datanucleus.readonlydatastore", "false");
        }
        else if (mode == SCHEMATOOL_VALIDATE_MODE)
        {
            props.put("datanucleus.autocreateschema", "false");
            props.put("datanucleus.autocreatetables", "false");
            props.put("datanucleus.autocreateconstraints", "false");
            props.put("datanucleus.autocreatecolumns", "false");
            props.put("datanucleus.validatetables", "true");
            props.put("datanucleus.validatecolumns", "true");
            props.put("datanucleus.validateconstraints", "true");
        }

        // Create the PMF that we will use to communicate with the datastore
        JDOPersistenceManagerFactory pmf = new JDOPersistenceManagerFactory(props);

        if (verbose)
        {
            String msg = LOCALISER.msg(false, "014020");
            LOGGER.info(msg);
            System.out.println(msg);

            Map<String,Object> pmfProps = pmf.getConfiguration().getPersistenceProperties();
            Set<String> keys = pmfProps.keySet();
            List<String> keyNames = new ArrayList<String>(keys);
            Collections.sort(keyNames);
            Iterator keyNamesIter = keyNames.iterator();
            while (keyNamesIter.hasNext())
            {
                String key = (String)keyNamesIter.next();
                Object value = pmfProps.get(key);
                boolean display = true;
                if (!key.startsWith("datanucleus"))
                {
                    display = false;
                }
                else if (key.equals("datanucleus.connectionpassword"))
                {
                    // Don't show passwords
                    display = false;
                }
                else if (value == null)
                {
                    display = false;
                }
                else if (value instanceof String && StringUtils.isWhitespace((String)value))
                {
                    display = false;
                }

                if (display)
                {
                    // Print the property to sysout
                    msg = LOCALISER.msg(false, "014022", key, value);
                    LOGGER.info(msg);
                    System.out.println(msg);
                }
            }
            System.out.println();            
        }
        return pmf;
    }

    /**
     * Method to take the input for SchemaTool and returns the FileMetaData that it implies.
     * The input should either be a persistence-unit name, or a set of input files.
     * @param metaDataMgr Manager for MetaData
     * @param clr ClassLoader resolver
     * @param verbose Whether to put message verbosely
     * @param persistenceUnitName Name of the "persistence-unit"
     * @param inputFiles Input metadata/class files
     * @return The FileMetaData for the input
     * @throws NucleusException Thrown if error(s) occur in processing the input
     */
    protected static FileMetaData[] getFileMetaDataForInput(MetaDataManager metaDataMgr, ClassLoaderResolver clr, 
            boolean verbose, String persistenceUnitName, String[] inputFiles)
    {
        FileMetaData[] filemds = null;

        String msg = null;
        if (inputFiles == null && persistenceUnitName == null)
        {
            msg = LOCALISER.msg(false, "014007");
            LOGGER.error(msg);
            System.out.println(msg);
            throw new NucleusUserException(msg);
        }

        if (persistenceUnitName != null)
        {
            // Schema management via "persistence-unit"
            msg = LOCALISER.msg(false, "014015", persistenceUnitName);
            LOGGER.info(msg);
            if (verbose)
            {
                System.out.println(msg);
                System.out.println();
            }

            // The PMF will have initialised the MetaDataManager with the persistence-unit
            filemds = metaDataMgr.getFileMetaData();
        }
        else
        {
            // Schema management via "Input Files" (metadata/class)
            msg = LOCALISER.msg(false, "014009");
            LOGGER.info(msg);
            if (verbose)
            {
                System.out.println(msg);
            }
            for (int i = 0; i < inputFiles.length; i++)
            {
                String entry = LOCALISER.msg(false, "014010", inputFiles[i]);
                LOGGER.info(entry);
                if (verbose)
                {
                    System.out.println(entry);
                }
            }
            if (verbose)
            {
                System.out.println();
            }

            // Read in the specified MetaData files - errors in MetaData will return exceptions and so we stop
            try
            {
                // Split the input files into MetaData files and classes
                LOGGER.debug(LOCALISER.msg(false, "014011", "" + inputFiles.length));
                HashSet metadataFiles = new HashSet();
                HashSet classNames = new HashSet();
                for (int i=0;i<inputFiles.length;i++)
                {
                    if (inputFiles[i].endsWith(".class"))
                    {
                        // Class file
                        URL classFileURL = null;
                        try
                        {
                            classFileURL = new URL("file:" + inputFiles[i]);
                        }
                        catch (Exception e)
                        {
                            msg = LOCALISER.msg(false, "014013", inputFiles[i]);
                            LOGGER.error(msg);
                            throw new NucleusUserException(msg);
                        }

                        String className = ClassUtils.getClassNameForFileURL(classFileURL);
                        classNames.add(className);
                    }
                    else
                    {
                        // MetaData file
                        metadataFiles.add(inputFiles[i]);
                    }
                }

                // Initialise the MetaDataManager using the mapping files and class names
                FileMetaData[] filemds1 = metaDataMgr.loadMetadataFiles(
                    (String[])metadataFiles.toArray(new String[metadataFiles.size()]), null);
                FileMetaData[] filemds2 = metaDataMgr.loadClasses(
                    (String[])classNames.toArray(new String[classNames.size()]), null);
                filemds = new FileMetaData[filemds1.length + filemds2.length];
                int pos = 0;
                for (int i=0;i<filemds1.length;i++)
                {
                    filemds[pos++] = filemds1[i];
                }
                for (int i=0;i<filemds2.length;i++)
                {
                    filemds[pos++] = filemds2[i];
                }
                LOGGER.debug(LOCALISER.msg(false, "014012", "" + inputFiles.length));
            }
            catch (Exception e)
            {
                // Error reading input files
                msg = LOCALISER.msg(false, "014014", e.getMessage());
                LOGGER.error(msg, e);
                System.out.println(msg);
                if (e instanceof NucleusException)
                {
                    throw (NucleusException)e;
                }
                throw new NucleusUserException(msg, e);
            }
        }

        return filemds;
    }

    /**
     * Method to generate a set of class names using the input list.
     * If no input class names are provided then uses the list of classes known to have metadata.
     * @param pmf PersistenceManager factory
     * @param inputClassNames Class names to start from
     * @return The set of class names
     */
    protected static Set<String> cleanInputClassNames(JDOPersistenceManagerFactory pmf, Set<String> inputClassNames) 
    {
        Set<String> classNames = new TreeSet<String>();
        if (inputClassNames == null || inputClassNames.size() == 0)
        {
            // Use all "known" persistable classes
            MetaDataManager mmgr = pmf.getOMFContext().getMetaDataManager();
            Collection classesWithMetadata = mmgr.getClassesWithMetaData();
            classNames.addAll(classesWithMetadata);
        }
        else
        {
            // Use all input classes
            classNames.addAll(inputClassNames);
        }
        return classNames;
    }

    /**
     * Method to handle the creation of the schema for a set of classes.
     * If no classes are supplied then assumes that all "known" classes (for this PMF) should be processed.
     * @param pmf PersistenceManagerFactory to use when generating the schema
     * @param inputClassNames names of all classes whose schema is to be created
     * @param autoStart Whether we need to create "SchemaTable" autostart table
     * @throws Exception Thrown when either an error occurs parsing the MetaData, or the DB definition is not defined.
     */
    public void createSchema(JDOPersistenceManagerFactory pmf, Set<String> inputClassNames, boolean autoStart)
    throws Exception
    {
        Set<String> classNames = cleanInputClassNames(pmf, inputClassNames);
        if (classNames.size() > 0)
        {
            ExecutionContext ec = ((JDOPersistenceManager) pmf.getPersistenceManager()).getObjectManager().getExecutionContext();
            ClassLoaderResolver clr = ec.getClassLoaderResolver();
            RDBMSStoreManager storeMgr = (RDBMSStoreManager)ec.getStoreManager();
            FileWriter ddlFileWriter = null;
            String ddlFilename = getDdlFile();
            if (ddlFilename != null)
            {
                // Open the DDL file for writing
                File ddlFile = StringUtils.getFileForFilename(ddlFilename);
                if (ddlFile.exists())
                {
                    // Delete existing file
                    ddlFile.delete();
                }
                if (ddlFile.getParentFile() != null && !ddlFile.getParentFile().exists())
                {
                    // Make sure the directory exists
                    ddlFile.getParentFile().mkdirs();
                }
                ddlFile.createNewFile();
                ddlFileWriter = new FileWriter(ddlFile);

                SimpleDateFormat fmt = new SimpleDateFormat("dd/MM/yyyy HH:mm:ss");
                ddlFileWriter.write("------------------------------------------------------------------\n");
                ddlFileWriter.write("-- DataNucleus SchemaTool " + 
                    "(version " + ObjectManagerFactoryImpl.getVersionNumber() + ")" +
                    " ran at " + fmt.format(new java.util.Date()) +
                    "\n");
                ddlFileWriter.write("------------------------------------------------------------------\n");
                if (getCompleteDdl())
                {
                    ddlFileWriter.write("-- Complete schema required for the following classes:-\n");
                }
                else  
                {
                    ddlFileWriter.write("-- Schema diff for " + pmf.getConnectionURL()+ " and the following classes:-\n");
                }
                Iterator classNameIter = classNames.iterator();
                while (classNameIter.hasNext())
                {
                    ddlFileWriter.write("--     " + classNameIter.next() + "\n");
                }
                ddlFileWriter.write("--\n");
            }

            try
            {
                String[] classNameArray = classNames.toArray(new String[classNames.size()]);
                storeMgr.addClassesForSchemaTool(classNameArray, clr, ddlFileWriter, getCompleteDdl(), autoStart);

                if (ddlFileWriter != null)
                {
                    try
                    {
                        ddlFileWriter.write("\n");
                        ddlFileWriter.write("------------------------------------------------------------------\n");
                        ddlFileWriter.write("-- Sequences and SequenceTables\n");
                    }
                    catch (IOException ioe)
                    {
                    }
                }
                createSchemaSequences(pmf, classNames, clr, ddlFileWriter);
            }
            finally
            {
                if (ddlFileWriter != null)
                {
                    ddlFileWriter.close();
                }
            }
        }
        else
        {
            String msg = LOCALISER.msg(false, "014039");
            LOGGER.error(msg);
            System.out.println(msg);

            throw new Exception(msg);
        }
    }

    protected void createSchemaSequences(JDOPersistenceManagerFactory pmf, Set<String> classNames, 
            ClassLoaderResolver clr, FileWriter ddlWriter)
    {
        RDBMSStoreManager rdbmsMgr = (RDBMSStoreManager) pmf.getOMFContext().getStoreManager();

        // Check for datastore-based value-generator usage
        if (classNames != null && classNames.size() > 0)
        {
            Set<String> seqTablesGenerated = new HashSet();
            Set<String> sequencesGenerated = new HashSet();
            Iterator<String> classNameIter = classNames.iterator();
            while (classNameIter.hasNext())
            {
                String className = classNameIter.next();
                AbstractClassMetaData cmd = rdbmsMgr.getMetaDataManager().getMetaDataForClass(className, clr);
                if (cmd.getIdentityMetaData() != null && cmd.getIdentityMetaData().getValueStrategy() != null)
                {
                    if (cmd.getIdentityMetaData().getValueStrategy() == IdentityStrategy.INCREMENT)
                    {
                        addSequenceTableForMetaData(cmd.getIdentityMetaData(), rdbmsMgr, clr, seqTablesGenerated);
                    }
                    else if (cmd.getIdentityMetaData().getValueStrategy() == IdentityStrategy.SEQUENCE)
                    {
                        String seqName = cmd.getIdentityMetaData().getSequence();
                        if (StringUtils.isWhitespace(seqName))
                        {
                            seqName = cmd.getIdentityMetaData().getValueGeneratorName();
                        }
                        if (!StringUtils.isWhitespace(seqName))
                        {
                            addSequenceForMetaData(cmd.getIdentityMetaData(), seqName,
                                rdbmsMgr, clr, sequencesGenerated, ddlWriter);
                        }
                    }
                }

                AbstractMemberMetaData[] mmds = cmd.getManagedMembers();
                for (int j=0;j<mmds.length;j++)
                {
                    IdentityStrategy str = mmds[j].getValueStrategy();
                    if (str == IdentityStrategy.INCREMENT)
                    {
                        addSequenceTableForMetaData(mmds[j], rdbmsMgr, clr, seqTablesGenerated);
                    }
                    else if (str == IdentityStrategy.SEQUENCE)
                    {
                        String seqName = mmds[j].getSequence();
                        if (StringUtils.isWhitespace(seqName))
                        {
                            seqName = mmds[j].getValueGeneratorName();
                        }
                        if (!StringUtils.isWhitespace(seqName))
                        {
                            addSequenceForMetaData(mmds[j], seqName, rdbmsMgr, clr, sequencesGenerated, ddlWriter);
                        }
                    }
                }
            }
        }
    }

    protected void addSequenceTableForMetaData(MetaData md, RDBMSStoreManager rdbmsMgr, ClassLoaderResolver clr,
            Set<String> seqTablesGenerated)
    {
        String catName = null;
        String schName = null;
        String tableName = TableGenerator.DEFAULT_TABLE_NAME;
        String seqColName = TableGenerator.DEFAULT_SEQUENCE_COLUMN_NAME;
        String nextValColName = TableGenerator.DEFAULT_NEXTVALUE_COLUMN_NAME;
        if (md.hasExtension("sequence-catalog-name"))
        {
            catName = md.getValueForExtension("sequence-catalog-name");
        }
        if (md.hasExtension("sequence-schema-name"))
        {
            schName = md.getValueForExtension("sequence-schema-name");
        }
        if (md.hasExtension("sequence-table-name"))
        {
            tableName = md.getValueForExtension("sequence-table-name");
        }
        if (md.hasExtension("sequence-name-column-name"))
        {
            seqColName = md.getValueForExtension("sequence-name-column-name");
        }
        if (md.hasExtension("sequence-nextval-column-name"))
        {
            nextValColName = md.getValueForExtension("sequence-nextval-column-name");
        }

        if (!seqTablesGenerated.contains(tableName))
        {
            ManagedConnection mconn = rdbmsMgr.getConnection(UserTransaction.TRANSACTION_NONE);
            Connection conn = (Connection) mconn.getConnection();
            try
            {
                DatastoreIdentifier tableIdentifier = rdbmsMgr.getIdentifierFactory().newDatastoreContainerIdentifier(tableName);
                if (catName != null)
                {
                    tableIdentifier.setCatalogName(catName);
                }
                if (schName != null)
                {
                    tableIdentifier.setSchemaName(schName);
                }
                SequenceTable seqTable = new SequenceTable(tableIdentifier, rdbmsMgr, seqColName, nextValColName);
                seqTable.initialize(clr);
                seqTable.exists(conn, true);
            }
            catch (Exception e)
            {
            }
            finally
            {
                mconn.close();
            }
            seqTablesGenerated.add(tableName);
        }
    }

    protected void addSequenceForMetaData(MetaData md, String seq, RDBMSStoreManager rdbmsMgr, ClassLoaderResolver clr,
            Set<String> sequencesGenerated, FileWriter ddlWriter)
    {
        String seqName = seq;
        Integer min = null;
        Integer max = null;
        Integer start = null;
        Integer increment = null;
        Integer cacheSize = null;

        SequenceMetaData seqmd = rdbmsMgr.getMetaDataManager().getMetaDataForSequence(clr, seq);
        if (seqmd != null)
        {
            seqName = seqmd.getDatastoreSequence();
            if (seqmd.getAllocationSize() > 0)
            {
                increment = new Integer(seqmd.getAllocationSize());
            }
            if (seqmd.getInitialValue() >= 0)
            {
                start = new Integer(seqmd.getInitialValue());
            }
            md = seqmd;
        }
        if (md.hasExtension("key-min-value"))
        {
            min = Integer.valueOf(md.getValueForExtension("key-min-value"));
        }
        if (md.hasExtension("key-max-value"))
        {
            max = Integer.valueOf(md.getValueForExtension("key-max-value"));
        }
        if (md.hasExtension("key-increment-by"))
        {
            increment = Integer.valueOf(md.getValueForExtension("key-increment-by"));
        }
        if (md.hasExtension("key-start-with"))
        {
            start = Integer.valueOf(md.getValueForExtension("key-start-with"));
        }
        if (md.hasExtension("key-database-cache-size"))
        {
            cacheSize = Integer.valueOf(md.getValueForExtension("key-database-cache-size"));
        }
        if (!sequencesGenerated.contains(seqName))
        {
            String stmt = ((RDBMSAdapter)rdbmsMgr.getDatastoreAdapter()).getSequenceCreateStmt(
                seqName, min, max, start, increment, cacheSize);
            if (ddlWriter != null)
            {
                try
                {
                    ddlWriter.write(stmt + ";\n");
                }
                catch (IOException ioe)
                {
                }
            }
            else
            {
                SQLController sqlController = rdbmsMgr.getSQLController();
                PreparedStatement ps = null;
                ManagedConnection mconn = rdbmsMgr.getConnection(UserTransaction.TRANSACTION_NONE);
                try
                {
                    ps = sqlController.getStatementForUpdate(mconn, stmt, false);
                    sqlController.executeStatementUpdate(mconn, stmt, ps, true);
                }
                catch (SQLException e)
                {
                }
                finally
                {
                    try
                    {
                        if (ps != null)
                        {
                            sqlController.closeStatement(mconn, ps);
                        }
                    }
                    catch (SQLException e)
                    {
                    }
                    mconn.close();
                }
            }
            sequencesGenerated.add(seqName);
        }
    }

    /**
     * Method to handle the deletion of a schema's tables.
     * If no classes are supplied then assumes that all "known" classes (for this PMF) should be processed.
     * @param pmf PersistenceManagerFactory to use when generating the schema
     * @param inputClassNames names of all classes whose schema is to be created
     * @throws Exception Thrown when either an error occurs parsing the MetaData, or the DB definition is not defined. 
     */
    public void deleteSchema(JDOPersistenceManagerFactory pmf, Set<String> inputClassNames)
    throws Exception
    {
        Set<String> classNames = cleanInputClassNames(pmf, inputClassNames);
        if (classNames.size() > 0)
        {
            // Create a PersistenceManager for this store and delete the tables
            ExecutionContext ec = ((JDOPersistenceManager) pmf.getPersistenceManager()).getObjectManager().getExecutionContext();
            StoreManager storeMgr = ec.getStoreManager();

            String[] classNameArray = classNames.toArray(new String[classNames.size()]);
            storeMgr.addClasses(classNameArray, ec.getClassLoaderResolver()); // Add them to mgr first
            storeMgr.removeAllClasses(ec.getClassLoaderResolver());
        }
        else
        {
            String msg = LOCALISER.msg(false, "014039");
            LOGGER.info(msg);
            System.out.println(msg);

            throw new Exception(msg);
        }
    }

    /**
     * Method to handle the validation of a schema's tables.
     * If no classes are supplied then assumes that all "known" classes (for this PMF) should be processed.
     * @param pmf PersistenceManagerFactory to use when generating the schema
     * @param inputClassNames names of all classes whose schema is to be created
     * @throws Exception Thrown when either an error occurs parsing the MetaData, or the DB definition is not defined. 
     */
    public void validateSchema(JDOPersistenceManagerFactory pmf, Set<String> inputClassNames)
    throws Exception
    {
        Set<String> classNames = cleanInputClassNames(pmf, inputClassNames);
        if (classNames != null && classNames.size() > 0)
        {
            // Create a PersistenceManager for this store and validate
            ExecutionContext ec = ((JDOPersistenceManager) pmf.getPersistenceManager()).getObjectManager().getExecutionContext();
            StoreManager storeMgr = ec.getStoreManager();

            String[] classNameArray = classNames.toArray(new String[classNames.size()]);
            storeMgr.addClasses(classNameArray, ec.getClassLoaderResolver()); // Validates since we have the flags set
        }
        else
        {
            String msg = LOCALISER.msg(false, "014039");
            LOGGER.error(msg);
            System.out.println(msg);

            throw new Exception(msg);
        }
    }

    /**
     * Acessor for the API (JDO, JPA)
     * @return the API
     */
    public String getApi()
    {
        return apiName;
    }

    /**
     * Mutator for the API (JDO, JPA)
     * @param api the API
     * @return The SchemaTool instance
     */
    public SchemaTool setApi(String api)
    {
        this.apiName = api;
        return this;
    }

    /**
     * @return the verbose
     */
    public boolean isVerbose()
    {
        return verbose;
    }

    /**
     * @param verbose the verbose to set
     * @return The SchemaTool instance
     */
    public SchemaTool setVerbose(boolean verbose)
    {
        this.verbose = verbose;
        return this;
    }

    /**
     * Accessor for the DDL filename
     * @return the file to use when outputing the DDL
     */
    public String getDdlFile()
    {
        return ddlFilename;
    }

    /**
     * Mutator for the DDL file
     * @param file the file to use when outputting the DDL
     * @return The SchemaTool instance
     */
    public SchemaTool setDdlFile(String file)
    {
        this.ddlFilename = file;
        return this;
    }

    /**
     * Mutator for the flag to output complete DDL (when using DDL file)
     * @param completeDdl Whether to return complete DDL
     * @return The SchemaTool instance
     */
    public SchemaTool setCompleteDdl(boolean completeDdl)
    {
        this.completeDdl = completeDdl;
        return this;
    }

    /**
     * @return whether to use generate DDL (or just update DDL)
     */
    public boolean getCompleteDdl()
    {
        return completeDdl;
    }
}