/**********************************************************************
Copyright (c) 2008 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.rdbms.query;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.datanucleus.exceptions.NucleusDataStoreException;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.query.QueryUtils;
import org.datanucleus.query.evaluator.JDOQLEvaluator;
import org.datanucleus.query.evaluator.JavaQueryEvaluator;
import org.datanucleus.query.expression.Expression;
import org.datanucleus.query.symbol.Symbol;
import org.datanucleus.store.ExecutionContext;
import org.datanucleus.store.connection.ManagedConnection;
import org.datanucleus.store.connection.ManagedConnectionResourceListener;
import org.datanucleus.store.mapped.DatastoreAdapter;
import org.datanucleus.store.mapped.DatastoreClass;
import org.datanucleus.store.mapped.StatementClassMapping;
import org.datanucleus.store.mapped.StatementMappingIndex;
import org.datanucleus.store.mapped.StatementNewObjectMapping;
import org.datanucleus.store.mapped.StatementResultMapping;
import org.datanucleus.store.mapped.mapping.AbstractContainerMapping;
import org.datanucleus.store.mapped.mapping.JavaTypeMapping;
import org.datanucleus.store.query.AbstractJDOQLQuery;
import org.datanucleus.store.query.CandidateIdsQueryResult;
import org.datanucleus.store.query.Query;
import org.datanucleus.store.query.QueryInterruptedException;
import org.datanucleus.store.query.QueryManager;
import org.datanucleus.store.query.QueryResult;
import org.datanucleus.store.query.ResultObjectFactory;
import org.datanucleus.store.rdbms.RDBMSStoreManager;
import org.datanucleus.store.rdbms.SQLController;
import org.datanucleus.store.rdbms.adapter.RDBMSAdapter;
import org.datanucleus.store.rdbms.sql.SQLStatement;
import org.datanucleus.store.rdbms.sql.SQLStatementHelper;
import org.datanucleus.store.rdbms.sql.expression.BooleanExpression;
import org.datanucleus.store.rdbms.sql.expression.SQLExpression;
import org.datanucleus.util.ClassUtils;
import org.datanucleus.util.NucleusLogger;

/**
 * RDBMS representation of a JDOQL query for use by DataNucleus.
 * The query can be specified via method calls, or via a single-string form.
 * This implementation uses the generic query compilation in "org.datanucleus.query".
 * There are the following main ways of running a query here
 * <ul>
 * <li>Totally in the datastore (no candidate collection specified, and no in-memory eval).</li>
 * <li>Totally in-memory (candidate collection specified, and in-memory eval)</li>
 * <li>Retrieve candidates from datastore (no candidate collection), and evaluate in-memory</li>
 * </ul>
 */
public class JDOQLQuery extends AbstractJDOQLQuery
{
    /** The compilation of the query for this datastore. Not applicable if totally in-memory. */
    protected transient RDBMSQueryCompilation datastoreCompilation = null;

    boolean statementReturnsEmpty = false;

    /**
     * Constructs a new query instance that uses the given object manager.
     * @param ec execution context
     */
    public JDOQLQuery(ExecutionContext ec)
    {
        this(ec, (JDOQLQuery) null);
    }

    /**
     * Constructs a new query instance having the same criteria as the given query.
     * @param ec execution context
     * @param q The query from which to copy criteria.
     */
    public JDOQLQuery(ExecutionContext ec, JDOQLQuery q)
    {
        super(ec, q);
    }

    /**
     * Constructor for a JDOQL query where the query is specified using the "Single-String" format.
     * @param ec execution context
     * @param query The single-string query form
     */
    public JDOQLQuery(ExecutionContext ec, String query)
    {
        super(ec, query);
    }

    /**
     * Utility to remove any previous compilation of this Query.
     */
    protected void discardCompiled()
    {
        super.discardCompiled();

        datastoreCompilation = null;
    }

    /**
     * Method to return if the query is compiled.
     * @return Whether it is compiled
     */
    protected boolean isCompiled()
    {
        if (evaluateInMemory())
        {
            // Don't need datastore compilation here since evaluating in-memory
            return compilation != null;
        }
        else
        {
            // Need both to be present to say "compiled"
            if (compilation == null || datastoreCompilation == null)
            {
                return false;
            }
            if (!datastoreCompilation.isPrecompilable())
            {
                NucleusLogger.GENERAL.info("Query compiled but not precompilable so ditching datastore compilation");
                datastoreCompilation = null;
                return false;
            }
            return true;
        }
    }

    /**
     * Convenience method to return whether the query should be evaluated in-memory.
     * @return Use in-memory evaluation?
     */
    protected boolean evaluateInMemory()
    {
        if (candidateCollection != null)
        {
            if (compilation != null && compilation.getSubqueryAliases() != null)
            {
                // TODO In-memory evaluation of subqueries isn't fully implemented yet, so remove this when it is
                NucleusLogger.QUERY.warn("In-memory evaluator doesn't currently handle subqueries completely so evaluating in datastore");
                return false;
            }

            Object val = getExtension("datanucleus.query.evaluateInMemory");
            if (val == null)
            {
                return true;
            }
            Boolean bool = Boolean.valueOf((String)val);
            if (bool != null && !bool.booleanValue())
            {
                // User has explicitly said to not evaluate in-memory
                return false;
            }
            return true;
        }
        return super.evaluateInMemory();
    }

    /**
     * Method to compile the JDOQL query.
     * Uses the superclass to compile the generic query populating the "compilation", and then generates
     * the datastore-specific "datastoreCompilation".
     * @param parameterValues Map of param values keyed by param name (if available at compile time)
     */
    protected synchronized void compileInternal(Map parameterValues)
    {
        if (isCompiled())
        {
            return;
        }

        // Compile the generic query expressions
        super.compileInternal(parameterValues);

        boolean inMemory = evaluateInMemory();
        if (candidateCollection != null && inMemory)
        {
            // Querying a candidate collection in-memory, so just return now (don't need datastore compilation)
            // TODO Maybe apply the result class checks ?
            return;
        }

        // Create the SQL statement, and its result/parameter definitions
        RDBMSStoreManager storeMgr = (RDBMSStoreManager)getStoreManager();
        if (candidateClass == null)
        {
            throw new NucleusUserException(LOCALISER.msg("021009", candidateClassName));
        }

        // Make sure any persistence info is loaded
        ec.hasPersistenceInformationForClass(candidateClass);
        AbstractClassMetaData acmd = getObjectManager().getMetaDataManager().getMetaDataForClass(candidateClass, clr);
        if (candidateClass.isInterface())
        {
            // Query of interface
            String[] impls = getObjectManager().getMetaDataManager().getClassesImplementingInterface(candidateClass.getName(), clr);
            if (impls.length == 1 && acmd.isImplementationOfPersistentDefinition())
            {
                // Only the generated implementation, so just use its metadata
            }
            else
            {
                // Use metadata for the persistent interface
                acmd = getObjectManager().getMetaDataManager().getMetaDataForInterface(candidateClass, clr);
                if (acmd == null)
                {
                    throw new NucleusUserException("Attempting to query an interface yet it is not declared 'persistent'." +
                        " Define the interface in metadata as being persistent to perform this operation, and make sure" +
                    " any implementations use the same identity and identity member(s)");
                }
            }
        }

        if (parameterValues != null)
        {
            // Check for null values on primitive parameters
            Set paramNames = parameterValues.keySet();
            Iterator iter = paramNames.iterator();
            while (iter.hasNext())
            {
                Object paramName = iter.next();
                if (paramName instanceof String)
                {
                    Symbol sym = compilation.getSymbolTable().getSymbol((String)paramName);
                    Object value = parameterValues.get(paramName);
                    if (value == null)
                    {
                        // When we have parameters supplied and have the flag 
                        // "datanucleus.query.checkUnusedParameters" set to false, the symbol may be null
                        // so omit the check for that case
                        if (sym != null && sym.getValueType() != null && sym.getValueType().isPrimitive())
                        {
                            throw new NucleusUserException(LOCALISER.msg("021117", paramName,
                                sym.getValueType().getName()));
                        }
                    }
                }
            }
        }

        QueryManager qm = getQueryManager();
        String datastoreKey = storeMgr.getQueryCacheKey();
        if (useCaching())
        {
            // Check if we have any parameters set to null, since this can invalidate a datastore compilation
            // e.g " field == :val" can be "COL IS NULL" or "COL = <val>"
            boolean nullParameter = false;
            if (parameterValues != null)
            {
                Iterator iter = parameterValues.values().iterator();
                while (iter.hasNext())
                {
                    Object val = iter.next();
                    if (val == null)
                    {
                        nullParameter = true;
                        break;
                    }
                }
            }

            if (!nullParameter)
            {
                // Allowing caching so try to find compiled (datastore) query
                datastoreCompilation = (RDBMSQueryCompilation)qm.getDatastoreQueryCompilation(datastoreKey,
                    getLanguage(), toString());
                if (datastoreCompilation != null)
                {
                    // Cached compilation exists for this datastore so reuse it
                    setResultDistinct(compilation.getResultDistinct());
                    return;
                }
            }
        }

        // Compile the query for the datastore since not cached
        if (type == Query.BULK_UPDATE)
        {
            datastoreCompilation = new RDBMSQueryCompilation();
            compileQueryUpdate(parameterValues, acmd);
        }
        else if (type == Query.BULK_DELETE)
        {
            datastoreCompilation = new RDBMSQueryCompilation();
            compileQueryDelete(parameterValues, acmd);
        }
        else
        {
            datastoreCompilation = new RDBMSQueryCompilation();
            synchronized (datastoreCompilation)
            {
                if (inMemory)
                {
                    // Generate statement to just retrieve all candidate objects for later processing
                    compileQueryToRetrieveCandidates(parameterValues, acmd);
                }
                else
                {
                    // Generate statement to perform the full query in the datastore
                    compileQueryFull(parameterValues, acmd);

                    if (result != null)
                    {
                        // Check existence of invalid selections in the result
                        StatementResultMapping resultMapping = datastoreCompilation.getResultDefinition();
                        for (int i=0;i<resultMapping.getNumberOfResultExpressions();i++)
                        {
                            Object stmtMap = resultMapping.getMappingForResultExpression(i);
                            if (stmtMap instanceof StatementMappingIndex)
                            {
                                StatementMappingIndex idx = (StatementMappingIndex)stmtMap;
                                AbstractMemberMetaData mmd = idx.getMapping().getMemberMetaData();
                                if (mmd != null)
                                {
                                    if ((mmd.hasCollection() || mmd.hasMap() || mmd.hasArray()) &&
                                        idx.getMapping() instanceof AbstractContainerMapping)
                                    {
                                        throw new NucleusUserException(LOCALISER.msg("021213"));
                                    }
                                }
                            }
                        }
                        if (resultClass != null)
                        {
                            // Check validity of resultClass for the result (PrivilegedAction since uses reflection)
                            AccessController.doPrivileged(new PrivilegedAction()
                            {
                                public Object run()
                                {
                                    // Check that this class has the necessary constructor/setters/fields to be used
                                    StatementResultMapping resultMapping = datastoreCompilation.getResultDefinition();
                                    if (QueryUtils.resultClassIsSimple(resultClass.getName()))
                                    {
                                        if (resultMapping.getNumberOfResultExpressions() > 1)
                                        {
                                            // Invalid number of result expressions
                                            throw new NucleusUserException(LOCALISER.msg("021201", resultClass.getName()));
                                        }

                                        Object stmtMap = resultMapping.getMappingForResultExpression(0);
                                        if (stmtMap instanceof StatementMappingIndex)
                                        {
                                            StatementMappingIndex idx = (StatementMappingIndex)stmtMap;
                                            Class exprType = idx.getMapping().getJavaType();
                                            boolean typeConsistent = false;
                                            if (exprType == resultClass)
                                            {
                                                typeConsistent = true;
                                            }
                                            else if (exprType.isPrimitive())
                                            {
                                                Class resultClassPrimitive = ClassUtils.getPrimitiveTypeForType(resultClass);
                                                if (resultClassPrimitive == exprType)
                                                {
                                                    typeConsistent = true;
                                                }
                                            }
                                            if (!typeConsistent)
                                            {
                                                // Inconsistent expression type not matching the result class type
                                                throw new NucleusUserException(LOCALISER.msg("021202", resultClass.getName(), exprType));
                                            }
                                        }
                                        else
                                        {
                                            // TODO Handle StatementClassMapping
                                            // TODO Handle StatementNewObjectMapping
                                            throw new NucleusUserException("Don't support result clause of " + 
                                                result + " with resultClass of " + resultClass.getName());
                                        }
                                    }
                                    else if (QueryUtils.resultClassIsUserType(resultClass.getName()))
                                    {
                                        // Check for valid constructor (either using param types, or using default ctr)
                                        Class[] ctrTypes = new Class[resultMapping.getNumberOfResultExpressions()];
                                        for (int i=0;i<ctrTypes.length;i++)
                                        {
                                            Object stmtMap = resultMapping.getMappingForResultExpression(i);
                                            if (stmtMap instanceof StatementMappingIndex)
                                            {
                                                ctrTypes[i] = ((StatementMappingIndex)stmtMap).getMapping().getJavaType();
                                            }
                                            else if (stmtMap instanceof StatementNewObjectMapping)
                                            {
                                                // TODO Handle this
                                            }
                                        }
                                        Constructor ctr = ClassUtils.getConstructorWithArguments(resultClass, ctrTypes);
                                        if (ctr == null && !ClassUtils.hasDefaultConstructor(resultClass))
                                        {
                                            // No valid constructor found!
                                            throw new NucleusUserException(LOCALISER.msg("021205", resultClass.getName()));
                                        }
                                        else if (ctr == null)
                                        {
                                            // We are using default constructor, so check the types of the result expressions for means of input
                                            for (int i=0;i<resultMapping.getNumberOfResultExpressions();i++)
                                            {
                                                Object stmtMap = resultMapping.getMappingForResultExpression(i);
                                                if (stmtMap instanceof StatementMappingIndex)
                                                {
                                                    StatementMappingIndex mapIdx = (StatementMappingIndex)stmtMap;
                                                    AbstractMemberMetaData mmd = mapIdx.getMapping().getMemberMetaData();
                                                    String fieldName = mapIdx.getColumnAlias();
                                                    Class fieldType = mapIdx.getMapping().getJavaType();
                                                    if (fieldName == null && mmd != null)
                                                    {
                                                        fieldName = mmd.getName();
                                                    }

                                                    if (fieldName != null)
                                                    {
                                                        // Check for the field of that name in the result class
                                                        Class resultFieldType = null;
                                                        boolean publicField = true;
                                                        try
                                                        {
                                                            Field fld = resultClass.getDeclaredField(fieldName);
                                                            resultFieldType = fld.getType();

                                                            // Check the type of the field
                                                            if (!ClassUtils.typesAreCompatible(fieldType, resultFieldType) && 
                                                                    !ClassUtils.typesAreCompatible(resultFieldType, fieldType))
                                                            {
                                                                throw new NucleusUserException(LOCALISER.msg("021211", 
                                                                    fieldName, fieldType.getName(), resultFieldType.getName()));
                                                            }
                                                            if (!Modifier.isPublic(fld.getModifiers()))
                                                            {
                                                                publicField = false;
                                                            }
                                                        }
                                                        catch (NoSuchFieldException nsfe)
                                                        {
                                                            publicField = false;
                                                        }

                                                        // Check for a public set method
                                                        if (!publicField)
                                                        {
                                                            Method setMethod = QueryUtils.getPublicSetMethodForFieldOfResultClass(resultClass, fieldName, resultFieldType);
                                                            if (setMethod == null)
                                                            {
                                                                // No setter, so check for a public put(Object, Object) method
                                                                Method putMethod = QueryUtils.getPublicPutMethodForResultClass(resultClass);
                                                                if (putMethod == null)
                                                                {
                                                                    throw new NucleusUserException(LOCALISER.msg("021212", 
                                                                        resultClass.getName(), fieldName));
                                                                }
                                                            }
                                                        }
                                                    }
                                                }
                                                else if (stmtMap instanceof StatementNewObjectMapping)
                                                {
                                                    // TODO Handle this
                                                }
                                            }
                                        }
                                    }

                                    return null;
                                }
                            });
                        }
                    }
                }

                if (NucleusLogger.QUERY.isDebugEnabled())
                {
                    NucleusLogger.QUERY.debug(LOCALISER.msg("021085", this, datastoreCompilation.getSQL()));
                }

                boolean hasParams = false;
                if (explicitParameters != null)
                {
                    hasParams = true;
                }
                else if (parameterValues != null && parameterValues.size() > 0)
                {
                    hasParams = true;
                }

                if (!statementReturnsEmpty && compilation.getSubqueryAliases() == null)
                {
                    // TODO Allow caching of queries with subqueries
                    if (!datastoreCompilation.isPrecompilable() || (datastoreCompilation.getSQL().indexOf('?') < 0 && hasParams))
                    {
                        // Some parameters had their clauses evaluated during compilation so the query
                        // didn't gain any parameters, so don't cache it
                        NucleusLogger.QUERY.debug(LOCALISER.msg("021075"));
                    }
                    else
                    {
                        qm.addDatastoreQueryCompilation(datastoreKey, getLanguage(), toString(), datastoreCompilation);
                    }
                }
            }
        }
    }

    /**
     * Convenience accessor for the SQL to invoke in the datastore for this query.
     * @return The SQL.
     */
    public String getSQL()
    {
        if (datastoreCompilation != null)
        {
            return datastoreCompilation.getSQL();
        }
        return null;
    }

    protected Object performExecute(Map parameters)
    {
        if (statementReturnsEmpty)
        {
            return Collections.EMPTY_LIST;
        }

        boolean inMemory = evaluateInMemory();
        if (candidateCollection != null)
        {
            // Supplied collection of instances, so evaluate in-memory
            if (candidateCollection.isEmpty())
            {
                return Collections.EMPTY_LIST;
            }
            else if (inMemory)
            {
                List candidates = new ArrayList(candidateCollection);
                JavaQueryEvaluator resultMapper = new JDOQLEvaluator(this, candidates, compilation, 
                    parameters, clr);
                return resultMapper.execute(true, true, true, true, true);
            }
        }
        else if (type == Query.SELECT)
        {
            // Query results are cached, so return those
            List<Object> cachedResults = getQueryManager().getDatastoreQueryResult(this, parameters);
            if (cachedResults != null)
            {
                return new CandidateIdsQueryResult(this, cachedResults);
            }
        }

        Object results = null;
        ManagedConnection mconn = ec.getStoreManager().getConnection(ec);
        try
        {
            // Execute the query
            long startTime = System.currentTimeMillis();
            if (NucleusLogger.QUERY.isDebugEnabled())
            {
                NucleusLogger.QUERY.debug(LOCALISER.msg("021046", getLanguage(), getSingleStringQuery(),
                    null));
            }

            RDBMSStoreManager storeMgr = (RDBMSStoreManager)getStoreManager();
            AbstractClassMetaData acmd = getObjectManager().getMetaDataManager().getMetaDataForClass(candidateClass, clr);
            SQLController sqlControl = storeMgr.getSQLController();
            PreparedStatement ps = null;
            try
            {
                if (type == Query.SELECT)
                {
                    // Create PreparedStatement and apply parameters, result settings etc
                    ps = RDBMSQueryUtils.getPreparedStatementForQuery(mconn,
                        datastoreCompilation.getSQL(), this);
                    SQLStatementHelper.applyParametersToStatement(ps, ec,
                        datastoreCompilation.getStatementParameters(), 
                        datastoreCompilation.getParameterNameByPosition(), parameters);
                    RDBMSQueryUtils.prepareStatementForExecution(ps, this, true);

                    ResultSet rs = null;
                    if (executeInSeparateThread())
                    {
                        // Execute the query as a task so we can allow timeout/cancel
                        rs = (ResultSet)performExecuteTask(new Object[] {sqlControl, mconn, ps});
                    }
                    else
                    {
                        // Execute the query in the current thread
                        rs = (ResultSet)performExecuteInternal(new Object[] {sqlControl, mconn, ps});
                    }

                    QueryResult qr = null;
                    try
                    {
                        if (inMemory)
                        {
                            // IN-MEMORY EVALUATION
                            ResultObjectFactory rof = storeMgr.newResultObjectFactory(acmd, 
                                datastoreCompilation.getResultDefinitionForClass(),
                                RDBMSQueryUtils.useUpdateLockForQuery(this), getFetchPlan(), 
                                candidateClass);

                            // Just instantiate the candidates for later in-memory processing
                            // TODO Use a queryResult rather than an ArrayList so we load when required
                            List candidates = new ArrayList();
                            while (rs.next())
                            {
                                candidates.add(rof.getObject(ec, rs));
                            }

                            // Perform in-memory filter/result/order etc
                            JavaQueryEvaluator resultMapper = 
                                new JDOQLEvaluator(this, candidates, compilation, parameters, clr);
                            results = resultMapper.execute(true, true, true, true, true);
                        }
                        else
                        {
                            // IN-DATASTORE EVALUATION
                            ResultObjectFactory rof = null;
                            if (result != null)
                            {
                                // Each result row is of a result type
                                rof = new ResultClassROF(resultClass, datastoreCompilation.getResultDefinition());
                            }
                            else if (resultClass != null)
                            {
                                rof = new ResultClassROF(resultClass, datastoreCompilation.getResultDefinitionForClass());
                            }
                            else
                            {
                                // Each result row is a candidate object
                                rof = storeMgr.newResultObjectFactory(acmd,
                                    datastoreCompilation.getResultDefinitionForClass(),
                                    RDBMSQueryUtils.useUpdateLockForQuery(this), getFetchPlan(), 
                                    candidateClass);
                            }

                            // Create the required type of QueryResult
                            String resultSetType = RDBMSQueryUtils.getResultSetTypeForQuery(this);
                            if (resultSetType.equals("scroll-insensitive") ||
                                    resultSetType.equals("scroll-sensitive"))
                            {
                                qr = new ScrollableQueryResult(this, rof, rs, 
                                    getResultDistinct() ? null : candidateCollection);
                            }
                            else
                            {
                                qr = new ForwardQueryResult(this, rof, rs, 
                                    getResultDistinct() ? null : candidateCollection);
                            }

                            final QueryResult qr1 = qr;
                            final ManagedConnection mconn1 = mconn;
                            ManagedConnectionResourceListener listener =
                                new ManagedConnectionResourceListener()
                            {
                                public void managedConnectionPreClose(){}
                                public void managedConnectionPostClose(){}
                                public void managedConnectionFlushed()
                                {
                                    // Disconnect the query from this ManagedConnection (read in unread rows etc)
                                    qr1.disconnect();
                                }
                                public void resourcePostClose()
                                {
                                    mconn1.removeListener(this);
                                }
                            };
                            mconn.addListener(listener);
                            ((AbstractRDBMSQueryResult)qr).addConnectionListener(listener);
                            results = qr;
                        }
                    }
                    finally
                    {
                        if (qr == null)
                        {
                            rs.close();
                        }
                    }
                }
                else if (type == Query.BULK_UPDATE)
                {
                    // Create PreparedStatement and apply parameters, result settings etc
                    ps = sqlControl.getStatementForUpdate(mconn, datastoreCompilation.getSQL(), false);
                    SQLStatementHelper.applyParametersToStatement(ps, ec,
                        datastoreCompilation.getStatementParameters(),
                        datastoreCompilation.getParameterNameByPosition(), parameters);
                    RDBMSQueryUtils.prepareStatementForExecution(ps, this, false);

                    // Execute the query as a task so we can allow timeout/cancel
                    results = performExecuteTask(new Object[] {sqlControl, mconn, ps});
                }
                else if (type == Query.BULK_DELETE)
                {
                    // Create PreparedStatement and apply parameters, result settings etc
                    // TODO Cater for multiple DELETE statements
                    ps = sqlControl.getStatementForUpdate(mconn, datastoreCompilation.getSQL(), false);
                    SQLStatementHelper.applyParametersToStatement(ps, ec,
                        datastoreCompilation.getStatementParameters(),
                        datastoreCompilation.getParameterNameByPosition(), parameters);
                    RDBMSQueryUtils.prepareStatementForExecution(ps, this, false);

                    // Execute the query as a task so we can allow timeout/cancel
                    results = performExecuteTask(new Object[] {sqlControl, mconn, ps});
                }
            }
            catch (QueryInterruptedException qie)
            {
                // Execution was cancelled so cancel the PreparedStatement
                try
                {
                    ps.cancel();
                }
                catch (SQLException sqle)
                {
                    // Some JDBC are too lazy to implement this
                }
                throw qie;
            }
            catch (SQLException sqle)
            {
                throw new NucleusException(LOCALISER.msg("021042"), sqle);
            }

            if (NucleusLogger.QUERY.isDebugEnabled())
            {
                NucleusLogger.QUERY.debug(LOCALISER.msg("021074", getLanguage(), 
                    "" + (System.currentTimeMillis() - startTime)));
            }
            return results;
        }
        finally
        {
            mconn.release();
        }
    }

    /**
     * Method that will throw an {@link UnsupportedOperationException} if the query implementation doesn't
     * support cancelling queries.
     */
    protected void assertSupportsCancel()
    {
        // We support cancel via JDBC PreparedStatement.cancel();
    }

    /**
     * Convenience method for whether this query supports timeouts.
     * @return Whether timeouts are supported.
     */
    protected boolean supportsTimeout()
    {
        return true;
    }

    /**
     * Method executing the query in the destination environment.
     * This performs the actual PreparedStatement.execute() and nothing else.
     * Any SQLException from the execution is wrapped as a NucleusDataStoreException.
     * @param args Arguments to the execution (SQLController, ManagedConnection, PreparedStatement)
     * @return The result
     */
    protected Object performExecuteInternal(final Object... args)
    {
        SQLController sqlControl = (SQLController)args[0];
        ManagedConnection mconn = (ManagedConnection)args[1];
        PreparedStatement ps = (PreparedStatement)args[2];
        try
        {
            if (type == Query.SELECT)
            {
                return sqlControl.executeStatementQuery(mconn, toString(), ps);
            }
            else if (type == Query.BULK_UPDATE || type == Query.BULK_DELETE)
            {
                int[] results = sqlControl.executeStatementUpdate(mconn, toString(), ps, true);

                // Evict all objects of this type from the cache
                ec.getOMFContext().getLevel2Cache().evictAll(candidateClass, subclasses);

                return Long.valueOf(results[0]);
            }
            else
            {
                throw new NucleusException("Query type not supported");
            }
        }
        catch (SQLException sqle)
        {
            throw new NucleusDataStoreException(LOCALISER.msg("021042"), sqle);
        }
    }

    /**
     * Method to set the (native) query statement for the compiled query as a whole.
     * @param parameters Input parameters (if known)
     * @param candidateCmd Metadata for the candidate class
     */
    private void compileQueryFull(Map parameters, AbstractClassMetaData candidateCmd)
    {
        if (type != Query.SELECT)
        {
            return;
        }

        long startTime = 0;
        if (NucleusLogger.QUERY.isDebugEnabled())
        {
            startTime = System.currentTimeMillis();
            NucleusLogger.QUERY.debug(LOCALISER.msg("021083", getLanguage(), toString()));
        }

        if (result != null)
        {
            datastoreCompilation.setResultDefinition(new StatementResultMapping());
        }
        else
        {
            datastoreCompilation.setResultDefinitionForClass(new StatementClassMapping());
        }

        // Generate statement for candidate(s)
        SQLStatement stmt = null;
        try
        {
            stmt = RDBMSQueryUtils.getStatementForCandidates(null, candidateCmd,
                datastoreCompilation.getResultDefinitionForClass(), ec, candidateClass, subclasses, result, null, 
                null);
        }
        catch (NucleusException ne)
        {
            // Statement would result in no results, so just catch it and avoid generating the statement
            NucleusLogger.QUERY.warn("Query for candidates of " + candidateClass.getName() +
                (subclasses ? " and subclasses" : "") + " resulted in no possible candidates", ne);
            statementReturnsEmpty = true;
            return;
        }

        // Update the SQLStatement with filter, ordering, result etc
        QueryToSQLMapper sqlMapper = new QueryToSQLMapper(stmt, compilation, parameters,
            datastoreCompilation.getResultDefinitionForClass(), datastoreCompilation.getResultDefinition(),
            candidateCmd, getFetchPlan(), ec, getParsedImports(), false, false, extensions);
        sqlMapper.compile();
        datastoreCompilation.setParameterNameByPosition(sqlMapper.getParameterNameByPosition());
        datastoreCompilation.setPrecompilable(sqlMapper.isPrecompilable());
        if (!getResultDistinct() && stmt.isDistinct())
        {
            setResultDistinct(true);
            compilation.setResultDistinct();
        }

        if (candidateCollection != null)
        {
            // Restrict to the supplied candidate ids
            BooleanExpression candidateExpr = null;
            Iterator iter = candidateCollection.iterator();
            JavaTypeMapping idMapping = stmt.getPrimaryTable().getTable().getIdMapping();
            while (iter.hasNext())
            {
                Object candidate = iter.next();
                SQLExpression idExpr = stmt.getSQLExpressionFactory().newExpression(stmt, stmt.getPrimaryTable(),
                    idMapping);
                SQLExpression idVal = stmt.getSQLExpressionFactory().newLiteral(stmt, idMapping, candidate);
                if (candidateExpr == null)
                {
                    candidateExpr = idExpr.eq(idVal);
                }
                else
                {
                    candidateExpr = candidateExpr.ior(idExpr.eq(idVal));
                }
            }
            stmt.whereAnd(candidateExpr, true);
        }

        // Apply any range
        if (range != null)
        {
            long lower = fromInclNo;
            long upper = toExclNo;
            if (fromInclParam != null)
            {
                if (parameters.containsKey(fromInclParam))
                {
                    lower = ((Number)parameters.get(fromInclParam)).longValue();
                }
                else
                {
                    // Must be numbered input so take penultimate
                    int pos = parameters.size()-2;
                    lower = ((Number)parameters.get(new Integer(pos))).longValue();
                }
            }
            if (toExclParam != null)
            {
                if (parameters.containsKey(fromInclParam))
                {
                    upper = ((Number)parameters.get(toExclParam)).longValue();
                }
                else
                {
                    // Must be numbered input so take ultimate
                    int pos = parameters.size()-1;
                    upper = ((Number)parameters.get(new Integer(pos))).longValue();
                }
            }
            stmt.setRange(lower, upper-lower);
        }

        // Set any extensions
        boolean useUpdateLock = RDBMSQueryUtils.useUpdateLockForQuery(this);
        stmt.addExtension("lock-for-update", new Boolean(useUpdateLock));

        datastoreCompilation.setSQL(stmt.getSelectStatement().toString());
        datastoreCompilation.setStatementParameters(stmt.getSelectStatement().getParametersForStatement());

        if (NucleusLogger.QUERY.isDebugEnabled())
        {
            NucleusLogger.QUERY.debug(LOCALISER.msg("021084", getLanguage(), System.currentTimeMillis()-startTime));
        }
    }

    /**
     * Method to set the statement (and parameter/results definitions) to retrieve all candidates.
     * This is used when we want to evaluate in-memory and so just retrieve all possible candidates
     * first.
     * @param parameters Input parameters (if known)
     * @param candidateCmd Metadata for the candidate class
     */
    private void compileQueryToRetrieveCandidates(Map parameters, AbstractClassMetaData candidateCmd)
    {
        if (type != Query.SELECT)
        {
            return;
        }

        StatementClassMapping resultsDef = new StatementClassMapping();
        datastoreCompilation.setResultDefinitionForClass(resultsDef);

        // Generate statement for candidate(s)
        SQLStatement stmt = null;
        try
        {
            stmt = RDBMSQueryUtils.getStatementForCandidates(null, candidateCmd,
                datastoreCompilation.getResultDefinitionForClass(), ec, candidateClass, subclasses, result, null,
                null);
        }
        catch (NucleusException ne)
        {
            // Statement would result in no results, so just catch it and avoid generating the statement
            NucleusLogger.QUERY.warn("Query for candidates of " + candidateClass.getName() +
                (subclasses ? " and subclasses" : "") + " resulted in no possible candidates", ne);
            statementReturnsEmpty = true;
            return;
        }

        if (stmt.allUnionsForSamePrimaryTable())
        {
            // Select fetch-plan fields of candidate class
            SQLStatementHelper.selectFetchPlanOfCandidateInStatement(stmt,
                datastoreCompilation.getResultDefinitionForClass(), candidateCmd, getFetchPlan(), 1);
        }
        else
        {
            // Select id only since tables don't have same mappings or column names
            // TODO complete-table will come through here but maybe ought to be treated differently
            SQLStatementHelper.selectIdentityOfCandidateInStatement(stmt,
                datastoreCompilation.getResultDefinitionForClass(), candidateCmd);
        }

        datastoreCompilation.setSQL(stmt.getSelectStatement().toString());
        datastoreCompilation.setStatementParameters(stmt.getSelectStatement().getParametersForStatement());
    }

    /**
     * Method to compile the query for RDBMS for a bulk update.
     * @param parameterValues The parameter values (if any)
     * @param candidateCmd Meta-data for the candidate class
     */
    protected void compileQueryUpdate(Map parameterValues, AbstractClassMetaData candidateCmd)
    {
        Expression[] updateExprs = compilation.getExprUpdate();
        if (updateExprs == null || updateExprs.length == 0)
        {
            // Nothing to update
            return;
        }

        // Generate statement for candidate
        StatementClassMapping resultsDef = new StatementClassMapping();
        SQLStatement stmt = RDBMSQueryUtils.getStatementForCandidates(null, candidateCmd,
            resultsDef, ec, candidateClass, subclasses, result, null, null);

        QueryToSQLMapper sqlMapper = new QueryToSQLMapper(stmt, compilation, parameterValues,
            resultsDef, null, candidateCmd, getFetchPlan(), ec, getParsedImports(), false, false, extensions);
        sqlMapper.compile();

        // The assumption here is that the SQL UPDATE statement generated will not need to use multiple
        // tables. This means that the SET clause only refers to the table of the candidate, and that
        // the WHERE clause only refers to the table of the candidate.
        // It is possible with some RDBMS e.g MySQL that support multiple table update syntax
        // that these 2 conditions don't need to be enforced.
        DatastoreAdapter dba = ((RDBMSStoreManager)ec.getStoreManager()).getDatastoreAdapter();
        if (stmt.getNumberOfTables() > 0 && !dba.supportsOption(RDBMSAdapter.UPDATE_MULTITABLE))
        {
            throw new NucleusDataStoreException("Bulk update requires use of multiple tables yet datastore doesnt allow multiple table syntax");
        }
        datastoreCompilation.setSQL(stmt.getUpdateStatement().toString());
        datastoreCompilation.setStatementParameters(stmt.getSelectStatement().getParametersForStatement());
    }

    /**
     * Method to compile the query for RDBMS for a bulk delete.
     * @param parameterValues The parameter values (if any)
     * @param candidateCmd Meta-data for the candidate class
     */
    protected void compileQueryDelete(Map parameterValues, AbstractClassMetaData candidateCmd)
    {
        HashSet<String> subclassNames = 
            ec.getStoreManager().getSubClassesForClass(candidateCmd.getFullClassName(), true, clr);
        if (subclassNames != null && !subclassNames.isEmpty())
        {
            // Check for subclasses having their own tables and hence needing multiple DELETEs
            RDBMSStoreManager storeMgr = (RDBMSStoreManager)ec.getStoreManager();
            DatastoreClass candidateTbl = storeMgr.getDatastoreClass(candidateCmd.getFullClassName(), clr);
            Iterator<String> iter = subclassNames.iterator();
            while (iter.hasNext())
            {
                String subclassName = iter.next();
                DatastoreClass subclassTbl = storeMgr.getDatastoreClass(subclassName, clr);
                if (candidateTbl != null && candidateTbl != subclassTbl)
                {
                    throw new NucleusException("Bulk delete doesn't currently support deletion "+ 
                        "where the candidate table also has subclasses in their own tables");
                }
            }
            if (candidateTbl.getSuperDatastoreClass() != null)
            {
                throw new NucleusException("Bulk delete doesn't currently support deletion " +
                    "where the candidate table also has superclass table(s)");
            }
        }

        // Generate statement for candidate
        StatementClassMapping resultsDef = new StatementClassMapping();
        SQLStatement stmt = RDBMSQueryUtils.getStatementForCandidates(null, candidateCmd,
            resultsDef, ec, candidateClass, subclasses, result, null, null);

        QueryToSQLMapper sqlMapper = new QueryToSQLMapper(stmt, compilation, parameterValues,
            resultsDef, null, candidateCmd, getFetchPlan(), ec, getParsedImports(), false, false, extensions);
        sqlMapper.compile();

        // This currently assumes that we only need to delete from one table.
        // TODO Delete entity from all populated tables
        datastoreCompilation.setSQL(stmt.getDeleteStatement().toString());
        datastoreCompilation.setStatementParameters(stmt.getSelectStatement().getParametersForStatement());
    }

    /**
     * Method to return the names of the extensions supported by this query.
     * To be overridden by subclasses where they support additional extensions.
     * @return The supported extension names
     */
    public Set<String> getSupportedExtensions()
    {
        Set<String> supported = super.getSupportedExtensions();
        supported.add("datanucleus.rdbms.query.resultSetType");
        supported.add("datanucleus.rdbms.query.resultSetConcurrency");
        supported.add("datanucleus.rdbms.query.useUpdateLock");
        return supported;
    }

    /**
     * Method to return if the query results should have the range checked and unnecessary rows
     * discarded. This is for where the query language has specified a range but the datastore doesn't
     * allow removal of unnecessary results in the query itself (so has to be done in post-processing).
     * This implementation returns false and so should be overridden by query languages to match their
     * capabilities.
     * @return Whether to apply range checks in post-processing of results.
     */
    protected boolean applyRangeChecks()
    {
        if (range == null)
        {
            // No range specified so don't apply checks!
            return false;
        }

        RDBMSStoreManager storeMgr = (RDBMSStoreManager)ec.getStoreManager();
        RDBMSAdapter dba = (RDBMSAdapter)storeMgr.getDatastoreAdapter();
        boolean using_limit_where_clause = (dba.getRangeByLimitEndOfStatementClause(fromInclNo, toExclNo).length() > 0);
        boolean using_rownum = (dba.getRangeByRowNumberColumn().length() > 0);

        return (range != null && !using_limit_where_clause && !using_rownum);
    }

    /**
     * Add a vendor-specific extension this query.
     * Intercepts any setting of in-memory evaluation, so we can throw away any datastore compilation.
     * @param key the extension key
     * @param value the extension value
     */
    public void addExtension(String key, Object value)
    {
        if (key != null && key.equals("datanucleus.query.evaluateInMemory"))
        {
            datastoreCompilation = null;
            getQueryManager().deleteDatastoreQueryCompilation(getStoreManager().getQueryCacheKey(), getLanguage(), 
                toString());
        }
        super.addExtension(key, value);
    }

    /**
     * Set multiple extensions, or use null to clear extensions.
     * Intercepts any settong of in-memory evaluation, so we can throw away any datastore compilation.
     * @param extensions
     */
    public void setExtensions(Map extensions)
    {
        if (extensions != null && extensions.containsKey("datanucleus.query.evaluateInMemory"))
        {
            datastoreCompilation = null;
            getQueryManager().deleteDatastoreQueryCompilation(getStoreManager().getQueryCacheKey(), getLanguage(), 
                toString());
        }
        super.setExtensions(extensions);
    }
}