/**********************************************************************
Copyright (c) 2008 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.rdbms.sql.method;

import java.util.List;

import org.datanucleus.api.ApiAdapter;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.store.mapped.mapping.JavaTypeMapping;
import org.datanucleus.store.mapped.mapping.PersistableMapping;
import org.datanucleus.store.rdbms.RDBMSStoreManager;
import org.datanucleus.store.rdbms.sql.expression.IllegalExpressionOperationException;
import org.datanucleus.store.rdbms.sql.expression.NullLiteral;
import org.datanucleus.store.rdbms.sql.expression.ObjectExpression;
import org.datanucleus.store.rdbms.sql.expression.ObjectLiteral;
import org.datanucleus.store.rdbms.sql.expression.SQLExpression;
import org.datanucleus.store.rdbms.sql.expression.SQLLiteral;
import org.datanucleus.util.NucleusLogger;

/**
 * Expression handler to evaluate JDOHelper.getVersion({expression}).
 * Returns an ObjectExpression or NullLiteral.
 */
public class JDOHelperGetVersionMethod extends AbstractSQLMethod
{
    /* (non-Javadoc)
     * @see org.datanucleus.store.rdbms.sql.method.SQLMethod#getExpression(org.datanucleus.store.rdbms.sql.expression.SQLExpression, java.util.List)
     */
    public SQLExpression getExpression(SQLExpression ignore, List args)
    {
        if (args == null || args.size() == 0)
        {
            throw new NucleusUserException("Cannot invoke JDOHelper.getVersion without an argument");
        }

        SQLExpression expr = (SQLExpression)args.get(0);
        if (expr == null)
        {
            throw new NucleusUserException("Cannot invoke JDOHelper.getVersion on null expression");
        }
        if (expr instanceof SQLLiteral)
        {
            RDBMSStoreManager storeMgr = stmt.getRDBMSManager();
            ApiAdapter api = storeMgr.getApiAdapter();
            Object obj = ((SQLLiteral)expr).getValue();
            if (obj == null || !api.isPersistable(obj))
            {
                return new NullLiteral(stmt, null, null, null);
            }
            else
            {
                Object ver = stmt.getRDBMSManager().getApiAdapter().getVersionForObject(obj);
                JavaTypeMapping m = getMappingForClass(ver.getClass());
                return new ObjectLiteral(stmt, m, ver, null);
            }
        }
        else if (ObjectExpression.class.isAssignableFrom(expr.getClass()))
        {
            if (((ObjectExpression)expr).getJavaTypeMapping() instanceof PersistableMapping)
            {
                NucleusLogger.QUERY.info(">> JDOHelper.getVersion for " + expr + " table=" + expr.getSQLTable() + " mapping=" + expr.getJavaTypeMapping());
                throw new NucleusUserException("Dont currently support JDOHelper.getVersion(ObjectExpression)");
                // TODO Implement this
            }
            return expr;
        }

        throw new IllegalExpressionOperationException("JDOHelper.getVersion", expr);
    }
}
