/*
 * Decompiled with CFR 0.152.
 */
package dmg.cells.nucleus;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import com.google.common.io.BaseEncoding;
import com.google.common.primitives.Longs;
import dmg.cells.nucleus.Cell;
import dmg.cells.nucleus.CellAddressCore;
import dmg.cells.nucleus.CellEvent;
import dmg.cells.nucleus.CellEventListener;
import dmg.cells.nucleus.CellInfo;
import dmg.cells.nucleus.CellMessage;
import dmg.cells.nucleus.CellNucleus;
import dmg.cells.nucleus.CellPath;
import dmg.cells.nucleus.CellRoute;
import dmg.cells.nucleus.CellRoutingTable;
import dmg.cells.nucleus.CellTunnel;
import dmg.cells.nucleus.CellTunnelInfo;
import dmg.cells.nucleus.KillEvent;
import dmg.cells.nucleus.MessageEvent;
import dmg.cells.nucleus.NoRouteToCellException;
import dmg.cells.nucleus.RoutedMessageEvent;
import dmg.cells.nucleus.SerializationException;
import dmg.util.TimebasedCounter;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.apache.curator.framework.CuratorFramework;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class CellGlue {
    private static final Logger LOGGER = LoggerFactory.getLogger(CellGlue.class);
    private final String _cellDomainName;
    private final ConcurrentMap<String, CellNucleus> _cellList = Maps.newConcurrentMap();
    private final Set<CellNucleus> _killedCells = Collections.newSetFromMap(Maps.newConcurrentMap());
    private final Map<String, List<CellEventListener>> _cellEventListener = new ConcurrentHashMap<String, List<CellEventListener>>();
    private final Map<String, Object> _cellContext = new ConcurrentHashMap<String, Object>();
    private final TimebasedCounter _uniqueCounter = new TimebasedCounter();
    private final BaseEncoding COUNTER_ENCODING = BaseEncoding.base64Url().omitPadding();
    private CellNucleus _systemNucleus;
    private CellRoutingTable _routingTable = new CellRoutingTable();
    private ThreadGroup _masterThreadGroup;
    private ThreadGroup _killerThreadGroup;
    private final Executor _killerExecutor;
    private final ThreadPoolExecutor _emergencyKillerExecutor;
    private final CellAddressCore _domainAddress;
    private final CuratorFramework _curatorFramework;
    private static final int MAX_ROUTE_LEVELS = 16;

    CellGlue(String cellDomainName, CuratorFramework curatorFramework) {
        String cellDomainNameLocal = cellDomainName;
        if (cellDomainName == null || cellDomainName.equals("")) {
            cellDomainNameLocal = "*";
        }
        if (cellDomainNameLocal.charAt(cellDomainNameLocal.length() - 1) == '*') {
            cellDomainNameLocal = cellDomainNameLocal.substring(0, cellDomainNameLocal.length()) + System.currentTimeMillis();
        }
        this._cellDomainName = cellDomainNameLocal;
        this._curatorFramework = curatorFramework;
        this._domainAddress = new CellAddressCore("*", this._cellDomainName);
        this._masterThreadGroup = new ThreadGroup("Master-Thread-Group");
        this._killerThreadGroup = new ThreadGroup("Killer-Thread-Group");
        ThreadFactory killerThreadFactory = r -> new Thread(this._killerThreadGroup, r);
        this._killerExecutor = Executors.newCachedThreadPool(killerThreadFactory);
        this._emergencyKillerExecutor = new ThreadPoolExecutor(1, 1, 0L, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<Runnable>(), killerThreadFactory);
        this._emergencyKillerExecutor.prestartCoreThread();
    }

    ThreadGroup getMasterThreadGroup() {
        return this._masterThreadGroup;
    }

    void addCell(String name, CellNucleus cell) throws IllegalArgumentException {
        if (this._cellList.putIfAbsent(name, cell) != null) {
            throw new IllegalArgumentException("Name Mismatch ( cell " + name + " exist )");
        }
        this.sendToAll(new CellEvent(name, 3));
    }

    void setSystemNucleus(CellNucleus nucleus) {
        Preconditions.checkState((this._systemNucleus == null ? 1 : 0) != 0);
        this._systemNucleus = nucleus;
    }

    CellNucleus getSystemNucleus() {
        return this._systemNucleus;
    }

    void consume(CellNucleus cell, String queue) {
        this.routeAdd(new CellRoute(queue, cell.getThisAddress(), 2));
    }

    void subscribe(CellNucleus cell, String topic) {
        this.routeAdd(new CellRoute(topic, cell.getThisAddress(), 7));
    }

    Map<String, Object> getCellContext() {
        return this._cellContext;
    }

    Object getCellContext(String str) {
        return this._cellContext.get(str);
    }

    public void routeAdd(CellRoute route) {
        CellAddressCore target = route.getTarget();
        if (target.getCellDomainName().equals(this.getCellDomainName()) && !this._cellList.containsKey(target.getCellName())) {
            throw new IllegalArgumentException("No such cell: " + target);
        }
        this._routingTable.add(route);
        this.sendToAll(new CellEvent(route, 7));
    }

    public void routeDelete(CellRoute route) {
        this._routingTable.delete(route);
        this.sendToAll(new CellEvent(route, 8));
    }

    CellRoutingTable getRoutingTable() {
        return this._routingTable;
    }

    CellRoute[] getRoutingList() {
        return this._routingTable.getRoutingList();
    }

    List<CellTunnelInfo> getCellTunnelInfos() {
        ArrayList<CellTunnelInfo> v = new ArrayList<CellTunnelInfo>();
        for (CellNucleus cellNucleus : this._cellList.values()) {
            Cell c = cellNucleus.getThisCell();
            if (!(c instanceof CellTunnel)) continue;
            v.add(((CellTunnel)((Object)c)).getCellTunnelInfo());
        }
        return v;
    }

    List<String> getCellNames() {
        return new ArrayList<String>(this._cellList.keySet());
    }

    String getUnique() {
        return this.COUNTER_ENCODING.encode(Longs.toByteArray((long)this._uniqueCounter.next()));
    }

    CellInfo getCellInfo(String name) {
        CellNucleus nucleus = this.getCell(name);
        return nucleus == null ? null : nucleus._getCellInfo();
    }

    Thread[] getThreads(String name) {
        CellNucleus nucleus = this.getCell(name);
        return nucleus == null ? null : nucleus.getThreads();
    }

    private void sendToAll(CellEvent event) {
        for (List<CellEventListener> listners : this._cellEventListener.values()) {
            for (CellEventListener hallo : listners) {
                if (hallo == null) {
                    LOGGER.trace("event distributor found NULL");
                    continue;
                }
                try {
                    switch (event.getEventType()) {
                        case 3: {
                            hallo.cellCreated(event);
                            break;
                        }
                        case 4: {
                            hallo.cellDied(event);
                            break;
                        }
                        case 7: {
                            hallo.routeAdded(event);
                            break;
                        }
                        case 8: {
                            hallo.routeDeleted(event);
                        }
                    }
                }
                catch (Exception e) {
                    LOGGER.info("Error while sending {}: {}", (Object)event, (Object)e);
                }
            }
        }
    }

    String getCellDomainName() {
        return this._cellDomainName;
    }

    void kill(CellNucleus nucleus) {
        this._kill(nucleus, nucleus, 0L);
    }

    void kill(CellNucleus sender, String cellName) throws IllegalArgumentException {
        CellNucleus nucleus = (CellNucleus)this._cellList.get(cellName);
        if (nucleus == null || this._killedCells.contains(nucleus)) {
            throw new IllegalArgumentException("Cell Not Found : " + cellName);
        }
        this._kill(sender, nucleus, 0L);
    }

    void threadGroupList(String cellName) {
        CellNucleus nucleus = (CellNucleus)this._cellList.get(cellName);
        if (nucleus != null) {
            nucleus.threadGroupList();
        } else {
            LOGGER.warn("cell {} is not running", (Object)cellName);
        }
    }

    CellNucleus getCell(String cellName) {
        return (CellNucleus)this._cellList.get(cellName);
    }

    synchronized boolean join(String cellName, long timeout) throws InterruptedException {
        if (timeout == 0L) {
            while (this.getCell(cellName) != null) {
                this.wait();
            }
            return true;
        }
        while (this.getCell(cellName) != null && timeout > 0L) {
            long time = System.currentTimeMillis();
            this.wait(timeout);
            timeout -= System.currentTimeMillis() - time;
        }
        return timeout > 0L;
    }

    synchronized void destroy(CellNucleus nucleus) {
        this._cellEventListener.remove(nucleus.getCellName());
        this._cellList.remove(nucleus.getCellName());
        this._killedCells.remove(nucleus);
        LOGGER.trace("destroy : sendToAll : killed {}", (Object)nucleus.getCellName());
        this.notifyAll();
    }

    private void _kill(CellNucleus source, CellNucleus destination, long to) {
        for (CellRoute route : this._routingTable.delete(destination.getThisAddress())) {
            this.sendToAll(new CellEvent(route, 8));
        }
        String cellToKill = destination.getCellName();
        if (!this._killedCells.add(destination)) {
            LOGGER.trace("Cell is being killed: {}", (Object)cellToKill);
            return;
        }
        CellPath sourceAddr = new CellPath(source.getCellName(), this.getCellDomainName());
        KillEvent killEvent = new KillEvent(sourceAddr, to);
        this.sendToAll(new CellEvent(cellToKill, 4));
        Runnable command = () -> destination.shutdown(killEvent);
        try {
            this._killerExecutor.execute(command);
        }
        catch (OutOfMemoryError e) {
            this._emergencyKillerExecutor.execute(command);
        }
    }

    void sendMessage(CellMessage msg, boolean resolveLocally, boolean resolveRemotely) throws SerializationException {
        if (!msg.isStreamMode()) {
            msg = msg.encode();
        }
        CellPath destination = msg.getDestinationPath();
        LOGGER.trace("sendMessage : {} send to {}", (Object)msg.getUOID(), (Object)destination);
        this.sendMessage(msg, destination.getCurrent(), resolveLocally, resolveRemotely, 16);
    }

    private void sendMessage(CellMessage msg, CellAddressCore address, boolean resolveLocally, boolean resolveRemotely, int steps) {
        CellPath destination = msg.getDestinationPath();
        boolean hasDestinationChanged = false;
        boolean hasTopicRoutes = false;
        while (steps > 0) {
            while (address.equals(this._domainAddress)) {
                if (!destination.next()) {
                    this.sendException(msg, "*");
                    return;
                }
                address = destination.getCurrent();
                hasDestinationChanged = true;
            }
            LOGGER.trace("sendMessage : next hop at {}: {}", (Object)steps, (Object)address);
            if (address.getCellDomainName().equals(this._cellDomainName)) {
                if (!this.deliverLocally(msg, address)) {
                    this.sendException(msg, address.toString());
                }
                return;
            }
            if (address.isLocalAddress()) {
                if (resolveLocally && this.deliverLocally(msg, address)) {
                    return;
                }
                for (CellRoute route : this._routingTable.findTopicRoutes(address)) {
                    boolean isLocalSubscriber;
                    CellAddressCore target = route.getTarget();
                    boolean bl = isLocalSubscriber = !target.isDomainAddress();
                    if (isLocalSubscriber || resolveRemotely) {
                        CellMessage m = msg.clone();
                        if (isLocalSubscriber) {
                            m.getDestinationPath().replaceCurrent(target);
                        }
                        this.sendMessage(m, target, true, resolveRemotely, steps - 1);
                    }
                    hasTopicRoutes = true;
                }
            }
            if (!hasDestinationChanged && msg.getSourcePath().getDestinationAddress().equals(address)) {
                if (!hasTopicRoutes) {
                    this.sendException(msg, address.toString());
                }
                return;
            }
            CellRoute route = this._routingTable.find(address, resolveRemotely);
            if (route == null) {
                LOGGER.trace("sendMessage : no route destination for : {}", (Object)address);
                if (!hasTopicRoutes) {
                    this.sendException(msg, address.toString());
                }
                return;
            }
            LOGGER.trace("sendMessage : using route : {}", (Object)route);
            address = route.getTarget();
            if (route.getRouteType() == 6 || route.getRouteType() == 2 && !address.isDomainAddress()) {
                destination.replaceCurrent(address);
                hasDestinationChanged = true;
            }
            resolveLocally = true;
            resolveRemotely = true;
            --steps;
        }
        LOGGER.trace("sendMessage : max route iteration reached: {}", (Object)destination);
        this.sendException(msg, address.toString());
    }

    private boolean deliverLocally(CellMessage msg, CellAddressCore address) {
        CellNucleus destNucleus = (CellNucleus)this._cellList.get(address.getCellName());
        if (destNucleus != null && !this._killedCells.contains(destNucleus)) {
            CellPath destinationPath = msg.getDestinationPath();
            if (address.equals(destinationPath.getCurrent())) {
                try {
                    destNucleus.addToEventQueue(new MessageEvent(msg.decode()));
                }
                catch (SerializationException e) {
                    LOGGER.error("Received malformed message from {} with UOID {} and session [{}]: {}", new Object[]{msg.getSourcePath(), msg.getUOID(), msg.getSession(), e.getMessage()});
                    this.sendException(msg, address.toString());
                }
            } else if (msg.getSourcePath().hops() > 30) {
                LOGGER.error("Hop count exceeds 30: {}", (Object)msg);
                this.sendException(msg, address.toString());
            } else {
                msg.addSourceAddress(this._domainAddress);
                destNucleus.addToEventQueue(new RoutedMessageEvent(msg));
            }
            return true;
        }
        return false;
    }

    private void sendException(CellMessage msg, String routeTarget) throws SerializationException {
        if (msg.getSourceAddress().getCellName().equals("*")) {
            Serializable messageObject = msg.decode().getMessageObject();
            if (messageObject instanceof NoRouteToCellException) {
                LOGGER.warn("Unable to notify {} about delivery failure of message sent to {}: No route for {}\u00a0in {}.", new Object[]{msg.getDestinationPath(), ((NoRouteToCellException)messageObject).getDestinationPath(), routeTarget, this._cellDomainName});
            } else {
                LOGGER.warn("Message from {} could not be delivered because no route to {}\u00a0is known.", (Object)msg.getSourcePath(), (Object)routeTarget);
            }
        } else {
            LOGGER.debug("Message from {} could not be delivered because no route to {}\u00a0is known; the sender will be notified.", (Object)msg.getSourcePath(), (Object)routeTarget);
            CellMessage envelope = new CellMessage(msg.getSourcePath().revert(), (Serializable)new NoRouteToCellException(msg, "Route for >" + routeTarget + "< not found at >" + this._cellDomainName + "<"));
            envelope.setLastUOID(msg.getUOID());
            envelope.addSourceAddress(this._domainAddress);
            this.sendMessage(envelope, true, true);
        }
    }

    void addCellEventListener(CellNucleus nucleus, CellEventListener listener) {
        this._cellEventListener.computeIfAbsent(nucleus.getCellName(), k -> new CopyOnWriteArrayList()).add(listener);
    }

    public String toString() {
        return this._cellDomainName;
    }

    public CuratorFramework getCuratorFramework() {
        return this._curatorFramework;
    }

    public void shutdown() {
        this._curatorFramework.close();
    }
}

