/*
 * Decompiled with CFR 0.152.
 */
package dmg.cells.zookeeper;

import com.google.common.base.Preconditions;
import com.google.common.collect.Ordering;
import java.io.Closeable;
import java.io.IOException;
import java.util.HashSet;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.api.BackgroundCallback;
import org.apache.curator.framework.api.BackgroundPathable;
import org.apache.curator.framework.api.GetDataWatchBackgroundStatable;
import org.apache.curator.framework.api.Pathable;
import org.apache.curator.framework.listen.ListenerContainer;
import org.apache.curator.framework.recipes.cache.ChildData;
import org.apache.curator.framework.recipes.cache.PathChildrenCacheEvent;
import org.apache.curator.framework.recipes.cache.PathChildrenCacheListener;
import org.apache.curator.framework.state.ConnectionState;
import org.apache.curator.framework.state.ConnectionStateListener;
import org.apache.curator.utils.PathUtils;
import org.apache.curator.utils.ThreadUtils;
import org.apache.curator.utils.ZKPaths;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PathChildrenCache
implements Closeable {
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final CuratorFramework client;
    private final String path;
    private final boolean cacheData;
    private final boolean dataIsCompressed;
    private final ListenerContainer<PathChildrenCacheListener> listeners = new ListenerContainer();
    private final ConcurrentMap<String, ChildData> currentData = new ConcurrentHashMap<String, ChildData>();
    private final AtomicReference<State> state = new AtomicReference<State>(State.LATENT);
    private final Watcher childrenWatcher = event -> {
        try {
            this.refresh(RefreshMode.NORMAL);
        }
        catch (Exception e) {
            this.handleException(e);
        }
    };
    private final Watcher dataWatcher = event -> {
        try {
            if (event.getType() == Watcher.Event.EventType.NodeDeleted) {
                this.remove(event.getPath());
            } else if (event.getType() == Watcher.Event.EventType.NodeDataChanged) {
                this.getDataAndStat(event.getPath());
            } else {
                this.log.debug("Data watcher ignored {}", (Object)event);
            }
        }
        catch (Exception e) {
            this.handleException(e);
        }
    };
    private final ConnectionStateListener connectionStateListener = (client, newState) -> this.handleStateChange(newState);

    public PathChildrenCache(CuratorFramework client, String path, boolean cacheData) {
        this(client, path, cacheData, false);
    }

    public PathChildrenCache(CuratorFramework client2, String path, boolean cacheData, boolean dataIsCompressed) {
        this.client = client2;
        this.path = PathUtils.validatePath((String)path);
        this.cacheData = cacheData;
        this.dataIsCompressed = dataIsCompressed;
    }

    public void start() throws Exception {
        Preconditions.checkState((boolean)this.state.compareAndSet(State.LATENT, State.STARTED), (Object)"already started");
        this.client.getConnectionStateListenable().addListener((Object)this.connectionStateListener);
        this.refresh(RefreshMode.NORMAL);
    }

    @Override
    public void close() throws IOException {
        if (this.state.compareAndSet(State.STARTED, State.CLOSED)) {
            this.client.getConnectionStateListenable().removeListener((Object)this.connectionStateListener);
            this.listeners.clear();
            this.client.clearWatcherReferences(this.childrenWatcher);
            this.client.clearWatcherReferences(this.dataWatcher);
        }
    }

    public ListenerContainer<PathChildrenCacheListener> getListenable() {
        return this.listeners;
    }

    public List<ChildData> getCurrentData() {
        return Ordering.natural().immutableSortedCopy(this.currentData.values());
    }

    public ChildData getCurrentData(String fullPath) {
        return (ChildData)this.currentData.get(fullPath);
    }

    void refresh(RefreshMode mode) throws Exception {
        BackgroundCallback callback = (client, event) -> {
            if (this.state.get() != State.CLOSED) {
                if (event.getResultCode() == KeeperException.Code.OK.intValue()) {
                    this.processChildren(event.getChildren(), mode);
                } else if (event.getResultCode() == KeeperException.Code.NONODE.intValue()) {
                    this.ensurePathAndThenRefresh(mode);
                } else if (event.getResultCode() == KeeperException.Code.CONNECTIONLOSS.intValue() || event.getResultCode() == KeeperException.Code.SESSIONEXPIRED.intValue()) {
                    this.log.debug("Refresh callback ignored {}", (Object)event);
                } else {
                    this.handleException((Throwable)KeeperException.create((int)event.getResultCode()));
                }
            }
        };
        ((Pathable)((BackgroundPathable)this.client.getChildren().usingWatcher(this.childrenWatcher)).inBackground(callback)).forPath(this.path);
    }

    void ensurePathAndThenRefresh(RefreshMode mode) throws Exception {
        BackgroundCallback callback = (client, event) -> {
            if (event.getResultCode() == KeeperException.Code.OK.intValue() || event.getResultCode() == KeeperException.Code.NONODE.intValue()) {
                this.refresh(mode);
            }
        };
        ((Pathable)this.client.checkExists().creatingParentContainersIfNeeded().inBackground(callback)).forPath(ZKPaths.makePath((String)this.path, (String)"ignored"));
    }

    void callListeners(PathChildrenCacheEvent event) {
        this.listeners.forEach(listener -> {
            try {
                listener.childEvent(this.client, event);
            }
            catch (Exception e) {
                this.handleException(e);
            }
            return null;
        });
    }

    void getDataAndStat(String fullPath) throws Exception {
        BackgroundCallback callback = (client, event) -> {
            if (event.getResultCode() == KeeperException.Code.OK.intValue()) {
                this.updateCache(fullPath, event.getStat(), this.cacheData ? event.getData() : null);
            }
        };
        if (this.dataIsCompressed && this.cacheData) {
            ((Pathable)((BackgroundPathable)((GetDataWatchBackgroundStatable)this.client.getData().decompressed()).usingWatcher(this.dataWatcher)).inBackground(callback)).forPath(fullPath);
        } else {
            ((Pathable)((BackgroundPathable)this.client.getData().usingWatcher(this.dataWatcher)).inBackground(callback)).forPath(fullPath);
        }
    }

    protected void handleException(Throwable e) {
        if (e instanceof RuntimeException) {
            this.log.error("", e);
        } else {
            this.log.error(e.getMessage());
        }
        ThreadUtils.checkInterrupted((Throwable)e);
    }

    protected void remove(String fullPath) {
        ChildData data = (ChildData)this.currentData.remove(fullPath);
        if (data != null) {
            this.callListeners(new PathChildrenCacheEvent(PathChildrenCacheEvent.Type.CHILD_REMOVED, data));
        }
    }

    private void handleStateChange(ConnectionState newState) {
        switch (newState) {
            case SUSPENDED: {
                this.callListeners(new PathChildrenCacheEvent(PathChildrenCacheEvent.Type.CONNECTION_SUSPENDED, null));
                break;
            }
            case LOST: {
                this.callListeners(new PathChildrenCacheEvent(PathChildrenCacheEvent.Type.CONNECTION_LOST, null));
                break;
            }
            case CONNECTED: 
            case RECONNECTED: {
                this.callListeners(new PathChildrenCacheEvent(PathChildrenCacheEvent.Type.CONNECTION_RECONNECTED, null));
                try {
                    this.refresh(RefreshMode.REBUILD);
                    break;
                }
                catch (Exception e) {
                    this.handleException(e);
                }
            }
        }
    }

    private void processChildren(List<String> children, RefreshMode mode) throws Exception {
        HashSet removedNodes = new HashSet(this.currentData.keySet());
        for (String child : children) {
            removedNodes.remove(ZKPaths.makePath((String)this.path, (String)child));
        }
        for (String fullPath : removedNodes) {
            this.remove(fullPath);
        }
        for (String child : children) {
            String fullPath = ZKPaths.makePath((String)this.path, (String)child);
            if (mode != RefreshMode.REBUILD && this.currentData.containsKey(fullPath)) continue;
            this.getDataAndStat(fullPath);
        }
    }

    private void updateCache(String fullPath, Stat stat, byte[] bytes) {
        ChildData data = new ChildData(fullPath, stat, bytes);
        ChildData previousData = this.currentData.put(fullPath, data);
        if (previousData == null) {
            this.callListeners(new PathChildrenCacheEvent(PathChildrenCacheEvent.Type.CHILD_ADDED, data));
        } else if (previousData.getStat().getVersion() != stat.getVersion()) {
            this.callListeners(new PathChildrenCacheEvent(PathChildrenCacheEvent.Type.CHILD_UPDATED, data));
        }
    }

    private static enum RefreshMode {
        NORMAL,
        REBUILD;

    }

    private static enum State {
        LATENT,
        STARTED,
        CLOSED;

    }
}

