/*
 * Decompiled with CFR 0.152.
 */
package dmg.cells.nucleus;

import com.google.common.base.Preconditions;
import com.google.common.io.BaseEncoding;
import com.google.common.primitives.Longs;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import dmg.cells.nucleus.Cell;
import dmg.cells.nucleus.CellAddressCore;
import dmg.cells.nucleus.CellEvent;
import dmg.cells.nucleus.CellInfo;
import dmg.cells.nucleus.CellMessage;
import dmg.cells.nucleus.CellNucleus;
import dmg.cells.nucleus.CellPath;
import dmg.cells.nucleus.CellRoute;
import dmg.cells.nucleus.CellRoutingTable;
import dmg.cells.nucleus.CellTunnel;
import dmg.cells.nucleus.CellTunnelInfo;
import dmg.cells.nucleus.KillEvent;
import dmg.cells.nucleus.MessageEvent;
import dmg.cells.nucleus.NoRouteToCellException;
import dmg.cells.nucleus.RoutedMessageEvent;
import dmg.cells.nucleus.SerializationException;
import dmg.cells.nucleus.SerializationHandler;
import dmg.cells.nucleus.SystemCell;
import dmg.util.TimebasedCounter;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nonnull;
import org.apache.curator.framework.CuratorFramework;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class CellGlue {
    private static final Logger LOGGER = LoggerFactory.getLogger(CellGlue.class);
    private final String _cellDomainName;
    private final ConcurrentMap<String, CellNucleus> _cellList = new ConcurrentHashMap<String, CellNucleus>();
    private final ConcurrentMap<String, CellNucleus> _publicCellList = new ConcurrentHashMap<String, CellNucleus>();
    private final ConcurrentMap<CellNucleus, CompletableFuture<?>> _killedCells = new ConcurrentHashMap();
    private final Map<String, Object> _cellContext = new ConcurrentHashMap<String, Object>();
    private final TimebasedCounter _uniqueCounter = new TimebasedCounter();
    private final BaseEncoding COUNTER_ENCODING = BaseEncoding.base64Url().omitPadding();
    private CellNucleus _systemNucleus;
    private final CellRoutingTable _routingTable = new CellRoutingTable();
    private final ThreadGroup _masterThreadGroup;
    private final ThreadGroup _killerThreadGroup;
    private final ListeningExecutorService _killerExecutor;
    private final ListeningExecutorService _emergencyKillerExecutor;
    private final CellAddressCore _domainAddress;
    private final CuratorFramework _curatorFramework;
    private final Optional<String> _zone;
    private final SerializationHandler.Serializer _serializer;
    private static final int MAX_ROUTE_LEVELS = 16;

    CellGlue(String cellDomainName, @Nonnull CuratorFramework curatorFramework, Optional<String> zone, SerializationHandler.Serializer serializer) {
        this._serializer = serializer;
        this._zone = Objects.requireNonNull(zone);
        Object cellDomainNameLocal = cellDomainName;
        if (cellDomainName == null || cellDomainName.isEmpty()) {
            cellDomainNameLocal = "*";
        }
        if (((String)cellDomainNameLocal).charAt(((String)cellDomainNameLocal).length() - 1) == '*') {
            cellDomainNameLocal = ((String)cellDomainNameLocal).substring(0, ((String)cellDomainNameLocal).length()) + System.currentTimeMillis();
        }
        this._cellDomainName = cellDomainNameLocal;
        this._curatorFramework = curatorFramework;
        this._domainAddress = new CellAddressCore("*", this._cellDomainName);
        this._masterThreadGroup = new ThreadGroup("Master-Thread-Group");
        this._killerThreadGroup = new ThreadGroup("Killer-Thread-Group");
        ThreadFactory killerThreadFactory = new ThreadFactoryBuilder().setNameFormat("killer-%d").setThreadFactory(r -> CellGlue.newThread(this._killerThreadGroup, r)).build();
        this._killerExecutor = MoreExecutors.listeningDecorator((ExecutorService)new ThreadPoolExecutor(1, Integer.MAX_VALUE, 3L, TimeUnit.SECONDS, new SynchronousQueue<Runnable>(), killerThreadFactory));
        ThreadPoolExecutor emergencyKillerExecutor = new ThreadPoolExecutor(1, 1, 0L, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<Runnable>(), killerThreadFactory);
        emergencyKillerExecutor.prestartCoreThread();
        this._emergencyKillerExecutor = MoreExecutors.listeningDecorator((ExecutorService)emergencyKillerExecutor);
    }

    static Thread newThread(ThreadGroup threadGroup, Runnable r) {
        Thread thread = new Thread(threadGroup, r);
        if (thread.isDaemon()) {
            thread.setDaemon(false);
        }
        if (thread.getPriority() != 5) {
            thread.setPriority(5);
        }
        return thread;
    }

    static Thread newThread(ThreadGroup threadGroup, Runnable r, String name) {
        Thread thread = new Thread(threadGroup, r, name);
        if (thread.isDaemon()) {
            thread.setDaemon(false);
        }
        if (thread.getPriority() != 5) {
            thread.setPriority(5);
        }
        return thread;
    }

    ThreadGroup getMasterThreadGroup() {
        return this._masterThreadGroup;
    }

    synchronized void registerCell(CellNucleus cell) throws IllegalStateException {
        String name;
        if (cell.getThisCell() instanceof SystemCell) {
            Preconditions.checkState((this._systemNucleus == null ? 1 : 0) != 0);
            this._systemNucleus = cell;
        }
        if (this._cellList.putIfAbsent(name = cell.getCellName(), cell) != null) {
            throw new IllegalStateException("Cell " + name + " already exists.");
        }
        this.sendToAll(new CellEvent(name, 3));
    }

    synchronized void publishCell(CellNucleus cell) throws IllegalArgumentException {
        String name = cell.getCellName();
        if (this._cellList.get(name) != cell) {
            throw new IllegalStateException("Cell " + name + " does not exist.");
        }
        if (!this._killedCells.containsKey(cell) && this._publicCellList.putIfAbsent(name, cell) != null) {
            throw new IllegalStateException("Cell " + name + " is already published.");
        }
    }

    CellNucleus getSystemNucleus() {
        return this._systemNucleus;
    }

    void consume(CellNucleus cell, String queue) {
        this.routeAdd(new CellRoute(queue, cell.getThisAddress(), cell.getZone(), 2));
    }

    void subscribe(CellNucleus cell, String topic) {
        this.routeAdd(new CellRoute(topic, cell.getThisAddress(), cell.getZone(), 7));
    }

    Map<String, Object> getCellContext() {
        return this._cellContext;
    }

    @Nonnull
    Optional<String> getZone() {
        return this._zone;
    }

    SerializationHandler.Serializer getMessageSerializer() {
        return this._serializer;
    }

    Object getCellContext(String str) {
        return this._cellContext.get(str);
    }

    public synchronized void routeAdd(CellRoute route) {
        CellAddressCore target = route.getTarget();
        if (target.getCellDomainName().equals(this.getCellDomainName()) && !this._publicCellList.containsKey(target.getCellName())) {
            throw new IllegalArgumentException("No such cell: " + target);
        }
        this._routingTable.add(route);
        this.sendToAll(new CellEvent(route, 7));
    }

    public synchronized void routeDelete(CellRoute route) {
        this._routingTable.delete(route);
        this.sendToAll(new CellEvent(route, 8));
    }

    CellRoutingTable getRoutingTable() {
        return this._routingTable;
    }

    CellRoute[] getRoutingList() {
        return this._routingTable.getRoutingList();
    }

    List<CellTunnelInfo> getCellTunnelInfos() {
        ArrayList<CellTunnelInfo> v = new ArrayList<CellTunnelInfo>();
        for (CellNucleus cellNucleus : this._publicCellList.values()) {
            Cell c = cellNucleus.getThisCell();
            if (!(c instanceof CellTunnel)) continue;
            v.add(((CellTunnel)((Object)c)).getCellTunnelInfo());
        }
        return v;
    }

    List<String> getCellNames() {
        return new ArrayList<String>(this._cellList.keySet());
    }

    String getUnique() {
        return this.COUNTER_ENCODING.encode(Longs.toByteArray((long)this._uniqueCounter.next()));
    }

    CellInfo getCellInfo(String name) {
        CellNucleus nucleus = this.getCell(name);
        return nucleus == null ? null : nucleus._getCellInfo();
    }

    Thread[] getThreads(String name) {
        CellNucleus nucleus = this.getCell(name);
        return nucleus == null ? null : nucleus.getThreads();
    }

    private void sendToAll(CellEvent event) {
        for (CellNucleus nucleus : this._publicCellList.values()) {
            nucleus.addToEventQueue(event);
        }
    }

    String getCellDomainName() {
        return this._cellDomainName;
    }

    CompletableFuture<?> kill(CellNucleus nucleus) {
        return this.kill(nucleus, nucleus);
    }

    CompletableFuture<?> kill(CellNucleus sender, String cellName) {
        CellNucleus nucleus = (CellNucleus)this._cellList.get(cellName);
        if (nucleus == null) {
            return CompletableFuture.failedFuture(new NoSuchElementException("No such cell: " + cellName));
        }
        return this.kill(sender, nucleus);
    }

    void listThreadGroupOf(String cellName) {
        CellNucleus nucleus = (CellNucleus)this._cellList.get(cellName);
        if (nucleus != null) {
            nucleus.threadGroupList();
        }
    }

    Optional<CellNucleus> findCellNucleus(Thread thread) {
        ThreadGroup targetGroup = thread.getThreadGroup();
        Optional<CellNucleus> result = this.findCellNucleus(targetGroup);
        while (!result.isPresent() && targetGroup.getParent() != null) {
            targetGroup = targetGroup.getParent();
            result = this.findCellNucleus(targetGroup);
        }
        return result;
    }

    private Optional<CellNucleus> findCellNucleus(ThreadGroup targetGroup) {
        return this._cellList.values().stream().filter(n -> n.getThreadGroup().equals(targetGroup)).findAny();
    }

    void listKillerThreadGroup() {
        CellGlue.listThreadGroup(this._killerThreadGroup);
    }

    static void listThreadGroup(ThreadGroup threadGroup) {
        Thread[] threads = new Thread[threadGroup.activeCount()];
        int n = threadGroup.enumerate(threads);
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < n; ++i) {
            boolean isAlive;
            Thread thread = threads[i];
            StackTraceElement[] elements = thread.getStackTrace();
            boolean bl = isAlive = elements.length != 0;
            if ((!isAlive || thread.isDaemon()) && !LOGGER.isDebugEnabled()) continue;
            if (sb.length() > 0) {
                sb.append('\n');
            }
            sb.append("Thread: ").append(thread.getName()).append(" [");
            sb.append(isAlive ? "A" : "-");
            sb.append(thread.isDaemon() ? "D" : "-");
            sb.append(thread.isInterrupted() ? "I" : "-");
            sb.append("] (").append(thread.getPriority()).append(") ").append((Object)thread.getState()).append('\n');
            for (int j = 0; j < elements.length; ++j) {
                if (j > 0) {
                    sb.append('\n');
                }
                StackTraceElement el = elements[j];
                sb.append("    ").append(el);
            }
        }
        LOGGER.warn("Thread Group \"{}\":\n{}", (Object)threadGroup.getName(), (Object)sb.toString());
    }

    CellNucleus getCell(String cellName) {
        return (CellNucleus)this._cellList.get(cellName);
    }

    synchronized boolean join(String cellName, long timeout) throws InterruptedException {
        if (timeout == 0L) {
            while (this.getCell(cellName) != null) {
                this.wait();
            }
            return true;
        }
        while (this.getCell(cellName) != null && timeout > 0L) {
            long time = System.currentTimeMillis();
            this.wait(timeout);
            timeout -= System.currentTimeMillis() - time;
        }
        return timeout > 0L;
    }

    synchronized void destroy(CellNucleus nucleus) {
        String cellName = nucleus.getCellName();
        if (this._publicCellList.remove(cellName, nucleus)) {
            LOGGER.warn("Apparently cell {} wasn't unpublished before being destroyed. Please contact support@dcache.org.", (Object)cellName);
        }
        if (!this._cellList.remove(cellName, nucleus)) {
            LOGGER.warn("Apparently cell {} wasn't registered before being destroyed. Please contact support@dcache.org.", (Object)cellName);
        }
        if (this._killedCells.remove(nucleus) == null) {
            LOGGER.warn("Apparently cell {} wasn't killed before being destroyed. Please contact support@dcache.org.", (Object)cellName);
        }
        this.notifyAll();
    }

    private synchronized CompletableFuture<?> kill(CellNucleus source, CellNucleus destination) {
        String cellToKill = destination.getCellName();
        if (this._cellList.get(cellToKill) != destination) {
            return CompletableFuture.failedFuture(new NoSuchElementException("No such cell: " + cellToKill));
        }
        return this._killedCells.computeIfAbsent(destination, d -> this.doKill(source, (CellNucleus)d));
    }

    private synchronized CompletableFuture<?> doKill(CellNucleus source, CellNucleus destination) {
        String cellToKill = destination.getCellName();
        Collection<CellRoute> routes = this._routingTable.delete(destination.getThisAddress());
        this._publicCellList.remove(cellToKill, destination);
        for (CellRoute route : routes) {
            this.sendToAll(new CellEvent(route, 8));
        }
        CellPath sourceAddr = new CellPath(source.getCellName(), source.getCellDomainName());
        KillEvent killEvent = new KillEvent(cellToKill, sourceAddr);
        this.sendToAll(new CellEvent(cellToKill, 4));
        Runnable command = () -> destination.shutdown(killEvent);
        try {
            return CompletableFuture.runAsync(command, (Executor)this._killerExecutor);
        }
        catch (OutOfMemoryError e) {
            return CompletableFuture.runAsync(command, (Executor)this._emergencyKillerExecutor);
        }
    }

    void sendMessage(CellMessage msg, boolean resolveLocally, boolean resolveRemotely) throws SerializationException {
        if (!msg.isStreamMode()) {
            msg = msg.encodeWith(this._serializer);
        }
        CellPath destination = msg.getDestinationPath();
        LOGGER.trace("sendMessage : {} send to {}", (Object)msg.getUOID(), (Object)destination);
        this.sendMessage(msg, destination.getCurrent(), resolveLocally, resolveRemotely, 16);
    }

    private void sendMessage(CellMessage msg, CellAddressCore address, boolean resolveLocally, boolean resolveRemotely, int steps) {
        CellPath destination = msg.getDestinationPath();
        boolean hasDestinationChanged = false;
        boolean hasTopicRoutes = false;
        while (steps > 0) {
            while (address.equals(this._domainAddress)) {
                if (!destination.next()) {
                    this.sendException(msg, "*");
                    return;
                }
                address = destination.getCurrent();
                hasDestinationChanged = true;
            }
            LOGGER.trace("sendMessage : next hop at {}: {}", (Object)steps, (Object)address);
            if (address.getCellDomainName().equals(this._cellDomainName)) {
                if (!this.deliverLocally(msg, address)) {
                    this.sendException(msg, address.toString());
                }
                return;
            }
            if (address.isLocalAddress()) {
                if (resolveLocally && this.deliverLocally(msg, address)) {
                    return;
                }
                for (CellRoute route : this._routingTable.findTopicRoutes(address)) {
                    boolean isLocalSubscriber;
                    CellAddressCore target = route.getTarget();
                    boolean bl = isLocalSubscriber = !target.isDomainAddress();
                    if (isLocalSubscriber || resolveRemotely) {
                        CellMessage m = msg.clone();
                        if (isLocalSubscriber) {
                            m.getDestinationPath().replaceCurrent(target);
                        }
                        this.sendMessage(m, target, true, resolveRemotely, steps - 1);
                    }
                    hasTopicRoutes = true;
                }
            }
            if (!hasDestinationChanged && msg.getSourcePath().getDestinationAddress().equals(address)) {
                if (!hasTopicRoutes) {
                    this.sendException(msg, address.toString());
                }
                return;
            }
            CellRoute route = this._routingTable.find(address, this.getZone(), resolveRemotely);
            if (route == null) {
                LOGGER.trace("sendMessage : no route destination for : {}", (Object)address);
                if (!hasTopicRoutes) {
                    this.sendException(msg, address.toString());
                }
                return;
            }
            LOGGER.trace("sendMessage : using route : {}", (Object)route);
            address = route.getTarget();
            if (route.getRouteType() == 6 || route.getRouteType() == 2 && !address.isDomainAddress()) {
                destination.replaceCurrent(address);
                hasDestinationChanged = true;
            }
            resolveLocally = true;
            resolveRemotely = true;
            --steps;
        }
        LOGGER.trace("sendMessage : max route iteration reached: {}", (Object)destination);
        this.sendException(msg, address.toString());
    }

    private boolean deliverLocally(CellMessage msg, CellAddressCore address) {
        CellNucleus destNucleus = (CellNucleus)this._publicCellList.get(address.getCellName());
        if (destNucleus != null) {
            CellPath destinationPath = msg.getDestinationPath();
            if (address.equals(destinationPath.getCurrent())) {
                try {
                    destNucleus.addToEventQueue(new MessageEvent(msg.decode()));
                }
                catch (SerializationException e) {
                    LOGGER.error("Received malformed message from {} with UOID {} and session [{}]: {}", new Object[]{msg.getSourcePath(), msg.getUOID(), msg.getSession(), e.getMessage()});
                    this.sendException(msg, address.toString());
                }
            } else if (msg.getSourcePath().hops() > 30) {
                LOGGER.error("Hop count exceeds 30: {}", (Object)msg);
                this.sendException(msg, address.toString());
            } else {
                msg.addSourceAddress(this._domainAddress);
                destNucleus.addToEventQueue(new RoutedMessageEvent(msg));
            }
            return true;
        }
        return false;
    }

    private void sendException(CellMessage msg, String routeTarget) throws SerializationException {
        if (msg.getSourceAddress().getCellName().equals("*")) {
            Serializable messageObject = msg.decode().getMessageObject();
            if (messageObject instanceof NoRouteToCellException) {
                LOGGER.info("Unable to notify {} about delivery failure of message sent to {}: No route for {}\u00a0in {}.", new Object[]{msg.getDestinationPath(), ((NoRouteToCellException)messageObject).getDestinationPath(), routeTarget, this._cellDomainName});
            } else {
                LOGGER.warn("Message from {} could not be delivered because no route to {}\u00a0is known.", (Object)msg.getSourcePath(), (Object)routeTarget);
            }
        } else {
            LOGGER.debug("Message from {} could not be delivered because no route to {}\u00a0is known; the sender will be notified.", (Object)msg.getSourcePath(), (Object)routeTarget);
            CellMessage envelope = new CellMessage(msg.getSourcePath().revert(), (Serializable)new NoRouteToCellException(msg, "Route for >" + routeTarget + "< not found at >" + this._cellDomainName + "<"));
            envelope.setLastUOID(msg.getUOID());
            envelope.addSourceAddress(this._domainAddress);
            this.sendMessage(envelope, true, true);
        }
    }

    public String toString() {
        return this._cellDomainName;
    }

    @Nonnull
    public CuratorFramework getCuratorFramework() {
        return this._curatorFramework;
    }

    public void shutdown() {
        this._curatorFramework.close();
        this._killerExecutor.shutdown();
    }
}

