/*
 * Decompiled with CFR 0.152.
 */
package dmg.cells.services;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import com.google.common.net.HostAndPort;
import dmg.cells.network.LocationManagerConnector;
import dmg.cells.nucleus.CellAdapter;
import dmg.cells.nucleus.CellDomainRole;
import dmg.cells.nucleus.CellEvent;
import dmg.cells.nucleus.CellEventListener;
import dmg.cells.services.login.LoginManager;
import dmg.cells.zookeeper.LmPersistentNode;
import dmg.util.CommandException;
import dmg.util.command.Argument;
import dmg.util.command.Command;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.net.InetAddress;
import java.net.URI;
import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import javax.net.SocketFactory;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.api.ACLBackgroundPathAndBytesable;
import org.apache.curator.framework.recipes.cache.ChildData;
import org.apache.curator.framework.recipes.cache.NodeCache;
import org.apache.curator.framework.recipes.cache.NodeCacheListener;
import org.apache.curator.framework.recipes.cache.PathChildrenCache;
import org.apache.curator.framework.recipes.cache.PathChildrenCacheEvent;
import org.apache.curator.utils.CloseableUtils;
import org.apache.curator.utils.ZKPaths;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.data.Stat;
import org.dcache.ssl.CanlSslSocketCreator;
import org.dcache.util.Args;
import org.dcache.util.ColumnWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LocationManager
extends CellAdapter {
    private static final Logger LOGGER = LoggerFactory.getLogger(LocationManager.class);
    private static final String ZK_CORES_PLAIN = "/dcache/lm/cores";
    private static final String ZK_CORES_URI = "/dcache/lm/cores-uri";
    private static final String ZK_CORE_CONFIG = "/dcache/lm/core-config";
    private final CoreDomains coreDomains;
    private final CoreConfig coreConfig;
    private final Args args = this.getArgs();
    private final CellDomainRole role;
    private final Client client;

    public LocationManager(String name, String args) throws CommandException, IOException, BadConfigException {
        super(name, "System", args);
        this.coreDomains = CoreDomains.createWithMode(this.getCellDomainName(), this.getCuratorFramework(), this.args.getOpt("mode"));
        if (this.args.hasOption("role")) {
            this.role = CellDomainRole.valueOf(this.args.getOption("role").toUpperCase());
            switch (this.role) {
                case CORE: {
                    Preconditions.checkArgument((this.args.argc() >= 1 ? 1 : 0) != 0, (Object)"Listening port is required.");
                    this.client = new CoreClient();
                    this.coreDomains.onChange(this.client::update);
                    this.coreConfig = new CoreConfig(this.getCuratorFramework(), this.client::reset);
                    break;
                }
                default: {
                    this.client = new Client();
                    this.coreDomains.onChange(this.client::update);
                    this.coreConfig = null;
                    break;
                }
            }
        } else {
            this.role = null;
            this.client = null;
            this.coreConfig = null;
        }
    }

    @Override
    protected void started() {
        try {
            this.coreDomains.start();
            if (this.coreConfig != null) {
                this.coreConfig.start();
            }
            if (this.client != null) {
                this.client.start();
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        catch (RuntimeException e) {
            LOGGER.error("Failed to start location manager", (Throwable)e);
            this.kill();
        }
        catch (Exception e) {
            LOGGER.error("Failed to start location manager: {}", (Object)e.toString());
            this.kill();
        }
    }

    @Override
    public void stopping() {
        CloseableUtils.closeQuietly((Closeable)this.coreDomains);
        if (this.client != null) {
            this.client.close();
        }
    }

    private LoginManager startListener(String args) throws ExecutionException, InterruptedException {
        String cellName = "l*";
        String cellClass = "dmg.cells.network.LocationMgrTunnel";
        String cellArgs = args + " " + cellClass + " -prot=raw -role=" + this.role;
        LOGGER.info("Starting acceptor with arguments: {}", (Object)cellArgs);
        LoginManager c = new LoginManager(cellName, "System", cellArgs);
        c.start().get();
        LOGGER.info("Created : {}", (Object)c);
        return c;
    }

    private String startConnector(String remoteDomain, CoreDomainInfo domainInfo) throws ExecutionException, InterruptedException, BadConfigException {
        SocketFactory socketFactory;
        HostAndPort where;
        Preconditions.checkArgument((boolean)this.args.hasOption("mode"), (Object)"No mode specified to run connector");
        String cellName = "c-" + remoteDomain + "*";
        String clientKey = this.args.getOpt("clientKey");
        clientKey = clientKey != null && !clientKey.isEmpty() ? "-clientKey=" + clientKey : "";
        String clientName = this.args.getOpt("clientUserName");
        clientName = clientName != null && !clientName.isEmpty() ? "-clientUserName=" + clientName : "";
        Mode mode = Mode.fromString(this.args.getOption("mode"));
        block1 : switch (mode) {
            case PLAIN: {
                LOGGER.info("Starting Connection in mode: PLAIN with {}", (Object)this.args.getArguments());
                where = domainInfo.getEndpointForSchema("tcp").orElseThrow(BadConfigException::new);
                socketFactory = SocketFactory.getDefault();
                break;
            }
            case TLS: {
                LOGGER.info("Starting Connection in mode: TLS with {}", (Object)this.args.getArguments());
                where = domainInfo.getEndpointForSchema("tls").orElseThrow(BadConfigException::new);
                try {
                    switch (this.role) {
                        case CORE: {
                            socketFactory = new CanlSslSocketCreator(new Args((CharSequence)this.args.getOption("socketfactory")));
                            break block1;
                        }
                    }
                    this.args.shift(2);
                    socketFactory = new CanlSslSocketCreator(this.args);
                    break;
                }
                catch (IOException e) {
                    throw new IllegalArgumentException(String.format("Problem creating socket factory with arguments: %s", this.args.toString()));
                }
            }
            default: {
                throw new IllegalArgumentException("Invalid mode to start connector: " + this.args.getOption("mode"));
            }
        }
        String cellArgs = "-domain=" + remoteDomain + " -lm=" + this.getCellName() + " -role=" + this.role + " -where=" + where + " " + clientKey + " " + clientName;
        LOGGER.info("Starting connector with {}", (Object)cellArgs);
        LocationManagerConnector c = new LocationManagerConnector(cellName, cellArgs, socketFactory);
        c.start().get();
        return c.getCellName();
    }

    private void killConnector(String cell) {
        LOGGER.info("Killing connector {}", (Object)cell);
        this.getNucleus().kill(cell);
    }

    private void killListener(String cell) {
        LOGGER.info("Killing listener {}", (Object)cell);
        this.getNucleus().kill(cell);
    }

    @Command(name="get core-config", hint="get current mode of operation for CoreDomain", description="Get the current operating modes of the CoreDomain. It should be one of the following \"none\", \"tls\" or \"none,tls\".")
    class GetCoreConfigCommand
    implements Callable<String> {
        GetCoreConfigCommand() {
        }

        @Override
        public String call() throws Exception {
            return new String((byte[])LocationManager.this.getCuratorFramework().getData().forPath(LocationManager.ZK_CORE_CONFIG), StandardCharsets.UTF_8);
        }
    }

    @Command(name="set core-config", hint="set operating mode for CoreDomain", description="Specify the mode to be none, tls or none,tls in which the CoreDomain should run")
    class SetCoreConfigCommand
    implements Callable<String> {
        @Argument(index=0, usage="Mode in which CoreDomain should run.")
        String _modes;

        SetCoreConfigCommand() {
        }

        @Override
        public synchronized String call() throws Exception {
            CuratorFramework curator = LocationManager.this.getCuratorFramework();
            Set modes = Sets.newHashSet((Object[])this._modes.split(",")).stream().map(String::trim).filter(s -> !s.isEmpty()).collect(Collectors.toSet());
            if (modes.stream().allMatch(Mode::isValid)) {
                String config = Joiner.on((String)",").join(modes);
                byte[] data = config.getBytes(StandardCharsets.UTF_8);
                if (curator.checkExists().forPath(LocationManager.ZK_CORE_CONFIG) != null) {
                    curator.setData().forPath(LocationManager.ZK_CORE_CONFIG, data);
                } else {
                    ((ACLBackgroundPathAndBytesable)curator.create().creatingParentContainersIfNeeded().withMode(CreateMode.PERSISTENT)).forPath(LocationManager.ZK_CORE_CONFIG, data);
                }
                if (Arrays.equals((byte[])curator.getData().forPath(LocationManager.ZK_CORE_CONFIG), data)) {
                    return "Successfully updated CoreDomain mode configuration to " + config;
                }
                return "Could not change CoreDomain configuration to " + config;
            }
            throw new BadConfigException("Invalid Modes provided for CoreDomain configuration. Valid modes are \"none\", \"tls\" or \"none,tls\"");
        }
    }

    @Command(name="ls", hint="list core domain information", description="Provides information on available core domains.")
    class ListCommand
    implements Callable<String> {
        ListCommand() {
        }

        @Override
        public String call() {
            ColumnWriter writer = new ColumnWriter().header("NAME").left("name").space().header("PROTOCOL").left("protocol").space().header("HOST").left("host").space().header("PORT").right("port");
            for (Map.Entry<String, CoreDomainInfo> entry : LocationManager.this.coreDomains.cores().entrySet()) {
                CoreDomainInfo info = entry.getValue();
                info.getEndpointForSchema("tcp").ifPresent(tcp -> writer.row().value("name", entry.getKey()).value("protocol", (Object)"PLAIN").value("host", (Object)tcp.getHost()).value("port", (Object)tcp.getPort()));
                info.getEndpointForSchema("tls").ifPresent(tls -> writer.row().value("name", entry.getKey()).value("protocol", (Object)"TLS").value("host", (Object)tls.getHost()).value("port", (Object)tls.getPort()));
            }
            return writer.toString();
        }
    }

    public class CoreClient
    extends Client {
        private LoginManager lmTls;
        private LoginManager lmPlain;
        private volatile CoreDomainInfo info;

        public CoreClient() {
            this.info = new CoreDomainInfo();
        }

        @Override
        protected boolean shouldConnectTo(String domain) {
            return domain.compareTo(LocationManager.this.getCellDomainName()) < 0;
        }

        @Override
        public void start() throws Exception {
            switch (LocationManager.this.coreConfig.getMode()) {
                case PLAIN: {
                    this.startListenerWithTcp();
                    break;
                }
                case PLAIN_TLS: {
                    this.startListenerWithTcp();
                }
                case TLS: {
                    this.startListenerWithTls();
                    break;
                }
                default: {
                    throw new IllegalArgumentException("Mode " + LocationManager.this.coreConfig.getMode() + "not supported for Core Domain");
                }
            }
            LocationManager.this.coreDomains.setCoreDomainInfoUri(this.info);
        }

        @Override
        public void close() {
            LocationManager.this.coreConfig.close();
        }

        @Override
        public synchronized void reset(Mode mode, State state) {
            try {
                switch (mode) {
                    case PLAIN: {
                        switch (state) {
                            case BRING_UP: {
                                this.startListenerWithTcp();
                                break;
                            }
                            case TEAR_DOWN: {
                                this.stopListenerTcp();
                            }
                        }
                        break;
                    }
                    case TLS: {
                        switch (state) {
                            case BRING_UP: {
                                this.startListenerWithTls();
                                break;
                            }
                            case TEAR_DOWN: {
                                this.stopListenerTls();
                            }
                        }
                        break;
                    }
                }
                LocationManager.this.coreDomains.setCoreDomainInfoUri(this.info);
            }
            catch (LmPersistentNode.PersistentNodeException e) {
                LOGGER.error("Failed to reset location manager on CoreConfig update: {}", (Object)e.getMessage());
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            catch (ExecutionException e) {
                LOGGER.error("Failed to reset location manager on CoreConfig update: {}", (Object)e.getCause().toString());
                LocationManager.this.kill();
            }
            catch (RuntimeException e) {
                LOGGER.error("Failed to reset location manager on CoreConfig update", (Throwable)e);
                LocationManager.this.kill();
            }
            catch (Exception e) {
                LOGGER.error("Failed to reset location manager on CoreConfig update: {}", (Object)e.toString());
                LocationManager.this.kill();
            }
        }

        private void startListenerWithTcp() throws ExecutionException, InterruptedException, UnknownHostException {
            String cellArgs = String.format("%s -netmask='%s'", LocationManager.this.args.argv(0), LocationManager.this.args.getOption("netmask", ""));
            this.lmPlain = LocationManager.this.startListener(cellArgs);
            LOGGER.info("lmPlain: {}; port; {} ", (Object)this.lmPlain, (Object)this.lmPlain.getListenPort());
            this.info.addCore("tcp", InetAddress.getLocalHost().getCanonicalHostName(), this.lmPlain.getListenPort());
        }

        private void startListenerWithTls() throws ExecutionException, InterruptedException, UnknownHostException {
            Preconditions.checkArgument((boolean)LocationManager.this.args.hasOption("socketfactory"), (Object)"No Socketfactory provided to Core Domain for channel encryption");
            String cellArgs = String.format("%s -socketfactory='%s'", Integer.parseInt(LocationManager.this.args.argc() == 1 ? LocationManager.this.args.argv(0) : LocationManager.this.args.argv(1)), LocationManager.this.args.getOpt("socketfactory"));
            this.lmTls = LocationManager.this.startListener(cellArgs);
            LOGGER.info("lmTls: {}; port; {} ", (Object)this.lmTls, (Object)this.lmTls.getListenPort());
            this.info.addCore("tls", InetAddress.getLocalHost().getCanonicalHostName(), this.lmTls.getListenPort());
        }

        private void stopListenerTls() {
            if (this.lmTls != null) {
                LocationManager.this.killListener(this.lmTls.getCellName());
                this.lmTls = null;
            }
            this.info.removeTls();
        }

        private void stopListenerTcp() {
            if (this.lmPlain != null) {
                LocationManager.this.killListener(this.lmPlain.getCellName());
                this.lmPlain = null;
            }
            this.info.removeTcp();
        }
    }

    public class Client
    implements CellEventListener {
        private final Map<String, String> connectors = new HashMap<String, String>();

        public Client() {
            LocationManager.this.addCommandListener(this);
            LocationManager.this.addCellEventListener(this);
        }

        public void start() throws Exception {
        }

        public void close() {
        }

        public void reset(Mode mode, State state) {
        }

        public void update(PathChildrenCacheEvent event) {
            LOGGER.info("{}", (Object)event);
            switch (event.getType()) {
                case CHILD_REMOVED: {
                    String cell = this.connectors.remove(ZKPaths.getNodeFromPath((String)event.getData().getPath()));
                    if (cell == null) break;
                    LocationManager.this.killConnector(cell);
                    break;
                }
                case CHILD_UPDATED: {
                    String cell = this.connectors.remove(ZKPaths.getNodeFromPath((String)event.getData().getPath()));
                    if (cell != null) {
                        LocationManager.this.killConnector(cell);
                    }
                }
                case CHILD_ADDED: {
                    String cell;
                    CoreDomainInfo info = this.infoFromZKEvent(event);
                    String domain = ZKPaths.getNodeFromPath((String)event.getData().getPath());
                    try {
                        if (!this.shouldConnectTo(domain)) break;
                        cell = this.connectors.remove(domain);
                        if (cell != null) {
                            LOGGER.error("About to create tunnel to core domain {}, but to my surprise a tunnel called {} already exists. Will kill it. Please contact support@dcache.org.", (Object)domain, (Object)cell);
                            LocationManager.this.killConnector(cell);
                        }
                        if ((cell = this.connectors.put(domain, LocationManager.this.startConnector(domain, info))) == null) break;
                        LOGGER.error("Created a tunnel to core domain {}, but to my surprise a tunnel called {} already exists. Will kill it. Please contact support@dcache.org.", (Object)domain, (Object)cell);
                        LocationManager.this.killConnector(cell);
                        break;
                    }
                    catch (ExecutionException e) {
                        LOGGER.error("Failed to start tunnel connector to {}: {}", (Object)domain, (Object)e.getCause());
                        break;
                    }
                    catch (InterruptedException e) {
                        break;
                    }
                    catch (BadConfigException be) {
                        LOGGER.error("Invalid ports provided for starting connector in mode {}", (Object)LocationManager.this.args.getOpt("mode"));
                    }
                }
            }
        }

        protected CoreDomainInfo infoFromZKEvent(PathChildrenCacheEvent event) {
            return new CoreDomainInfo(event.getData().getData());
        }

        protected boolean shouldConnectTo(String domain) {
            return true;
        }

        @Override
        public void cellDied(CellEvent ce) {
            this.connectors.values().remove((String)ce.getSource());
        }
    }

    private class CoreConfig
    implements NodeCacheListener,
    Closeable {
        private final CuratorFramework _curator;
        private Mode _mode = Mode.PLAIN;
        private final NodeCache _cache;
        private Stat _current;
        private final BiConsumer<Mode, State> _reset;

        CoreConfig(CuratorFramework curator, BiConsumer<Mode, State> f) {
            this._curator = curator;
            this._cache = new NodeCache(this._curator, LocationManager.ZK_CORE_CONFIG);
            this._reset = f;
        }

        public Mode getMode() {
            return this._mode;
        }

        synchronized void start() throws Exception {
            this._cache.getListenable().addListener((Object)this);
            try {
                this._cache.start(true);
                this.apply(this._cache.getCurrentData());
            }
            catch (KeeperException.ConnectionLossException e) {
                LOGGER.warn("Failed to connect to zookeeper, using mode {} until connection reestablished", (Object)this._mode);
            }
        }

        private void apply(ChildData currentData) {
            if (currentData == null) {
                this._current = null;
                this._mode = Mode.PLAIN;
                LOGGER.info("CoreDomain config node /dcache/lm/core-config not present; assuming mode {}", (Object)this._mode);
            } else if (this._current == null || currentData.getStat().getVersion() > this._current.getVersion()) {
                this._mode = Mode.fromString(new String(currentData.getData(), StandardCharsets.UTF_8));
                LOGGER.info("CoreDomain config node /dcache/lm/core-config switching to mode {}", (Object)this._mode);
                this._current = currentData.getStat();
            } else {
                LOGGER.info("Ignoring spurious CoreDomain config node /dcache/lm/core-config updated");
            }
        }

        public synchronized void nodeChanged() throws Exception {
            Set<Mode> oldModes = this._mode.getModeAsSet();
            this.apply(this._cache.getCurrentData());
            Set<Mode> curModes = this._mode.getModeAsSet();
            Set up = Sets.difference(curModes, oldModes).copyInto(new HashSet());
            LOGGER.info("Following modes from CoreDomain are being brought up: [{}]", (Object)Joiner.on((char)',').join((Iterable)up));
            up.stream().forEach(u -> this._reset.accept((Mode)((Object)u), State.BRING_UP));
            Set down = Sets.difference(oldModes, curModes).copyInto(new HashSet());
            LOGGER.info("Following modes from CoreDomain are being taken down: [{}]", (Object)Joiner.on((char)',').join((Iterable)down));
            down.stream().forEach(d -> this._reset.accept((Mode)((Object)d), State.TEAR_DOWN));
        }

        @Override
        public void close() {
            CloseableUtils.closeQuietly((Closeable)this._cache);
        }
    }

    private static class CoreDomainLocations
    extends CoreDomains {
        private final ConnectionType connectionType;

        CoreDomainLocations(String domainName, CuratorFramework client, ConnectionType connectionType) {
            super(domainName, client, new PathChildrenCache(client, LocationManager.ZK_CORES_URI, true));
            this.connectionType = connectionType;
        }

        @Override
        Map<String, CoreDomainInfo> cores() {
            HashMap<String, CoreDomainInfo> coresInfo = new HashMap<String, CoreDomainInfo>();
            for (ChildData d : this.cores.getCurrentData()) {
                CoreDomainInfo urlInfo = new CoreDomainInfo(d.getData());
                if (!urlInfo.getEndpointForSchema(this.connectionType.getConnectionScheme()).isPresent()) continue;
                coresInfo.put(ZKPaths.getNodeFromPath((String)d.getPath()), urlInfo);
            }
            return coresInfo;
        }
    }

    private static abstract class CoreDomains
    implements Closeable {
        private final String domainName;
        private final CuratorFramework client;
        protected final PathChildrenCache cores;
        private LmPersistentNode<HostAndPort> localLegacy;
        private LmPersistentNode<CoreDomainInfo> localUri;

        protected CoreDomains(String domainName, CuratorFramework client, PathChildrenCache cores) {
            this.domainName = domainName;
            this.client = client;
            this.cores = cores;
        }

        public static CoreDomains createWithMode(String domainName, CuratorFramework client, String mode) throws BadConfigException {
            LOGGER.info("Creating CoreDomains: {}, {}", (Object)domainName, (Object)mode);
            ConnectionType type = ConnectionType.fromConfig(mode).orElseThrow(() -> new BadConfigException("Bad mode " + mode));
            return new CoreDomainLocations(domainName, client, type);
        }

        void onChange(Consumer<PathChildrenCacheEvent> consumer) {
            this.cores.getListenable().addListener((client, event) -> consumer.accept(event));
        }

        void start() throws Exception {
            this.cores.start();
        }

        @Override
        public void close() throws IOException {
            CloseableUtils.closeQuietly((Closeable)this.cores);
            if (this.localLegacy != null) {
                CloseableUtils.closeQuietly(this.localLegacy);
            }
            if (this.localUri != null) {
                CloseableUtils.closeQuietly(this.localUri);
            }
        }

        Map<String, CoreDomainInfo> cores() {
            return Collections.emptyMap();
        }

        String pathOf(String core, String domainName) {
            return ZKPaths.makePath((String)core, (String)domainName);
        }

        private void setCoreDomainInfoUri(CoreDomainInfo coreDomainInfo) throws LmPersistentNode.PersistentNodeException {
            this.localUri = LmPersistentNode.createOrUpdate(this.client, this.pathOf(LocationManager.ZK_CORES_URI, this.domainName), coreDomainInfo, CoreDomainInfo::toBytes, this.localUri);
            Optional<HostAndPort> plainConnection = coreDomainInfo.getEndpointForSchema("tcp");
            if (plainConnection.isPresent()) {
                this.localLegacy = LmPersistentNode.createOrUpdate(this.client, this.pathOf(LocationManager.ZK_CORES_PLAIN, this.domainName), plainConnection.get(), address -> address.toString().getBytes(StandardCharsets.US_ASCII), this.localLegacy);
            }
        }
    }

    private static class BadConfigException
    extends Exception {
        public BadConfigException() {
        }

        public BadConfigException(String message) {
            super(message);
        }
    }

    private static class CoreDomainInfo {
        private final Set<URI> endpoints = new HashSet<URI>();

        public CoreDomainInfo() {
        }

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        public CoreDomainInfo(byte[] bytes) {
            this();
            try (ByteArrayInputStream bais = new ByteArrayInputStream(bytes);
                 DataInputStream in = new DataInputStream(bais);){
                block19: while (in.available() > 0) {
                    String info = in.readUTF();
                    URI entry = URI.create(info);
                    switch (entry.getScheme()) {
                        case "tls": 
                        case "tcp": {
                            this.endpoints.add(entry);
                            continue block19;
                        }
                    }
                    LOGGER.warn("Unknown Scheme for LocationManager Cores: {}", (Object)entry);
                }
                return;
            }
            catch (IOException ie) {
                throw new IllegalArgumentException("Failed deserializing LocationManager Cores as uri: {}", ie.getCause());
            }
        }

        void addCore(String scheme, String host, int port) {
            switch (scheme) {
                case "tls": 
                case "tcp": {
                    this.endpoints.add(URI.create(String.format("%s://%s:%s", scheme, host, port)));
                    break;
                }
                default: {
                    throw new RuntimeException("Unknown Scheme " + scheme + " for LocationManager Cores");
                }
            }
        }

        public void removeTcp() {
            this.endpoints.removeIf(u -> u.getScheme().equalsIgnoreCase("tcp"));
        }

        public void removeTls() {
            this.endpoints.removeIf(u -> u.getScheme().equalsIgnoreCase("tls"));
        }

        /*
         * Enabled aggressive exception aggregation
         */
        byte[] toBytes() {
            try (ByteArrayOutputStream baos = new ByteArrayOutputStream();){
                Object object;
                try (DataOutputStream out = new DataOutputStream(baos);){
                    for (URI uri : this.endpoints) {
                        out.writeUTF(uri.toString());
                    }
                    object = baos.toByteArray();
                }
                return object;
            }
            catch (IOException ie) {
                LOGGER.warn("Failed to serialize CoreDomain Info {}", (Object)this);
                return new byte[0];
            }
        }

        public Optional<HostAndPort> getEndpointForSchema(String schema) {
            return this.endpoints.stream().filter(u -> u.getScheme().equalsIgnoreCase(schema)).findAny().map(u -> HostAndPort.fromParts((String)u.getHost(), (int)u.getPort()));
        }

        public boolean equals(Object other) {
            if (this == other) {
                return true;
            }
            if (!(other instanceof CoreDomainInfo)) {
                return false;
            }
            CoreDomainInfo that = (CoreDomainInfo)other;
            return Objects.equals(this.endpoints, that.endpoints);
        }

        public int hashCode() {
            return this.endpoints.hashCode();
        }

        public String toString() {
            return this.endpoints.toString();
        }
    }

    static enum Mode {
        PLAIN("none"),
        PLAIN_TLS("none,tls"),
        TLS("tls");

        private final String _mode;
        private static final ImmutableSet<Mode> tls;
        private static final ImmutableSet<Mode> plain;
        private static final ImmutableSet<Mode> plainAndTls;

        private Mode(String mode) {
            this._mode = mode;
        }

        public String toString() {
            return this._mode;
        }

        public String getMode() {
            return this._mode;
        }

        public Set<Mode> getModeAsSet() {
            switch (this) {
                case PLAIN_TLS: {
                    return plainAndTls;
                }
                case TLS: {
                    return tls;
                }
            }
            return plain;
        }

        public static Mode fromString(String mode) {
            String m = Mode.filterAndSort(mode);
            for (Mode b : Mode.values()) {
                if (!b._mode.equalsIgnoreCase(m)) continue;
                return b;
            }
            throw new IllegalArgumentException("No Mode of type: " + mode);
        }

        public static boolean isValid(String mode) {
            String m = Mode.filterAndSort(mode);
            for (Mode b : Mode.values()) {
                if (!b._mode.equalsIgnoreCase(m)) continue;
                return true;
            }
            return false;
        }

        private static String filterAndSort(String mode) {
            return Arrays.stream(mode.split(",")).map(String::trim).filter(s -> !s.isEmpty()).distinct().sorted().collect(Collectors.joining(","));
        }

        static {
            tls = ImmutableSet.of((Object)((Object)TLS));
            plain = ImmutableSet.of((Object)((Object)PLAIN));
            plainAndTls = ImmutableSet.of((Object)((Object)PLAIN), (Object)((Object)TLS));
        }
    }

    public static enum ConnectionType {
        PLAIN("none", "tcp"),
        TLS("tls", "tls");

        private static final ImmutableMap<String, ConnectionType> CONFIG_TO_VALUE;
        private final String config;
        private final String scheme;

        private ConnectionType(String config, String scheme) {
            this.config = config;
            this.scheme = scheme;
        }

        public static Optional<ConnectionType> fromConfig(String value) {
            return Optional.ofNullable((ConnectionType)((Object)CONFIG_TO_VALUE.get((Object)value)));
        }

        public String getConnectionScheme() {
            return this.scheme;
        }

        static {
            CONFIG_TO_VALUE = ImmutableMap.of((Object)ConnectionType.PLAIN.config, (Object)((Object)PLAIN), (Object)ConnectionType.TLS.config, (Object)((Object)TLS));
        }
    }

    static enum State {
        BRING_UP,
        TEAR_DOWN;

    }
}

