/*
 * Decompiled with CFR 0.152.
 */
package org.dcache.util;

import com.google.common.base.Strings;
import com.google.common.primitives.Ints;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

public class ColumnWriter {
    private final List<String> headers = new ArrayList<String>();
    private final List<Integer> spaces = new ArrayList<Integer>();
    private final List<Column> columns = new ArrayList<Column>();
    private final List<Row> rows = new ArrayList<Row>();
    private boolean abbrev;

    public ColumnWriter() {
        this.spaces.add(0);
    }

    private void addColumn(Column column) {
        this.columns.add(column);
        if (this.headers.size() < this.columns.size()) {
            this.headers.add(null);
        }
        this.spaces.add(0);
    }

    public ColumnWriter left(String name) {
        this.addColumn(new LeftColumn(name));
        return this;
    }

    public ColumnWriter right(String name) {
        this.addColumn(new RightColumn(name));
        return this;
    }

    public ColumnWriter bytes(String name) {
        this.addColumn(new ByteColumn(name));
        return this;
    }

    public ColumnWriter fixed(String value) {
        this.addColumn(new FixedColumn(value));
        return this;
    }

    public ColumnWriter space() {
        int last = this.spaces.size() - 1;
        this.spaces.set(last, this.spaces.get(last) + 1);
        return this;
    }

    public ColumnWriter abbreviateBytes(boolean abbrev) {
        this.abbrev = abbrev;
        return this;
    }

    public TabulatedRow row() {
        TabulatedRow row = new TabulatedRow();
        this.rows.add(row);
        return row;
    }

    public void row(String value) {
        this.rows.add(new LiteralRow(value));
    }

    public String toString() {
        if (this.rows.isEmpty()) {
            return "";
        }
        List<Integer> widths = this.calculateWidths();
        ArrayList<Integer> spaces = new ArrayList<Integer>(this.spaces);
        StringWriter result = new StringWriter();
        try (PrintWriter out = new PrintWriter(result);){
            String header = this.renderHeader(spaces, widths);
            if (!header.isEmpty()) {
                out.println(header);
            }
            for (Row row : this.rows) {
                row.render(this.columns, spaces, widths, out);
            }
        }
        return result.toString();
    }

    private List<Integer> calculateWidths() {
        int columnCount = this.columns.size();
        int[] widths = new int[columnCount];
        for (Row row : this.rows) {
            for (int i = 0; i < columnCount; ++i) {
                widths[i] = Math.max(widths[i], row.width(this.columns.get(i)));
            }
        }
        return Ints.asList((int[])widths);
    }

    private String renderHeader(List<Integer> spaces, List<Integer> widths) {
        int columnCount = this.columns.size();
        StringBuilder line = new StringBuilder();
        int columnEnd = 0;
        for (int i = 0; i < columnCount; ++i) {
            String header = this.headers.get(i);
            if (header != null) {
                int headerStart = this.columns.get(i).isLeftJustified() ? columnEnd + spaces.get(i) : columnEnd + spaces.get(i) + widths.get(i) - header.length();
                if (line.length() >= headerStart) {
                    int newHeaderStart = line.length() > 0 ? line.length() + 1 : 0;
                    spaces.set(i, spaces.get(i) + newHeaderStart - headerStart);
                    headerStart = newHeaderStart;
                }
                for (int c = line.length(); c < headerStart; ++c) {
                    line.append(' ');
                }
                line.append(header);
            }
            columnEnd = columnEnd + spaces.get(i) + widths.get(i);
        }
        return line.toString();
    }

    public ColumnWriter date(String name) {
        this.addColumn(new DateColumn(name));
        return this;
    }

    public ColumnWriter header(String text) {
        this.headers.add(text);
        return this;
    }

    private static class LiteralRow
    implements Row {
        private final String value;

        private LiteralRow(String value) {
            this.value = value;
        }

        @Override
        public int width(Column column) {
            return 0;
        }

        @Override
        public void render(List<Column> columns, List<Integer> spaces, List<Integer> widths, PrintWriter out) {
            out.println(this.value);
        }
    }

    public static class TabulatedRow
    implements Row {
        private final Map<String, Object> values = new HashMap<String, Object>();

        public TabulatedRow value(String column, Object value) {
            this.values.put(column, value);
            return this;
        }

        @Override
        public int width(Column column) {
            return column.width(this.values.get(column.name()));
        }

        @Override
        public void render(List<Column> columns, List<Integer> spaces, List<Integer> widths, PrintWriter out) {
            int size = columns.size();
            for (int i = 0; i < size; ++i) {
                for (int c = spaces.get(i).intValue(); c > 0; --c) {
                    out.append(' ');
                }
                Column column = columns.get(i);
                Object value = this.values.get(column.name());
                column.render(value, widths.get(i), out);
            }
            out.println();
        }
    }

    private static interface Row {
        public int width(Column var1);

        public void render(List<Column> var1, List<Integer> var2, List<Integer> var3, PrintWriter var4);
    }

    private static class DateColumn
    extends AbstractColumn {
        public static final String FORMAT = "%1$tF %1$tT";
        public static final int WIDTH_OF_FORMAT = 19;

        public DateColumn(String name) {
            super(name);
        }

        @Override
        public boolean isLeftJustified() {
            return false;
        }

        @Override
        public int width(Object value) {
            return 19;
        }

        @Override
        public void render(Object value, int actualWidth, PrintWriter out) {
            if (value == null) {
                while (actualWidth-- > 0) {
                    out.append(' ');
                }
            } else {
                out.format(FORMAT, value);
            }
        }
    }

    private static class FixedColumn
    implements Column {
        private final String value;

        public FixedColumn(String value) {
            this.value = value;
        }

        @Override
        public boolean isLeftJustified() {
            return true;
        }

        @Override
        public String name() {
            return null;
        }

        @Override
        public int width(Object o) {
            return this.value.length();
        }

        @Override
        public void render(Object o, int actualWidth, PrintWriter out) {
            out.append(Strings.padEnd((String)this.value, (int)actualWidth, (char)' '));
        }
    }

    private class ByteColumn
    extends AbstractColumn {
        public static final long PETA = 1000000000000000L;
        public static final long TERA = 1000000000000L;
        public static final long GIGA = 1000000000L;
        public static final long MEGA = 1000000L;
        public static final long KILO = 1000L;

        public ByteColumn(String name) {
            super(name);
        }

        @Override
        public boolean isLeftJustified() {
            return false;
        }

        @Override
        public int width(Object value) {
            return ColumnWriter.this.abbrev ? 4 : Objects.toString(value, "").length();
        }

        private void render(long value, long factor, char unit, PrintWriter out) {
            double tmp = (double)value / (double)factor;
            if (tmp >= 0.0 && tmp < 9.95) {
                out.format("%3.1f", tmp).append(unit);
            } else {
                out.format("%3.0f", tmp).append(unit);
            }
        }

        @Override
        public void render(Object o, int actualWidth, PrintWriter out) {
            if (o == null) {
                while (actualWidth-- > 0) {
                    out.append(' ');
                }
            } else {
                long value = (Long)o;
                long abs = Math.abs(value);
                if (!ColumnWriter.this.abbrev) {
                    out.format("%" + actualWidth + "d", value);
                } else if (abs >= 1000000000000000L) {
                    this.render(value, 1000000000000000L, 'P', out);
                } else if (abs >= 1000000000000L) {
                    this.render(value, 1000000000000L, 'T', out);
                } else if (abs >= 1000000000L) {
                    this.render(value, 1000000000L, 'G', out);
                } else if (abs >= 1000000L) {
                    this.render(value, 1000000L, 'M', out);
                } else if (abs >= 1000L) {
                    this.render(value, 1000L, 'k', out);
                } else {
                    out.format("%3d", value).append('B');
                }
            }
        }
    }

    private static class RightColumn
    extends RegularColumn {
        private RightColumn(String name) {
            super(name);
        }

        @Override
        public boolean isLeftJustified() {
            return false;
        }

        @Override
        public void render(Object value, int actualWidth, PrintWriter out) {
            out.append(Strings.padStart((String)Objects.toString(value, ""), (int)actualWidth, (char)' '));
        }
    }

    private static class LeftColumn
    extends RegularColumn {
        private LeftColumn(String name) {
            super(name);
        }

        @Override
        public boolean isLeftJustified() {
            return true;
        }

        @Override
        public void render(Object value, int actualWidth, PrintWriter out) {
            out.append(Strings.padEnd((String)Objects.toString(value, ""), (int)actualWidth, (char)' '));
        }
    }

    private static abstract class RegularColumn
    extends AbstractColumn {
        private RegularColumn(String name) {
            super(name);
        }

        @Override
        public int width(Object value) {
            return Objects.toString(value, "").length();
        }
    }

    private static abstract class AbstractColumn
    implements Column {
        protected final String name;

        public AbstractColumn(String name) {
            this.name = name;
        }

        @Override
        public String name() {
            return this.name;
        }
    }

    private static interface Column {
        public String name();

        public boolean isLeftJustified();

        public int width(Object var1);

        public void render(Object var1, int var2, PrintWriter var3);
    }
}

