/*
 * Decompiled with CFR 0.152.
 */
package org.dcache.util;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Properties;
import java.util.concurrent.TimeUnit;
import org.apache.commons.math3.stat.descriptive.StatisticalSummary;
import org.dcache.util.Strings;

public class TimeUtils {
    public static final String TIMESTAMP_FORMAT = "yyyy-MM-dd' 'HH:mm:ss.SSS";
    private static final ImmutableMap<TimeUnit, String> SHORT_TIMEUNIT_NAMES = ImmutableMap.builder().put((Object)TimeUnit.NANOSECONDS, (Object)"ns").put((Object)TimeUnit.MICROSECONDS, (Object)"\u00b5s").put((Object)TimeUnit.MILLISECONDS, (Object)"ms").put((Object)TimeUnit.SECONDS, (Object)"s").put((Object)TimeUnit.MINUTES, (Object)"min").put((Object)TimeUnit.HOURS, (Object)"hours").put((Object)TimeUnit.DAYS, (Object)"days").build();
    private static final ImmutableMap<TimeUnit, String> LONG_TIMEUNIT_NAMES = ImmutableMap.builder().put((Object)TimeUnit.NANOSECONDS, (Object)"nanoseconds").put((Object)TimeUnit.MICROSECONDS, (Object)"microseconds").put((Object)TimeUnit.MILLISECONDS, (Object)"milliseconds").put((Object)TimeUnit.SECONDS, (Object)"seconds").put((Object)TimeUnit.MINUTES, (Object)"minutes").put((Object)TimeUnit.HOURS, (Object)"hours").put((Object)TimeUnit.DAYS, (Object)"days").build();
    private static final DecreasingTimeUnitComparator comparator = new DecreasingTimeUnitComparator();

    private TimeUtils() {
    }

    public static CharSequence duration(long duration, TimeUnit units, TimeUnitFormat unitFormat) {
        return TimeUtils.appendDuration(new StringBuilder(), duration, units, unitFormat);
    }

    public static String getFormattedDuration(long duration, TimeUnit unit, String format) {
        return new DurationFormatter(format).format(duration, unit);
    }

    public static String unitStringOf(TimeUnit unit) {
        return (String)SHORT_TIMEUNIT_NAMES.get((Object)unit);
    }

    public static StringBuilder appendDuration(StringBuilder sb, Duration duration, TimeUnitFormat unitFormat) {
        return TimeUtils.appendDuration(sb, duration.toMillis(), TimeUnit.MILLISECONDS, unitFormat);
    }

    public static StringBuilder appendDuration(StringBuilder sb, long duration, TimeUnit units, TimeUnitFormat unitFormat) {
        Preconditions.checkArgument((duration >= 0L ? 1 : 0) != 0);
        ImmutableMap<TimeUnit, String> unitsFormat = unitFormat == TimeUnitFormat.SHORT ? SHORT_TIMEUNIT_NAMES : LONG_TIMEUNIT_NAMES;
        TimeUnit targetUnit = TimeUtils.displayUnitFor(duration, units);
        return sb.append(targetUnit.convert(duration, units)).append(' ').append((String)unitsFormat.get((Object)targetUnit));
    }

    public static TimeUnit displayUnitFor(long duration, TimeUnit units) {
        if (units == TimeUnit.NANOSECONDS && duration < TimeUnit.MICROSECONDS.toNanos(2L)) {
            return TimeUnit.NANOSECONDS;
        }
        if (units.toMicros(duration) < TimeUnit.MILLISECONDS.toMicros(2L) && units.compareTo(TimeUnit.MICROSECONDS) <= 0) {
            return TimeUnit.MICROSECONDS;
        }
        long durationInMillis = units.toMillis(duration);
        if (durationInMillis < TimeUnit.SECONDS.toMillis(2L) && units.compareTo(TimeUnit.MILLISECONDS) <= 0) {
            return TimeUnit.MILLISECONDS;
        }
        if (durationInMillis < TimeUnit.MINUTES.toMillis(2L) && units.compareTo(TimeUnit.SECONDS) <= 0) {
            return TimeUnit.SECONDS;
        }
        if (durationInMillis < TimeUnit.HOURS.toMillis(2L) && units.compareTo(TimeUnit.MINUTES) <= 0) {
            return TimeUnit.MINUTES;
        }
        if (durationInMillis < TimeUnit.DAYS.toMillis(2L) && units.compareTo(TimeUnit.HOURS) <= 0) {
            return TimeUnit.HOURS;
        }
        return TimeUnit.DAYS;
    }

    private static double seconds(Duration duration) {
        return (double)duration.getSeconds() + (double)duration.getNano() / 1.0E9;
    }

    private static String convert(Duration duration, ChronoUnit unit, String name) {
        double value = TimeUtils.seconds(duration) / TimeUtils.seconds(unit.getDuration());
        return Strings.toThreeSigFig(value, 2000.0) + " " + name;
    }

    private static Optional<String> inUnits(Duration duration, ChronoUnit unit, String name) {
        Duration cutoff = unit.getDuration().multipliedBy(2L);
        if (duration.compareTo(cutoff) >= 0) {
            return Optional.of(TimeUtils.convert(duration, unit, name));
        }
        return Optional.empty();
    }

    public static Optional<String> describe(Duration duration) {
        if (duration.isZero()) {
            return Optional.empty();
        }
        Optional<String> value = TimeUtils.inUnits(duration, ChronoUnit.DAYS, "days");
        if (!value.isPresent()) {
            value = TimeUtils.inUnits(duration, ChronoUnit.HOURS, "hours");
        }
        if (!value.isPresent()) {
            value = TimeUtils.inUnits(duration, ChronoUnit.MINUTES, "minutes");
        }
        if (!value.isPresent()) {
            value = TimeUtils.inUnits(duration, ChronoUnit.SECONDS, "s");
        }
        if (!value.isPresent()) {
            value = TimeUtils.inUnits(duration, ChronoUnit.MILLIS, "ms");
        }
        if (!value.isPresent()) {
            value = TimeUtils.inUnits(duration, ChronoUnit.MICROS, "\u00b5s");
        }
        if (!value.isPresent()) {
            value = Optional.of(TimeUtils.convert(duration, ChronoUnit.NANOS, "ns"));
        }
        return value;
    }

    public static String describeDuration(double duration, TimeUnit units) {
        TimeUnit targetUnits = TimeUtils.displayUnitFor(Math.round(duration), units);
        double scaledDuration = TimeUtils.convert(duration, units, targetUnits);
        return Strings.toThreeSigFig(scaledDuration, 2000.0) + " " + (String)SHORT_TIMEUNIT_NAMES.get((Object)targetUnits);
    }

    public static String describeDuration(StatisticalSummary duration, TimeUnit units) {
        Object meanDescription;
        double max;
        double min = duration.getMin();
        if (min == (max = duration.getMax())) {
            return TimeUtils.describeDuration(max, units);
        }
        double mean = duration.getMean();
        double sem = duration.getStandardDeviation() / Math.sqrt(duration.getN());
        if (sem == 0.0) {
            meanDescription = TimeUtils.describeDuration(mean, units);
        } else {
            TimeUnit targetUnits = TimeUtils.displayUnitFor(Math.round(mean), units);
            double scaledMean = TimeUtils.convert(mean, units, targetUnits);
            double scaledSem = TimeUtils.convert(sem, units, targetUnits);
            meanDescription = "(" + Strings.toThreeSigFig(scaledMean, 2000.0, scaledSem) + ") " + (String)SHORT_TIMEUNIT_NAMES.get((Object)targetUnits);
        }
        double sd = duration.getStandardDeviation();
        return " min. " + TimeUtils.describeDuration(min, units) + ", mean " + (String)meanDescription + ", SD " + TimeUtils.describeDuration(sd, units) + ", max. " + TimeUtils.describeDuration(max, units);
    }

    private static double convert(double source, TimeUnit sourceUnits, TimeUnit targetUnits) {
        return source / (double)sourceUnits.convert(1L, targetUnits);
    }

    public static CharSequence relativeTimestamp(long when, long current) {
        return TimeUtils.appendRelativeTimestamp(new StringBuilder(), when, current);
    }

    public static CharSequence relativeTimestamp(Instant when) {
        return TimeUtils.appendRelativeTimestamp(new StringBuilder(), when.toEpochMilli(), System.currentTimeMillis(), TimeUnitFormat.SHORT);
    }

    public static StringBuilder appendRelativeTimestamp(StringBuilder sb, long when, long current) {
        return TimeUtils.appendRelativeTimestamp(sb, when, current, TimeUnitFormat.LONG);
    }

    public static StringBuilder appendRelativeTimestamp(StringBuilder sb, long when, long current, TimeUnitFormat format) {
        Preconditions.checkArgument((when > 0L ? 1 : 0) != 0);
        Preconditions.checkArgument((current > 0L ? 1 : 0) != 0);
        SimpleDateFormat iso8601 = new SimpleDateFormat(TIMESTAMP_FORMAT);
        sb.append(iso8601.format(new Date(when)));
        long diff = Math.abs(when - current);
        sb.append(" (");
        if (diff > 0L) {
            TimeUtils.appendDuration(sb, diff, TimeUnit.MILLISECONDS, format);
            sb.append(' ');
            sb.append(when < current ? "ago" : "in the future");
        } else {
            sb.append("now");
        }
        sb.append(')');
        return sb;
    }

    public static long getMillis(Properties properties, String key) {
        return TimeUnit.valueOf(properties.getProperty(key + ".unit")).toMillis(Long.parseLong(properties.getProperty(key)));
    }

    public static Duration durationOf(long value, TimeUnit unit) {
        return Duration.of(value, unit.toChronoUnit());
    }

    public static enum TimeUnitFormat {
        SHORT,
        LONG;

    }

    public static class DurationFormatter {
        private final String format;
        private DurationParser durations;

        public DurationFormatter(String format) {
            Objects.requireNonNull(format, "Format string must be specified.");
            this.format = format;
        }

        public String format(long duration, TimeUnit unit) {
            Objects.requireNonNull(unit, "Duration time unit must be specified.");
            TimeUnit[] sortedDimensions = this.getSortedDimensions();
            this.durations = new DurationParser(duration, unit);
            for (TimeUnit dimension : sortedDimensions) {
                this.durations.parse(dimension);
            }
            StringBuilder builder = new StringBuilder();
            this.replace(builder);
            return builder.toString();
        }

        private TimeUnit[] getSortedDimensions() {
            HashSet<TimeUnit> units = new HashSet<TimeUnit>();
            char[] sequence = this.format.toCharArray();
            block11: for (int c = 0; c < sequence.length; ++c) {
                switch (sequence[c]) {
                    case '%': {
                        switch (sequence[++c]) {
                            case 'D': {
                                units.add(TimeUnit.DAYS);
                                continue block11;
                            }
                            case 'H': {
                                units.add(TimeUnit.HOURS);
                                continue block11;
                            }
                            case 'm': {
                                units.add(TimeUnit.MINUTES);
                                continue block11;
                            }
                            case 's': {
                                units.add(TimeUnit.SECONDS);
                                continue block11;
                            }
                            case 'S': {
                                units.add(TimeUnit.MILLISECONDS);
                                continue block11;
                            }
                            case 'N': {
                                units.add(TimeUnit.MILLISECONDS);
                                units.add(TimeUnit.MICROSECONDS);
                                units.add(TimeUnit.NANOSECONDS);
                                continue block11;
                            }
                        }
                        throw new IllegalArgumentException("No such formatting symbol " + c);
                    }
                }
            }
            TimeUnit[] sorted = (TimeUnit[])units.toArray(TimeUnit[]::new);
            Arrays.sort(sorted, comparator);
            return sorted;
        }

        private void replace(StringBuilder builder) {
            char[] sequence = this.format.toCharArray();
            block3: for (int c = 0; c < sequence.length; ++c) {
                switch (sequence[c]) {
                    case '%': {
                        ++c;
                        c = this.handlePlaceholder(c, sequence, builder);
                        continue block3;
                    }
                    default: {
                        builder.append(sequence[c]);
                    }
                }
            }
        }

        private int handlePlaceholder(int c, char[] sequence, StringBuilder builder) {
            switch (sequence[c]) {
                case 'D': {
                    builder.append(this.durations.get(TimeUnit.DAYS));
                    break;
                }
                case 'H': {
                    if (sequence[c + 1] == 'H') {
                        ++c;
                        builder.append(this.leadingZero(this.durations.get(TimeUnit.HOURS)));
                        break;
                    }
                    builder.append(this.durations.get(TimeUnit.HOURS));
                    break;
                }
                case 'm': {
                    if (sequence[c + 1] == 'm') {
                        ++c;
                        builder.append(this.leadingZero(this.durations.get(TimeUnit.MINUTES)));
                        break;
                    }
                    builder.append(this.durations.get(TimeUnit.MINUTES));
                    break;
                }
                case 's': {
                    if (sequence[c + 1] == 's') {
                        ++c;
                        builder.append(this.leadingZero(this.durations.get(TimeUnit.SECONDS)));
                        break;
                    }
                    builder.append(this.durations.get(TimeUnit.SECONDS));
                    break;
                }
                case 'S': {
                    builder.append(this.durations.get(TimeUnit.MILLISECONDS));
                    break;
                }
                case 'N': {
                    builder.append(this.durations.get(TimeUnit.MILLISECONDS)).append(this.durations.get(TimeUnit.MICROSECONDS)).append(this.durations.get(TimeUnit.NANOSECONDS));
                    break;
                }
                default: {
                    throw new IllegalArgumentException("No such formatting symbol " + c);
                }
            }
            return c;
        }

        private String leadingZero(Long value) {
            String valueString = String.valueOf(value);
            if (valueString.length() < 2) {
                return "0" + valueString;
            }
            return valueString;
        }
    }

    public static class DurationParser {
        private final Map<TimeUnit, Long> durations;
        private final Long duration;
        private final TimeUnit durationUnit;
        private TimeUnit current;
        private long remainder;

        public DurationParser(Long duration, TimeUnit durationUnit) {
            this.duration = Objects.requireNonNull(duration, "duration was null");
            this.durationUnit = Objects.requireNonNull(durationUnit, "durationUnit was null");
            this.durations = new HashMap<TimeUnit, Long>();
            this.remainder = durationUnit.toNanos(duration);
        }

        public DurationParser parseAll() {
            this.parse(TimeUnit.DAYS);
            this.parse(TimeUnit.HOURS);
            this.parse(TimeUnit.MINUTES);
            this.parse(TimeUnit.SECONDS);
            this.parse(TimeUnit.MILLISECONDS);
            this.parse(TimeUnit.MICROSECONDS);
            this.parse(TimeUnit.NANOSECONDS);
            return this;
        }

        public void parse(TimeUnit unit) throws IllegalStateException {
            this.checkStrictlyDecreasing(unit);
            long durationForUnit = unit.convert(this.remainder, TimeUnit.NANOSECONDS);
            long durationInNanos = unit.toNanos(durationForUnit);
            this.remainder -= durationInNanos;
            this.durations.put(unit, durationForUnit);
            this.current = unit;
        }

        public void clear() {
            this.durations.clear();
            this.current = null;
            this.remainder = this.durationUnit.toNanos(this.duration);
        }

        public long get(TimeUnit unit) {
            Long duration = this.durations.get((Object)unit);
            return duration == null ? 0L : duration;
        }

        private void checkStrictlyDecreasing(TimeUnit next) throws IllegalStateException {
            if (comparator.compare(next, this.current) <= 0) {
                throw new IllegalStateException(next + " is not strictly smaller than " + this.current);
            }
        }
    }

    public static class DecreasingTimeUnitComparator
    implements Comparator<TimeUnit> {
        @Override
        public int compare(TimeUnit unit1, TimeUnit unit2) {
            long nanos2;
            if (unit1 == null) {
                return -1;
            }
            if (unit2 == null) {
                return 1;
            }
            long nanos1 = unit1.toNanos(1L);
            if (nanos1 == (nanos2 = unit2.toNanos(1L))) {
                return 0;
            }
            if (nanos1 < nanos2) {
                return 1;
            }
            return -1;
        }
    }
}

