            CREATE OR REPLACE FUNCTION
                path2inodes(root bigint, path varchar, OUT inode t_inodes)
                RETURNS SETOF t_inodes AS $$
            DECLARE
                dir      bigint;
                elements text[] := string_to_array(path, '/');
                inodes   t_inodes[];
                parent   t_inodes;
                link     varchar;
            BEGIN
                -- Find the inode of the root
                SELECT * INTO inode FROM t_inodes WHERE inumber = root;
                IF NOT FOUND THEN
                    RETURN;
                END IF;

                -- We build an array of the inodes for the path
                inodes := ARRAY[inode];

                -- For each path element
                FOR i IN 1..array_upper(elements,1) LOOP
                    -- Return empty set if not a directory
                    IF inode.itype != 16384 THEN
                        RETURN;
                    END IF;

                    -- The ID of the directory
                    dir := inode.inumber;

                    -- Lookup the next path element
                    CASE
                    WHEN elements[i] = '.' THEN
                        CONTINUE;
                    WHEN elements[i] = '..' THEN
                        SELECT p.* INTO parent
                            FROM t_inodes p JOIN t_dirs d ON p.inumber = d.iparent
                            WHERE d.ichild = dir;
                        IF FOUND THEN
                            inode := parent;
                        ELSE
                            CONTINUE;
                        END IF;
                    ELSE
                        SELECT c.* INTO inode
                            FROM t_inodes c JOIN t_dirs d ON c.inumber = d.ichild
                            WHERE d.iparent = dir AND d.iname = elements[i];

                        -- Return the empty set if not found
                        IF NOT FOUND THEN
                            RETURN;
                        END IF;
                    END CASE;

                    -- Append the inode to the result set
                    inodes := array_append(inodes, inode);

                    -- If inode is a symbolic link
                    IF inode.itype = 40960 THEN
                        -- Read the link
                        SELECT encode(ifiledata,'escape') INTO STRICT link
                            FROM t_inodes_data WHERE inumber = inode.inumber;

                        -- If absolute path then resolve from the file system root
                        IF link LIKE '/%' THEN
                            dir := pnfsid2inumber('000000000000000000000000000000000000');
                            link := substring(link from 2);

                            -- Call recursively and add inodes to result set
                            FOR inode IN SELECT * FROM path2inodes(dir, link) LOOP
                                inodes := array_append(inodes, inode);
                            END LOOP;
                        ELSE
                            -- Call recursively and add inodes to result set; skip
                            -- first inode as it is the inode of dir
                            FOR inode IN SELECT * FROM path2inodes(dir, link) OFFSET 1 LOOP
                                inodes := array_append(inodes, inode);
                            END LOOP;
                        END IF;

                        -- Return empty set if link could not be resolved
                        IF NOT FOUND THEN
                            RETURN;
                        END IF;

                        -- Continue from the inode pointed to by the link
                        inode = inodes[array_upper(inodes,1)];
                    END IF;
                END LOOP;

                -- Output all inodes
                FOR i IN 1..array_upper(inodes,1) LOOP
                    inode := inodes[i];
                    RETURN NEXT;
                END LOOP;
            END;
            $$ LANGUAGE plpgsql;
