/*
 * Decompiled with CFR 0.152.
 */
package org.dcache.nfs.util;

import com.google.common.base.Preconditions;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.StampedLock;
import org.dcache.nfs.util.CacheElement;
import org.dcache.nfs.util.CacheEventListener;
import org.dcache.nfs.util.CacheMXBean;
import org.dcache.nfs.util.CacheMXBeanImpl;
import org.dcache.nfs.util.NopCacheEventListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Cache<K, V> {
    private static final Logger _log = LoggerFactory.getLogger(Cache.class);
    private final Clock _timeSource;
    private final String _name;
    private final Duration _defaultEntryMaxLifeTime;
    private final Duration _defaultEntryIdleTime;
    private final int _size;
    private final Map<K, CacheElement<V>> _storage;
    private final StampedLock _accessLock = new StampedLock();
    private final CacheEventListener<K, V> _eventListener;
    private final CacheMXBean<V> _mxBean;
    private final AtomicReference<Instant> _lastClean;

    public Cache(String name2, int size, Duration entryLifeTime, Duration entryIdleTime) {
        this(name2, size, entryLifeTime, entryIdleTime, new NopCacheEventListener());
    }

    public Cache(String name2, int size, Duration entryLifeTime, Duration entryIdleTime, CacheEventListener<K, V> eventListener) {
        this(name2, size, entryLifeTime, entryIdleTime, eventListener, Clock.systemDefaultZone());
    }

    public Cache(String name2, int size, Duration entryLifeTime, Duration entryIdleTime, CacheEventListener<K, V> eventListener, Clock clock) {
        Preconditions.checkArgument(entryLifeTime.compareTo(entryIdleTime) >= 0, "Entry life time cant be smaller that idle time");
        this._name = name2;
        this._size = size;
        this._defaultEntryMaxLifeTime = entryLifeTime;
        this._defaultEntryIdleTime = entryIdleTime;
        this._storage = new HashMap<K, CacheElement<V>>(this._size);
        this._eventListener = eventListener;
        this._mxBean = new CacheMXBeanImpl(this);
        this._timeSource = clock;
        this._lastClean = new AtomicReference<Instant>(this._timeSource.instant());
    }

    public String getName() {
        return this._name;
    }

    public void put(K k, V v) {
        this.put(k, v, this._defaultEntryMaxLifeTime, this._defaultEntryIdleTime);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void put(K k, V v, Duration entryMaxLifeTime, Duration entryIdleTime) {
        _log.debug("Adding new cache entry: key = [{}], value = [{}]", (Object)k, (Object)v);
        long stamp = this._accessLock.writeLock();
        try {
            if (this._storage.size() >= this._size && !this._storage.containsKey(k)) {
                _log.warn("Cache limit reached: {}", (Object)this._size);
                throw new MissingResourceException("Cache limit reached", Cache.class.getName(), "");
            }
            this._storage.put(k, new CacheElement<V>(v, this._timeSource, entryMaxLifeTime, entryIdleTime));
        }
        finally {
            this._accessLock.unlock(stamp);
        }
        this._eventListener.notifyPut(this, v);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public V get(K k) {
        V v;
        boolean valid;
        boolean removed = false;
        long stamp = this._accessLock.readLock();
        try {
            CacheElement<V> element = this._storage.get(k);
            if (element == null) {
                _log.debug("No cache hits for key = [{}]", (Object)k);
                V v2 = null;
                return v2;
            }
            valid = element.validAt(this._timeSource.instant());
            v = element.getObject();
            if (!valid) {
                _log.debug("Cache hits but entry expired for key = [{}], value = [{}]", (Object)k, (Object)v);
                long ws = this._accessLock.tryConvertToWriteLock(stamp);
                if (ws != 0L) {
                    stamp = ws;
                } else {
                    this._accessLock.unlock(stamp);
                    stamp = this._accessLock.writeLock();
                }
                removed = this._storage.remove(k) != null;
            } else {
                _log.debug("Cache hits for key = [{}], value = [{}]", (Object)k, (Object)v);
            }
        }
        finally {
            this._accessLock.unlock(stamp);
        }
        if (!valid) {
            if (removed) {
                this._eventListener.notifyExpired(this, v);
            }
            v = null;
        } else {
            this._eventListener.notifyGet(this, v);
        }
        return v;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public V remove(K k) {
        V v;
        boolean valid;
        long stamp = this._accessLock.writeLock();
        try {
            CacheElement<V> element = this._storage.remove(k);
            if (element == null) {
                V v2 = null;
                return v2;
            }
            valid = element.validAt(this._timeSource.instant());
            v = element.getObject();
        }
        finally {
            this._accessLock.unlock(stamp);
        }
        _log.debug("Removing entry: active = [{}] key = [{}], value = [{}]", valid, k, v);
        this._eventListener.notifyRemove(this, v);
        return (V)(valid ? v : null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    int size() {
        long stamp = this._accessLock.readLock();
        try {
            int n = this._storage.size();
            return n;
        }
        finally {
            this._accessLock.unlock(stamp);
        }
    }

    public Duration getEntryIdleTime() {
        return this._defaultEntryIdleTime;
    }

    public Duration getEntryLiveTime() {
        return this._defaultEntryMaxLifeTime;
    }

    public void clear() {
        _log.debug("Cleaning the cache");
        long stamp = this._accessLock.writeLock();
        try {
            this._storage.clear();
        }
        finally {
            this._accessLock.unlock(stamp);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void cleanUp() {
        ArrayList<V> expiredEntries = new ArrayList<V>();
        long stamp = this._accessLock.writeLock();
        try {
            Instant now = this._timeSource.instant();
            Iterator<Map.Entry<K, CacheElement<V>>> entries = this._storage.entrySet().iterator();
            while (entries.hasNext()) {
                Map.Entry<K, CacheElement<V>> entry = entries.next();
                CacheElement<V> cacheElement = entry.getValue();
                if (cacheElement.validAt(now)) continue;
                _log.debug("Cleaning expired entry key = [{}], value = [{}]", (Object)entry.getKey(), (Object)cacheElement.getObject());
                entries.remove();
                expiredEntries.add(cacheElement.getObject());
            }
            this._lastClean.set(now);
        }
        finally {
            this._accessLock.unlock(stamp);
        }
        expiredEntries.forEach(v -> this._eventListener.notifyExpired(this, v));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<CacheElement<V>> entries() {
        ArrayList<CacheElement<V>> entries;
        long stamp = this._accessLock.readLock();
        try {
            entries = new ArrayList<CacheElement<V>>(this._storage.size());
            entries.addAll(this._storage.values());
        }
        finally {
            this._accessLock.unlock(stamp);
        }
        return entries;
    }

    public Instant lastClean() {
        return this._lastClean.get();
    }
}

