/*
 * Decompiled with CFR 0.152.
 */
package org.dcache.xrootd.plugins.authn.gsi;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import java.io.IOException;
import java.util.Collection;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import org.dcache.xrootd.core.XrootdDecoder;
import org.dcache.xrootd.core.XrootdEncoder;
import org.dcache.xrootd.core.XrootdException;
import org.dcache.xrootd.plugins.authn.gsi.GSIBucket;
import org.dcache.xrootd.plugins.authn.gsi.GSIBucketContainer;
import org.dcache.xrootd.plugins.authn.gsi.NestedBucketBuffer;
import org.dcache.xrootd.plugins.authn.gsi.RawBucket;
import org.dcache.xrootd.plugins.authn.gsi.StringBucket;
import org.dcache.xrootd.plugins.authn.gsi.UnsignedIntBucket;
import org.dcache.xrootd.protocol.messages.AuthenticationRequest;
import org.dcache.xrootd.security.XrootdSecurityProtocol;
import org.dcache.xrootd.tpc.protocol.messages.InboundAuthenticationResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GSIBucketUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(GSIBucketUtils.class);
    private static final String[] BYTE_DUMP = new String[]{"//  0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x    %s\n", "//  0x%02x                                                     %s\n", "//  0x%02x 0x%02x                                              %s\n", "//  0x%02x 0x%02x 0x%02x                                       %s\n", "//  0x%02x 0x%02x 0x%02x 0x%02x                                %s\n", "//  0x%02x 0x%02x 0x%02x 0x%02x 0x%02x                         %s\n", "//  0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x                  %s\n", "//  0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x           %s\n"};

    public static GSIBucket deserialize(XrootdSecurityProtocol.BucketType type, ByteBuf buffer) throws IOException {
        GSIBucket bucket;
        switch (type) {
            case kXRS_main: {
                try {
                    bucket = GSIBucketUtils.deserializeNested(type, buffer);
                }
                catch (IOException e) {
                    bucket = RawBucket.deserialize(type, buffer);
                }
                break;
            }
            case kXRS_cryptomod: 
            case kXRS_issuer_hash: 
            case kXRS_rtag: 
            case kXRS_puk: 
            case kXRS_cipher_alg: 
            case kXRS_x509: 
            case kXRS_x509_req: 
            case kXRS_md_alg: 
            case kXRS_message: {
                bucket = StringBucket.deserialize(type, buffer);
                break;
            }
            case kXRS_version: 
            case kXRS_clnt_opts: {
                bucket = UnsignedIntBucket.deserialize(type, buffer);
                break;
            }
            default: {
                bucket = RawBucket.deserialize(type, buffer);
            }
        }
        return bucket;
    }

    public static Map<XrootdSecurityProtocol.BucketType, GSIBucket> deserializeBuckets(ByteBuf buffer) throws IOException {
        int bucketCode = buffer.readInt();
        XrootdSecurityProtocol.BucketType bucketType = XrootdSecurityProtocol.BucketType.get((int)bucketCode);
        EnumMap<XrootdSecurityProtocol.BucketType, GSIBucket> buckets = new EnumMap<XrootdSecurityProtocol.BucketType, GSIBucket>(XrootdSecurityProtocol.BucketType.class);
        while (bucketType != XrootdSecurityProtocol.BucketType.kXRS_none) {
            LOGGER.debug("Deserialized a bucket with code {}, type {}", (Object)bucketCode, (Object)bucketType);
            int bucketLength = buffer.readInt();
            LOGGER.debug("bucket type {} has length {}.", (Object)bucketType, (Object)bucketLength);
            GSIBucket bucket = GSIBucketUtils.deserialize(bucketType, buffer.slice(buffer.readerIndex(), bucketLength));
            buckets.put(bucketType, bucket);
            buffer.readerIndex(buffer.readerIndex() + bucketLength);
            bucketCode = buffer.readInt();
            bucketType = XrootdSecurityProtocol.BucketType.get((int)bucketCode);
        }
        return buckets;
    }

    public static BucketData deserializeData(AuthenticationRequest request) {
        BucketData data = new BucketData();
        ByteBuf buffer = request.getCredentialBuffer();
        data.protocol = GSIBucketUtils.deserializeProtocol(buffer);
        data.step = GSIBucketUtils.deserializeStep(buffer);
        try {
            data.bucketMap.putAll(GSIBucketUtils.deserializeBuckets(buffer));
        }
        catch (IOException ioex) {
            throw new IllegalArgumentException("Illegal credential format: {}", ioex);
        }
        request.releaseBuffer();
        UnsignedIntBucket versionBucket = (UnsignedIntBucket)data.bucketMap.get(XrootdSecurityProtocol.BucketType.kXRS_version);
        if (versionBucket != null) {
            data.version = versionBucket.getContent();
        }
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace(GSIBucketUtils.describe("//                Authentication Request", b -> GSIBucketUtils.dumpBuckets(b, data.bucketMap.values(), XrootdSecurityProtocol.getClientStep((int)data.step)), request.getStreamId(), request.getRequestId(), null));
        }
        return data;
    }

    public static BucketData deserializeData(InboundAuthenticationResponse response) throws XrootdException {
        BucketData data = new BucketData();
        ByteBuf buffer = response.getDataBuffer();
        data.protocol = GSIBucketUtils.deserializeProtocol(buffer);
        data.step = GSIBucketUtils.deserializeStep(buffer);
        try {
            data.bucketMap.putAll(GSIBucketUtils.deserializeBuckets(buffer));
            if (data.step != 2002) {
                RawBucket mainBucket = (RawBucket)data.bucketMap.remove(XrootdSecurityProtocol.BucketType.kXRS_main);
                ByteBuf mainBuffer = Unpooled.wrappedBuffer((byte[])mainBucket.getContent());
                mainBuffer.readerIndex(8);
                data.bucketMap.putAll(GSIBucketUtils.deserializeBuckets(mainBuffer));
            }
        }
        catch (IOException e) {
            throw new XrootdException(3007, e.toString());
        }
        response.releaseBuffer();
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace(GSIBucketUtils.describe("//           Inbound Authentication Response", b -> GSIBucketUtils.dumpBuckets(b, data.bucketMap.values(), XrootdSecurityProtocol.getServerStep((int)data.step)), response.getStreamId(), response.getRequestId(), response.getStatus()));
        }
        return data;
    }

    public static NestedBucketBuffer deserializeNested(XrootdSecurityProtocol.BucketType type, ByteBuf buffer) throws IOException {
        int readIndex = buffer.readerIndex();
        String protocol = GSIBucketUtils.deserializeProtocol(buffer);
        int step = GSIBucketUtils.deserializeStep(buffer);
        LOGGER.debug("NestedBucketBuffer protocol: {}, step {}", (Object)protocol, (Object)step);
        if (step < 1000 || step > 1003) {
            buffer.readerIndex(readIndex);
            throw new IOException("Buffer contents are not a nested buffer!");
        }
        return new NestedBucketBuffer(type, protocol, step, GSIBucketUtils.deserializeBuckets(buffer));
    }

    public static void dumpBuckets(StringBuilder builder, Collection<GSIBucket> buckets, String step) {
        int i = 0;
        for (GSIBucket bucket : buckets) {
            ++i;
            i = bucket.dump(builder, step, i);
        }
    }

    public static void dumpBytes(StringBuilder builder, byte[] data) {
        int i = 0;
        int D = data.length / 8;
        for (int d = 0; d < D; ++d) {
            builder.append(String.format(BYTE_DUMP[0], data[i], data[i + 1], data[i + 2], data[i + 3], data[i + 4], data[i + 5], data[i + 6], data[i + 7], GSIBucketUtils.getAscii(data, i, 8)));
            i += 8;
        }
        switch (data.length % 8) {
            case 7: {
                builder.append(String.format(BYTE_DUMP[7], data[i], data[i + 1], data[i + 2], data[i + 3], data[i + 4], data[i + 5], data[i + 6], GSIBucketUtils.getAscii(data, i, 7)));
                break;
            }
            case 6: {
                builder.append(String.format(BYTE_DUMP[6], data[i], data[i + 1], data[i + 2], data[i + 3], data[i + 4], data[i + 5], GSIBucketUtils.getAscii(data, i, 6)));
                break;
            }
            case 5: {
                builder.append(String.format(BYTE_DUMP[5], data[i], data[i + 1], data[i + 2], data[i + 3], data[i + 4], GSIBucketUtils.getAscii(data, i, 5)));
                break;
            }
            case 4: {
                builder.append(String.format(BYTE_DUMP[4], data[i], data[i + 1], data[i + 2], data[i + 3], GSIBucketUtils.getAscii(data, i, 4)));
                break;
            }
            case 3: {
                builder.append(String.format(BYTE_DUMP[3], data[i], data[i + 1], data[i + 2], GSIBucketUtils.getAscii(data, i, 3)));
                break;
            }
            case 2: {
                builder.append(String.format(BYTE_DUMP[2], data[i], data[i + 1], GSIBucketUtils.getAscii(data, i, 2)));
                break;
            }
            case 1: {
                builder.append(String.format(BYTE_DUMP[1], data[i], GSIBucketUtils.getAscii(data, i, 1)));
            }
        }
    }

    public static int getLengthForRequest(GSIBucketContainer container) {
        return 12 + container.getSize();
    }

    private static void writeBytes(ByteBuf buffer, String protocol, int step, List<GSIBucket> buckets) {
        XrootdEncoder.writeZeroPad((String)protocol, (ByteBuf)buffer, (int)4);
        buffer.writeInt(step);
        for (GSIBucket bucket : buckets) {
            bucket.serialize(buffer);
        }
        buffer.writeInt(XrootdSecurityProtocol.BucketType.kXRS_none.getCode());
    }

    private static String describe(String title, Consumer<StringBuilder> data, Integer streamId, Integer requestId, Integer stat) {
        StringBuilder builder = new StringBuilder("\n");
        builder.append("/////////////////////////////////////////////////////////\n");
        builder.append(title);
        builder.append("\n//\n");
        builder.append("//  stream:  ").append(streamId).append("\n");
        if (requestId != null) {
            builder.append("//  request: ").append(requestId).append("\n");
        }
        if (stat != null) {
            builder.append("//  stat:    ").append(stat).append("\n");
        }
        builder.append("//\n");
        data.accept(builder);
        builder.append("/////////////////////////////////////////////////////////\n");
        return builder.toString();
    }

    private static String deserializeProtocol(ByteBuf buffer) {
        return XrootdDecoder.readAscii((ByteBuf)buffer, (int)4);
    }

    private static int deserializeStep(ByteBuf buffer) {
        return buffer.readInt();
    }

    private static String getAscii(byte[] bytes, int from, int len) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < len; ++i) {
            byte b = bytes[from + i];
            if (32 < b && b < 127) {
                sb.append((char)b);
                continue;
            }
            sb.append('.');
        }
        return sb.toString();
    }

    private GSIBucketUtils() {
    }

    public static class BucketData {
        String protocol;
        int step;
        Integer version;
        final Map<XrootdSecurityProtocol.BucketType, GSIBucket> bucketMap = new EnumMap<XrootdSecurityProtocol.BucketType, GSIBucket>(XrootdSecurityProtocol.BucketType.class);

        public String getProtocol() {
            return this.protocol;
        }

        public int getStep() {
            return this.step;
        }

        public Integer getVersion() {
            return this.version;
        }

        public Map<XrootdSecurityProtocol.BucketType, GSIBucket> getBucketMap() {
            return this.bucketMap;
        }
    }

    public static class BucketSerializer
    implements Consumer<ByteBuf> {
        String title;
        Integer streamId;
        Integer requestId;
        Integer stat;
        String stepName;
        String protocol;
        int step;
        List<GSIBucket> buckets;

        @Override
        public void accept(ByteBuf byteBuf) {
            GSIBucketUtils.writeBytes(byteBuf, this.protocol, this.step, this.buckets);
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace(GSIBucketUtils.describe(this.title, b -> GSIBucketUtils.dumpBuckets(b, this.buckets, this.stepName), this.streamId, this.requestId, this.stat));
            }
        }
    }

    public static class BucketSerializerBuilder {
        private final BucketSerializer serializer = new BucketSerializer();

        public BucketSerializerBuilder withTitle(String title) {
            this.serializer.title = title;
            return this;
        }

        public BucketSerializerBuilder withStreamId(Integer streamId) {
            this.serializer.streamId = streamId;
            return this;
        }

        public BucketSerializerBuilder withRequestId(Integer requestId) {
            this.serializer.requestId = requestId;
            return this;
        }

        public BucketSerializerBuilder withStat(Integer stat) {
            this.serializer.stat = stat;
            return this;
        }

        public BucketSerializerBuilder withStepName(String stepName) {
            this.serializer.stepName = stepName;
            return this;
        }

        public BucketSerializerBuilder withProtocol(String protocol) {
            this.serializer.protocol = protocol;
            return this;
        }

        public BucketSerializerBuilder withStep(int step) {
            this.serializer.step = step;
            return this;
        }

        public BucketSerializerBuilder withBuckets(List<GSIBucket> buckets) {
            this.serializer.buckets = buckets;
            return this;
        }

        public BucketSerializer build() {
            return this.serializer;
        }
    }
}

