/*
 * Decompiled with CFR 0.152.
 */
package org.dcache.xrootd.tpc;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import java.io.Serializable;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.dcache.xrootd.core.XrootdException;
import org.dcache.xrootd.tpc.protocol.messages.InboundRedirectResponse;
import org.dcache.xrootd.util.FileStatus;
import org.dcache.xrootd.util.OpaqueStringParser;
import org.dcache.xrootd.util.ParseException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class XrootdTpcInfo {
    private static final Logger LOGGER = LoggerFactory.getLogger(XrootdTpcInfo.class);
    private final String key;
    private final long createdTime;
    private Delegation dlgon;
    private Long uid;
    private Long gid;
    private String org;
    private String dst;
    private String src;
    private String srcHost;
    private Integer srcPort;
    private String lfn;
    private Long ttl;
    private String cks;
    private Long asize;
    private Status status;
    private int fd;
    private long startTime;
    private String external;
    private String loginToken;
    private String sourceToken;
    private FileStatus fileStatus;
    private Serializable delegatedProxy;
    private ServerRole serverRole;
    private ClientRole clientRole;
    private OptionalLong fileSize = OptionalLong.empty();
    private Optional<String> sourceProtocol = Optional.empty();

    public XrootdTpcInfo(String key) {
        this.key = key;
        this.createdTime = System.currentTimeMillis();
        this.calculateRoles();
    }

    public XrootdTpcInfo(Map<String, String> opaque) throws ParseException {
        this(opaque.get(Cgi.RENDEZVOUS_KEY.key()));
        this.dlgon = Delegation.valueOf(opaque);
        this.lfn = opaque.get(Cgi.LOGICAL_NAME.key());
        this.dst = opaque.get(Cgi.DST.key());
        this.setSourceFromOpaque(opaque);
        this.cks = opaque.get(Cgi.CHECKSUM.key());
        this.org = opaque.get(Cgi.CLIENT.key());
        String asize = opaque.get(Cgi.SIZE_IN_BYTES.key());
        if (asize != null) {
            this.asize = Long.parseLong(asize);
        }
        this.status = Status.READY;
        this.sourceProtocol = Optional.ofNullable(opaque.get(Cgi.SPR.key()));
        this.findSourceToken(opaque);
        this.addExternal(opaque);
        this.calculateRoles();
    }

    public long computeFileSize() throws XrootdException {
        if (!this.fileSize.isPresent()) {
            if (this.fileStatus == null) {
                if (this.asize == null) {
                    throw new XrootdException(3001, "Cannot read source; file size is unknown.");
                }
                this.fileSize = OptionalLong.of(this.asize);
            } else {
                this.fileSize = OptionalLong.of(this.fileStatus.getSize());
            }
            LOGGER.debug("computeFileSize: file status {}, oss.asize {}, computed size {}.", new Object[]{this.fileStatus, this.asize, this.fileSize.getAsLong()});
        }
        return this.fileSize.getAsLong();
    }

    public ServerRole getServerRole() {
        return this.serverRole;
    }

    public ClientRole getClientRole() {
        return this.clientRole;
    }

    public boolean isTpcRequest() {
        return this.clientRole != ClientRole.NON_TPC;
    }

    public synchronized XrootdTpcInfo addInfoFromOpaque(String slfn, Map<String, String> opaque) throws ParseException {
        String value;
        if (this.lfn == null) {
            this.lfn = slfn;
        }
        this.dlgon = Delegation.valueOf(opaque);
        if (this.org == null) {
            this.org = opaque.get(Cgi.CLIENT.key());
        }
        if (this.dst == null) {
            this.dst = opaque.get(Cgi.DST.key());
        }
        if (this.src == null) {
            this.setSourceFromOpaque(opaque);
        }
        if (this.cks == null) {
            this.cks = opaque.get(Cgi.CHECKSUM.key());
        }
        if ((value = opaque.get(Cgi.TIME_TO_LIVE.key())) != null) {
            this.ttl = new Long(value);
            this.startTime = System.currentTimeMillis();
        }
        if ((value = opaque.get(Cgi.SIZE_IN_BYTES.key())) != null) {
            this.asize = Long.parseLong(value);
        }
        if (this.status == null) {
            this.status = Status.PENDING;
        }
        if (!this.sourceProtocol.isPresent()) {
            this.sourceProtocol = Optional.ofNullable(opaque.get(Cgi.SPR.key()));
        }
        if (this.sourceToken == null) {
            this.findSourceToken(opaque);
        }
        this.addExternal(opaque);
        this.calculateRoles();
        return this;
    }

    public XrootdTpcInfo copyForRedirect(InboundRedirectResponse response) throws ParseException {
        XrootdTpcInfo info = new XrootdTpcInfo(this.key);
        URL url = response.getUrl();
        if (url != null) {
            info.srcHost = url.getHost();
            info.srcPort = url.getPort();
            info.sourceProtocol = Optional.ofNullable(url.getProtocol());
        } else {
            info.srcHost = response.getHost();
            info.srcPort = response.getPort();
            info.sourceProtocol = this.sourceProtocol;
        }
        info.src = info.srcHost + ":" + info.srcPort;
        info.dlgon = this.dlgon;
        info.lfn = this.lfn;
        info.asize = this.asize;
        info.cks = this.cks;
        info.loginToken = response.getToken();
        info.delegatedProxy = this.delegatedProxy;
        info.uid = this.uid;
        info.gid = this.gid;
        String opaque = response.getOpaque();
        if (opaque == null && url != null) {
            opaque = url.getQuery();
        }
        if (opaque != null) {
            if (!opaque.startsWith("?")) {
                opaque = "?" + opaque;
            }
            Map<String, String> map = OpaqueStringParser.getOpaqueMap(opaque);
            info.sourceToken = map.get((Object)Cgi.AUTHZ);
            info.addExternal(map);
        }
        info.status = Status.READY;
        info.calculateRoles();
        return info;
    }

    public boolean isTls() {
        return this.sourceProtocol.filter("xroots"::equals).isPresent();
    }

    public synchronized Status verify(String dst, String slfn, String org) {
        if (this.status == Status.ERROR) {
            return this.status;
        }
        if (this.dst == null) {
            return Status.PENDING;
        }
        this.status = dst.equals(this.dst) && slfn.equals(this.lfn) && org.equals(this.org) && !this.isExpired() ? Status.READY : Status.CANCELLED;
        return this.status;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder().append("(dlgon ").append((Object)this.dlgon).append("(key ").append(this.key).append(")(dst ").append(this.dst).append(")(src ").append(this.srcHost).append(":").append(this.srcPort).append(")(org ").append(this.org).append(")(lfn ").append(this.lfn).append(")(ttl ").append(this.ttl).append(")(cks ").append(this.cks).append(")(asize ").append(this.asize).append(")(fhandle ").append(this.fd).append(')');
        this.sourceProtocol.ifPresent(p -> sb.append("(spr ").append((String)p).append(')'));
        return sb.append("(status ").append((Object)this.status).append(")(token ").append(this.loginToken).append(")(source token ").append(this.sourceToken).append(")(external [").append(this.external).append("])").toString();
    }

    public boolean isExpired() {
        return this.ttl != null && System.currentTimeMillis() > this.startTime + TimeUnit.SECONDS.toMillis(this.ttl);
    }

    public String getCks() {
        return this.cks;
    }

    public long getCreatedTime() {
        return this.createdTime;
    }

    public Serializable getDelegatedProxy() {
        return this.delegatedProxy;
    }

    public String getSourceToken() {
        return this.sourceToken;
    }

    public String getExternal() {
        return this.external;
    }

    public int getFd() {
        return this.fd;
    }

    public Long getGid() {
        return this.gid;
    }

    public String getKey() {
        return this.key;
    }

    public String getLfn() {
        return this.lfn;
    }

    public String getLoginToken() {
        return this.loginToken;
    }

    public String getSrc() {
        return this.src;
    }

    public String getSrcHost() {
        return this.srcHost;
    }

    public Integer getSrcPort() {
        return this.srcPort;
    }

    public synchronized Status getStatus() {
        return this.status;
    }

    public Long getUid() {
        return this.uid;
    }

    public Delegation getDlgon() {
        return this.dlgon;
    }

    public void setUid(Long uid) {
        this.uid = uid;
    }

    public void setGid(Long gid) {
        this.gid = gid;
    }

    public void setDelegatedProxy(Serializable delegatedProxy) {
        this.delegatedProxy = delegatedProxy;
    }

    public void setFileStatus(FileStatus fileStatus) {
        this.fileStatus = fileStatus;
    }

    public void setFd(int fd) {
        this.fd = fd;
    }

    public synchronized void setStatus(Status status) {
        this.status = status;
    }

    private void addExternal(Map<String, String> opaque) {
        HashMap<String, String> external = new HashMap<String, String>();
        for (Map.Entry<String, String> entry : opaque.entrySet()) {
            if (Cgi.keys().contains(entry.getKey())) continue;
            external.put(entry.getKey(), entry.getValue());
        }
        this.external = OpaqueStringParser.buildOpaqueString(external);
    }

    private void setSourceFromOpaque(Map<String, String> map) {
        this.src = map.get(Cgi.DLG.key());
        if (this.src == null) {
            this.src = map.get(Cgi.SRC.key());
        }
        if (this.src != null) {
            int at = this.src.indexOf("@");
            if (at >= 0) {
                this.src = this.src.substring(at + 1);
            }
            String[] source = this.src.split(":");
            this.srcHost = source[0];
            this.srcPort = source.length > 1 && Strings.emptyToNull((String)source[1]) != null ? Integer.valueOf(Integer.parseInt(source[1])) : Integer.valueOf(1094);
        }
    }

    public URI getSourceURL(String destinationPath) {
        Preconditions.checkState((this.src != null ? 1 : 0) != 0, (Object)"'tpc.src' element is missing");
        int port = this.srcPort == null || this.srcPort == 1094 ? -1 : this.srcPort;
        String sourcePath = this.lfn == null ? destinationPath : this.lfn;
        String scheme = this.sourceProtocol.orElse("xroot");
        try {
            return new URI(scheme, null, this.srcHost, port, sourcePath.startsWith("/") ? sourcePath : "/" + sourcePath, null, null);
        }
        catch (URISyntaxException e) {
            throw new RuntimeException(e.toString(), e);
        }
    }

    private void calculateRoles() {
        this.clientRole = this.org != null ? ClientRole.TPC_DESTINATION : (this.src != null || this.dst != null ? ClientRole.TPC_ORCHESTRATOR : ClientRole.NON_TPC);
        this.serverRole = this.src != null ? ServerRole.TPC_DESTINATION : (this.dst != null || this.org != null ? ServerRole.TPC_SOURCE : ServerRole.NON_TPC);
        if (this.serverRole == ServerRole.TPC_DESTINATION && this.clientRole != ClientRole.TPC_ORCHESTRATOR || this.serverRole == ServerRole.TPC_SOURCE && this.clientRole != ClientRole.TPC_ORCHESTRATOR && this.clientRole != ClientRole.TPC_DESTINATION || this.serverRole == ServerRole.NON_TPC && this.clientRole != ClientRole.NON_TPC) {
            LOGGER.warn("Inconsistent xrootd-TPC roles ServerRole={} ClientRole={}", (Object)this.serverRole, (Object)this.clientRole);
        }
    }

    private void findSourceToken(Map<String, String> opaque) throws ParseException {
        String scgi = opaque.get(Cgi.SCGI.key());
        if (scgi != null) {
            scgi = scgi.replaceAll("\\t+", String.valueOf('&'));
            Map<String, String> sourceOpaque = OpaqueStringParser.getOpaqueMap(scgi);
            this.sourceToken = sourceOpaque.get(Cgi.AUTHZ.key());
        }
    }

    static enum Delegation {
        OFF,
        ON;


        public static Delegation valueOf(Map<String, String> opaque) {
            String value = opaque.get(Cgi.DLGON.key);
            if (value == null) {
                return OFF;
            }
            switch (opaque.get(Cgi.DLGON.key)) {
                case "1": {
                    return ON;
                }
            }
            return OFF;
        }
    }

    public static enum ClientRole {
        NON_TPC,
        TPC_ORCHESTRATOR,
        TPC_DESTINATION;

    }

    public static enum ServerRole {
        NON_TPC,
        TPC_SOURCE,
        TPC_DESTINATION;

    }

    public static enum Status {
        PENDING,
        READY,
        CANCELLED,
        ERROR;

    }

    public static enum TpcStage {
        PLACEMENT("placement"),
        COPY("copy");

        private String key;

        private TpcStage(String key) {
            this.key = key;
        }

        public String key() {
            return this.key;
        }
    }

    public static enum CksumType {
        ADLER32("adler32"),
        CRC32("crc32"),
        MD5("md5");

        private String key;

        private CksumType(String key) {
            this.key = key;
        }

        public String key() {
            return this.key;
        }
    }

    public static enum Cgi {
        STAGE("tpc.stage"),
        RENDEZVOUS_KEY("tpc.key"),
        SRC("tpc.src"),
        DLG("tpc.dlg"),
        DLGON("tpc.dlgon"),
        DST("tpc.dst"),
        LOGICAL_NAME("tpc.lfn"),
        CLIENT("tpc.org"),
        CHECKSUM("tpc.cks"),
        TIME_TO_LIVE("tpc.ttl"),
        SIZE_IN_BYTES("oss.asize"),
        STR("tpc.str"),
        TPR("tpc.tpr"),
        SPR("tpc.spr"),
        SCGI("tpc.scgi"),
        AUTHZ("authz");

        private static final Set<String> KEYS;
        private String key;

        private Cgi(String key) {
            this.key = key;
        }

        public String key() {
            return this.key;
        }

        static Set<String> keys() {
            return KEYS;
        }

        static {
            KEYS = EnumSet.allOf(Cgi.class).stream().map(Cgi::key).collect(Collectors.toSet());
        }
    }
}

