package org.freehep.application.studio;

import java.io.File;
import java.io.IOException;
import java.util.Enumeration;
import java.util.Properties;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import org.freehep.util.VersionComparator;

/**
 * Library file descriptor.
 * <p>
 * Typically, instances of this class describe either libraries referenced in a plugin 
 * descriptor, or library files installed in one of the extension directories.
 * <p>
 * If a library descriptor is a part of a plugin descriptor (obtained from a list returned by 
 * {@link PluginInfo#getLibraries()}, its <tt>location</tt> and <tt>href</tt> are set
 * to corresponding fields in the plugin descriptor. <tt>version</tt> is only set if the 
 * <tt>location</tt> is in valid <tt>id-version.ext</tt> format. If not, it remains undefined,
 * end the ID is set to <tt>location</tt> without extension.
 * All other fields are undefined, and corresponding getters return <tt>null</tt>.
 * <p>
 * If a library descriptor corresponds to a file on disk, its <tt>location</tt> is set to
 * the file name (without extension). If the file name is in valid ID-version format, it is parsed to set both 
 * <tt>id</tt> and <tt>version</tt>. Otherwise, ID is set to the file name and the version 
 * remains undefined. <tt>dir</tt> corresponds to the installation directory, and <tt>file</tt>
 * is an absolute abstract path. All other fields are undefined, and corresponding getters 
 * return <tt>null</tt>.
 *
 * @author onoprien
 */
public class LibInfo {
    
// -- Private parts : ----------------------------------------------------------
    
    private String id;
    private String version;
    
    private String location;
    private String href;
    
    private PluginDir dir;
    private File file;

// -- Constructors : -----------------------------------------------------------

    /**
     * Constructs library descriptor from information found in a plugin descriptor.
     * If <tt>location</tt> is in valid <tt>ID-version.ext</tt> format, it is parsed
     * to set both ID and version. Otherwise, ID is set to <tt>location</tt> without extension, 
     * and the version remains unknown.
     */
    public LibInfo(String location, String href) {
        this.href = href;
        this.location = location;
        int i = location.lastIndexOf('.');
        if (i != -1) location = location.substring(0, i); // strip extension
        try {
            id = VersionComparator.stripVersion(location);
            version = location.substring(id.length()+1);
        } catch (IllegalArgumentException x) {
            id = location;
        }
    }
    
    /**
     * Constructs library descriptor given a file name and an installation directory.
     * If the file name is in valid ID-version format, it is parsed to set both ID and version.
     * Otherwise, ID is set to the file name (without extension) and the version remains unknown.
     */
    public LibInfo(File file, PluginDir dir) {
        this.file = file.getAbsoluteFile();
        this.dir = dir;
        String fileName = file.getName();
        int i = fileName.lastIndexOf('.');
        if (i != -1) fileName = fileName.substring(0, i); // strip extension
        try {
            id = VersionComparator.stripVersion(fileName);
            version = fileName.substring(id.length()+1);
        } catch (IllegalArgumentException x) {
            if (!checkMavenID()) id = fileName;
        }
    }
    
    /**
     * Copy constructor.
     * All fields are copied. 
     */
    public LibInfo(LibInfo other) {
        id = other.id;
        version = other.version;
        location = other.location;
        href = other.href;
        dir = other.dir;
        file = other.file;
    }

// -- Field getters and setters : ----------------------------------------------

    public String getId() {
        return id;
    }

    public void setId(String id) {
        this.id = id;
    }

    public String getVersion() {
        return version;
    }

    public void setVersion(String version) {
        this.version = version;
    }

    public String getLocation() {
        return location;
    }

    public void setLocation(String location) {
        this.location = location;
    }

    public String getHref() {
        return href;
    }

    public void setHref(String href) {
        this.href = href;
    }

    public PluginDir getDir() {
        return dir;
    }

    public void setDir(PluginDir dir) {
        this.dir = dir;
    }

    public File getFile() {
        return file;
    }

    public void setFile(File file) {
        this.file = file;
    }
    
    
// -- Utility methods : --------------------------------------------------------
    
    /**
     * Attempts to read Maven manifest and set this library id and version to maven artifact and version.
     * @return True if artifact and version were successfully extracted.
     */
    public final boolean checkMavenID() {
        if (file == null || !file.exists()) return false;
        try {
            JarFile jar = new JarFile(file);
            JarEntry je = jar.getJarEntry("META-INF/maven");
            if (je != null) {
                Enumeration<JarEntry> entries = jar.entries();
                while (entries.hasMoreElements()) {
                    je = entries.nextElement();
                    String name = je.getName();
                    if (name.startsWith("META-INF/maven") && name.endsWith("pom.properties")) {
                        Properties p = new Properties();
                        p.load(jar.getInputStream(je));
                        version = p.getProperty("version");
                        if (version != null) id = p.getProperty("artifactId");
                        return true;
                    }
                }
            }
        } catch (IOException x) {
        } catch (IllegalArgumentException x) {
        }
        return false;
    }
        
    
// -- Equality etc: -------------------------=----------------------------------

    @Override
    public boolean equals(Object o) {
        if (o instanceof LibInfo) {
            return getId().equals(((LibInfo) o).getId());
        }
        return false;
    }

    @Override
    public int hashCode() {
        return getId().hashCode();
    }
    
// -----------------------------------------------------------------------------  
}
