package org.freehep.application.studio;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.*;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import javax.swing.JOptionPane;
import org.freehep.util.VersionComparator;

/**
 * Data structure that supports operations on installed plugins and libraries.
 * 
 * FIXME: 
 *    1. getActivePlugins is called again from getActiveLibraries
 *    2. do I need version checking in getActivePlugins ?
 *
 * @author onoprien
 */
public class PluginMap {
    
// -- Private parts : ----------------------------------------------------------

    protected Studio app;
    
    // primary data structures
    
    protected EnumMap<PluginDir,Map<String,PluginInfo>> pluginMap = new EnumMap(PluginDir.class);
    protected EnumMap<PluginDir,Map<String,LibInfo>> libMap = new EnumMap(PluginDir.class);
    protected ArrayList<LibInfo> duplicateLibraries;
    
    // ID map
    
    protected EnumMap<PluginDir,Map<String,String>> name2id = new EnumMap(PluginDir.class); // file name --> id
    static private final String ID_MAP_NAME = "library.map";

    
// -- Constructors : -----------------------------------------------------------

    /**
     * Constructs PluginMap by scanning application extension directories.
     * Attempts de delete empty files while scanning.
     */
    public PluginMap(Studio application) {
        
        app = application;
        duplicateLibraries = new ArrayList<LibInfo>();
        
//        PluginDir[] extDirs = {PluginDir.SYSTEM, PluginDir.GROUP, PluginDir.USER};
//        for (PluginDir extdir : extDirs) {
        for (PluginDir extdir : PluginDir.values()) {
            scanDirectory(extdir);
        }
        
        if (duplicateLibraries.isEmpty()) {
            duplicateLibraries = null;
        } else {
            duplicateLibraries.trimToSize();
        }
        
    }
    
    /**
     * Constructs PluginMap from a collections of plugins.
     * Copies of plugin and library descriptors are made in the process.
     */
    protected PluginMap(Collection<PluginInfo> plugins) {
        for (PluginInfo plugin : plugins) {
            plugin = new PluginInfo(plugin);
            PluginDir dir = plugin.getDirectory();
            Map<String,PluginInfo> dirMapPlugin = pluginMap.get(dir);
            if (dirMapPlugin == null) {
                dirMapPlugin = new LinkedHashMap<String,PluginInfo>();
                pluginMap.put(dir, dirMapPlugin);
            }
            dirMapPlugin.put(plugin.getName(), plugin);
            Map<String,LibInfo> dirMapLib = libMap.get(dir);
            if (dirMapLib == null) {
                dirMapLib = new LinkedHashMap<String,LibInfo>();
                libMap.put(dir, dirMapLib);
            }
            for (LibInfo lib : plugin.getLibraries()) {
                lib = new LibInfo(lib);
                lib.setDir(dir);
                LibInfo other = dirMapLib.get(lib.getId());
                if ( other == null || other.getVersion() == null || 
                     (lib.getVersion() != null && (VersionComparator.compareVersion(lib.getVersion(), other.getVersion()) > 0)) ) {
                    dirMapLib.put(lib.getId(), lib);
                }
            }
        }
    }
    
    /**
     * Scans extensions directory and adds found items to this map.
     * Deletes files marked for deletion.
     */
    private void scanDirectory(PluginDir dir) {
//        JOptionPane.showMessageDialog(app, "Scanning "+ dir);
        
        String dirPath = app.getExtensionsDir(dir);
        if (dirPath == null) return;
        File extdir = new File(dirPath);
        if (!extdir.isDirectory()) return;
        
        // read (library name --> artifact id) map
        
        File infile = new File(dirPath, ID_MAP_NAME);
        try {
            BufferedReader reader = new BufferedReader(new FileReader(infile));
            HashMap<String,String> map = new HashMap<String,String>();
            name2id.put(dir, map);
            String line;
            while ((line = reader.readLine()) != null) {
                String[] mapping = line.trim().split("\\s");
                if (mapping.length == 2) { // true if line is in "name id" format
                    map.put(mapping[0], mapping[1]);
                }
            }
            reader.close();
        } catch (FileNotFoundException x) {
            name2id.remove(dir);
        } catch (IOException x) {
            throw new RuntimeException(x);
        }
        
        // scan for libraries
        
        HashMap<String,PluginInfo> loadedPlugins = new HashMap<String,PluginInfo>();
        for (PluginInfo plugin : app.getPlugins()) {
            if (plugin.getDirectory() == dir) {
                loadedPlugins.put(plugin.getName(), plugin);
            }
        }

        Map<String,PluginInfo> dirMapPlug = new LinkedHashMap<String,PluginInfo>();
        Map<String,LibInfo> dirMapLib = new LinkedHashMap<String,LibInfo>();
        String[] files = extdir.list();
        for (int i = 0; i < files.length; i++) {
            String file = files[i];
            if (file.endsWith(".jar") || file.endsWith(".tmp")) {
                File f = new File(extdir, file);
                if (f.length() > 0) {
                    try {
                        JarFile jarFile = new JarFile(f);
                        JarEntry manifest = jarFile.getJarEntry("PLUGIN-inf/plugins.xml");
                        if (manifest != null) {
                            InputStream in = jarFile.getInputStream(manifest);
                            List<PluginInfo> newPlugins = app.buildPluginList(in);
                            for (PluginInfo plugin : newPlugins) {
                                plugin.setDirectory(dir);
                                if (plugin.isApplicationCompatible(app) && plugin.isJavaCompatible()) {
                                    PluginInfo prev = dirMapPlug.get(plugin.getName());
                                    if ( prev == null || (VersionComparator.compareVersion(prev.getVersion(), plugin.getVersion()) < 0) ) {
                                        PluginInfo loaded = loadedPlugins.get(plugin.getName());
                                        if (loaded != null && VersionComparator.compareVersion(loaded.getVersion(), plugin.getVersion()) == 0) {
                                            plugin = loaded;
                                        }
                                        dirMapPlug.put(plugin.getName(), plugin);
                                    }
                                }
                            }
                        }
                        jarFile.close();
                        LibInfo library = new LibInfo(f, dir);
                        LibInfo prev = dirMapLib.get(library.getId());
                        if (prev == null) {
                            dirMapLib.put(library.getId(), library);
                        } else {
                            if ( prev.getVersion() == null || 
                                 (library.getVersion() != null && (VersionComparator.compareVersion(library.getVersion(), prev.getVersion()) > 0)) ) {
                                dirMapLib.put(library.getId(), library);
                                duplicateLibraries.add(prev);
                            } else {
                                duplicateLibraries.add(library);
                            }
                        }
                    } catch (IOException x) {
                        System.err.println("Error reading extension file " + file + " : " + x);
                    }
                } else { // Files with length 0 have been flagged for deletion by the plugin manager.
                    f.delete();
                }
            }
        }
        
        // remove plugins that miss dependencies or libraries
        
        boolean checkAgain = true;
        while (checkAgain) {
            checkAgain = false;
            List<PluginInfo> plugins = new ArrayList<PluginInfo>(dirMapPlug.values());
            for (PluginInfo plugin : plugins) {
                for (String name : plugin.getRequiredPluginNames()) {
                    PluginInfo req = dirMapPlug.get(name);
                    if (req == null) req = getPlugin(name, PluginDir.BUILTIN);
                    String min = plugin.getRequiredPluginMinVersion(name);
                    String max = plugin.getRequiredPluginMaxVersion(name);
                    if (req == null ||
                        (min != null && VersionComparator.compareVersion(min, req.getVersion()) > 0) ||
                        (max != null && VersionComparator.compareVersion(max, req.getVersion()) < 0) ) {
                            checkAgain = true;
                            dirMapPlug.remove(plugin.getName());
                            break;
                    }
                }
                for (LibInfo lib : plugin.getLibraries()) {
                    String id = getId(lib);
                    if (dirMapLib.get(id) == null && getLibrary(id, PluginDir.BUILTIN) == null) {
                        checkAgain = true;
                        dirMapPlug.remove(plugin.getName());
                        break;
                    }
                }
           }
        }
        
        // save maps
        
        if (!dirMapPlug.isEmpty() || !libMap.isEmpty()) {
            pluginMap.put(dir, dirMapPlug);
            libMap.put(dir, dirMapLib);
        }

    }


// -- Modification : -----------------------------------------------------------    
    
    /**
     * Creates a PluginMap that can be used to download and install the specified plugins.
     * This method should be called on the application master map, which is not modified as a 
     * result of the call. The returned map can be used to get a collection of required downloads 
     * through a call to {@link #getDownloads()}.
     * Once the files are downloaded, it can be passed to the master map's {@link #commit()}
     * method to install or update the plugins.
     * 
     * @param plugins Collection of plugins to be installed.
     *                Must contain all required dependencies.
     * @return PluginMap that contains only plugins that have to be installed.
     *         For each library, <tt>getFile()</tt> will return the path where the file needs to be downloaded,
     *         or <tt>null</tt> if the same or newer version of this library is already installed.
     */
    public PluginMap add(Collection<PluginInfo> plugins) {
        ArrayList<PluginInfo> pluginsToBeInstalled = new ArrayList<PluginInfo>(plugins.size());
        for (PluginInfo plugin : plugins) {
           PluginInfo installed = getPlugin(plugin.getName(), plugin.getDirectory());
           if (installed == null || VersionComparator.compareVersion(installed.getVersion(), plugin.getVersion()) < 0) {
               pluginsToBeInstalled.add(plugin);
           }
        }
        PluginMap updateMap = new PluginMap(pluginsToBeInstalled);
        for (LibInfo candidate : updateMap.getLibraries()) {
            LibInfo installed = getLibrary(candidate.getId(), candidate.getDir());
            String candidateVersion = candidate.getVersion();
            if (installed == null || installed.getVersion() == null || candidateVersion == null || 
                    (VersionComparator.compareVersion(installed.getVersion(), candidateVersion) < 0)) {
                candidate.setFile(makePath(candidate, "tmp"));
            }
        }
        return updateMap;
    }
    
    /**
     * Installs plugins and libraries contained in the <tt>update</tt> map.
     * 
     * @param update
     * @return True if the update cannot be immediately loaded and therefore requires restart.
     */
    public boolean commit(PluginMap update) {
        
        boolean restart = false;
        LinkedHashSet<File> classpath = new LinkedHashSet<File>();
        
        // add plugin descriptors
        
        for (PluginInfo plugin : update.getPlugins()) {
            restart = restart || isLoaded(plugin);
            addPlugin(plugin);
        }
        
        // install library files;
        
        for (LibInfo candidate : update.getLibraries()) {
            if (candidate.getFile() != null) {
                
                // set ID and version from maven manifest if necessary
                
                if (candidate.getVersion() == null) {
                    String oldId = candidate.getId();
                    if (candidate.checkMavenID()) {
                        String id = candidate.getId();
                        if (! id.equals(oldId)) {
                            File renameTo = makePath(candidate, "tmp");
                            if (candidate.getFile().renameTo(renameTo)) {
                                candidate.setFile(renameTo.getAbsoluteFile());
                                update.idMapPut(candidate.getDir(), oldId, id);
                            }
                        }
                    }
                }
                
                // replace existing library if necessary
                
                LibInfo installed = getLibrary(candidate.getId(), candidate.getDir());
                String candidateVersion = candidate.getVersion();
                if (installed == null || installed.getVersion() == null || candidateVersion == null || 
                          (VersionComparator.compareVersion(installed.getVersion(), candidateVersion) < 0)) {
                    restart = addLibrary(candidate) || restart;
                    if (!restart) classpath.add(candidate.getFile());
                } else {
                    candidate.getFile().delete();
                }
            }
        }
        
        // remove no longer needed libraries
        
        for (LibInfo lib : getLibraries()) {
            if (lib.getDir() != PluginDir.BUILTIN && getReferencingPlugins(lib).isEmpty()) {
                restart = removeLibrary(lib) || restart;
            }
        }
        
        // merge in ID map
        
        updateIdMap(update);
        
        return restart;
    }
    
    /**
     * Removes the specified plugins from this map and deletes (or marks for deletion) the files that are no longer needed.
     * 
     * @param plugins Collection of plugins to be removed. 
     *                Should not contain dependencies of other plugins - no checking is done by this method.
     * @return True if restart is required before the changes can take effect.
     */
    public boolean remove(Collection<PluginInfo> plugins) {
        
        boolean restart = false;
        
        // remove plugin descriptors from this map
        
        for (PluginInfo plugin : plugins) {
            restart = isLoaded(plugin) || restart;
            removePlugin(plugin);
        }

        // remove no longer needed files
        
        for (LibInfo lib : getLibraries()) {
            if (lib.getDir() != PluginDir.BUILTIN && getReferencingPlugins(lib).isEmpty()) {
                restart = removeLibrary(lib) || restart;
            }
        }
        
        // merge in ID map
        
        updateIdMap(null);
        
        return restart;
    }
    
    /**
     * Returns a "path to url" map of library files in this map.
     * This method should only be called on maps returned by {@link #add}.
     */
    public Map<File, String> getDownloads() {
        Map<File, String> out = new HashMap<File, String>();
        for (LibInfo lib : getLibraries()) {
            if (lib.getFile() != null) {
                out.put(lib.getFile(), lib.getHref());
            }
        }
        return out;
    }
    
    /**
     * Purges unused libraries.
     * Libraries not claimed by any plugins are deleted.
     */
    public void purge() {
        
        // remove older plugins
        
        // FIXME: might add an option for purging older plugins from all directories
        
        // remove unused libraries
        
        if (duplicateLibraries != null) {
            for (LibInfo lib : duplicateLibraries) {
                lib.getFile().delete();
            }
            duplicateLibraries = null;
        }
        
        for (LibInfo lib : getLibraries()) {
            if (lib.getDir() != PluginDir.BUILTIN && getReferencingPlugins(lib).isEmpty()) {
                removeLibrary(lib);
            }
        }
        
        // rename .tmp files
        
        for (LibInfo lib : getLibraries()) {
            File file = lib.getFile();
            String path = file.getPath();
            if (path.endsWith(".tmp")) {
                path = path.substring(0, path.length()-4) +".jar";
                File newFile = new File(path);
                if (!isLoaded(newFile)) {
                    if (file.renameTo(newFile)) lib.setFile(newFile.getAbsoluteFile());
                }
            }
        }
        
    }

// -- Public getters : ---------------------------------------------------------
    
    /**
     * Returns a map of names to descriptors for all currently active plugins.
     * A plugin is active if it would be loaded in the application was starting now.
     */
    public Map<String,PluginInfo> getActivePlugins() {
        
        // compile the output map based on installation directories
        
        Map<String,PluginInfo> out = new HashMap<String,PluginInfo>();
        for (PluginDir dir : PluginDir.inverseSearchOrder()) {
            Map<String,PluginInfo> dirMap = pluginMap.get(dir);
            if (dirMap != null) {
                for (Map.Entry<String, PluginInfo> entry : dirMap.entrySet()) {
                    out.put(entry.getKey(), entry.getValue());
                }
            }
        }
        
        // verify that all plugin dependencies are satisfied
        
        boolean checkAgain = true;
        while (checkAgain) {
            checkAgain = false;
            PluginInfo problem = null;
            for (PluginInfo plugin : out.values()) {
                if (plugin.getDirectory() != PluginDir.BUILTIN) {
                    Set<String> dep = plugin.getRequiredPluginNames();
                    if (!dep.isEmpty()) {
                        for (String name : dep) {
                            PluginInfo active = out.get(name);
                            if (active == null || !plugin.isRequiredPluginValid(active)) {
                                checkAgain = true;
                                break;
                            }
                        }
                        if (checkAgain) {
                            problem = plugin;
                            out.remove(problem.getName());
                            break;
                        }
                    }
                }
            }
            if (checkAgain) {
                for (PluginDir dir : PluginDir.inverseSearchOrder()) {
                    PluginInfo replacement = getPlugin(problem.getName(), dir);
                    if (replacement != null && replacement != problem) {
                        Set<String> dep = replacement.getRequiredPluginNames();
                        if (!dep.isEmpty()) {
                            for (String name : dep) {
                                PluginInfo active = out.get(name);
                                if (active == null || !replacement.isRequiredPluginValid(active)) {
                                    replacement = null;
                                    break;
                                }
                            }
                        }
                    }
                    if (replacement != null) {
                        out.put(replacement.getName(), replacement);
                        break;
                    }
                }
            }
        }
        
        return out;
    }
    
    public Map<String,LibInfo> getActiveLibraries() {
        
        // map library IDs to sets of directories where plugins need them
        
        Map<String,EnumSet<PluginDir>> id2plugins = new HashMap<String,EnumSet<PluginDir>>();
        for (PluginInfo plugin : getActivePlugins().values()) {
            for (LibInfo lib : plugin.getLibraries()) {
                String id = getId(lib);
                PluginDir dir = plugin.getDirectory();
                EnumSet<PluginDir> dirSet = id2plugins.get(id);
                if (dirSet == null) {
                    dirSet = EnumSet.of(dir);
                    id2plugins.put(id, dirSet);
                } else {
                    dirSet.add(dir);
                }
            }
        }
        
        // for each ID, choose library
        
        Map<String,LibInfo> out = new HashMap<String,LibInfo>();
        for (Map.Entry<String,EnumSet<PluginDir>> e : id2plugins.entrySet()) {
            String id = e.getKey();
            LibInfo lib = getLibrary(id, PluginDir.BUILTIN);
            if (lib == null) {
                EnumSet<PluginDir> dirSet = e.getValue();
                for (PluginDir dir : PluginDir.searchOrder()) {
                    if (dirSet.contains(dir)) {
                        LibInfo candidate = getLibrary(id, dir);
                        if (candidate != null) {
                            if (lib == null || (candidate.getVersion() != null && 
                                                VersionComparator.compareVersion(lib.getVersion(), candidate.getVersion()) < 0)) {
                                lib = candidate;
                                if (lib.getVersion() == null) break;
                            }
                        }
                    }
                }
            }
            out.put(id, lib);
        }
        
        return out;
    }
    
    public URL[] getExtensionClasspath() {
        Map<String,LibInfo> al = getActiveLibraries();
        URL[] out = new URL[al.size()];
        int i = out.length;
        try {
            for (LibInfo lib : al.values()) {
                out[--i] = lib.getFile().toURI().toURL();
            }
        } catch (MalformedURLException x) {
            throw new RuntimeException(x);
        }
        return out;
    }
    
    /** Returns a list of all plugin descriptors in this map. */
    public List<PluginInfo> getPlugins() {
        ArrayList<PluginInfo> out = new ArrayList<PluginInfo>(20);
        for (Map<String,PluginInfo> dirMap : pluginMap.values()) {
            if (dirMap != null) {
                for (PluginInfo plugin : dirMap.values()) {
                    out.add(plugin);
                }
            }
        }
        return out;
    }
    
    /** Returns a map of plugin names to descriptors in the specified directory. */
    public Map<String,PluginInfo> getPlugins(PluginDir dir) {
        if (dir == null) return getActivePlugins();
        Map<String,PluginInfo> out = pluginMap.get(dir);
        return out == null ? Collections.<String,PluginInfo>emptyMap() : Collections.unmodifiableMap(out);
    }
    
    /** 
     * Returns active plugin descriptor with the specified name.
     * Returns <tt>null</tt> if there is no such plugin in this map.
     */
    public PluginInfo getPlugin(String name) {
        for (PluginDir dir : PluginDir.searchOrder()) {
            PluginInfo plugin = getPlugin(name, dir);
            if (plugin != null) return plugin;
        }
        return null;
    }
    
    /**
     * Returns a plugin descriptor for the specified plugin name and directory.
     * Returns <tt>null</tt> if there is no such plugin in this map.
     */
    public PluginInfo getPlugin(String name, PluginDir dir) {
        Map<String,PluginInfo> dirMap = pluginMap.get(dir);
        return dirMap == null ? null : dirMap.get(name);
    }
    
    /** Returns a list of all library descriptors in this map. */
    public List<LibInfo> getLibraries() {
        ArrayList<LibInfo> out = new ArrayList<LibInfo>(50);
        for (Map<String,LibInfo> dirMap : libMap.values()) {
            if (dirMap != null) {
                for (LibInfo lib : dirMap.values()) {
                    out.add(lib);
                }
            }
        }
        return out;
    }
    
    /** 
     * Returns library descriptor for the specified ID and directory.
     * Returns <tt>null</tt> if there is no such library in this map.
     */
    public LibInfo getLibrary(String id, PluginDir dir) {
        Map<String,LibInfo> dirMap = libMap.get(dir);
        return dirMap == null ? null : dirMap.get(id);
    }
    
    /**
     * Returns a set of dependents (direct and transient) of the specified plugin.
     * Only looks for dependents installed in the same directory. No version checking is done.
     */
    public Set<PluginInfo> getDependentPlugins(PluginInfo plugin) {
        Set<PluginInfo> out = new HashSet<PluginInfo>();
        Map<String,PluginInfo> all = getPlugins(plugin.getDirectory());
        findDependentPlugins(plugin, out, all);
        return out;
    }

    
// -- Utility methods : --------------------------------------------------------
    
    private void findDependentPlugins(PluginInfo plugin, Set<PluginInfo> out, Map<String,PluginInfo> all) {
        for (PluginInfo p : all.values()) {
            if (p.getRequiredPluginNames().contains(plugin.getName()) && out.add(p)) {
                findDependentPlugins(p, out, all);
            }
        }
    }
    
    /** Removes the specified plugin descriptor from this map. */
    protected void removePlugin(PluginInfo plugin) {
        Map<String,PluginInfo> dirMap = pluginMap.get(plugin.getDirectory());
        if (dirMap != null) {
            dirMap.remove(plugin.getName());
            if (dirMap.isEmpty()) pluginMap.remove(plugin.getDirectory());
        }
    }
    
    /** Adds the specified plugin descriptor to this map. */
    protected void addPlugin(PluginInfo plugin) {
        Map<String,PluginInfo> dirMap = pluginMap.get(plugin.getDirectory());
        if (dirMap == null) {
            dirMap = new LinkedHashMap<String,PluginInfo>();
            pluginMap.put(plugin.getDirectory(), dirMap);
        }
        dirMap.put(plugin.getName(), plugin);
    }
    
    /**
     * Remove the specified descriptor from this map, delete the corresponding file if it is 
     * not on the classpath, mark for deletion otherwise.
     * 
     * @return True if the file was not actually deleted.
     */
    protected boolean removeLibrary(LibInfo library) {
        boolean restart = false;
        Map<String,LibInfo> dirMap = libMap.get(library.getDir());
        if (dirMap != null) {
            LibInfo lib = dirMap.remove(library.getId());
            if (dirMap.isEmpty()) libMap.remove(library.getDir());
            if (lib != null) {
                File file = lib.getFile();
                if (file != null) {
                    if (isLoaded(file)) {
                        markFileForDeletion(file);
                        restart = true;
                    } else {
                        if (file.delete()) {
                            restart = false;
                        } else {
                            markFileForDeletion(file);
                            restart = true;
                        }
                    }
                }
            }
        }
        return restart;
    }
    
    /**
     * Add or replace the specified descriptor to this map. 
     * If a library with the same ID and installation directory is already present in this
     * map, that library is removed, and the corresponding file is
     * deleted unless it is currently on the classpath. If the library being added corresponds
     * to a file with ".tmp" extension and is expected to be immediately loadable, the 
     * extension is changed to ".jar".
     * 
     * @return True if the replaced library was on the classpath or its file could not be deleted.
     */
    protected boolean addLibrary(LibInfo library) {
        boolean restart = false;
        Map<String,LibInfo> dirMap = libMap.get(library.getDir());
        if (dirMap == null) {
            dirMap = new LinkedHashMap<String,LibInfo>();
            libMap.put(library.getDir(), dirMap);
        } else {
            LibInfo old = dirMap.get(library.getId());
            if (old != null) {
                File oldFile = old.getFile();
                if (isLoaded(oldFile)) {
                    restart = true;
                } else {
                    restart = !oldFile.delete() || restart;
                }
            }
        }
        dirMap.put(library.getId(), library);
        if (!restart && library.getFile().getName().endsWith(".tmp")) {
            File renameTo = makePath(library, "jar");
            restart = isLoaded(renameTo) || !library.getFile().renameTo(renameTo);
            if (!restart) library.setFile(renameTo);
        }
        return restart;
    }
    
    /**
     * Returns an identity set of plugin descriptors referencing the specified library.
     * A plugin references a library if both are installed in the same directory, and the 
     * plugin descriptor specifies a library with a matching (or mapped) ID.
     */
    protected Set<PluginInfo> getReferencingPlugins(LibInfo library) {
        Set<PluginInfo> out = Collections.newSetFromMap(new IdentityHashMap<PluginInfo,Boolean>());
        PluginDir dir = library.getDir();
        Map<String,PluginInfo> dirMap = getPlugins(dir);
        for (PluginInfo plugin : dirMap.values()) {
            for (LibInfo lib : plugin.getLibraries()) {
                if (library.getId().equals(getId(lib))) out.add(plugin);
            }
        }
        return out;
    }
    
    /**
     * Puts a mapping from name to ID into the ID map.
     * @return Previous value.
     */
    protected String idMapPut(PluginDir dir, String name, String id) {
        Map<String,String> dirMap = name2id.get(dir);
        if (dirMap == null) {
            dirMap = new HashMap<String,String>();
            name2id.put(dir, dirMap);
        }
        return dirMap.put(name, id);
    }
    
    /**
     * Removes a mapping for the specified name from the ID map.
     * @return Previous value.
     */
    protected String idMapRemove(PluginDir dir, String name) {
        Map<String,String> dirMap = name2id.get(dir);
        if (dirMap == null) return null;
        String out = dirMap.remove(name);
        if (dirMap.isEmpty()) name2id.remove(dir);
        return out;
    }
    
    /**
     * Returns library ID to which the specified name is mapped in the ID map.
     * Returns <tt>name</tt> if the ID map contains no such key.
     */
    protected String idMapGet(PluginDir dir, String name) {
        Map<String,String> dirMap = name2id.get(dir);
        if (dirMap == null) {
            return name;
        } else {
            String id = dirMap.get(name);
            return id == null ? name : id;
        }
    }
    
    /**
     * Returns an ID of the required library, taking ID map into account if necessary.
     * @param library Library descriptor that is a part of plugin descriptor.
     * @return Library ID to be used in identifying the jar file.
     */
    protected String getId(LibInfo library) {
        String id = library.getId();
        if (library.getVersion() == null) id = idMapGet(library.getDir(), id);
        return id;
    }
    
    /**
     * Constructs abstract absolute path to a library file based on its ID and version.
     * 
     * @param library
     * @param extension File extension.
     */
    protected File makePath(LibInfo library, String extension) {
        String version = library.getVersion();
        StringBuilder sb = new StringBuilder(library.getId());
        if (version != null) sb.append('-').append(version);
        sb.append('.').append(extension);
        File out = new File(app.getExtensionsDir(library.getDir()), sb.toString());
        return out.getAbsoluteFile();
    }

    /**
     * Checks whether the file is on the extensions class loader classpath.
     * This method should only be called on the master map.
     * FIXME: might want to switch to faster implementation.
     */
    protected boolean isLoaded(File file) {
        ExtensionClassLoader loader = app.getExtensionLoader();
        if (loader == null) return false;
        URL[] classpath = loader.getURLs();
        try {
            URL url = file.toURI().toURL();
            for (URL u : classpath) {
                if (u.equals(url)) return true;
            }
            return false;
        } catch (MalformedURLException x) {
            throw new IllegalArgumentException(x);
        }
    }
    
    /**
     * Returns <tt>true</tt> if a plugin with the same name and directory as the argument is currently loaded.
     */
    protected boolean isLoaded(PluginInfo plugin) {
        String name = plugin.getName();
        PluginDir dir = plugin.getDirectory();
        for (PluginInfo loaded : app.getPlugins()) {
            if (name.equals(loaded.getName()) && dir == loaded.getDirectory()) {
                return true;
            }
        }
        return false;
    }
    
    protected void updateIdMap(PluginMap update) {
        
        EnumSet<PluginDir> changed = EnumSet.noneOf(PluginDir.class);
        
        // merge in id map from update
        
        if (update != null && !update.name2id.isEmpty()) {
            for (PluginDir dir : PluginDir.values()) {
                Map<String,String> dirMap = update.name2id.get(dir);
                if (dirMap != null) {
                    for (Map.Entry<String,String> e : dirMap.entrySet()) {
                        String old = idMapPut(dir, e.getKey(), e.getValue());
                        if (old == null || !old.equals(e.getValue())) {
                            changed.add(dir);
                        }
                    }
                }
            }
        }
        
        // remove unnecessary entries
        
        for (PluginDir dir : changed) {
            Map<String,PluginInfo> pn2pi = pluginMap.get(dir);
            if (pn2pi == null) name2id.remove(dir);
            Map<String,String> n2id = name2id.get(dir);
            if (n2id == null) break;
            Set<String> ids = new HashSet<String>();
            for (PluginInfo plugin : pn2pi.values()) {
                for (LibInfo lib : plugin.getLibraries()) {
                    if (lib.getVersion() == null) {
                        ids.add(lib.getId());
                    }
                }
            }
            Iterator<Map.Entry<String,String>> it = n2id.entrySet().iterator();
            while (it.hasNext()) {
                Map.Entry<String,String> e = it.next();
                if (!ids.contains(e.getKey())) {
                    it.remove();
                }
            }
            if (n2id.isEmpty()) pluginMap.remove(dir);
        }
        
        // save into files
        
        for (PluginDir dir : changed) {
            File f = new File(app.getExtensionsDir(dir), ID_MAP_NAME);
            f.delete();
            Map<String,String> n2id = name2id.get(dir);
            if (n2id != null) {
                try {
                    PrintWriter pw = new PrintWriter(f);
                    for (Map.Entry<String, String> e : n2id.entrySet()) {
                        pw.println(e.getKey() +" "+ e.getValue());
                    }
                    pw.close();
                } catch (FileNotFoundException x) {
                    throw new IllegalArgumentException(x);
                }
            }
        }
        
    }
    
    protected boolean markFileForDeletion(File file) {
        try {
            (new FileOutputStream(file)).close();
            return true;
        } catch (IOException x) {
            return false;
        }
    }
    
// -----------------------------------------------------------------------------    
}
