package org.freehep.application.studio;

import java.awt.Component;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.*;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import org.freehep.application.ApplicationEvent;
import org.freehep.application.mdi.InternalFramePageManager;
import org.freehep.application.mdi.MDIApplication;
import org.freehep.application.mdi.PageContext;
import org.freehep.application.mdi.PageManager;
import org.freehep.application.mdi.TabbedPageManager;
import org.freehep.util.FreeHEPLookup;
import org.freehep.util.commandline.CommandLine;
import org.freehep.xml.util.ClassPathEntityResolver;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.openide.util.Lookup;

/**
 * Swing application that supports plugins.
 *
 * @author tonyj
 * @version $Id: Studio.java 14533 2013-04-18 22:14:05Z onoprien $
 */
public class Studio extends MDIApplication {
    
// -- Private parts : ----------------------------------------------------------

    private FreeHEPLookup lookup;
    private EventSender sender = new EventSender();
    private boolean debugExtensions = System.getProperty("debugExtensions") != null;
    private SAXBuilder builder;
    
    private ArrayList<PluginInfo> loadedPlugins = new ArrayList<PluginInfo>();
    private ExtensionClassLoader extensionLoader;
    public static final String LOADDIR = "loaddir";
    private boolean isApplicationVisible = false;
    private boolean isApplicationInitialized = false;
    private boolean atLeastOnePluginFailedToLoad = false;
    
    private PluginMap pluginMap;
    
// -- Construction and initialization : ----------------------------------------

    protected Studio(String name) {
        super(name);
        // For the moment at least we will use JDOM for parsing the plugin.xml files
        builder = new SAXBuilder(true);
        builder.setEntityResolver(new ClassPathEntityResolver("plugin.dtd", Studio.class));

        getLookup().add(new TabbedPageManager(), "Tabbed Panes");
        getLookup().add(new InternalFramePageManager(), "Internal Frames");
    }

    private Studio() {
        this("Studio");
    }

    @Override
    protected CommandLine createCommandLine() {
        CommandLine cl = super.createCommandLine();
        // register standard options
        cl.addOption("extdir", null, "directory", "Sets the directory to scan for plugins");
        return cl;
    }

    @Override
    protected void init() {
        super.init();
        setStatusMessage("Loading extensions...");
        loadExtensions();
        // Now load the real page manager
        setStatusMessage("Setting page manager...");
        setPageManager(createRealPageManager());
        if (atLeastOnePluginFailedToLoad) {
            error("At least one plugin failed to load, see Plugin Manager for details");
        }
    }

    /** Called from <tt>init()</tt> to load extensions. */
    private void loadExtensions() {
        
        // Scan for extensions and purge unused libraries

        getPluginMap().purge();

        // Create the extension Loader.

        extensionLoader = new ExtensionClassLoader(pluginMap.getExtensionClasspath());
        createLookup().setClassLoader(extensionLoader);
        // Make sure the extensionClassLoader is set as the contextClassLoader
        // so that services etc can be looked up in jar files from the extension directory.
        Runnable lola = new Runnable() {
            @Override
            public void run() {
                Thread.currentThread().setContextClassLoader(extensionLoader);
            }
        };
        lola.run(); // for the main (this) thread
        SwingUtilities.invokeLater(lola); // for the event dispatch thread

        // Load plugins
        
        loadPlugins(pluginMap.getActivePlugins().values(), extensionLoader);
    }

    public static void main(String[] args) {
        new Studio().createFrame(args).setVisible(true);
    }


// -- Getters : ----------------------------------------------------------------

    public EventSender getEventSender() {
        return sender;
    }

    public final FreeHEPLookup getLookup() {
        if (lookup == null) lookup = createLookup();
        return lookup;
    }

    @Deprecated
    public String getUserExtensionsDir() {
        return getExtensionsDir(PluginDir.USER);
    }

    @Deprecated
    public String getGroupExtensionsDir() {
        return getExtensionsDir(PluginDir.GROUP);
    }

    @Deprecated
    public String getSystemExtensionsDir() {
        return getExtensionsDir(PluginDir.SYSTEM);
    }
    
    /**
     * Returns a path to the specified extension directory.
     * <p>
     * We look for extensions:<ul>
     *   <li>USER: In the directory specified by the org.freehep.application.studio.user.extensions property
     *   <li>GROUP: In the directory specified by the org.freehep.application.studio.group.extensions property
     *   <li>SYSTEM: In the directory specified by the org.freehep.application.studio.system.extensions property</ul>
     * The following defaults apply if the property is not specified
     *   <li>USER: {user.home}/.FreeHEPPlugins
     *   <li>GROUP: none
     *   <li>SYSTEM: {java.home}/.FreeHEPPlugins</ul>
     */
    public String getExtensionsDir(PluginDir dir) {
        String out = null;
        switch (dir) {
            case SYSTEM:
                out = getAppProperties().getProperty("org.freehep.application.studio.system.extensions", "{java.home}/FreeHEPPlugins");
                break;
            case GROUP:
                out = getAppProperties().getProperty("org.freehep.application.studio.group.extensions");
                break;
            case USER:
                out = getCommandLine().getOption("extdir");
                if (out == null) {
                  out = getAppProperties().getProperty("org.freehep.application.studio.user.extensions", "{user.home}/.FreeHEPPlugins");
                }
                break;
            default:
                return null;
        }
        if (out != null) {
            try {
                out = (new File(out)).getCanonicalPath();
            } catch (IOException x) {
                out = null;
            }
        }
        return out;
    }

    public ExtensionClassLoader getExtensionLoader() {
        return extensionLoader;
    }

    /** Return a list of loaded plugins. */
    public List<PluginInfo> getPlugins() {
        return Collections.unmodifiableList(loadedPlugins);
    }
    
    /** 
     * Returns plugin descriptor from the list of loaded plugins with the specified name.
     * Returns <tt>null</tt> if there is no loaded plugin with matching name.
     */
    public PluginInfo getPlugin(String name) {
        for (PluginInfo plugin : loadedPlugins) {
            if (plugin.getName().equals(name)) return plugin;
        }
        return null;
    }
    
    /** Returns a map of installed plugins. */
    public PluginMap getPluginMap() {
        if (pluginMap == null) {
            pluginMap = new PluginMap(this);
        }
        return pluginMap;
    }
    
    
// -- Operations on plugins : --------------------------------------------------

    /**
     * Stops a plugin.
     * Calls <tt>stop()</tt> method on the specified plugin and removes a reference
     * to a {@link Plugin} object from the <tt>PluginInfo</tt>. 
     * The <tt>PluginInfo</tt> object supplied as an argument is used to identify a loaded
     * plugin with the same name. If no matching loaded plugin is found, this method returns 
     * immediately without doing anything.
     * 
     * @throws IllegalArgumentException if the plugin cannot be shut down.
     */
    public void stopPlugin(PluginInfo plugin) {
        plugin = getPlugin(plugin.getName());
        if (plugin == null) return;
        Plugin plug = plugin.getPlugin();
        if (plug == null || !plug.canBeShutDown()) {
            throw new IllegalArgumentException("Plugin can not be stopped");
        }
        plug.stop();
        plugin.setPlugin(null);
    }

    /**
     * Starts and initializes a plugin.
     * Loads the plugin class with the extension class loader, creates an instance, 
     * and calls its initialization methods
     * The <tt>PluginInfo</tt> object supplied as an argument is used to identify a loaded
     * plugin with the same name. If no matching loaded plugin is found, this method returns 
     * immediately without doing anything.
     * 
     * @throws Throwable Re-throws any exceptions thrown by the plugin class code.
     */
    public void startPlugin(PluginInfo plugin) throws Throwable {
        plugin = getPlugin(plugin.getName());
        if (plugin == null) return;
        initializePlugin(plugin, extensionLoader);
        revalidate();
    }

    private Plugin initializePlugin(PluginInfo plugin, ClassLoader loader) throws Throwable {
        try {
            getAppProperties().putAll(plugin.getProperties());
            Class c = loader.loadClass(plugin.getMainClass());
            Plugin plug = (Plugin) c.newInstance();
            plug.setContext(this);
            plugin.setPlugin(plug);
            if (isApplicationInitialized) plug.postInit();
            if (isApplicationVisible) plug.applicationVisible();
            plugin.setErrorStatus(null);
            return plug;
        } catch (Throwable t) {
            plugin.setErrorStatus(t);
            throw t;
        }
    }

    /**
     * Reads the specified stream and creates a list of <tt>PluginInfo</tt> instances.
     */
    protected List<PluginInfo> buildPluginList(InputStream in) throws IOException {
        Properties user = getUserProperties();
        List<PluginInfo> result = new ArrayList<PluginInfo>();
        try {
            Document doc = builder.build(in);
            List<Element> rootChildren = doc.getRootElement().getChildren();
            for (Element node : rootChildren) {
                PluginInfo plugin = new PluginInfo(node);
                plugin.loadUserProperties(user);
                result.add(plugin);
                if (debugExtensions) System.out.println("\t\tPlugin: " + plugin.getName());
            }
        } catch (JDOMException x) {
            if (debugExtensions) x.printStackTrace();
        } finally {
            in.close();
        }
        return result;
    }

    /**
     * Loads and initializes the specified plugins.
     * Only plugins whose load-at-start property is set to <tt>true</tt> are started.
     * Class loader should contain all required classes on its classpath before this method is called.
     * 
     * @return True if at least one plugin failed to load.
     */
    public boolean loadPlugins(Collection<PluginInfo> plugins, ClassLoader loader) {
        for (PluginInfo plugin : plugins) {
            if (! loadedPlugins.contains(plugin)) {
                loadedPlugins.add(plugin);
                if (plugin.isLoadAtStart()) {
                    try {
                        setStatusMessage("Loading " + plugin.getName() + "...");
                        initializePlugin(plugin, loader);
                        plugin.setErrorStatus(null);
                    } catch (Throwable t) {
                        plugin.setErrorStatus(t);
                        atLeastOnePluginFailedToLoad = true;
                    }
                }
//            } else {
//                atLeastOnePluginFailedToLoad = true;
            }
        }
        loadedPlugins.trimToSize();
        revalidate(); // plugins may have added menus etc, so for good measure!
        return atLeastOnePluginFailedToLoad;
    }


// -- Calling Plugin and PluginInfo SPI methods on loaded plugins : ------------
    
    @Override
    protected void fireInitializationComplete(ApplicationEvent event) {
        super.fireInitializationComplete(event);
        getEventSender().broadcast(event);
        for (PluginInfo info : loadedPlugins) {
            Plugin plugin = info.getPlugin();
            if (plugin != null) {
                plugin.postInit();
            }
        }
        isApplicationInitialized = true;
    }

    @Override
    protected void fireApplicationVisible(ApplicationEvent event) {
        super.fireApplicationVisible(event);
        getEventSender().broadcast(event);
        for (PluginInfo info : loadedPlugins) {
            Plugin plugin = info.getPlugin();
            if (plugin != null) {
                plugin.applicationVisible();
            }
        }
        isApplicationVisible = true;
    }

    @Override
    protected void fireAboutToExit(ApplicationEvent event) {
        for (PluginInfo info : loadedPlugins) {
            Properties user = getUserProperties();
            info.saveUserProperties(user);
        }
        super.fireAboutToExit(event);
        getEventSender().broadcast(event);
    }


// -- Utility methods : --------------------------------------------------------

    protected FreeHEPLookup createLookup() {
        return FreeHEPLookup.instance();
    }
    
    
// -- Page management : --------------------------------------------------------

    protected PageManager createRealPageManager() {
        String name = getUserProperties().getProperty("pageManagerName", "Tabbed Panes");
        Lookup.Template template = new Lookup.Template(PageManager.class, name, null);
        Lookup.Result result = getLookup().lookup(template);
        if (!result.allInstances().isEmpty()) {
            return (PageManager) result.allInstances().iterator().next();
        } else {
            // Previously we used the class name as pageManager, so this is just for backward compatibility.
            try {
                return super.createPageManager();
            } catch (Throwable t) {
                // Last chance, use whatever page manager we can find.
                PageManager pm = (PageManager) getLookup().lookup(PageManager.class);
                if (pm != null) return pm;
                return new TabbedPageManager();
            }
        }
    }

    @Override
    protected PageManager createPageManager() {
        // We initially create a dummy page manager, so we can delay creating the 
        // real page manager until after the plugins have been loaded (so that 
        // a plugin can register a plugin manager)
        return new DummyPageManager();
    }

    private static class DummyPageManager extends PageManager {

        private JPanel panel = new JPanel();

        @Override
        protected Component getEmbodiment() {
            return panel;
        }

        @Override
        protected void iconChanged(PageContext page) {
        }

        @Override
        protected void show(PageContext page) {
        }

        @Override
        protected void titleChanged(PageContext page) {
        }
    }
}