package org.freehep.application.studio.pluginmanager;

import java.awt.Component;
import java.io.File;
import java.net.URL;
import java.util.*;
import java.util.logging.Logger;
import javax.swing.Box;
import javax.swing.JCheckBox;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.EventListenerList;
import org.freehep.application.Application;
import org.freehep.application.PropertyUtilities;
import org.freehep.application.studio.ExtensionClassLoader;
import org.freehep.application.studio.Plugin;
import org.freehep.application.studio.PluginDir;
import org.freehep.application.studio.PluginInfo;
import org.freehep.application.studio.PluginMap;
import org.freehep.application.studio.Studio;
import org.freehep.util.VersionComparator;

/**
 * A Plugin which provides a Plugin Manager.
 * 
 * TODO:
 *   1. Standardize property keys.
 *   2. Inform user if additional plugins are being uninstalled.
 *   3. Do I need update(Component parent, Collection<PluginInfo> plugins) ?
 *
 * @author tonyj
 * @version $Id: PluginManager.java 14533 2013-04-18 22:14:05Z onoprien $
 */
public class PluginManager extends Plugin implements Runnable {
    
// -- Private parts : ----------------------------------------------------------

    private boolean checkForPluginsAtStart; // property - should we check for updates when the application starts
    private URL checkURL; // property - URL from which the list of available plugins should be downloaded
    private boolean removeJavaIncompatible; // property - shoul we remove plugins incompatible with Java version when purging
    private boolean removeAppIncompatible; // property - shoul we remove plugins incompatible with application version when purging
    
    protected PluginListHandler pluginListHandler; // IoC service - handles fetching a list of available plugins
    protected Thread pluginListReaderThread; // thread used to download the list of available plugins
    protected volatile boolean pluginListIsReady; // flag - set to true once the list of available plugins has been downloaded
    
    protected List<PluginInfo> availablePlugins; // available plugins as reported by PluginListHandler
    
    protected ArrayList<PluginInfo> installablePlugins;  // available plugins not currently installed
    protected ArrayList<PluginInfo>  updatablePlugins;  // installed plugins for which newer version is available
    protected Map<PluginInfo,PluginInfo>  updateMap; // mapping from installed to available plugins, regardless of version
    
    protected PluginMap masterMap;
    
    protected boolean dialogVisible = false;
    
    private EventListenerList listeners = new EventListenerList();
    
    static final Logger logger = Logger.getLogger(PluginManager.class.getName());

    
// -- Plugin SPI : -------------------------------------------------------------

    @Override
    protected void init() throws org.xml.sax.SAXException, java.io.IOException {
        
        final Studio app = getApplication();
        Properties user = app.getUserProperties();
        String pluginURL = user.getProperty("PluginManager.URL");
        checkURL = pluginURL == null ? null : new URL(pluginURL);
        checkForPluginsAtStart = PropertyUtilities.getBoolean(user, "checkForPluginsAtStart", false);
        
        pluginListIsReady = false;
    }

    @Override
    protected void applicationVisible() {
        masterMap = getApplication().getPluginMap();
        if (checkForPluginsAtStart && checkURL != null) {
            startPluginListDownload();
        }
    }

    
// -- Getters : ----------------------------------------------------------------
    
    /** Returns <tt>true</tt> if the list of available plugins has been downloaded. */
    protected boolean isPluginListIsReady() {
        return pluginListIsReady;
    }
    
    /** Returns a map of currently installed plugins. */
    protected PluginMap getPluginMap() {
        return masterMap;
    }

    /** Returns the list of available plugins, or <tt>null</tt> if the list has not yet been downloaded. */
    protected List<PluginInfo> getAvailablePlugins() {
        return pluginListIsReady ? availablePlugins : null;
    }

    /**
     * Returns a map from installed plugins to available plugins with the same name.
     * The map contains plugin pairs regardless of whether the available version is greater than the 
     * installed one. Installed or available plugins that do not have a counterpart are not in the map.
     * Returns <tt>null</tt> if the list of available plugins is not ready yet.
     */
    Map<PluginInfo,PluginInfo> getUpdateMap() {
        if (!pluginListIsReady) return null;
        if (updateMap == null) processAvailablePlugins();
        return updateMap;
    }

    /**
     * Returns a list of active plugins.
     * A plugin is active if its manifest is found in a preferential position in one of the
     * extension directories. Note that this does not necessarily mean the plugin is loaded.
     * If the plugin was installed after the application was started, and loading it required 
     * a restart, then the plugin will be active but not loaded.
     */
    protected List<PluginInfo> getPlugins() {
        return new ArrayList<PluginInfo>(masterMap.getActivePlugins().values());
    }

    /**
     * Returns a list of available plugins that do not have installed counterparts with the same name.
     */
    List<PluginInfo> getInstallablePlugins() {
        if (!pluginListIsReady) return null;
        if (installablePlugins == null) processAvailablePlugins();
        return installablePlugins;
    }

    /**
     * Returns the list of updatable installed plugins.
     * A plugin is updatable if the list of available plugins contains a newer version of it. 
     */
    List<PluginInfo> getUpdatablePlugins() {
        if (!pluginListIsReady) return null;
        if (updatablePlugins == null) processAvailablePlugins();
        return updatablePlugins;
    }

    /**
     * Getter for property checkAtStart.
     * If <tt>true</tt>, the plugin manager will start downloading a list of available
     * and updatable plugins when the application is started.
     */
    public boolean isCheckAtStart() {
        return checkForPluginsAtStart;
    }

    /**
     * Getter for property checkURL.
     * URL from which the list of available and updatable plugins should be downloaded.
     */
    public URL getCheckURL() {
        return this.checkURL;
    }

    /**
     * Getter for property removeJavaIncompatible.
     * If <tt>true</tt>, plugins incompatible with the currently used java runtime version 
     * are uninstalled when extension directories are purged.
     */
    public boolean isRemoveJavaIncompatible() {
        return removeJavaIncompatible;
    }

    /**
     * Getter for property removeAppIncompatible.
     * If <tt>true</tt>, plugins incompatible with the currently used application version 
     * are uninstalled when extension directories are purged.
     */
    public boolean isRemoveAppIncompatible() {
        return removeAppIncompatible;
    }
    
    
// -- Setters : ----------------------------------------------------------------

    /**
     * Setter for property checkAtStart.
     * If <tt>true</tt>, the plugin manager will start downloading a list of available
     * plugins when the application is started.
     */
    public void setCheckAtStart(boolean checkAtStart) {
        this.checkForPluginsAtStart = checkAtStart;
        PropertyUtilities.setBoolean(getApplication().getUserProperties(), "checkForPluginsAtStart", checkAtStart);
    }

    /**
     * Setter for property checkURL.
     * URL from which the list of available and plugins should be downloaded.
     */
    public void setCheckURL(URL checkURL) {
        this.checkURL = checkURL;
        getApplication().getUserProperties().setProperty("PluginManager.URL", checkURL.toExternalForm());
    }

    /**
     * Setter for property removeJavaIncompatible.
     * If <tt>true</tt>, plugins incompatible with the currently used java runtime version 
     * are uninstalled when extension directories are purged.
     */
    public void setRemoveJavaIncompatible(boolean removeJavaIncompatible) {
        this.removeJavaIncompatible = removeJavaIncompatible;
    }

    /**
     * Setter for property removeJavaIncompatible.
     * If <tt>true</tt>, plugins incompatible with the currently used application version 
     * are uninstalled when extension directories are purged.
     */
    public void setRemoveAppIncompatible(boolean removeAppIncompatible) {
        this.removeAppIncompatible = removeAppIncompatible;
    }
    
    /* Setter for PluginListHandler service. */
    public void setPluginListHandler(PluginListHandler pluginListHandler) {
        this.pluginListHandler = pluginListHandler;
    }
    
    
// -- GUI display : ------------------------------------------------------------

    /** 
     * Displays PluginManagerDialog.
     * Should be called from the event processing thread.
     */
    public void showPluginManager() {
        if (!pluginListIsReady && pluginListReaderThread == null) startPluginListDownload();
        JFrame frame = (JFrame) SwingUtilities.getAncestorOfClass(JFrame.class, getApplication());
        JDialog dlg = new PluginManagerDialog(frame, PluginManager.this);
        dlg.setModal(true);
        dlg.setTitle("Plugin Manager");
        dlg.pack();
        dlg.setLocationRelativeTo(getApplication());
        try {
            dialogVisible = true;
            dlg.setVisible(true);
        } finally {
            dialogVisible = false;
            cleanup();
        }
    }

    /** 
     * Offers the user to update installed plugins and executes user's command.
     * Should be called from the event processing thread.
     */
    public void offerUpdate() {
        Box message = Box.createVerticalBox();
        JLabel label = new JLabel("Updated plugins available");
        label.setAlignmentX(1.0f);
        message.add(label);
        JCheckBox ask = new JCheckBox("Don't show me this again");
        message.add(ask);
        String[] options = {"Install now", "Plugin Manager...", "Cancel"};
        int rc = JOptionPane.showOptionDialog(getApplication(), message, "Updates available", JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.INFORMATION_MESSAGE, null, options, options[0]);
        if (ask.isSelected()) {
            getApplication().getUserProperties().setProperty("notifyPluginUpdates", "false");
        }
        if (rc == 0) { // install now
            updateInstalledPlugins(getApplication());
        } else if (rc == 1) { // plugin manager
            showPluginManager();
        }
    }
    
// -- Downloading and installing plugins : -------------------------------------

    /** Launch downloading the list of available plugins in a new thread. */
    protected void startPluginListDownload() {
        pluginListReaderThread = new Thread(this);
        pluginListReaderThread.setDaemon(true);
        pluginListReaderThread.setPriority(Thread.NORM_PRIORITY);
        pluginListReaderThread.start();
    }

    @Override
    /** Download and process the list of available plugins.*/
    public void run() {
        
        PluginListHandler handler = (pluginListHandler == null ? new PluginListHandler() : pluginListHandler);
        availablePlugins = handler.getAvailablePlugins(checkURL, logger, getApplication());
        
        Runnable r = new Runnable() {
            public void run() {
                processAvailablePlugins();
                boolean notify = PropertyUtilities.getBoolean(getApplication().getUserProperties(), "notifyPluginUpdates", true);
                if (!updatablePlugins.isEmpty() && !dialogVisible && notify) {
                    offerUpdate();
                }
                cleanup();
                pluginListReaderThread = null;
                pluginListIsReady = true;
                fireStateChanged();
            }
        };
        SwingUtilities.invokeLater(r);
    }

    /** 
     * Fills lists of updatable and installable plugins.
     * Called once the list of available plugins is downloaded.
     */
    protected void processAvailablePlugins() {
        installablePlugins = new ArrayList<PluginInfo>(availablePlugins.size());
        updatablePlugins = new ArrayList<PluginInfo>(availablePlugins.size());
        updateMap = new HashMap<PluginInfo,PluginInfo>();
        Map<String,PluginInfo> activePlugins = masterMap.getActivePlugins();
        for (PluginInfo info : availablePlugins) {
            PluginInfo old = activePlugins.get(info.getName());
            if (old != null) {
                updateMap.put(old, info);
                if (VersionComparator.compareVersion(info.getVersion(), old.getVersion()) > 0) {
                    updatablePlugins.add(old);
                }
            } else {
                installablePlugins.add(info);
            }
        }
    }

    /**
     * Updates all installed plugins for which newer versions are available.
     * 
     * @param parent Component to be used as parent by any GUI windows displayed while 
     *               executing this method, if any. If <tt>null</tt>, no error notifications 
     *               will be displayed, and <tt>IllegalArgumentException</tt> will be thrown instead.
     * @return true if restarting the application is required for the changes to take effect.
     * @throws IllegalArgumentException if errors occur while updating, and <tt>parent</tt> was not specified.
     */
    public boolean updateInstalledPlugins(Component parent) {
        return install(parent, updatablePlugins);
    }
    
    
    /**
     * Installs specified plugins into the application extensions directories.
     * 
     * @param parent Component to be used as parent by any GUI windows displayed while 
     *               executing this method, if any. If <tt>null</tt>, no error notifications 
     *               will be displayed, and <tt>IllegalArgumentException</tt> will be thrown instead.
     * @param plugins Plugins to be installed. May or may not contain required dependencies.
     * @return true if restarting the application is required for the changes to take effect.
     * @throws IllegalArgumentException if errors occur while updating, and <tt>parent</tt> was not specified.
     */
    public boolean install(Component parent, Collection<PluginInfo> plugins) {
        
        // Compile a list of plugins to install, taking dependencies into account
        
        ArrayList<PluginInfo> verified = new ArrayList<PluginInfo>(plugins.size());
        try {
            for (PluginDir dir : PluginDir.sgu()) {
                // compile a set off all plugins that need to be installed, including dependencies
                Map<String, PluginInfo> required = new HashMap<String, PluginInfo>();
                Set<String> already = new HashSet<String>();
                for (PluginInfo plugin : plugins) {
                    if (plugin.getDirectory() == dir) {
                        addNameWithDependencies(dir, plugin.getName(), required, already);
                    }
                }
                // calculate acceptable version ranges for all plugins
                Map<String, PluginInfo> after = new HashMap<String, PluginInfo>(masterMap.getPlugins(dir));
                after.putAll(required);
                Map<String, String[]> n2range = new HashMap<String, String[]>();
                for (PluginInfo plugin : after.values()) {
                    for (String name : plugin.getRequiredPluginNames()) {
                        String min = plugin.getRequiredPluginMinVersion(name);
                        String max = plugin.getRequiredPluginMaxVersion(name);
                        if (min != null || max != null) {
                            String[] range = n2range.get(name);
                            if (range == null) {
                                range = new String[2];
                                n2range.put(name, range);
                            }
                            if (min != null && (range[0] == null || VersionComparator.compareVersion(min, range[0]) > 0)) {
                               range[0] = min; 
                            }
                            if (max != null && (range[1] == null || VersionComparator.compareVersion(max, range[1]) < 0)) {
                               range[1] = max; 
                            }
                        }
                    }
                }
                // check that all plugin versions are within ranges
                for (PluginInfo plugin : after.values()) {
                    String version = plugin.getVersion();
                    String[] range = n2range.get(plugin.getName());
                    if (range != null) {
                        if (range[0] != null && VersionComparator.compareVersion(version, range[0]) < 0) {
                            throw new IllegalStateException(plugin.getName());
                        }
                        if (range[1] != null && VersionComparator.compareVersion(version, range[1]) > 0) {
                            throw new IllegalStateException(plugin.getName());
                        }
                    }
                }
                // add to the list of plugins to install
                verified.addAll(required.values());
            }
        } catch (IllegalStateException x) {
            String message = "Installation cancelled: missing or incompatible dependencies";
            if (parent == null) {
                throw new IllegalArgumentException(message);
            } else {
                JOptionPane.showMessageDialog(parent, message, "Installation error", JOptionPane.ERROR_MESSAGE);
                return false;
            }
        }
        
        // Create PluginMap for installation
        
        PluginMap update = masterMap.add(verified);
        
        // Download files
        
        Map<File,String> downloads = update.getDownloads(); // File(resource) --> href
        PluginDownload download = new PluginDownload(downloads);
        Thread t = new Thread(download);
        t.start();
        JOptionPane.showMessageDialog(parent, download, "Downloading...", JOptionPane.PLAIN_MESSAGE);

        Throwable status = download.getStatus();
        if (status != null) {
            download.cleanUp();
            String message = "Download failed: " + status;
            if (parent == null) throw new IllegalArgumentException(message);
            JOptionPane.showMessageDialog(parent, message, "Download error", JOptionPane.ERROR_MESSAGE);
            return false;
        } else {
            download.commit();
        }
        
        // Install files
        
        boolean restart = masterMap.commit(update);
        
        // Load new plugins
        
        if (!restart) {
            ExtensionClassLoader loader = getApplication().getExtensionLoader();
            for (URL url : update.getExtensionClasspath()) {
                loader.addURL(url);
            }
            boolean failure = getApplication().loadPlugins(update.getPlugins(), loader);
            if (failure) {
                String message = "At least one plugin failed to load, see Plugin Manager for details";
                if (parent == null) throw new IllegalArgumentException(message);
                Application.error(parent, message);
            }
        }
        
        // Update PluginManager and its listeners
        
        installablePlugins.removeAll(plugins);
        fireStateChanged();
        return restart;
    }
    
    /**
     * Uninstall specified plugins.
     * 
     * @param parent Component to be used as parent by any GUI windows displayed while 
     *               executing this method, if any. If <tt>null</tt>, no error notifications 
     *               will be displayed, and <tt>IllegalArgumentException</tt> will be thrown instead.
     * @param plugins Plugins to be uninstalled. May or may not contain required dependencies.
     * @return true if restarting the application is required for the changes to take effect.
     * @throws IllegalArgumentException if errors occur while updating, and <tt>parent</tt> was not specified.
     */
    public boolean uninstall(Component parent, Collection<PluginInfo> plugins) {
        
        // Compile a set of plugins to remove, taking dependencies between plugins into account
        
        ArrayList<PluginInfo> verified = new ArrayList<PluginInfo>(plugins.size());
        for (PluginDir dir : PluginDir.sgu()) {
            HashSet<PluginInfo> verDir = new HashSet<PluginInfo>();
            for (PluginInfo plugin : plugins) {
                if (plugin.getDirectory() == dir) {
                    PluginInfo installed = masterMap.getPlugin(plugin.getName(), dir);
                    if (installed != null) {
                        verDir.add(installed);
                        verDir.addAll(masterMap.getDependentPlugins(installed));
                    }
                }
            }
            verified.addAll(verDir);
        }
        
        // remove plugins from the master map
        
        masterMap.remove(verified);
        
        // if a removed plugin was loaded, attempt to stop it
        
        boolean restart = false;
        for (PluginInfo plugin : verified) {
            try {
                PluginInfo loadedPlugin = getApplication().getPlugin(plugin.getName());
                if (loadedPlugin != null && loadedPlugin.getDirectory() == plugin.getDirectory()) {
                    getApplication().stopPlugin(plugin);
                    restart = true;
                }
            } catch (IllegalArgumentException x) {
                restart = true;
            }
            updatablePlugins.remove(plugin);
        }
        fireStateChanged();
        return restart;
    }

    /**
     * Downloads, installs, and loads latest versions of files for the specified plugins.
     * 
     * @param parent Component to be used as parent by any GUI windows displayed while 
     *               executing this method, if any. If <tt>null</tt>, no error notifications 
     *               will be displayed, and <tt>IllegalArgumentException</tt> will be thrown instead.
     * @param plugins Plugins to update.
     * @return true if restarting the application is required for the changes to take effect.
     * @throws IllegalArgumentException if the update fails, and <tt>parent</tt> was not specified.
     */
    public boolean update(Component parent, Collection<PluginInfo> plugins) {
        HashMap<String,PluginInfo> available = new HashMap<String,PluginInfo>();
        for (PluginInfo plugin : availablePlugins) {
            available.put(plugin.getName(), plugin);
        }
        Map<String,PluginInfo> active = masterMap.getActivePlugins();
        ArrayList<PluginInfo> updates = new ArrayList<PluginInfo>(plugins.size());
        for (PluginInfo plugin : plugins) {
            PluginInfo availablePlugin = available.get(plugin.getName());
            PluginInfo installedPlugin = active.get(plugin.getName());
            
            if (availablePlugin != null && installedPlugin != null && 
                VersionComparator.compareVersion(availablePlugin.getVersion(), installedPlugin.getVersion()) > 0) {
                PluginInfo update = new PluginInfo(availablePlugin);
                update.setDirectory(installedPlugin.getDirectory());
                updates.add(update);
            }
        }
        return install(parent, updates);
    }


// -- Utility methods : --------------------------------------------------------
    
    /** Remove data structures only needed while plugin manager dialog is open. */
    protected void cleanup() {
        installablePlugins = null;
        updatablePlugins = null;
        updateMap = null;
    }

    /**
     * 
     * @param dir
     * @param name
     * @param required
     * @param ranges
     * @throws IllegalStateException if plugin with required name is not found.
     */
    private void addNameWithDependencies(PluginDir dir, String name, Map<String,PluginInfo> required, Set<String> already) {

        if (required.containsKey(name) || already.contains(name)) return;
        
        PluginInfo installed = masterMap.getPlugin(name, dir);
        PluginInfo available = null;
        for (PluginInfo p : availablePlugins) {
            if (p.getName().equals(name)) {
                available = new PluginInfo(p);
                available.setDirectory(dir);
                break;
            }
        }
        
        PluginInfo effective = null;
        if (installed == null) {
            if (available != null) {
                required.put(name, available);
                effective = available;
            }
        } else {
            if (available != null && VersionComparator.compareVersion(available.getVersion(), installed.getVersion()) > 0) {
                required.put(name, available);
                effective = available;
            } else {
                effective = installed;
                already.add(name);
            }
        }
        if (effective == null) throw new IllegalStateException(name);
        
        for (String req : effective.getRequiredPluginNames()) {
            addNameWithDependencies(dir, req, required, already);
        }
    }
    
    
// -- Handling listeners : -----------------------------------------------------

    void addChangeListener(ChangeListener l) {
        listeners.add(ChangeListener.class, l);
    }

    void removeChangeListener(ChangeListener l) {
        listeners.remove(ChangeListener.class, l);
    }

    private void fireStateChanged() {
        ChangeEvent event = new ChangeEvent(this);
        for (ChangeListener listener : listeners.getListeners(ChangeListener.class)) {
            listener.stateChanged(event);
        }
    }

}